(*        Copyright (C) 1996 Tomas Hajny, XHajT03@vse.cz on Internet         *)
(*   (temporary e-mail address from March/April 1996 is ps-axon@login.cz,    *)
(*             my new address will be XHajT03@vol.cz probably).              *)
(*                                                                           *)
(* You can freely use this unit as long as you distribute this source along. *)
(*        If you modify it, you must state so in the source file.            *)
(*                                                                           *)
(*           Many thanks to Ralf Brown for his Interrupt List                *)
(*      (can be found in msdos/info/inter*.zip at SIMTEL mirrors)            *)
(*    - it's the source of informations for most of these routines.          *)
(*                                                                           *)
(*  I've tried to check the type without accessing the drive if possible     *)
(* (mainly due to slower access to some drives - e.g. floppies and CD-ROMs), *)
(* but sometimes the access is needed - that's case of differentiation       *)
(* between hard disks and ramdrives or some special drives (e.g. encrypted   *)
(* or compressed).                                                           *)
(*  If you know how to recognize other drive types or if you find that under *)
(* some conditions the type is determined incorrectly (e.g. some ramdrives   *)
(* could be recognized incorrectly as other type), let me know, please. I'm  *)
(* interested as well if it determines Stacker drives under OS/2 correctly - *)
(* I couldn't check this myself.                                             *)
(*                                                                           *)
(*    Remarks (and problems):                                                *)
(*  I've tried to make the routines as compatible as possible, so I don't    *)
(* use any undocumented DOS calls (well, the DOS function 32h was documented *)
(* only for DOS 4.0+, but should work anywhere). They seem to work well in   *)
(* OS/2 compatibility box (OS/2 Warp tested) and should work under Windows   *)
(* NT as well (not tested). But I'm not sure about DR-DOS and checking for   *)
(* ASSIGNed and SUBSTed drives - I have no access to it; if you do, send me  *)
(* a report, please.                                                         *)
(*  4DOS steals the multiplex interrupt 2Fh each time new shell is invoked   *)
(* and this causes that ASSIGN (and maybe others like InterLink and SUBST as *)
(* well) run in the parent shell cannot be recognized correctly under 4DOS   *)
(* as a secondary shell - that's an incompatibility of 4DOS, sorry.          *)
(*  DriveNo in Is... is always a number from 1 (i.e. A = 1 and so on).       *)
(* Calls to some of Is... have meaning only if other tests were passed (e.g. *)
(* IsRAMDrive is this case).                                                 *)
(*  If a drive B: is created using SUBST on a PC with only one floppy drive, *)
(* the system doesn't recognize it as a SUBSTed drive (at least when using   *)
(* SUBST from MS-DOS 5.0) - that's a bug in DOS, not in my routine.          *)
(*  There are troubles with some magnetooptical drives (resp. their drivers) *)
(* - although the drivers should report them as optical devices, they're     *)
(* claimed to be hard drives in some cases. I don't know a fully reliable    *)
(* way how to distinct them.                                                 *)
(*  I don't recommend to change the routines substantially, because other    *)
(* routines sometimes depend on values in registers (not only on the result  *)
(* returned for Pascal) - so if you want to change something, do it very     *)
(* carefully, please, and state what you changed in the source. As well do   *)
(* not change the meaning of dt... constants.                                *)
(*  There are no warranties (expressed or implied) about this source.        *)

unit DrvTypes;

interface

{$IFNDEF MSDOS}
 Sorry, this works in DOS real mode only. I don't have time to rewrite
 it for protected mode now. If you do so, I'd really like to see it.
{$ENDIF}

const
 dtNone = 0;         (* common constants for IsDrive and DriveType *)
 dtCurrent = 1;
 dtSubsted = 2; {*}
 dtOther = 3;

 dtRemote = 4;       (* constants for IsDrive *)
 dtOneLetter = 5;
 dtThisLetter = 6;
 dtAnother = 7;

 dtAssigned = 4; {*} (* constants for DriveType *)
 dtPhantom = 5; {*}
 dtFixed = 6; {*}
 dtNetwork = 7;
 dtCDRom = 8; {*}
 dtRAMDrive = 9; {*}
 dtInterLink = 10;
 dtJam = 11;
 dtStacker = 12;
 dtStackerAnywhere = 13;
 dtDoubleOrDriveSpace = 14;
 dtSuperStore = 15;
 dtOtherCompressed = 16;
 dtDiskreet = 17; {*}
 dtLapLink = 18;
 dtPCMCIA = 19;
 dtDriveMap = 20;
 dtSecureFileSystem = 21;
 dtDoubleDisk = 22;
 dtError = 23;
 dtRemovable = 24;
 dtPlaceHolder = 25;
 dtFloppy525DD = 26; (* constants for RemovableType *)
 dtFloppy525HD = 27;
 dtFloppy35DD = 28;
 dtFloppy35HD = 29; {*}
 dtFloppy35ED = 30; (* 2.88MB diskette *)
 dtFloppy8SD = 31;
 dtFloppy8DD = 32;
 dtTape = 33;
 dtOptical = 34; {*}
 dtPseudoFixed = 35;

(* types marked with {*} were tested *)

type
 String7 = string [7];

function DriveValid (Drive: char): boolean;
(* Does DOS accept this drive letter? *)

function IsDrive (Drive: char): byte;
(* Basic drive type determination via IOCTL *)

function IsPhantom (DriveNo: byte): boolean;
(* Is this only another letter for some other drive *)
(* (e.g. drive B: on systems with only one floppy)? *)

function IsCDRom (DriveNo: byte): boolean;
(* If MSCDEX or similar found, as addition to normal function returns *)
(* starting CD-ROM drive in CL and number of CD-ROM drives in BL. *)

function CDRomReady (DriveNo: byte): boolean;
(* Is there a CD in the CD-ROM? *)

function ExtIsRAMDrive (DriveNo: byte): boolean;
(* Does the drive report one head only? This is true on a single side *)
(* diskettes as well, i.e. meaningful only on nonremovable drives! *)

function IsRAMDrive (DriveNo: byte): boolean;
(* If only 1 copy of FAT reported then true, otherwise checks the number *)
(* of heads - see the remark above. *)

function IsAssigned (DriveNo: byte): boolean;
(* Was the drive assigned to another drive using the DOS external command *)
(* ASSIGN (see remark above)? *)

function IsFixed (DriveNo: byte): boolean;
(* Is removable or fixed? *)

function IsDiskreet (DriveNo: byte): boolean;
(* Is it an encrypted drive created using DISKREET from Norton Utilities? *)

function IsSecureFileSystem (DriveNo: byte): boolean;
(* Is it an encrypted disk maintained by SFS? *)

function IsEncrypted (DriveNo: byte): byte;     (* may be treated as boolean *)
(* Checks if it is an encrypted drive (and returns type if so, 0 otherwise). *)

function IsJam (DriveNo: byte): boolean;
(* Was the drive created using online compressor JAM? *)

function JamVersion: word;
(* Returns the version of installed JAM driver or 0 if not installed *)

function SuperStoreInstalled: boolean;
(* Is the online compressor SuperStore installed? *)

function IsDoubleOrDriveSpace (DriveNo: byte): boolean;
(* Was the drive created using online compressor from MS-DOS 6.x? *)

function IsStackerAnywhere (DriveNo: byte): boolean;
(* Is it an online compressed removable media (using Stacker Anywhere)? *)

function IsStacker (DriveNo: byte): boolean;
(* Was the drive created using Stacker online compressor? *)

function StackerVersion (DriveNo: byte): word;
(* Returns the Stacker version * 100 (64h) or 0 if neither Stacker *)
(* nor Stacker Anywhere control this drive. *)

function IsPCMCIA (DriveNo: byte): boolean;
(* Is the drive a PCMCIA card? *)

function IsInterLink (DriveNo: byte): boolean;
(* Was the drive created using INTERLNK device driver (part of MS-DOS 6.x)? *)

function IsLapLink (DriveNo: byte): boolean;
(* Was the drive created using LapLink III device driver DD.BIN? *)

function IsDriveMap (DriveNo: byte): boolean;
(* Is it a remote drive connected using DriveMap from PC Tools 8.0? *)

function IsSerialOrParallelConnected (DriveNo: byte): byte;
(* Is it a drive on another computer connected with serial or parallel port? *)
(* Returns the type if so, 0 otherwise (i.e. may be treated as boolean).     *)

function IsDoubleDisk (DriveNo: byte): boolean;
(* Was the drive created using driver DUBLDISK.SYS? *)

function NormDrive (Drive: char): char;
(* uppercase the letter - see remarks bellow *)

procedure SetFirstLetters;
(* The first letters will be used *)
(* for access to drives with multiple letters from now. *)

function RemovableType (DriveNo: byte): byte;
(* checks the type of drive with removable media *)

function IsCompressed (DriveNo: byte): byte;    (* may be treated as boolean *)
(* Checks if it is a compressed drive (and returns type if so, 0 otherwise). *)

function GetDriverSignature (DriveNo: byte): String7;
(* Returns the signature (if any) from the device driver, or returns  *)
(* dtPlaceHolder in the byte 0 (the length) of the string if there is *)
(* no driver for this drive (i.e. the drive doesn't exist).           *)

function DriveType (Drive: char): byte; (* the main, universal type tester *)

implementation

const
 RemovableTypes: array [0..9] of byte =
       (dtFloppy525DD, dtFloppy525HD, dtFloppy35DD, dtFloppy8SD, dtFloppy8DD,
                 dtPseudoFixed, dtTape, dtFloppy35HD, dtOptical, dtFloppy35ED);

 SuperStoreSignature: string [23] = 'Universal Data Exchange';

function IsDrive2: byte; assembler;
(* Cannot be called from Pascal code, expects drive number in BL!!!          *)
(* Well, maybe it should be an external assembler routine but the overhead   *)
(* isn't as big.                                                             *)
asm
 mov ax, 4409h
 int 21h
 mov al, dtNone
 mov bh, 0
 jc @0
 test dh, 10h
 mov al, dtRemote
 jnz @0
 test dh, 80h
 mov al, dtSubsted
 jnz @0
@3:
 mov ax, 440Eh
 int 21h
 jc @2
 mov bh, 0
 test dh, 2
 jz @4         (* direct I/O allowed *)
 mov bh, dtOther
@4:
 or al, al
 jz @1
 cmp al, bl
 mov al, dtThisLetter
 jz @0
 mov al, dtAnother
 jmp @0
@2:
 test dh, 2
 jz @5         (* direct I/O allowed *)
 mov bh, dtOther
@5:
 cmp ax, 1
 mov al, dtNone
 jnz @0
@1:
 mov al, dtOneLetter
@0:
end;

function NormDrive (Drive: char): char; assembler;
(* normalize the drive letter - uppercase, return drive number in BL or set  *)
(* the carry flag if the character is neither a letter nor '@' (which is     *)
(* a shortcut for the current drive)                                         *)
asm
 mov al, Drive
 cmp al, '@'
 jz @0
 and al, not (20h)
 mov bl, al
 sub bl, '@'
 jbe @1
 cmp al, 'Z' + 1
 cbw
 dec ah   (* clear the zero flag, don't change the carry flag *)
 cmc      (* negate the carry flag *)
 jmp @0
@1:
 stc
@0:
end;

procedure SetFirstLetters; assembler;
(* all drives will expect to be called using their first letters from now *)
asm
 xor si, si
 mov cx, 26
@0:
 mov di, cx
 mov cx, 1Ah
@1:
 mov bx, di
 call IsDrive2
 cmp al, dtAnother
 jnz @2
 mov ax, 440Fh
 int 21h
 loop @1
 inc si
@2:
 mov cx, di
 loop @0
end;

function DriveValid (Drive: char): boolean; assembler;
asm
 push word ptr Drive
 call NormDrive
 mov al, false
 jbe @0
 mov ax, 4409h
 int 21h
 mov al, false
 jc @0
 inc al
@0:
end;

function IsDrive (Drive: char): byte; assembler;
asm
 push word ptr Drive
 call NormDrive
 mov al, dtCurrent
 jz @0
 dec al  (* doesn't change the carry flag! *)
 jc @0
 call IsDrive2
@0:
end;

function IsPhantom (DriveNo: byte): boolean; assembler;
asm
 call SetFirstLetters
 mov bl, DriveNo
 call IsDrive2
 cmp al, dtAnother
 mov al, false
 jnz @0
 mov al, dtPhantom
@0:
end;

function IsCDRom (DriveNo: byte): boolean; assembler;
asm
 xor bx, bx
 mov ax, 1500h
 int 2Fh
 mov ax, bx
 or ax, ax
 jz @0         (* MSCDEX or similar not found *)
 mov al, false
 mov dl, DriveNo
 inc cl        (* 0 = A:, 1 = B:, ... for MSCDEX *)
 sub dl, cl
 jc @0
 cmp dl, bl
 jnc @0
 mov al, dtCDRom
@0:
end;

function CDRomReady (DriveNo: byte): boolean; assembler;
asm
 mov ax, 150Bh
 mov cl, DriveNo
 dec cl        (* 0 = A:, 1 = B:, ... for MSCDEX *)
 int 2Fh
 mov ax, 150Bh
 int 2Fh       (* must be called twice - has a bit slower reaction :-) *)
end;

function ExtIsRAMDrive (DriveNo: byte): boolean;
var
 P: pointer;
 OK: boolean;
 Size: word;
begin
 OK := true;
 ExtIsRAMDrive := false;
 asm
  push bp
  push ds
  mov dl, DriveNo
  mov ah, 1Ch     (* read the sector size to determine the memory for buffer *)
  int 21h
  jnc @0
  mov OK, false
@0:
  mov Size, cx
  pop ds
  pop bp
 end;
 if OK then
 begin
  if MemAvail < Size then
  begin
   asm
    mov @Result, dtError  (* cannot assign the value directly because *)
   end;                   (* Pascal treats the result as boolean      *)
   Exit;
  end;
  GetMem (P, Size);
  asm
   push ds
   lds bx, P
   mov al, DriveNo
   dec al           (* expects A = 1, B = 2, ... *)
   mov cx, 1        (* number of sectors *)
   mov dx, 0        (* sector 0 = boot sector *)
   int 25h          (* read logical sectors *)
   pop ax           (* remove a word from the stack (stays after INT 25h) *)
   jc @0            (* cannot read the boot sector => not ramdrive *)
   mov al, [bx + 1Ah] (* number of heads is at this offset *)
   cmp al, 1
   ja @0
   mov @Result, dtRAMDrive
@0:
   pop ds
  end;
  FreeMem (P, Size);
 end;
end;

function IsRAMDrive (DriveNo: byte): boolean; assembler;
asm
 push ds
 mov ah, 32h
 mov dl, DriveNo
 int 21h
 mov al, false
 jc @0
 cmp byte ptr [bx + 8], 1
 mov al, dtRamDrive
 jz @0
 pop ds
 push ds
 push dx
 call ExtIsRamDrive
@0:
 pop ds
end;

function IsAssigned (DriveNo: byte): boolean; assembler;
asm
 mov ax, 600h
 int 2Fh
 mov ah, al
 mov al, false
 cmp ah, 0FFh
 jnz @0         (* ASSIGN not installed (or 4DOS stole the 2F vector :-( ) *)
 mov ax, 601h
 int 2Fh
 mov al, DriveNo
 cbw
 mov di, 102h
 add di, ax
 mov ah, es: [di]
 xchg ah, al
 cmp al, ah
 jnz @0
 mov al, false
@0:
end;

function IsInterLink (DriveNo: byte): boolean; assembler;
asm
 mov ax, 5601h
 mov dx, 0FFFFh
 mov bh, DriveNo
 dec bh
 mov bl, 0
 int 2Fh
 inc al
 mov al, false
 jnz @0             (* INTERLNK not installed *)
 or bl, bl
 jz @0
 mov al, dtInterLink
@0:
end;

function IsFixed (DriveNo: byte): boolean; assembler;
asm
 mov bl, DriveNo
 mov ax, 4408h
 int 21h        (* returns 0 or 1 in AL; seems to be prepared for Pascal :-) *)
end;

function JamVersion: word; assembler;
asm
 mov ax, 5200h
 int 2Fh
 cmp ah, 80h
 mov ax, bx
 jz @0          (* JAM driver installed *)
 xor ax, ax
@0:
end;

function IsJam (DriveNo: byte): boolean;
var
 P: pointer;
 Size: word;
 OK: boolean;
begin
 asm
  mov OK, true
  mov @Result, false
  call JamVersion
  mov Size, cx
  or ax, ax
  jnz @0        (* JAM driver found *)
  mov OK, false
@0:
 end;
 if OK then
 begin
  if MemAvail < Size then
  begin
   asm
    mov @Result, dtError  (* cannot assign the value directly because *)
   end;                   (* Pascal treats the result as boolean      *)
   Exit;
  end;
  GetMem (P, Size);
  asm
   push ds
   mov dl, DriveNo
   mov ax, 5201h
   int 2Fh
   pop ds
   or ah, ah
   jnz @1
   mov @Result, dtJam
@1:
  end;
  FreeMem (P, Size);
 end;
end;

function GetDriverSignature (DriveNo: byte): String7; assembler;
(* This function never returns anything under OS/2, but it should be OK   *)
(* - external DOS device drivers for block devices aren't allowed there.  *)
(* Only PCMCIA under OS/2 would be interesting - isn't distinguished now. *)
asm
 les di, @Result
 mov byte ptr es: [di], 0
 push ds
 mov dl, DriveNo
 mov ah, 32h
 int 21h
 jnc @4
 mov al, dtPlaceHolder
 jmp @1
@4:
 or al, al
 jnz @0
 mov si, [bx + 13h]
 add si, 0Bh
 jc @0        (* breaks segment boundary (e.g. offset = 0FFFFh) => error *)
 mov cx, [bx + 15h]
 mov ds, cx
 or cx, cx
 jz @0        (* segment is 0000 => error *)
 mov bx, 1
@3:
 lodsb
 cmp al, ' '
 jz @2
 or al, al
 jz @2
 mov es: [di + bx], al
 inc bx
 cmp bx, 7
 jbe @3
@2:
 mov al, bl
 dec al
@1:
 stosb
@0:
 pop ds
end;

function IsLapLink (DriveNo: byte): boolean;
var
 S: string;
begin
 S := GetDriverSignature (DriveNo);
 if byte (S [0]) = dtPlaceHolder then
 asm
  mov @Result, dtPlaceHolder
 end else if S = 'GFS' then
 asm
  mov @Result, dtLapLink
 end else
 asm
  mov @Result, false
 end;
end;

function IsDiskreet (DriveNo: byte): boolean;
var
 OK: boolean;
 S: string;
begin
 OK := true;
 IsDiskreet := false;
 asm
  mov ax, 0FE00h
  mov di, 4E55h
  mov si, 4443h
  int 2Fh
  cmp si, 6463h
  jz @0
  mov OK, false
@0:
 end;
 if OK then
 begin
  S := GetDriverSignature (DriveNo);
  if byte (S [0]) = dtPlaceHolder then
  asm
   mov @Result, dtPlaceHolder
  end else if S = 'DSKREET' then
  asm
   mov @Result, dtDiskreet
  end else
  asm
   mov @Result, false
  end;
 end;
end;

function SuperStoreInstalled: boolean; assembler;
asm
 mov ax, 0F800h
 mov cx, 4455h
 mov dl, 45h
 int 2Fh
 inc al
 mov al, false
 jnz @0
 push ds
 lea si, SuperStoreSignature
 lodsb
 cbw
 mov cx, ax
@2:
 lodsb
 cmp al, es: [bx]
 jnz @1
 loop @2
 pop ds
 mov al, true
 jmp @0
@1:
 mov al, false
 pop ds
@0:
end;

function IsDoubleOrDriveSpace (DriveNo: byte): boolean;
var
 S: string;
begin
 S := GetDriverSignature (DriveNo);
 if byte (S [0]) = dtPlaceHolder then
 asm
  mov @Result, dtPlaceHolder
 end else if S = 'DBLSPAC' then
 asm
  mov @Result, dtDoubleOrDriveSpace
 end else
 asm
  mov @Result, false
 end;
end;

function StackerVersion (DriveNo: byte): word; assembler;
var
 P: pointer;
asm
 push ds
 mov bl, DriveNo
 mov ax, 4404h
 mov cx, 4
 push ss
 pop ds
 lea dx, P
 mov si, dx
 xor di, di
 mov word ptr [si], di
 mov word ptr [si + 2], di
 int 2Fh
 mov ax, 0
 jc @0
 lodsw
 mov di, ax
 lodsw
 mov ds, ax
 or ax, di
 jz @0
 mov bh, dtStacker
 cmp byte ptr [si], 5Ah
 jz @1
 inc bh
 inc si
@1:
 lodsw
 cmp ax, 0A55Ah
 mov ax, 0
 jz @0
 lodsw
@0:
 pop ds
end;

function IsStackerAnywhere (DriveNo: byte): boolean; assembler;
asm
 push word ptr DriveNo
 call StackerVersion
 or ax, ax
 jz @0
 mov al, bh
 cmp al, dtStackerAnywhere
 jz @0
 mov al, false
@0:
end;

function IsStacker (DriveNo: byte): boolean; assembler;
asm
 push word ptr DriveNo
 call StackerVersion
 or ax, ax
 jz @0
 mov al, bh
 cmp al, dtStacker
 jz @0
 mov al, false
@0:
end;

function IsPCMCIA (DriveNo: byte): boolean;
var
 S: string;
begin
 S := GetDriverSignature (DriveNo);
 if byte (S [0]) = dtPlaceHolder then
 asm
  mov @Result, dtPlaceHolder
 end else if (S = '$PCMATA') or
                          (Copy (S, 1, 4) = 'SIDE') and (byte (S [0]) = 7) then
 asm
  mov @Result, dtPCMCIA
 end else
 asm
  mov @Result, false
 end;
end;

function IsDriveMap (DriveNo: byte): boolean; assembler;
asm
 mov ax, 9204h
 int 2Fh
 cmp ax, 9200h
 mov al, false
 jnz @0         (* DriveMap not installed *)
 inc al         (* AX := 9201h ... check drive *)
 mov dl, DriveNo
 int 2Fh
 cmp al, 92h
 mov al, false
 jz @0          (* not mapped drive *)
 mov al, dtDriveMap
@0:
end;

function IsSecureFileSystem (DriveNo: byte): boolean; assembler;
var
 Buffer: array [1..26] of byte;
asm
 push ds
 push ss
 pop ds
 lea dx, Buffer
 mov ax, 4404h
 mov cx, 1Ah
 mov bl, DriveNo
 int 2Fh
 mov al, false
 jc @0
 mov si, dx
 cmp word ptr [si], 'FS'
 jnz @0
 cmp word ptr [si + 2], '1S'
 jnz @0
 mov al, dtSecureFileSystem
@0:
 pop ds
end;

function IsDoubleDisk (DriveNo: byte): boolean; assembler;
var
 Buffer: array [1..10] of byte;
asm
 push ds
 push ss
 pop ds
 lea dx, Buffer
 mov si, dx
 mov cx, 0Ah
 mov bl, DriveNo
 mov word ptr [si], 4444h
 mov byte ptr [si + 2], 1
 int 2Fh
 mov bx, ax
 mov al, false
 jc @0
 cmp bx, cx
 jnz @0
 cmp word ptr [si], 4444h
 jnz @0
 cmp word ptr [si + 2], 4444h
 jnz @0
 mov al, dtDoubleDisk
@0:
 pop ds
end;

function IsCompressed (DriveNo: byte): byte; assembler;
asm
 mov bh, DriveNo
 push bx
 call IsDoubleDisk
 or al, al
 jnz @0
 push bx
 push bx
 call StackerVersion
 pop di
 or ax, ax
 mov al, bh
 jnz @0
 push di
 push di
 call IsJam
 pop di
 or al, al
 jnz @0
 push di
 push di
 call IsDoubleOrDriveSpace
 pop di
 or al, al
 jnz @0
 mov ax, 4A11h
 xor bx, bx
 push di
 int 2Fh
 pop di
 or ax, ax
 mov al, false
 jnz @0
 cmp bx, 444Dh
 jnz @0
 mov bx, di
 sub cl, 40h
 jb @0
 cmp bh, cl
 jc @0
 add cl, ch
 cmp bh, cl
 jnc @0
 call SuperStoreInstalled
 or al, al
 mov al, dtSuperStore        (* assume only one driver *)
 jnz @0                      (* for on-line compression installed *)
 mov al, dtOtherCompressed
@0:
end;

function IsEncrypted (DriveNo: byte): byte; assembler;
asm
 mov bh, DriveNo
 push bx
 push bx
 call IsDiskreet
 pop bx
 or al, al
 jnz @0
 push bx
 call IsSecureFileSystem
@0:
end;

function IsSerialOrParallelConnected (DriveNo: byte): byte; assembler;
asm
 mov bh, DriveNo
 push bx
 push bx
 call IsInterLink
 pop bx
 or al, al
 jnz @0
 push bx
 push bx
 call IsLapLink
 pop bx
 or al, al
 jnz @0
 push bx
 call IsDriveMap
@0:
end;

function RemovableType (DriveNo: byte): byte; assembler;
var
 Buffer: array [1..38] of byte;
asm
 mov dl, DriveNo
 dec dl         (* for BIOS it's zero based *)
 mov ah, 8
 int 13h
 cmp dl, DriveNo
 jb @1
 mov al, bl
 add al, dtFloppy525DD - 1
 cmp al, dtFloppy35ED
 jbe @0
 dec al
 cmp al, dtFloppy35ED
 jz @0
@1:
 push ds
 mov ax, 440Dh
 mov bl, DriveNo
 mov cx, 0860h
 mov si, ss
 mov ds, si
 lea di, Buffer
 mov dx, di
 mov byte ptr [di], 0Dh
 int 21h
 mov bl, [di + 1]
 pop ds
 mov al, dtRemovable
 jc @0
 mov al, dtOther
 cmp bl, 0Ah
 jnc @0
 mov al, bl
 mov bx, OFFSET RemovableTypes
 xlat
@0:
end;

function DriveType (Drive: char): byte; assembler;
asm
 push word ptr Drive
 call IsDrive
 cmp al, dtRemote
 jb @0              (* dtNone, dtCurrent or dtSubsted *)
 push ax
 push bx
 call IsAssigned
 or al, al
 mov al, dtAssigned
 jnz @0
 mov al, dtOther
 or bh, bh
 jnz @0
 pop ax
 cmp al, dtRemote
 jz @1
 cmp al, dtOneLetter
 jz @4
 push bx
 call IsPhantom
 or al, al
 jnz @0             (* dtPhantom in AL already *)
@4:
 push bx
 call IsFixed
 jc @3              (* some device drivers return this (maybe a ramdrive) *)
 or al, al
 jz @2
 push bx
 push bx
 call IsCompressed
 pop bx
 or al, al
 jnz @0
 push bx
 push bx
 call IsEncrypted
 pop bx
 or al, al
 jnz @0
 push bx
 push bx
 call IsPCMCIA
 pop bx
 or al, al
 jnz @0
 push bx
 call ExtIsRamDrive
 or al, al
 jnz @0             (* dtRAMDrive or dtError in AL already *)
 mov al, dtFixed
 jmp @0
@3:
 push bx
 push bx
 call IsRamDrive
 pop bx
 or al, al
 jnz @0
@2:
 push bx
 push bx
 call IsStackerAnywhere
 pop bx
 or al, al
 jnz @0
 push bx
 call RemovableType
 jmp @0
@1:
 push bx
 push bx
 call IsSerialOrParallelConnected
 pop di
 or al, al
 jnz @0             (* correct type in AL already *)
 push di
 call IsCDRom
 or al, al
 jnz @0             (* dtCDRom in AL already *)
 push di
 call IsRamDrive
 or al, al
 jnz @0             (* dtRamDrive in AL already *)
 mov al, dtNetwork
@0:
end;

end.
