"""
Simple message board system
"""
import os
import sys
import time

class Door:
    """
    A simple base class for creating BBS doors.  It works (hopefully) with telnet connections,
    as well as real live COM ports.  
    """
    def __init__(self, Session):
        self.Session = Session
        if self.Session:
            self.Socket = self.Session.request
            self.GetKey = self.GetKeySocket
            self.GetLine = self.GetLineSocket
            self.Send = self.SendSocket
        else:
            self.GetKey = self.GetKeyLocal
            self.GetLine = self.GetLineLocal
            self.Send = self.SendLocal
        self.KeepRunning = 1
        self.ClientText = ""
    def GetKeyLocal(self):
        Key = sys.stdin.read(1)
        return Key
    def GetKeySocket(self, Timeout = 60):
        Start = time.clock()
        while 1:
            try:
                Text = self.Socket.recv(1)
                self.Socket.send(Text) # echo
                return Text
            except:
                time.sleep(0.1) # Don't crush the CPU
                Elapsed = time.clock() - Start
                if Elapsed > Timeout:
                    return None
    def Run(self):
        "Main method - override this with your door's code"
        pass
    def Flush(self):
        self.ClientText = ""
    def GetLineLocal(self):
        return sys.stdin.readline()
    def GetLineSocket(self, Timeout = 60):
        Start = time.clock()
        while 1:
            try:
                Text = self.Socket.recv(1024)
                self.ClientText += Text
                self.Socket.send(Text) # echo
            except:
                time.sleep(0.1) # Don't crush the CPU
                Elapsed = time.clock() - Start
                if Elapsed > Timeout:
                    return None
            Pos = self.ClientText.find("\r\n")
            if Pos!=-1:
                Line = self.ClientText[:Pos]
                self.ClientText = self.ClientText[Pos+2:]
                return Line
            Pos = self.ClientText.find("\n")
            if Pos!=-1:
                Line = self.ClientText[:Pos]
                self.ClientText = self.ClientText[Pos+1:]
                return Line
    def SendSocket(self, Text):
        if not Text:
            return
        self.Socket.send(Text)
    def SendLocal(self, Text):
        print Text, 
Escape = chr(27)

CRLF = "\r\n"

class ANSI:
    # Foreground:
    Red = Escape + "[31m"
    Green = Escape + "[32m"
    Yellow = Escape + "[33m"
    Blue = Escape + "[34m"
    Purple = Escape + "[35m"
    Cyan = Escape + "[36m"
    White = Escape + "[37m"
    Default = Escape + "[37m"
    # Background:
    BlackBack = Escape + "[40m"
    RedBack = Escape + "[41m"
    GreenBack = Escape + "[42m"
    YellowBack = Escape + "[43m"
    BlueBack = Escape + "[44m"
    PurpleBack = Escape + "[45m"
    CyanBack = Escape + "[46m"
    WhiteBack = Escape + "[47m"
    DefaultBack = Escape + "[40m"
    # Special:
    Reset = Escape + "[0m"
    BoldOn = Escape + "[1m"
    BoldOff = Escape + "[22m"
##NormalColors = ["Red","Blue","Cyan","Magenta","Yellow","Green","White"]    

class MessageForum:
    def __init__(self, ID, Name, Path):
        self.Name = Name
        self.ID = ID
        self.Path = Path
        self.MessageCount = 0
        self.Messages = []
        
class MessageServerClass:
    """
    This class serves up messages to all the user threads that are reading or writing messages
    """
    def __init__(self, MessageDir = "."):
        self.MessageDir = MessageDir
        self.NextForumID = 1
        self.IDToForum = {}
        self.GetFora()
    def GetFora(self):
        "Find all the message boards (as configured in Boards.txt)"
        PrintedReassignMessage = 0
        BoardList = os.path.join(self.MessageDir, "Boards.txt")
        if os.path.exists(BoardList):
            File = open(BoardList, "r")
            LineNumber = 0
            for FileLine in File.xreadlines():
                LineNumber += 1
                FileLine = FileLine.strip()
                # Skip blank lines, comment lines:
                if (not FileLine) or FileLine[0]=="#":
                    continue
                # Expect to see: ID\tName\tPath\n
                Bits = FileLine.split("\t")
                if len(Bits) < 3:
                    print "Skipping bad record on line %d (no tabs)"%LineNumber
                    continue
                # Get the ID:
                try:
                    ID = int(Bits[0])
                except:
                    print "Skipping bad record on line %d: Bad ID %s"%(LineNumber, Bits[0])
                    continue
                # Add the beast:
                self.AddForum(ID, Bits[1], Bits[2])
        # Make a default board, if we have no boards:
        if not self.IDToForum.keys():
            Path = os.path.join(self.MessageDir, "General.txt")
            self.AddForum(1, "General Chat", Path)
            return
    def AddForum(self, ID, Name, FilePath):
        "Add a new message board."
        # Build the file, if it's not already there:
        if not os.path.exists(FilePath):
            File = open(FilePath,"wb")
            File.close()
        while self.IDToForum.has_key(ID):
            ID += 1
        Forum = MessageForum(ID, Name, FilePath)
        self.NextForumID += 1
        self.IDToForum[Forum.ID] = Forum
    def GetFirstBoard(self):
        IDs = self.IDToForum.keys()
        IDs.sort()
        return self.IDToForum[IDs[0]]
    def GetBoard(self, ID):
        return self.IDToForum.get(ID, None)
    
MessageServer = MessageServerClass()
    
class ColorHolder:
    pass
    
class MessageBoardDoor(Door):
    DefaultTranslator = {"@@":"@",
                  "@E":ANSI.BoldOn + ANSI.Red, # Error
                  "@P":ANSI.BoldOff + ANSI.Yellow, # prompt
                  "@W":ANSI.BoldOff + ANSI.White, # parens
                  "@X":ANSI.BoldOn + ANSI.Green, # Hilite key
                  "@Z":ANSI.BoldOn + ANSI.Blue, # User input
                  "@T":ANSI.BoldOn + ANSI.Red + ANSI.WhiteBack, # Title
                  }
    NoANSITranslator ={"@@":"@",
                       }
    def __init__(self, Session):
        Door.__init__(self, Session)
        self.InitColors()
    def InitColors(self):
        self.Colors = ColorHolder()
        self.Colors.Title = ANSI.BoldOn + ANSI.Red + ANSI.WhiteBack
        self.Colors.Prompt = ANSI.BoldOn + ANSI.Green
        self.Colors.Entry = ANSI.BoldOff + ANSI.White
        self.Translator = self.NoANSITranslator
    def Run(self):
        self.Send(self.Colors.Title + "Message board subsystem [alpha version - no worky yet]" + ANSI.BlackBack + CRLF)
        #self.CurrentBoard = MessageServer.GetFirstBoard()
        self.DoMainMenu()
    def FormatStr(self, Str):
        AtPos = Str.find("@")
        while AtPos!=-1:
            if AtPos == len(Str)-1:
                break
            Code = Str[AtPos:AtPos+2]
            Str = Str[:AtPos] + self.Translator.get(Code, "") + Str[AtPos+2:]
            AtPos = Str.find("@", AtPos + 1)
        return Str
    def DoMainMenu(self):
        PromptStr = "@PMain lobby.\r\nEnter board number, @W(@XL@W)@P to list, @W(@XX@W)@P to exit, @W(@X?@W)@P for help: @Z"
        PromptStr = self.FormatStr(PromptStr)
        while (1):
            self.Send(PromptStr+CRLF)
            Line = self.GetLine().strip()
            if not Line:
                continue
            if Line[0].upper()=="X": # Exit
                Str = self.FormatStr("@TReturning you to the BBS...")
                self.Send(Str+CRLF)
                return
            if Line[0]=="?":
                Str = self.FormatStr("@ESorry - no help available yet")
                self.Send(Str+CRLF)
                continue
            try:
                BoardNumber = int(Line)
            except:
                continue
            Board = MessageServer.GetBoard(BoardNumber)
            if not Board:
                ErrorStr = self.FormatStr("@EUnknown board - type @W(@XL@W)@E for a list.")
                self.Send(ErrorStr+CRLF)
            self.DoMessageBoard(Board)
    def DoMessageBoard(self, Board):
        pass
            
def DumpleBoard(Session):
    Door = MessageBoardDoor(Session)
    Door.Run()

def LocalTest():
    Door = MessageBoardDoor(None)
    Door.Run()

if __name__=="__main__":
    LocalTest()

