#include <QMessageBox>
#include <QCloseEvent>
#include <QJsonDocument>
#include <QJsonObject>
#include <iostream>
#include "logitem.h"
#include "portdialog.h"
#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "logitemdelegate.h"

#ifdef Q_OS_MAC
#include <QtMac>
#include "CocoaInitializer.h"
#endif

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{

#ifdef Q_OS_MAC
    initializer = new CocoaInitializer();
    initializer->setMainWindow(this);
#endif

    ui->setupUi(this);
    ui->logListWidget->setItemDelegate(new logitemdelegate);
    unseenNotifications = 0;
    listening = false;
    udpsocket = new QUdpSocket(this);
    connect(udpsocket, SIGNAL(readyRead()), this, SLOT(processReadyRead()));
    pdialog = new PortDialog(this);
#ifdef Q_OS_MACOS
    setWindowIcon(QIcon(":/icons/cauldron.icns"));
    connect(this, SIGNAL(setupDockEventMonitor()), SLOT(onSetupDockEventMonitor()), Qt::QueuedConnection);
    emit setupDockEventMonitor();
#else
    setWindowIcon(QIcon(":/icons/cauldron.png"));
    sysTrayIcon = new QSystemTrayIcon();
    sysTrayIcon->setIcon(QIcon(":/icons/cauldron.png"));
    connect(sysTrayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)), this, SLOT(iconActivated(QSystemTrayIcon::ActivationReason)));
    sysTrayIcon->show();
#endif
    settings = new QSettings("MagickaBBS", "MagickaNotify");
    port = settings->value("port", 2027).toInt();

    this->resize(settings->value("size", this->size()).toSize());
    this->move(settings->value("pos", this->pos()).toPoint());
}

MainWindow::~MainWindow()
{
    delete ui;
}

#ifdef Q_OS_MAC
void MainWindow::onSetupDockEventMonitor() {
    initializer->onSetupDockEventMonitor();
}
#endif

void MainWindow::addItem(QString item) {
    QListWidgetItem *witem;
    if (ui->logListWidget->count() > 100 ) {
        witem = ui->logListWidget->takeItem(0);
        delete witem;
    }
    ui->logListWidget->addItem(item);
    ui->logListWidget->scrollToBottom();
}

void MainWindow::on_actionListen_toggled(bool arg1)
{
    QString item;
    if (arg1) {
        item = "Starting to listen on port " + QString::number(port);
        udpsocket->bind(port, QUdpSocket::ShareAddress);
        listening = true;
    } else {
        item = "Stopping listening on port " + QString::number(port);
        udpsocket->close();
        listening = false;
    }
    addItem(item);
}

void MainWindow::setPort(quint16 p) {
    port = p;
    QString item;
    if (listening) {
        udpsocket->close();
        udpsocket->bind(port, QUdpSocket::ShareAddress);
        item = "Restarting on port " + QString::number(port);
        addItem(item);
    }
    settings->setValue("port", port);
}

void MainWindow::processReadyRead()
{
    while (udpsocket->hasPendingDatagrams()) {
        QByteArray datagram;
        QJsonDocument doc;
        datagram.resize(udpsocket->pendingDatagramSize());
        udpsocket->readDatagram(datagram.data(), datagram.size());

        doc = QJsonDocument::fromJson(datagram.data());

        if (doc.isObject()) {
            QString message = doc.object().value("Message").toString();
            QString system = doc.object().value("System").toString();
            QString program = doc.object().value("Program").toString();

            QListWidgetItem *item = new QListWidgetItem;

            item->setData(0, QVariant::fromValue(LogItem(system, program, message)));
            QListWidgetItem *witem;
            if (ui->logListWidget->count() > 100 ) {
               witem = ui->logListWidget->takeItem(0);
               delete witem;
            }
            ui->logListWidget->addItem(item);
            ui->logListWidget->scrollToBottom();
            if (this->isHidden()) {
                setUnseenNotifications(unseenNotifications + 1);
            }
        }
    }
}

void MainWindow::on_actionChoose_Port_triggered()
{
    pdialog->setPort(port);
    pdialog->show();
}

void MainWindow::iconActivated(QSystemTrayIcon::ActivationReason reason)
{
    switch (reason) {
    case QSystemTrayIcon::Trigger:
    case QSystemTrayIcon::DoubleClick:
        setUnseenNotifications(0);
        this->show();
        break;
    default:
        ;
    }
}

void MainWindow::closeEvent(QCloseEvent *event)
{
#ifdef Q_OS_MAC
    this->hide();
    event->ignore();
#else
    if (sysTrayIcon->isVisible()) {
        this->hide();
        event->ignore();
    }
#endif
}

void MainWindow::on_actionQuit_triggered()
{
    if (listening) {
        udpsocket->close();
    }

    settings->setValue("size", this->size());
    settings->setValue("pos", this->pos());
    QApplication::quit();
}

void MainWindow::setUnseenNotifications(int i) {
    unseenNotifications = i;
#ifdef Q_OS_MAC
    if (i > 0) {
        QtMac::setBadgeLabelText(QString::number(i));
    } else {
        QtMac::setBadgeLabelText("");
    }
#endif
}
