// Filename:   sysexec
// Contents:   the object that encapsulates unusual system operations
// Author: Greg Shaw
// Created:    5/18/94

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef _SYSEXEC_C_
#define _SYSEXEC_C_

#include "bbshdr.h"

static int child_exited = 0;    // *(%(@%&#*(@* C/C++ and signals!
static int running_pid = 0;     // more *%(#*@(isms.
static int sec_violation = 0;	// security violation?
static int exit_status = 0;	// exit status of child

// Function:   catch_child
// Function:   catch the child dying signal so that the BBS knows to "wake up"
// Input:  sig - the signal that was caught
// Output: will reset object if pid correct.
// Author: Greg Shaw
// Notes:  this is a 'plain c' function, as that is what signal() wants.
//     GRRRR!  Wish I could remember that typecast that goes from
//     C++ to C!
// Created:    10/16/94

void catch_child(int sig)       // catch a child dying
{
	int        status;

	if (sig == SIGCHLD)
	{
		if (wait(&status) == running_pid)
		{
			child_exited = 0;
		}
#ifdef SECURE	// be anal about security?
		else
		{
			sec_violation++;
		}
#endif
		if (WIFEXITED(status))	// exited normally?
			exit_status = WEXITSTATUS(status);	// save status
		else
			exit_status = -1;	// non-normal exit

	}
	if (sig == SIGHUP)
	{
		child_exited = -1;
	}
}


// Method: constructor
// Function:   initialize the object
// Author: Greg Shaw
// Created:    5/20/94

sysexec::sysexec()
{
	running_pid = 0;            // nothing running
	child_exited = 1;           // no child
}


// Method: destructor
// Function:   clean up the object
// Author: Greg Shaw
// Created:    5/20/94

sysexec::~sysexec()
{
	if (running_pid > 0)        // child running?
	{
		kill(running_pid,SIGHUP);
	}
	child_exited = 0;
}

// Method: sysint
// Function:   serve as a system() command that correctly handles SIGHUP
// Input:  command - the command to execute
// Output: (command is executed)
// Author: Greg Shaw
// Notes:  I have to use printf() in this function to talk to the user
//     because bbsint (the next highest object, and the one which
//     handles i/o) requires sysint to display a file.
// Created:    5/18/94

int sysexec::sysint(char *command, time_t logoff_time, int quiet)
{
	sigset_t   mask;            // old signal mask
	time_t     now;             // current time
	int        warning;         // which warning?
	char   tmpstr[255];

	if (command == NULL)        // null command?
		return(1);              // error

	if (log_externals())        // log externals?
	{
		sprintf(tmpstr,"called %s",command);
		ap_log(tmpstr);
	}
	warning = 0;                // no warnings yet
	signal(SIGCHLD, &catch_child);
// Changed 3/24/96 - sigint in an external, if hit twice causes the BBS to
// abort without saving the user's changes...
//	signal(SIGINT, &catch_child);	// ignore?
	child_exited = 1;
	// get current blocked sigs mask
	sigprocmask(SIG_BLOCK, NULL, &mask);
	sigaddset(&mask, SIGQUIT);
	// set new mask
	sigprocmask(SIG_SETMASK, &mask, NULL);
	if (!(running_pid = fork()))
	{


		setgid(getgid());
		setuid(getuid());

		#ifdef HAVE_RESTRICTED_BASH
		execl("/bin/sh", "sh", "-rc", command, (char *) 0);
		#else
		execl("/bin/sh", "sh", "-c", command, (char *) 0);
		#endif
		ap_log("system: exec failure");
		exit(127);
	}
	if (running_pid == -1)
	{
		ap_log("system: exec failure");
		exit_status = -1;
	}
	else
	{
		while (child_exited != 0)
		{
			sleep(10);          // just sleep for now
			time(&now);         // get date
			// less than 3 minutes?
			if (logoff_time != 0 && logoff_time - now <= 180)
			{
				// first warning
				if (warning == 0 && !quiet)
				{
					printf(EXPIRE3MINUTES);
					printf(PREPARETOLOGOFF);
					warning++;
				}
				else if (warning == 1 && logoff_time - now <= 60 && !quiet)
				{
					printf(YOUWILLBEOFF);
					printf(PLEASELOGOFF);
					warning++;
				}
				else if (warning > 1 && logoff_time - now < 0)
				{
					// nuke him
					kill(running_pid,SIGHUP);
					ap_log("forced user logoff - time expired");
					return(NUKEHIM);
				}
			}
			// he hung up
			if (child_exited == -1)
			{
				kill(running_pid,SIGHUP);
				ap_log("User hangup in external");
				return(NUKEHIM);
			}
		}
		if (sec_violation)
		{	// log potential security violation
			sprintf(tmpstr,"Caught %d %s dying.  Potential security violation!",sec_violation,sec_violation>1?"children":"child");
			ap_log(tmpstr);
			sec_violation = 0;
		}
	}
	running_pid = 0;
	// return to ignoring signals
	signal(SIGINT, SIG_IGN);	// ignore?
	return(exit_status);

}

#endif                          // _SYSEXEC_C_
