/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* chat.c - Chat and yell system */

/* Todo: OS/2 and DOS version yell, chat logging, split screen chat BUGS */

#include <stdio.h>
#include <string.h>

#ifdef __linux__
#  include <sys/time.h>
#  include <unistd.h>
#endif

#include "os.h"
#include "memory.h"
#include "output.h"
#include "userbase.h"
#include "ask_str.h"
#include "lastcall.h"
#include "timeslic.h"
#include "logfile.h"
#include "showfile.h"
#include "files.h"
#include "keyb.h"
#include "nodes.h"
#include "modem.h"
#include "config.h"
#include "language.h"
#include "chat.h"

static chatting = 0;

void yell_sysop(char *data)
{
    char reason[61];
#ifdef __linux__
    char str[256];
    int num;
    int F;
#endif
    int sound, wait;

    if (data == NULL)
        reason[0] = '\0';
    else
    {
        reason[sizeof(reason-1)] = '\0';
        strncpy(reason, data, sizeof(reason)-1);
    }
    if (!ask_string(lang[LANG_YELL_SYSOP], reason, sizeof(reason)-1, 0, NULL)) return;
    if (reason[0] == '\0') return;

    write_log("Yelling sysop with reason: %s",reason);
    current_lastrec.done_flags |= DONE_YELL;
    loc_output("\r\nPress enter to start chat.\r\n");
    output("\r\n@X07Press ctrl-c to abort yell..");

    wait = 0;
    while (wait < 30)
    {
        sleep(1);
        if (mdm_kbhit() && mdm_getch() == 3) return;
        if (kbhit_nowait())
        {
            switch (getch())
            {
                case 3:
                    return;
                case 13:
                    line_chat();
                    return;
            }
        }
        outchr('.');
        fprintf(Fout, "\x07"); /* beep! */
        wait++;
        if (wait == 2)
        {
#ifdef __linux__
            /* Send yell-message to /dev/tty1..6 */
            for (num=1; num<=6; num++)
            {
                sprintf(str,"/dev/tty%d", num);
                F = FileOpen(str, O_WRONLY, SH_DENYNO);
                if (F != -1)
                {
                    sprintf(str, "\r\n%s yelling sysop: '%s'\r\n", usrsub.name, reason);
                    FileWrite(F, str, strlen(str));
                    FileClose(F);
                }
            }
#endif
        }
        if (wait == 6)
        {
            sound = 0;
            if (in_group("yell"))
            {
                output("\r\nWhoa, you were in yell group, starting mp3 yell..");
                sound = 1;
            }
            if (sound == 0 && stristr(reason, "sky") != NULL) sound = 1;
            if (sound == 0 && stristr(reason, "bug") != NULL) sound = 1;
            if (sound == 0 && stristr(reason, "vika") != NULL) sound = 1;
            if (sound == 0 && stristr(reason, "ei toimi") != NULL) sound = 1;
            if (sound == 0 && stristr(reason, "yak") != NULL) sound = 1;
            if (sound == 0 && stristr(reason, "smodem") != NULL) sound = 1;
            if (sound == 0 && stristr(reason, "smotu") != NULL) sound = 1;
            if (sound == 0 && stristr(reason, "rekka") != NULL) sound = 1;
            if (sound == 0 && stristr(reason, "reg") != NULL) sound = 1;

            if (sound)
            {
#ifdef __linux__
#if 0
                if (fork() == 0)
                {
                    char *arg[3];

                    arg[0] = "mpg123";
                    arg[1] = "/files/music/mp3/WHEN_I.MP3";
                    arg[2] = NULL;
                    execvp(arg[0],&arg[0]);
                }
#endif
#endif
            }
        }
    }

    output("\r\n@X07Beware! Sysop is out there...\r\n");
}

int get_event(void)
{
#ifdef __linux__
    static fd_set fdset;
    struct timeval wait;

    wait.tv_sec = 1;
    wait.tv_usec = 0;

    FD_ZERO(&fdset);
    FD_SET(0, &fdset);
    FD_SET(hCom, &fdset);

    if (select(hCom+1, &fdset, NULL, NULL, &wait) >= 1)
    {
        if (FD_ISSET(0, &fdset)) return 1;
        if (FD_ISSET(hCom, &fdset)) return 2;
    }

    return 0;
#else
    if (kbhit()) return 1;
    if (mdm_kbhit()) return 2;
    return 0;
#endif
}

/* Line chat */
void line_chat(void)
{
    char line[256],*strp;
    int ch, xpos;

    if (chatting)
    {
        /* Pressed chat-key while already in chat.. */
        return;
    }

    noderec.doing = DOING_CHATTING;
    update_nodefile(0);

    output(lang[LANG_SYSOP_HERE]);
    xpos = 0; chatting = 1; ch = -1;

    while (ch != -2)
    {
        switch (get_event())
        {
            case 1:
                ch = (unsigned char) getch();
                output("@X0F");
                if (ch == 27 || ch == 26) ch = -2;
                break;
            case 2:
                ch = (unsigned char) mdm_getch();
                output("@X07");
                break;
            default:
                if (!carr_det()) return;
                give_timeslice();
                ch = -1;
                break;
        }

        switch (ch)
        {
            case -1:
                break;
            case 8:
                if (xpos > 0)
                {
                    output("\x08 \x08");
                    xpos--;
                }
                break;
            case 13:
                output("\r\n");
                xpos = 0;
                break;
            default:
                if (ch < 32) break;
                if (xpos == 79)
                {
                    line[xpos] = '\0';
                    strp = strrchr(line,' ');
                    if (strp == NULL || strp == line)
                    {
                        xpos = 0;
                        output("\r\n"); /* Nextline */
                    }
                    else
                    {
                        strp++;
                        xpos = 79-(int) (strp-line);
                        memset(line,8,xpos); line[xpos] = '\0';
                        output(line); /* backspaces */
                        memset(line,' ',xpos);
                        output(line); /* Spaces */
                        output("\r\n"); /* Nextline */
                        xpos = strlen(strp);
                        memmove(line,strp,xpos+1);
                        output(line);
                    }
                }
                outchr((unsigned char) ch);
                line[xpos++] = (unsigned char) ch;
                break;
        }
    }

    output(lang[LANG_SYSOP_LEFT]);
    chatting = 0;

    noderec.doing = DOING_NOTHING;
    update_nodefile(0);
}

static int palettes;
static char palbuf[5][100],*palette;
static unsigned short *chatscr;
static unsigned short *chatbuf,*chatbuf1,*chatbuf2;
static int xlen, ylen;
static int xstart,ystart,xstart1,ystart1,xstart2,ystart2;
static int xbox1,ybox1,xbox2,ybox2,max1,max2,maxlen;
static int xpos,ypos,xpos1,ypos1,xpos2,ypos2,maxpos,oldx,oldy,oldcolor;
static int bufpos,buf1pos,buf2pos;

/* Display chat background and read chat configuration */
int load_chat_screen(char *fname)
{
    char tmp[50],ansiname[256];
    int f;

    /* Read configuration */
    sprintf(tmp,"%s"SSLASH"%s.cfg", data_path, fname);
    f = FileOpen(tmp, O_RDONLY, SH_DENYNO);
    if (f == -1)
    {
        printf("Couldn't open file '%s'\n",fname);
        return 0;
    }

    /* Read first line (ansi name) */
    if (_fgets(ansiname,sizeof(ansiname),f) == NULL)
    {
        close(f);
        printf("Couldn't read first line of configuration file\n");
        return 0;
    }

    /* Read second line (configuration) */
    if (_fgets(tmp,sizeof(tmp),f) == NULL)
    {
        close(f);
        printf("Couldn't read second line of configuration file\n");
        return 0;
    }
    close(f);

    if (sscanf(tmp,"%d %d %d %d %d %d %d %d %d %d %d %d",
               &xlen,&ylen,
               &xstart1,&ystart1,&xstart2,&ystart2,
               &xbox1,&ybox1,&xbox2,&ybox2,&max1,&max2) != 12)
    {
        /* Error in configuration file */
        printf("Error in configuration file\n");
        return 0;
    }

    /* Display ansi */
    output("@CLR@");
    show_file(ansiname);

    return 1;
}

int load_palette(char *fname)
{
    char tmp[256],*strp;
    int pals, f;

    palettes = 0;

    /* Open */
    f = FileOpen(fname, O_RDONLY, SH_DENYNO);
    if (f == -1) return 0;

    /* Read */
    while (_fgets(tmp,sizeof(tmp),f) != NULL)
    {
        if (tmp[0] == '\0' || tmp[0] == ';') continue;

        /* Get palettes */
        strp = tmp; pals = 0; palbuf[palettes][0] = 0;
        while (*strp != '\0')
        {
            if (*strp == ' ')
            {
                if (palbuf[palettes][pals] == 0) break;
                palbuf[palettes][++pals] = 0;
                if (pals == sizeof(palbuf[palettes])) break;
            }
            else
            {
                if (*strp < '0' || *strp > '9')
                {
                    /* Error in line */
                    pals = 0;
                    break;
                }
                palbuf[palettes][pals] = (char) (palbuf[palettes][pals]*10 + (*strp-'0'));
            }
            strp++;
        }

        /* No palette here.. */
        if (pals == 0 || palbuf[palettes][0] == 0) continue;

        /* Each palette ends with zero */
        if (palbuf[palettes][pals] != 0) palbuf[palettes][++pals] = 0;

        palettes++;
        if (palettes == 5) break; /* Max. 5 palettes */
    }

    close(f);
    return 1;
}

static void next_line(int local, int just_clear)
{
    int xp,xp2,_oldx,_oldy,bg;

    _oldx = xpos; _oldy = ypos;

    while (xpos > 1 && (chatscr[xpos-1+(ypos-1)*xlen] & 0xff) != ' ') xpos--;
    ypos++;
    if (ypos > maxlen)
    {
        /* End of screen, jump back to start */
        xpos = xstart; ypos = ystart;
    }
    else
    {
        if ((chatscr[xpos-1+(ypos-1)*xlen] & 0xff) != ' ')
        {
            /* Search forward.. */
            while (xpos < maxpos && (chatscr[xpos-1+(ypos-1)*xlen] & 0xff) != ' ') xpos++;
            if (xpos == maxpos)
            {
                /* Jump back to start */
                xpos = xstart; ypos = ystart;
            }
        }
        else
        {
            /* Search backward */
            while (xpos > 1 && (chatscr[xpos-2+(ypos-1)*xlen] & 0xff) == ' ') xpos--;
        }
    }
    output("@GOTO:%d,%d@",xpos,ypos); oldx = xpos; oldy = ypos;

    /* Clear line */
    xp = xpos; bg = chatscr[xp-1+(ypos-1)*xlen] >> 12;
    while (xp < xlen && (chatscr[xp-1+(ypos-1)*xlen] & 0xff) == ' ')
    {
        if (bg != -1 && (chatscr[xp-1+(ypos-1)*xlen] >> 12) != bg)
        {
            /* Line contains different colored background, can't use @CLREOL@ */
            bg = -1;
        }
        xp++;
    }
    if (xp == xlen && bg != -1)
    {
        /* Can clear to end of line */
        if ((bg << 8)+7 != oldcolor)
        {
            output("@X%X7",bg);
            oldcolor = (bg << 8)+7;
        }
        output("@CLREOL@");
    }
    else
    {
        xp2 = xp; xp = xpos;
        while (xp < xp2)
        {
            bg = ((chatscr[xp-1+(ypos-1)*xlen] >> 8) & 0xf0)+7;
            if (bg != oldcolor)
            {
                oldcolor = bg;
                output("@X%02X",oldcolor);
            }
            outchr(' ');
            xp++;
        }
        oldx = -1; oldy = -1;
        if (!just_clear)
        {
            output("@GOTO:%d,%d@",xpos,ypos);
            oldx = xpos; oldy = ypos;
        }
    }

    if (just_clear)
    {
        xpos = _oldx;
        ypos = _oldy;
        output("@GOTO:%d,%d@",xpos,ypos);
        oldx = xpos; oldy = ypos;
    }
    else
    {
        /* Clear also next line */
        next_line(local,1);
        chatbuf[(ypos-2)*xlen+bufpos] = 0;
        bufpos = 0;
    }
}

void put_char(unsigned char ch, char col, int local)
{
    unsigned short *tmp;
    unsigned char color;
    int tmplen,pos,xp,bg;

    /* Get color to use */
    color = (unsigned char) (((chatscr[xpos-1+(ypos-1)*xlen] >> 8) & 0xf0) + col);

    /* Add to buffer */
    chatbuf[(ypos-1)*xlen+bufpos++] = (unsigned short) (ch + ((unsigned) color << 8));

    if (color != oldcolor)
    {
        /* Change color */
        output("@X%02X",color);
        oldcolor = color;
    }
    if (oldx != xpos && oldy != ypos)
    {
        /* Move cursor */
        output("@GOTO:%d,%d@",xpos,ypos);
    }

    outchr(ch);
    xpos++; oldx = xpos; oldy = ypos;

    if (xpos == xlen || (chatscr[xpos-1+(ypos-1)*xlen] & 0xff) != ' ')
    {
        tmp = NULL; tmplen = 0;
        if (ch != ' ')
        {
            /* End of line, move last word to next line */
            xp = bufpos-1;
            while (xp > 0 && (chatbuf[(ypos-1)*xlen+xp] & 0xff) != ' ') xp--;
            if (xp > 0)
            {
                /* Got word.. */
                tmplen = bufpos-xp-1;
                tmp = (unsigned short *) _malloc(tmplen*sizeof(unsigned short));
                memcpy(tmp,&chatbuf[(ypos-1)*xlen+xp+1],tmplen*sizeof(tmp[0]));
                bufpos = xp;
                output("@GOTO:%d,%d@",xpos-tmplen,ypos);
                for (xp = 0; xp < tmplen; xp++)
                {
                    bg = ((chatscr[xpos-tmplen+xp-1+(ypos-1)*xlen] >> 8) & 0xf0)+7;
                    if (bg != oldcolor)
                    {
                        oldcolor = bg;
                        output("@X%02X",oldcolor);
                    }
                    outchr(' ');
                }
                oldx = -1; oldy = -1;
            }
        }
        xpos--; maxpos = xpos;
        next_line(local,0);

        if (tmp != NULL)
        {
            /* Move last word to this line.. */
            pos = 0;
            while (pos < tmplen)
            {
                put_char((unsigned char) (tmp[pos] & 0xff),(char) ((tmp[pos] >> 8) & 0x0f),local);
                pos++;
            }
            _free(tmp);
        }
    }
}

/* Split screen chat */
void split_screen_chat(void)
{
    unsigned short *tmpbuf,*tmpbuf1,*tmpbuf2;
    unsigned char ch,color,coloring,coloring1,coloring2;
    int xbox,ybox,xp,yp,maxx,colx,colx1,colx2,coly,coly1,coly2,hit;
    char pal_pos;

    if (chatting)
    {
        /* Pressed chat-key while already in chat.. */
        return;
    }

    /* Display background */
    if (!load_chat_screen("chat")) return;

    /* Load palette configuration */
    load_palette("chatpal.cfg");

    chatting = 1;
    noderec.doing = DOING_CHATTING;
    update_nodefile(0);

    /* Text buffers */
    chatbuf1 = (unsigned short *) _malloc(xlen*ylen*sizeof(unsigned short));
    chatbuf2 = (unsigned short *) _malloc(xlen*ylen*sizeof(unsigned short));
    memset(chatbuf1,1,xlen*ylen*sizeof(unsigned short));
    memset(chatbuf2,1,xlen*ylen*sizeof(unsigned short));
    buf1pos = 0; buf2pos = 0;

    coloring1 = 0; coloring2 = 0; tmpbuf = NULL; tmpbuf1 = NULL; tmpbuf2 = NULL;

    /* Chat screen */
    chatscr = (unsigned short *) _malloc(xlen*ylen*sizeof(unsigned short));
    if (xlen == scrwidth)
    {
        memcpy(chatscr,screen,xlen*ylen*sizeof(screen[0]));
    }
    else
    {
        for (yp = 0; yp < ylen; yp++)
        {
            memcpy(chatscr+yp*xlen,screen+yp*scrwidth,xlen*sizeof(tmpbuf[0]));
        }
    }

    pal_pos = 0; colx1 = 15; colx2 = 15; coly1 = 0; coly2 = 0;

    xpos1 = xstart1; ypos1 = ystart1;
    xpos2 = xstart2; ypos2 = ystart2;
    output("@GOTO:%d,%d@",xpos1,ypos1);
    oldx = xpos1; oldy = ypos1; oldcolor = -1;
    for (;;)
    {
        if (!carr_det()) break; /* No carrier */

        hit = kbhit();
        if (hit || mdm_kbhit())
        {
            if (hit)
            {
                /* Local hit */
                ch = (unsigned char) getch();
                xpos = xpos1; ypos = ypos1; xstart = xstart1; ystart = ystart1;
                bufpos = buf1pos; chatbuf = chatbuf1;
                xbox = xbox1; ybox = ybox1; coloring = coloring1;
                colx = colx1; coly = coly1; maxlen = max1;
                if (ch == 27 || ch == 26) break; /* Quit */
            }
            else
            {
                /* Remote hit */
                ch = (unsigned char) mdm_getch();
                xpos = xpos2; ypos = ypos2; xstart = xstart2; ystart = ystart2;
                bufpos = buf2pos; chatbuf = chatbuf2;
                xbox = xbox2; ybox = ybox2; coloring = coloring2;
                colx = colx2; coly = coly2; maxlen = max2;
            }

            switch (ch)
            {
                case 0:
                    /* Cursor? */
                    ch = (unsigned char) (hit ? getch() : mdm_getch());
                    if (!coloring) break;

                    switch (ch)
                    {
                        case 'M':
                        case 'K':
                            /* Right/left */
                            if (colx == 0)
                                output("@GOTO:%d,%d@@X0F ",xbox+1,ybox+1+coly);
                            else
                                output("@GOTO:%d,%d@@X0F ",xbox+16+colx*2,ybox+3);

                            if (ch == 'K')
                            {
                                /* Left */
                                if (colx == 0) colx = 15; else colx--;
                            }
                            else
                            {
                                /* Right */
                                if (colx == 15) colx = 0; else colx++;
                            }

                            if (colx == 0)
                                output("@GOTO:%d,%d@@X0F>",xbox+1,ybox+1+coly);
                            else
                                output("@GOTO:%d,%d@@X0F>",xbox+16+colx*2,ybox+3);
                            break;
                        case 'P':
                            /* Down */
                            if (colx != 0) break;
                            output("@GOTO:%d,%d@@X0F ",xbox+1,ybox+1+coly);
                            if (coly == palettes-1) coly = 0; else coly++;
                            palette = palbuf[coly];
                            output("@GOTO:%d,%d@@X0F>",xbox+1,ybox+1+coly);
                            break;
                        case 'H':
                            /* Up */
                            if (colx != 0) break;
                            output("@GOTO:%d,%d@@X0F ",xbox+1,ybox+1+coly);
                            if (coly == 0) coly = palettes-1; else coly--;
                            palette = palbuf[coly];
                            output("@GOTO:%d,%d@@X0F>",xbox+1,ybox+1+coly);
                            break;
                    }
                    oldx = -1; oldy = -1;
                    break;
                case 3:
                    /* Ctrl-C = Change color */
                    if (coloring) break;
                    coloring = 1;
                    tmpbuf = (unsigned short *) _malloc(xlen*7*sizeof(unsigned short));
                    if (hit) tmpbuf1 = tmpbuf; else tmpbuf2 = tmpbuf;
                    for (yp = 0; yp < 7; yp++)
                    {
                        memcpy(tmpbuf+yp*xlen,screen+(ybox-1+yp)*scrwidth,xlen*sizeof(tmpbuf[0]));
                    }
                    output("@GOTO:%d,%d@@X0F@X08[@X0BPalette@X08]Ŀ",xbox,ybox);
                    output("@GOTO:%d,%d@@X0F             @X08",xbox,ybox+1);
                    output("@GOTO:%d,%d@@X07             ",xbox,ybox+2);
                    output("@GOTO:%d,%d@@X08             @X07",xbox,ybox+3);
                    output("@GOTO:%d,%d@             ",xbox,ybox+4);
                    output("@GOTO:%d,%d@@X08             ",xbox,ybox+5);
                    output("@GOTO:%d,%d@@X08@X07@X08@X07@X08@X07@X0F@X07@X0F",xbox,ybox+6);
                    output("@GOTO:%d,%d@@X0F@X08[@X0BSingle Color@X08]Ŀ",xbox+17,ybox+2);
                    output("@GOTO:%d,%d@ @X01* @X02* @X03* @X04* @X05* @X06* @X07* @X08* @X09* @X0A* @X0B* @X0C* @X0D* @X0E* @X0F* @X07",xbox+17,ybox+3);
                    output("@GOTO:%d,%d@@X08@X07@X08@X07@X08@X07@X0F@X07@X0F",xbox+17,ybox+4);

                    /* Draw palettes */
                    for (yp = 0; yp < palettes; yp++)
                    {
                        output("@GOTO:%d,%d@",xbox+2,ybox+yp+1);
                        palette = palbuf[yp];
                        for (xp = 0; xp < 6; xp++)
                        {
                            if (palette[xp] == 0) break;
                            output("@X%02X* ",palette[xp]);
                        }
                    }

                    if (colx == 0)
                    {
                        palette = palbuf[coly];
                        output("@GOTO:%d,%d@@X0F>",xbox+1,ybox+1+coly);
                    }
                    else
                        output("@GOTO:%d,%d@@X0F>",xbox+16+colx*2,ybox+3);

                    oldx = 0; oldy = 0; oldcolor = 15;
                    break;
                case 8:
                    /* Backspace */
                    if (coloring) break;
                    ch = (xpos > 1 && (chatscr[xpos-2+(ypos-1)*xlen] & 0xff) == ' ');
                    if (ch || ypos > ystart)
                    {
                        if (ch)
                        {
                            /* Just go back to previous character.. */
                            color = (unsigned char) ((chatscr[xpos-2+(ypos-1)*xlen] >> 8) & 0xff);
                            if (color != oldcolor)
                            {
                                /* Change color */
                                output("@X%02X",color);
                                oldcolor = color;
                            }
                            xpos--; bufpos--;
                            output("@GOTO:%d,%d@ @GOTO:%d,%d@",xpos,ypos,xpos,ypos);
                            oldx = xpos; oldy = ypos;
                        }
                        else
                        {
                            /* Jump to previous line */
                            while (xpos < xlen && (chatscr[xpos-1+(ypos-1)*xlen] & 0xff) == ' ') xpos++;
                            xpos--; maxpos = xpos;
                            ypos--;
                            while (xpos > 1 && (chatscr[xpos-1+(ypos-1)*xlen] & 0xff) != ' ') xpos--;
                            bufpos = 0;
                            while ((chatbuf[(ypos-1)*xlen+bufpos] & 0xff) != 0) bufpos++;
                            while (xpos > 1 && (chatscr[xpos-1+(ypos-1)*xlen] & 0xff) != ' ') xpos--;
                            while (xpos > 1 && (chatscr[xpos-1+(ypos-1)*xlen] & 0xff) == ' ') xpos--;
                            xpos += bufpos;

                            color = (unsigned char) ((chatscr[xpos-1+(ypos-1)*xlen] >> 8) & 0xff);
                            if (color != oldcolor)
                            {
                                /* Change color */
                                output("@X%02X",color);
                                oldcolor = color;
                            }

                            output("@GOTO:%d,%d@ @GOTO:%d,%d@",xpos,ypos,xpos,ypos);
                            oldx = xpos; oldy = ypos;
                        }
                    }
                    break;
                case 13:
                    if (coloring)
                    {
                        /* Color selected */
                        if (colx == 0)
                        {
                            /* Palette selected */
                            pal_pos = 0;
                        }
                        else
                        {
                            /* Single color */
                            palette = NULL;
                        }

                        /* Restore screen */
                        for (yp = 0; yp < 7; yp++)
                        {
                            output("@GOTO:%d,%d@",xbox,yp+ybox);
                            maxx = (yp >= 2 && yp <= 4) ? 49 : 14;
                            for (xp = xbox-1; xp < xbox+maxx; xp++)
                            {
                                color = (unsigned char) (tmpbuf[xp+yp*xlen] >> 8);
                                if (color != oldcolor)
                                {
                                    oldcolor = color;
                                    output("@X%02X",color);
                                }
                                outchr((unsigned char) (tmpbuf[xp+yp*xlen] & 0xff));
                            }
                        }
                        if (hit) _free(tmpbuf1); else _free(tmpbuf2);
                        output("@GOTO:%d,%d@",xpos,ypos); oldx = xpos; oldy = ypos;
                        coloring = 0;
                        break;
                    }

                    /* Trace to end of line.. */
                    while (xpos < xlen && (chatscr[xpos-1+(ypos-1)*xlen] & 0xff) == ' ') xpos++;
                    xpos--; maxpos = xpos;
                    /* Move to next line */
                    next_line(hit,0);
                    break;
                case 18:
                    /* Ctrl-R = Redraw screen */
                    break;
                case 23:
                    /* Ctrl-W = Clear screen */
                    xpos = xstart; ypos = ystart;
                    for (yp = 0; yp < maxlen; yp++)
                    {
                        /* Trace to end of line.. */
                        while (xpos < xlen && (chatscr[xpos-1+(ypos-1)*xlen] & 0xff) == ' ') xpos++;
                        xpos--; maxpos = xpos;
                        next_line(hit,0);
                    }
                    xpos = xstart; ypos = ystart;
                    output("@GOTO:%d,%d@",xpos,ypos); oldx = xpos; oldy = ypos;
                    break;
                default:
                    if (coloring) break;
                    if (ch < 32) break;

                    if (colx != 0)
                    {
                        /* Use single color */
                        color = (unsigned char) colx;
                    }
                    else
                    {
                        /* Use palette */
                        color = palette[pal_pos++];
                        if (palette[pal_pos] == 0) pal_pos = 0;
                    }
                    put_char(ch,color,hit);
                    break;
            }

            if (hit)
            {
                xpos1 = xpos; ypos1 = ypos; buf1pos = bufpos;
                coloring1 = coloring; colx1 = colx; coly1 = coly;
            }
            else
            {
                xpos2 = xpos; ypos2 = ypos; buf2pos = bufpos;
                coloring2 = coloring; colx2 = colx; coly2 = coly;
            }
        }
    }
    chatting = 0;

    _free(chatscr);
    _free(chatbuf1);
    _free(chatbuf2);

    noderec.doing = DOING_NOTHING;
    update_nodefile(0);
}
