/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* mareas.c - Message area handling */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "os.h"
#include "memory.h"
#include "output.h"
#include "files.h"
#include "jam.h"
#include "logfile.h"
#include "keyb.h"
#include "threads.h"
#include "modem.h"
#include "timeslic.h"
#include "bbs_func.h"
#include "userbase.h"
#include "mareas.h"
#include "messages.h"
#include "msg_jam.h"
#include "ask_str.h"
#include "language.h"
#include "config.h"

MAREA_REC marea;                /* Current file area record */
MAREA_REC *marea_rec;           /* Current file area record .. pointer */
static MAREA_REC *first_marea;  /* First msg area record in linked list */

unsigned msg_areas;             /* Number of msg areas */
unsigned current_marea;         /* Current msg area number */

/* Saved lastread pointers */
unsigned long *ptrs_this_call;  /* Start of this call */

MAREA_REC msg_area;             /* Message area record.. Again.. Used by @MAREA_xxx@ macros */

/* Read message areas */
int init_mareas(void)
{
    int Farea;
    char str[512], rsec[81], wsec[81], flags[11], charset, *strp;
    int nodeaddr;
    unsigned long line,num;
    MAREA_REC *tmp_rec,tmp_marea_rec;
    CHARSET_REC *cset;

    msg_areas = 0;

    /* Open message area configureation file */
    Farea = FileOpen(mareas_bbs, O_RDONLY | O_BINARY, SH_DENYNO);
    if (Farea == -1) return 0;

    /* Read it */
    line = 0;
    num = 0;
    while (_fgets(str,sizeof(str),Farea) != NULL)
    {
        line++;
        if ((str[0] == ';') || (str[0] == '\0')) continue;

        memset(&tmp_marea_rec,0,sizeof(tmp_marea_rec));
        if (sscanf(str,"%s %s %s %d %s %s %s %c %[^\n]",
                   tmp_marea_rec.path,
                   tmp_marea_rec.name,
                   tmp_marea_rec.bw_tag,
                   &nodeaddr,
                   rsec, wsec,
                   flags, &charset,
                   tmp_marea_rec.description
                  ) < 8)
        {
            write_log("Error in message area configuration file '%s' line %lu", mareas_bbs, line);
            printf("Error in message area configuration file '%s' line %lu\n", mareas_bbs, line);
            continue;
        }

        num++;

        /* Not even read rights.. */
        sprintf(str, "%s@r", tmp_marea_rec.name);
        if (!in_group(rsec) || !in_group(str)) continue;

        tmp_rec = (MAREA_REC *) _malloc(sizeof(MAREA_REC));
        if (tmp_rec == NULL)
        {
            printf("Not enough memory!\n");
            deinit_mareas();
            return 0;
        }

        if (msg_areas > 0)
            marea_rec->next = tmp_rec;
        else
            first_marea = tmp_rec;
        msg_areas++;

        /* Make message area record */
        tmp_rec->prev = marea_rec;
        marea_rec = tmp_rec;
        marea_rec->next = NULL;

        marea_rec->number = num;
        strcpy(marea_rec->path, tmp_marea_rec.path);
#if SLASH == '\\'
        strp = marea_rec->path;
        while (*strp != '\0')
        {
            if (*strp == '/') *strp = '\\';
            strp++;
        }
#endif
        strcpy(marea_rec->name, tmp_marea_rec.name);
        if (tmp_marea_rec.bw_tag[0] == '-' && tmp_marea_rec.bw_tag[1] == '\0')
        {
            /* BW tag = Area Tag */
            if (strlen(tmp_marea_rec.name) > 20)
            {
                /* Too long area tag */
                printf("Too long BW tag name in msg area configuration file '%s' line %lu\n", mareas_bbs, line);
                strncpy(marea_rec->bw_tag, tmp_marea_rec.name,20);
                marea_rec->bw_tag[20] = '\0';
            }
            else
            {
                strcpy(marea_rec->bw_tag, tmp_marea_rec.name);
            }
        }
        else
        {
            strcpy(marea_rec->bw_tag,tmp_marea_rec.bw_tag);
        }
        strcpy(marea_rec->description, tmp_marea_rec.description);
        marea_rec->nodeaddr = (unsigned char) nodeaddr;
        marea_rec->flags = MAREA_FLAG_PUBLIC_MSGS|MAREA_FLAG_READ_RIGHTS;

        sprintf(str, "%s@w", tmp_marea_rec.name);
        if (in_group(wsec) && in_group(str)) marea_rec->flags |= MAREA_FLAG_WRITE_RIGHTS;

        /* Get character set */
        marea_rec->charset = firstcharset;
        cset = firstcharset;
        while (cset != NULL)
        {
            if (cset->key == charset)
            {
                marea_rec->charset = cset;
                break;
            }
            cset = cset->next;
        }

        /* Get flags */
        strupr(flags);
        strp = flags;
        while (*strp != '\0')
        {
            switch (*strp++)
            {
                case 'P':
                    /* Only private messages */
                    marea_rec->flags |= MAREA_FLAG_PRIVATE_MSGS;
                    marea_rec->flags &= ~MAREA_FLAG_PUBLIC_MSGS;
                    break;
                case 'B':
                    /* Private and public messages */
                    marea_rec->flags |= MAREA_FLAG_PRIVATE_MSGS;
                    break;
                case 'E':
                    /* Echo area */
                    marea_rec->flags |= MAREA_FLAG_ECHO;
                    break;
                case 'N':
                    /* Netmail area */
                    marea_rec->flags |= MAREA_FLAG_NETMAIL;
                    break;
                case 'F':
                    /* Forced */
                    marea_rec->flags |= MAREA_FLAG_FORCED|MAREA_FLAG_SELECTED;
                    break;
                case 'T':
                    /* No taglines */
                    marea_rec->flags |= MAREA_FLAG_NO_TAGS;
                    break;
            }
        }
    }

    /* Close message area configuration file */
    FileClose(Farea);

    current_marea = 1;
    marea_rec = first_marea;

    memcpy(&marea,marea_rec,sizeof(marea));
    msg_init();

    return 1;
}

/* Deallocate memory used by message areas */
void deinit_mareas(void)
{
    while (first_marea != NULL)
    {
        marea_rec = first_marea->next;
        _free(first_marea);
        first_marea = marea_rec;
    }

    /* Close opened message area */
    close_msgarea();
    msg_deinit();
}

/* Read message area record # */
int read_marea_record(unsigned long marea_num)
{
    /* Too big area number? */
    if (marea_num > msg_areas || marea_num == 0) return 0;

    if (current_marea == 0 || marea_num == 1)
    {
        current_marea = 1;
        marea_rec = first_marea;
    }

    /* Go to right position */
    while (marea_num > current_marea)
    {
        marea_rec = marea_rec->next;
        if (marea_rec == NULL)
        {
            current_marea = 0;
            return 0;
        }
        current_marea++;
    }
    while (marea_num < current_marea)
    {
        marea_rec = marea_rec->prev;
        if (marea_rec == NULL)
        {
            current_marea = 0;
            return 0;
        }
        current_marea--;
    }

    memcpy(&marea,marea_rec,sizeof(marea));
    return 1;
}

int more_msg(void)
{
    int ch,modem;

    for (;;)
    {
        for (;;)
        {
            if (mdm_kbhit())
            {
                ch = mdm_getch();
                modem = 1;
                break;
            }
            else if (kbhit())
            {
                ch = getch();
                modem = 0;
                break;
            }
            else
            {
                if (!carr_det()) return 0;
                give_timeslice();
            }
        }

        switch (toupper(ch))
        {
            case 'Q':
                return 0;
            case 13:
                return 1;
            case ' ':
                if (user.Emulation != USER_EMULATION_ASCII)
                {
                    if (marea_rec->flags & MAREA_FLAG_FORCED && marea_rec->flags & MAREA_FLAG_SELECTED) break;

                    marea_rec->flags ^= MAREA_FLAG_SELECTED;
                    if (marea_rec->flags & MAREA_FLAG_SELECTED)
                        output(lang[LANG_MSGAREA_ON]);
                    else
                        output(lang[LANG_MSGAREA_OFF]);
                    outchr('\r');
                    return 2;
                }
                break;
            case 5:
                /* Up */
                return 3;
            case 24:
                /* Down */
                return 2;
            case 18:
                /* PgUp */
                return 4;
            case 3:
                /* PgDn */
                return 5;
            case 23:
                /* Home */
                return 6;
            case 26:
                /* End */
                return 7;
            case 0:
                ch = modem ? mdm_getch() : getch();
                switch (ch)
                {
                    case 'P':
                        /* Down */
                        return 2;
                    case 'H':
                        /* Up */
                        return 3;
                    case 'I':
                        /* PgUp */
                        return 4;
                    case 'Q':
                        /* PgDn */
                        return 5;
                    case 'G':
                        /* Home */
                        return 6;
                    case 'O':
                        /* End */
                        return 7;
                }
                break;
        }
    }
}

static int cur_marea;

void print_line(void)
{
    memcpy(&msg_area,&marea,sizeof(msg_area));
    if (marea.flags & MAREA_FLAG_FORCED)
        output(lang[LANG_MSGAREA_FORCED]);
    else if (marea.flags & MAREA_FLAG_SELECTED)
        output(lang[LANG_MSGAREA_ON]);
    else
        output(lang[LANG_MSGAREA_OFF]);
    output(lang[LANG_MSGAREA_ENTRY]);
}

void print_areas(void)
{
    unsigned long rec;
    int lines;

    output(lang[LANG_MSGAREA_HEADER]);

    lines = 1;
    for (rec=cur_marea; rec<=msg_areas; rec++)
    {
        if (read_marea_record(rec) == 0) continue;
        if ((marea_rec->flags & MAREA_FLAG_READ_RIGHTS) == 0) continue;

        if (lines++ == user.ScreenLen) break;
        output("\r\n");

        print_line();
    }
}

void areas_back(int num)
{
    unsigned long rec;

    for (rec=cur_marea; rec>=1; rec--)
    {
        if (read_marea_record(rec) == 0) continue;
        if ((marea_rec->flags & MAREA_FLAG_READ_RIGHTS) == 0) continue;

        cur_marea--;
        num--;
        if (num == 0) break;
    }
}

void areas_forward(int num)
{
    unsigned long rec;

    for (rec=cur_marea; rec<=msg_areas; rec++)
    {
        if (read_marea_record(rec) == 0) continue;
        if ((marea_rec->flags & MAREA_FLAG_READ_RIGHTS) == 0) continue;

        cur_marea++;
        num--;
        if (num == 0) break;
    }
}

int search_areaname(char *name)
{
    unsigned long rec,old_rec;

    old_rec = current_marea;
    for (rec=1; rec<=msg_areas; rec++)
    {
        read_marea_record(rec);
        if ((marea_rec->flags & MAREA_FLAG_READ_RIGHTS) == 0) continue;
        if (stricmp(marea.name,name) == 0) return 1;
    }

    read_marea_record(old_rec);
    return 0;
}

int search_bwtag(char *bw_tag)
{
    unsigned long rec,old_rec;

    old_rec = current_marea;
    for (rec=1; rec<=msg_areas; rec++)
    {
        read_marea_record(rec);
        if ((marea_rec->flags & MAREA_FLAG_READ_RIGHTS) == 0) continue;
        if (stricmp(marea.bw_tag,bw_tag) == 0) return 1;
    }

    read_marea_record(old_rec);
    return 0;
}

int search_areanum(unsigned long num)
{
    unsigned long rec,old_rec;

    old_rec = current_marea;
    for (rec=1; rec<=msg_areas; rec++)
    {
        read_marea_record(rec);
        if ((marea_rec->flags & MAREA_FLAG_READ_RIGHTS) == 0) continue;
        if (num == marea.number) return 1;
    }

    read_marea_record(old_rec);
    return 0;
}

/* Select new message area */
int select_marea(char *marea_name)
{
    MAREA_REC *closest;
    unsigned long old_rec,rec,closest_num;
    unsigned ret,ypos;
    unsigned upy,old,last;
    char tmp[81],*strp;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return 0;
    }

    old_rec = current_marea;

    if (marea_name[0] == '\0')
    {
        /* No parameters */
        cur_marea = current_marea;
        upy = cur_marea-1;
        ypos = 1;
        print_areas();

        do
        {
            output("@GOTO:1,%d@",ypos+1);
            read_marea_record(ypos+upy);
            ret = more_msg();
            switch (ret)
            {
                case 2:
                    /* Down */
                    if (ypos+upy < msg_areas)
                    {
                        if (ypos+1 < user.ScreenLen && user.Emulation != USER_EMULATION_ASCII)
                        {
                            ypos++;
                        }
                        else
                        {
                            if (user.Emulation == USER_EMULATION_ANSI_X364)
                            {
                                /* Scroll one line down */
                                output("@SCROLL_UP:2@@GOTO:1,%d@",ypos+1);
                                upy++;
                                areas_forward(1);
                                read_marea_record(cur_marea+ypos-1);
                                print_line();
                            }
                            else
                            {
                                old = upy;
                                upy += user.ScreenLen-1;
                                if (upy >= msg_areas) upy = msg_areas-1;
                                areas_forward(upy-old);
                                print_areas();
                                ypos = 1;
                            }
                        }
                    }
                    break;
                case 3:
                    /* Up */
                    if (ypos+upy > 1)
                    {
                        if (ypos > 1 && user.Emulation != USER_EMULATION_ASCII)
                        {
                            ypos--;
                        }
                        else
                        {
                            if (user.Emulation == USER_EMULATION_ANSI_X364)
                            {
                                /* Scroll one line up */
                                output("@SCROLL_DOWN:2@@GOTO:1,2@");
                                upy--;
                                areas_back(1);
                                read_marea_record(cur_marea);
                                print_line();
                            }
                            else
                            {
                                last = ypos+upy;
                                old = upy;
                                if (upy+1 <= user.ScreenLen) upy = 0; else upy -= user.ScreenLen-1;
                                areas_back(old-upy);
                                print_areas();
                                ypos = last-upy-1;
                            }
                        }
                    }
                    break;
                case 4:
                    /* PgUp */
                    if (ypos+upy > 1)
                    {
                        if (ypos > 1 && user.Emulation != USER_EMULATION_ASCII)
                        {
                            ypos = 1;
                        }
                        else
                        {
                            old = upy;
                            if (upy+1 <= user.ScreenLen) upy = 0; else upy -= user.ScreenLen-1;
                            areas_back(old-upy);
                            print_areas();
                        }
                    }
                    break;
                case 5:
                    /* PgDn */
                    if (ypos+upy < msg_areas)
                    {
                        if (ypos+1 < user.ScreenLen && user.Emulation != USER_EMULATION_ASCII)
                        {
                            ypos = user.ScreenLen-1;
                            if (ypos+upy > msg_areas) ypos = msg_areas-upy;
                        }
                        else
                        {
                            old = upy;
                            upy += user.ScreenLen-1;
                            if (upy >= msg_areas) upy = msg_areas-1;
                            areas_forward(upy-old);
                            print_areas();
                            if (ypos+upy > msg_areas) ypos = msg_areas-upy;
                        }
                    }
                    break;
                case 6:
                    /* Home */
                    if (ypos+upy > 1)
                    {
                        if (upy == 0)
                        {
                            ypos = 1;
                        }
                        else
                        {
                            areas_back(upy);
                            ypos = 1; upy = 0;
                            print_areas();
                        }
                    }
                    break;
                case 7:
                    /* End */
                    if (ypos+upy < msg_areas)
                    {
                        if (msg_areas-upy+1 <= user.ScreenLen)
                        {
                            ypos = msg_areas-upy;
                        }
                        else
                        {
                            old = upy;
                            upy = msg_areas-user.ScreenLen+1;
                            ypos = msg_areas-upy;
                            areas_forward(upy-old);
                            print_areas();
                        }
                    }
                    break;
            }
        } while (ret > 1);
        output("@GOTO:1,%d@",user.ScreenLen);
    }
    else
    {
        strp = marea_name;
        while (isdigit(*strp)) strp++;
        if ((*strp == '\0' || *strp == ' ') && sscanf(marea_name,"%lu",&rec) == 1)
        {
            /* Select area by number */
            ret = read_marea_record(rec) != 0;
            if (!ret) output(lang[LANG_MSGAREA_NOT_FOUND], rec);
        }
        else
        {
            /* Select area by name */
            ret = 0; closest = NULL; closest_num = 0;
            for (rec=1; rec<=msg_areas; rec++)
            {
                read_marea_record(rec);
                if ((marea.flags & MAREA_FLAG_READ_RIGHTS) == 0) continue;

                if (stricmp(marea.name,marea_name) == 0)
                {
                    /* Found */
                    ret = 1;
                    break;
                }
                else
                {
                    strupr(marea.name); strupr(strcpy(tmp,marea_name));
                    if (strstr(marea.name,tmp) != NULL)
                    {
                        /* This one was close .. */
                        if (closest == NULL || strlen(closest->name) > strlen(marea.name))
                        {
                            /* Whoa, closest one yet! */
                            closest = marea_rec;
                            closest_num = rec;
                        }
                    }
                }
            }

            if (ret == 0 && closest != NULL)
            {
                /* Select closest one.. */
                read_marea_record(closest_num);
                ret = 1;
            }

            if (ret != 1)
            {
                read_marea_record(old_rec);
                output(lang[LANG_MSGAREA_NAME_NOT_FOUND], marea_name);
                return 0;
            }
        }
    }

    if (ret == 1)
    {
        /* Open message area */
        open_msgarea(marea.path,TYPE_JAM);
        memcpy(&msg_area,&marea,sizeof(msg_area));
        if (marea.lr_ptr == msg_messages)
            output(lang[LANG_NO_UNREAD_AREA_LEFT]);
        return 1;
    }

    read_marea_record(old_rec);
    memcpy(&msg_area,&marea,sizeof(msg_area));
    return 0;
}

/* Update lastread pointers */
int update_pointers(unsigned long msgnum)
{
    marea_rec->lr_ptr = msgnum;
    return 1;
}

/* Save lastread pointers */
int save_pointers(unsigned long *ptrs, int forced)
{
    unsigned long old_area,rec;

    if (restrict_usage & NO_MESSAGES && !forced)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return 0;
    }

    old_area = current_marea;
    for (rec=1; rec<=msg_areas; rec++)
    {
        if (read_marea_record(rec) == 0) marea.lr_ptr = 0;
        ptrs[rec-1] = marea.lr_ptr;
    }
    read_marea_record(old_area);
    return 1;
}

/* Restore lastread pointers */
int restore_pointers(unsigned long *ptrs)
{
    unsigned long old_area,rec;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return 0;
    }

    old_area = current_marea;
    for (rec=1; rec<=msg_areas; rec++)
    {
        if (read_marea_record(rec) == 0) marea.lr_ptr = 0;
        update_pointers(ptrs[rec-1]);
    }
    read_marea_record(old_area);
    return 1;
}

/* Restore lastread pointers */
void clear_pointers(void)
{
    unsigned long old_area,rec;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return;
    }

    old_area = current_marea;
    for (rec=1; rec<=msg_areas; rec++)
    {
        if (read_marea_record(rec) == 0) continue;
        update_pointers(0);
    }
    read_marea_record(old_area);
}

/* Set lastread pointers ## days back */
void pointer_days(char *data)
{
    unsigned long old_area,rec,mnum,msgpos;
    unsigned days_back;
    time_t _tim;
    struct tm *tim;
    int oldp, pros, tries, max;
    char str[20], *strp;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return;
    }

    str[0] = '\0'; if (!ask_string(lang[LANG_DAYS_BACK], str, 10, 0, &data)) return;

    if (sscanf(str, "%u", &days_back) != 1) return;

    _tim = time(NULL)-days_back*24*60*60;
    tim = localtime(&_tim);
    _tim -= (unsigned) tim->tm_hour*60*60 + (unsigned) tim->tm_min*60 + tim->tm_sec;

    strncpy(str, lang[LANG_DAYS_DATA], 19); str[19] = '\0';
    strp = strchr(str, ';');
    if (strp != NULL)
    {
        *strp = '\0';
        if (sscanf(str, "%d", &max) != 1) max = 0;
    }
    strp = strchr(lang[LANG_DAYS_DATA], ';');
    if (strp != NULL) strp++;

    output(lang[LANG_DAYS_SETTING]);
    old_area = current_marea;
    oldp = 0;
    for (rec=1; rec<=msg_areas; rec++)
    {
        if (max > 0 && strp != NULL)
        {
            pros = (int) ((rec*max)/msg_areas);
            if (oldp != pros)
            {
                oldp = pros;
                output(strp, pros);
            }
        }
        if (read_marea_record(rec) == 0) continue;
        if ((marea.flags & MAREA_FLAG_SELECTED) == 0) continue;
        if (open_msgarea(marea.path,TYPE_JAM) == 0) continue;

        msgpos = msg_messages; tries = 20;
        for (mnum = msg_messages; mnum > 0; mnum--)
        {
            if (msg_read_message(mnum) == 0) continue;
            if ((long) msg_date >= _tim && (msg_date & 0x80000000) == 0)
            {
                msgpos = mnum-1;
                tries = 20;
            }
            else
                if (tries > 0 && --tries == 0) break;
        }
        update_pointers(msgpos);
    }
    output("\r\n");
    read_marea_record(old_area);
    open_msgarea(marea.path,TYPE_JAM);
}

/* JAM's lastread pointer file .JLR -> YAK */
void update_jam_lastread(unsigned long NameCRC, int forced)
{
    int Fjlr;
    char path[256];
    unsigned long old_area,rec;
    JAMLREAD jamlr;

    if (restrict_usage & NO_MESSAGES && !forced)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return;
    }

    old_area = current_marea;
    for (rec=1; rec<=msg_areas; rec++)
    {
        if (read_marea_record(rec) == 0) continue;

        sprintf(path,"%s.jlr",marea.path);
        Fjlr = FileOpen(path, O_RDONLY | O_BINARY, SH_DENYNO);
        if (Fjlr != -1)
        {
            while (FileRead(Fjlr,&jamlr,sizeof(jamlr)) == sizeof(jamlr))
            {
                if (jamlr.UserCRC == NameCRC)
                {
                    update_pointers(jamlr.LastReadMsg);
                    break;
                }
            }
            FileClose(Fjlr);
        }
    }
    read_marea_record(old_area);
}

/* YAK -> JAM's lastread pointer file .JLR */
void update_rbbs_lastread(unsigned long NameCRC)
{
    int Fjlr;
    char path[256];
    unsigned long old_area,rec;
    JAMLREAD jamlr;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return;
    }

    old_area = current_marea;
    for (rec=1; rec<=msg_areas; rec++)
    {
        if (read_marea_record(rec) == 0) continue;

        sprintf(path,"%s.jlr",marea.path);
        Fjlr = FileOpen(path, O_RDWR | O_BINARY, SH_DENYNO);
        if (Fjlr == -1)
        {
            Fjlr = FileCreate(path, CREATE_MODE);
            FileMode(Fjlr,O_BINARY);
        }

        if (Fjlr != -1)
        {
            while (FileRead(Fjlr,&jamlr,sizeof(jamlr)) == sizeof(jamlr))
            {
                if (jamlr.UserCRC == NameCRC)
                {
                    FileSeek(Fjlr,-sizeof(jamlr),SEEK_CUR);
                    break;
                }
            }

            memset(&jamlr,0,sizeof(jamlr));
            jamlr.UserCRC = NameCRC;
            jamlr.UserID = NameCRC;
            jamlr.LastReadMsg = marea.lr_ptr;
            jamlr.HighReadMsg = marea.lr_ptr;

            FileWrite(Fjlr,&jamlr,sizeof(jamlr));
            FileClose(Fjlr);
        }
    }
    read_marea_record(old_area);
}

void show_mareas_status(void)
{
    unsigned long old_area,rec;
    unsigned long tot_unread,tot_personal;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return;
    }

    tot_unread = 0;
    tot_personal = 0;
    output(lang[LANG_MSTAT_HEADER]);

    old_area = current_marea;
    for (rec=1; rec<=msg_areas; rec++)
    {
        if (read_marea_record(rec) == 0) continue;
        if ((marea.flags & MAREA_FLAG_SELECTED) == 0) continue;
        if (msgarea_stat(marea.path,TYPE_JAM) == 0) continue;
        if (marea_stat.unread_msgs == 0) continue;

        memcpy(&msg_area,&marea,sizeof(msg_area));
        num1 = marea_stat.unread_msgs; num2 = marea_stat.personal_msgs;
        if (!output(lang[LANG_MSTAT_ENTRY])) break;
        tot_unread += marea_stat.unread_msgs;
        tot_personal += marea_stat.personal_msgs;
    }
    read_marea_record(old_area);
    memcpy(&msg_area,&marea,sizeof(msg_area));
    open_msgarea(marea.path,TYPE_JAM);
    num1 = tot_unread; num2 = tot_personal;
    output(lang[LANG_MSTAT_BOTTOM]);
}

int str2addr(char *addrstrp, unsigned short *zone, unsigned short *net, unsigned short *node, unsigned short *point)
{
    char *pstr;
    char addrstr[50],pos;

    strcpy(addrstr,addrstrp); pos = 0;
    for (pstr = addrstr; ; pstr++)
    {
        if (*pstr >= '0' && *pstr <= '9') continue;
        if (*pstr == ':' && pos == 0) { *pstr = ' '; pos = 1; continue; }
        if (*pstr == '/' && pos == 1) { *pstr = ' '; pos = 2; continue; }
        if (*pstr == '.' && pos == 2) { *pstr = ' '; pos = 3; continue; }
        if (*pstr == '@') { *pstr = '\0'; break; }
        if (*pstr == '\0') break;
        printf("Error : str2addr(\"%s\") position %d ('%c')\n",addrstrp,(int) (pstr-addrstr)+1,*pstr);
        return 0;
    }

    *zone = 0; *net = 0; *node = 0; *point = 0;
    return sscanf(addrstr,"%hu %hu %hu %hu",zone,net,node,point) >= 3;
}

NETADDR_REC *get_nearest_net(unsigned short d_zone, unsigned short d_net, unsigned short d_node, unsigned short d_point)
{
    int match;
    NETADDR_REC *net, *best;

    switch (d_zone)
    {
        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
            return firstnet;
        default:
            match = 0;
            best = firstnet;

            net = firstnet;
            for (; net != NULL; net = net->next)
            {
                if (d_zone == net->zone)
                {
                    if (d_net == net->net)
                    {
                        if (d_node == net->node)
                        {
                            if (d_point == net->point)
                            {
                                return net;
                            }
                            else
                            {
                                if (match <= 3)
                                {
                                    match = 3;
                                    best = net;
                                }
                            }
                        }
                        else
                        {
                            if (match <= 1)
                            {
                                match = 2;
                                best = net;
                            }
                        }
                    }
                    else
                    {
                        if (match == 0)
                        {
                            match = 1;
                            best = net;
                        }
                    }
                }
            }
            return best;
    }
}
