/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* output.c - All display management */

#ifdef __OS2__
#  define INCL_VIO
#  include <os2.h>
#endif

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

#include "os.h"
#include "memory.h"
#include "modem.h"
#include "menutype.h"
#include "output.h"
#include "nodes.h"
#include "keyb.h"
#include "messages.h"
#include "msg_jam.h"
#include "fareas.h"
#include "userbase.h"
#include "bbs_func.h"
#include "files.h"
#include "threads.h"
#include "mareas.h"
#include "filelist.h"
#include "lastcall.h"
#include "language.h"
#include "logfile.h"

#if !defined (__DOS__) && !defined (__OS2__)
#  define WRITE_OUTPUT
#endif

#define TYPE_STR        1
#define TYPE_STRP       2
#define TYPE_FUNC       3
#define TYPE_FUNC2      4
#define TYPE_CHAR       5
#define TYPE_BYTE       6
#define TYPE_SHORT      7
#define TYPE_INT        8
#define TYPE_LONG       9
#define TYPE_DATE       10
#define TYPE_TIME       11

char *month[12] =
{
    "Jan","Feb","Mar","Apr","May","Jun",
    "Jul","Aug","Sep","Oct","Nov","Dec"
};

int wherex, wherey;

typedef void (*OUTPUT_FUNC)(char *);
typedef void (*OUTPUT_FUNC2)(char *, char *);

FILE *Fout;

int ttynum;

static char CR_str[] = "\r\n";
static char PLAIN_CR_str[] = "\r";
static char CLR_str[] = "\x1b[1;1H\x1b[2J";
static char CLREOL_str[] = "\x1b[K";
static char insline_str[] = "\x1b[L";
static char delline_str[] = "\x1b[M";

static int old_fg, old_bg, cur_fg, cur_bg;
static int lines, more;

static int col_saved;

unsigned char chr_out[256];
unsigned char chr_in[256];

unsigned char local_in[256];
unsigned char local_out[256];

void time_func(char *output);
void msgdate_func(char *output);
void last_done_func(char *output);
void who_doing_func(char *output);

void clrscr_func(char *output);
void clreol_func(char *output);
void goto_func(char *output, char *parms);
void left_func(char *output, char *parms);
void right_func(char *output, char *parms);
void cup_func(char *output, char *parms);
void cdown_func(char *output, char *parms);
void scroll_down_func(char *output, char *parms);
void scroll_up_func(char *output, char *parms);
void inschar_func(char *output, char *parms);
void delchar_func(char *output, char *parms);
void bar_func(char *output, char *parms);
void restcol_func(char *output);
void vt100_func(char *output);

typedef struct
{
    char *name;
    char type;
    void *ptr;
}
MACRO_TYPE;

MACRO_TYPE macro_str[] =
{
    { "CR", TYPE_STR, CR_str },
    { "C", TYPE_STR, PLAIN_CR_str },
    { "OS", TYPE_STR, OP_SYSTEM },
    { "VER", TYPE_STR, BBS_VERSION },
    { "TIME", TYPE_FUNC, &time_func },
    { "TMP1", TYPE_STR, tmp1_str },
    { "TMP2", TYPE_STR, tmp2_str },
    { "NUM1", TYPE_LONG, &num1 },
    { "NUM2", TYPE_LONG, &num2 },
    { "NUM3", TYPE_LONG, &num3 },
    { "NUM4", TYPE_LONG, &num4 },
    { "NUM5", TYPE_LONG, &num5 },
    { "TIME_LEFT", TYPE_INT, &time_left },
    { "FILE_NAME", TYPE_STRP, &frec.name },
    { "FILE_DATE", TYPE_STR, frec.date },
    { "FILE_SIZE", TYPE_LONG, &frec.size },
    { "FILE_DOWNS", TYPE_SHORT, &frec.downs },
    { "TEMP_FAREA_NAME", TYPE_STR, hold_farea.name },
    { "TEMP_FAREA_PATH", TYPE_STR, hold_farea.path },
    { "TEMP_FAREA_DESC", TYPE_STR, hold_farea.description },
    { "FAREA_NAME", TYPE_STR, farea.name },
    { "FAREA_PATH", TYPE_STR, farea.path },
    { "FAREA_DESC", TYPE_STR, farea.description },
    { "FAREA_HISTORY", TYPE_STR, farea_history },
    { "MAREA_NUM", TYPE_LONG, &msg_area.number },
    { "MAREA_NAME", TYPE_STR, msg_area.name },
    { "MAREA_PATH", TYPE_STR, msg_area.path },
    { "MAREA_DESC", TYPE_STR, msg_area.description },
    { "THREAD_NUM", TYPE_LONG, &cur_thread_num },
    { "THREAD_NAME", TYPE_STR, cur_thread_name },
    { "THREAD_RESTR", TYPE_STR, cur_thread_restr },
    { "MSG_FROM", TYPE_STR, msg_from },
    { "MSG_TO", TYPE_STR, msg_to },
    { "MSG_SUBJ", TYPE_STR, msg_subj },
    { "MSG_NUM", TYPE_LONG, &msg_num },
    { "MSG_DATE", TYPE_FUNC, msgdate_func },
    { "MSG_LINES", TYPE_LONG, &msg_txtlines },
    { "ANSWER", TYPE_STR, answer },
    { "PARMS", TYPE_STR, parms },
    { "NODE", TYPE_INT, &nodenum },
    { "PORT", TYPE_STR, devname },
    { "HANDLE", TYPE_LONG, &hCom },
    { "BPSRATE", TYPE_LONG, &bpsrate },
    { "USER_NAME", TYPE_STR, usrsub.name },
    { "USER_ALIAS", TYPE_STR, usrsub.alias },
    { "USER_ADDR1", TYPE_STR, usrsub.address1 },
    { "USER_ADDR2", TYPE_STR, usrsub.address2 },
    { "USER_ADDR3", TYPE_STR, usrsub.address3 },
    { "USER_ADDR4", TYPE_STR, usrsub.address4 },
    { "USER_ADDR5", TYPE_STR, usrsub.address5 },
    { "USER_VOICE", TYPE_STR, usrsub.voice },
    { "USER_DATA", TYPE_STR, usrsub.data },
    { "USER_CITY", TYPE_STR, usrsub.city },
    { "USER_BIRTHDAY", TYPE_STR, usrsub.birthday },
    { "USER_SCRLEN", TYPE_BYTE, &user.ScreenLen },
    { "USER_SCRWIDTH", TYPE_BYTE, &user.ScreenWidth },
    { "USER_FIRSTTIME", TYPE_DATE, &user.FirstTime },
    { "USER_LASTTIME1", TYPE_DATE, &user.LastTime[0] },
    { "USER_LASTTIME2", TYPE_DATE, &user.LastTime[1] },
    { "USER_LASTTIME3", TYPE_DATE, &user.LastTime[2] },
    { "USER_LASTTIME4", TYPE_DATE, &user.LastTime[3] },
    { "USER_LASTTIME5", TYPE_DATE, &user.LastTime[4] },
    { "USER_CALLS", TYPE_LONG, &user.TotalCalls },
    { "USER_MINS", TYPE_LONG, &user.TotalMinutes },
    { "USER_PAGES", TYPE_LONG, &user.TotalPages },
    { "USER_UPBYTES", TYPE_LONG, &user.UploadBytes },
    { "USER_UPFILES", TYPE_LONG, &user.UploadFiles },
    { "USER_DOWNBYTES", TYPE_LONG, &user.DownloadBytes },
    { "USER_DOWNFILES", TYPE_LONG, &user.DownloadFiles },
    { "USER_TODAYCALLS", TYPE_LONG, &user.TodayCalls },
    { "USER_TODAYMINS", TYPE_LONG, &user.TodayMinutes },
    { "USER_TODAYPAGES", TYPE_LONG, &user.TodayPages },
    { "USER_TODAYUPBYTES", TYPE_LONG, &user.TodayUploadBytes },
    { "USER_TODAYUPFILES", TYPE_LONG, &user.TodayUploadFiles },
    { "USER_TODAYDOWNBYTES", TYPE_LONG, &user.TodayDownloadBytes },
    { "USER_TODAYDOWNFILES", TYPE_LONG, &user.TodayDownloadFiles },
    { "USER_PROTOCOL", TYPE_CHAR, &user.Protocol },
    { "USER_EDITOR", TYPE_CHAR, &user.Editor },
    { "USER_PACKER", TYPE_CHAR, &user.Packer },
    { "USER_CHARSET", TYPE_CHAR, &user.CharSet },
    { "USER_CHARSET_NAME", TYPE_STR, current_charset },
    { "USER_LANGUAGE", TYPE_CHAR, &user.Language },
    { "USER_EMULATION", TYPE_CHAR, &user.Emulation },
    { "USER_EMULATION_NAME", TYPE_STR, current_emulation },
    { "USER_VT100KEYB", TYPE_FUNC, vt100_func },
    { "WHO_NAME", TYPE_STR, noderec.name },
    { "WHO_CITY", TYPE_STR, noderec.city },
    { "WHO_LOGIN", TYPE_TIME, &noderec.login },
    { "WHO_BPS", TYPE_LONG, &noderec.BPS },
    { "WHO_DOING", TYPE_FUNC, who_doing_func },
    { "LAST_NODE", TYPE_SHORT, &lastrec.node },
    { "LAST_NAME", TYPE_STR, lastrec.name },
    { "LAST_ALIAS", TYPE_STR, lastrec.alias },
    { "LAST_CITY", TYPE_STR, lastrec.city },
    { "LAST_BPSRATE", TYPE_LONG, &lastrec.bpsrate },
    { "LAST_LOGIN", TYPE_TIME, &lastrec.login },
    { "LAST_LOGOFF", TYPE_TIME, &lastrec.logoff },
    { "LAST_DONE", TYPE_FUNC, last_done_func },
    { "PACKER_NAME", TYPE_STR, user_packer.name },
    { "PACKER_KEY", TYPE_CHAR, &user_packer.key },
    { "PACKER_ID", TYPE_STR, user_packer.id },
    { "PACKER_PACK", TYPE_STRP, &user_packer.pack },
    { "PACKER_UNPACK", TYPE_STRP, &user_packer.unpack },
    { "PROTO_NAME", TYPE_STR, user_proto.name },
    { "PROTO_KEY", TYPE_CHAR, &user_proto.key },
    { "PROTO_EFFICIENCY", TYPE_INT, &user_proto.efficiency },
    { "PROTO_DOWNLOAD", TYPE_STRP, &user_proto.download },
    { "PROTO_UPLOAD", TYPE_STRP, &user_proto.upload },
    { "PROTO_DLCHAR", TYPE_CHAR, &user_proto.dlchar },
    { "LANG_NAME", TYPE_STR, user_lang.name },
    { "LANG_KEY", TYPE_CHAR, &user_lang.key },
    { "LANG_FNAME", TYPE_STRP, &user_lang.fname },
    { "ANSIPATH", TYPE_STRP, &user_lang.ansipath },
    { "MENUPATH", TYPE_STRP, &user_lang.menupath },
    { "MAINMENU", TYPE_STRP, &user_lang.mainmenu },

    /* "commands" */
    { "CLR", TYPE_FUNC, clrscr_func },
    { "CLREOL", TYPE_FUNC, clreol_func },
    { "LEFT", TYPE_FUNC2, left_func },
    { "RIGHT", TYPE_FUNC2, right_func },
    { "UP", TYPE_FUNC2, cup_func },
    { "DOWN", TYPE_FUNC2, cdown_func },
    { "GOTO", TYPE_FUNC2, goto_func },
    { "SCROLL_DOWN", TYPE_FUNC2, scroll_down_func },
    { "SCROLL_UP", TYPE_FUNC2, scroll_up_func },
    { "INSCHR", TYPE_FUNC2, inschar_func },
    { "DELCHR", TYPE_FUNC2, delchar_func },
    { "INSLINE", TYPE_STR, insline_str },
    { "DELLINE", TYPE_STR, delline_str },
    { "BAR", TYPE_FUNC2, bar_func },
    { "MON", TYPE_STR, "\x02" },
    { "MOFF", TYPE_STR, "\x03" },
    { "SCOL", TYPE_STR, "\x04" },
    { "RCOL", TYPE_FUNC, restcol_func },
};

#ifndef __DOS__
unsigned short *screen;
#define _fmemcpy memcpy
#else
#ifdef __386__
unsigned short *screen = (unsigned short *) 0xb8000;
#else
unsigned short far *screen = (unsigned short far *) 0xb8000000;
#endif
#endif

int scrlength, scrwidth;

void gotoxy(int x, int y)
{
#ifdef __DOS__
    union REGS regs;

    regs.hireg.dx = (unsigned short) ((y << 8) + x - 0x0101);
    regs.loreg.bh = 0;
    regs.loreg.ah = 2;
    intr(0x10,&regs,&regs);
#elif defined (__OS2__)
    VioSetCurPos((unsigned short) (y-1),(unsigned short) (x-1),0);
#else
    x = y;
#endif
}

void tclrscr(void)
{
#ifdef __OS2__
    char cell[3] = { ' ', 7, 0 };
#endif
    int nro;

    for (nro=0; nro<scrwidth*scrlength; nro++)
        screen[nro] = (7<<8)+' ';
#ifdef __OS2__
    VioScrollUp(0,0,65535,65535,65535,cell,0);
    VioSetCurPos(0,0,0);
#endif

    gotoxy(1,1);
}

void tbar(int x1, int y1, int x2, int y2, unsigned col)
{
    int x,y,addr;
#ifdef __OS2__
    char cell[3] = { ' ', 7, 0 };
    cell[1] = (char) col;
#endif

    for (y=y1-1; y<y2; y++)
    {
#ifdef __OS2__
        VioWrtNCell(cell,(unsigned short) (x2-x1+1),(unsigned short) y,(unsigned short) (x1-1),0);
#endif
        addr = (x1-1)+y*scrwidth;
        for (x=x1-1; x<=x2-1; x++)
            screen[addr++] = (unsigned short) ((col << 8) + ' ');
    }
}

void writechr(int x, int y, unsigned char chr, unsigned col)
{
#ifdef __OS2__
    VioWrtCharStrAtt((char *) &chr, 1, (unsigned short) (y-1),
                     (unsigned short) (x-1), (char *) &col, 0);
#endif
    screen[(y-1)*scrwidth + (x-1)] = (unsigned short) ((col << 8) + chr);
}

void scroll_up(int x1, int y1, int x2, int y2)
{
#ifdef __OS2__
    char cell[3] = { ' ', 7, 0 };
#endif
    int y;

    for (y=y1-1; y<=y2-2; y++)
        _fmemcpy(screen+x1-1+y*scrwidth,screen+x1-1+(y+1)*scrwidth,(x2-x1+1)*2);
#ifdef __OS2__
    VioScrollUp((unsigned short) (y1-1), (unsigned short) (x1-1),
                (unsigned short) (y2-1), (unsigned short) (x2-1), 1, cell, 0);
#endif
}

void scroll_down(int x1, int y1, int x2, int y2)
{
#ifdef __OS2__
    char cell[3] = { ' ', 7, 0 };
#endif
    int y;

    for (y=y2-1; y>=y1; y--)
        _fmemcpy(screen+x1-1+y*scrwidth,screen+x1-1+(y-1)*scrwidth,(x2-x1+1)*2);
#ifdef __OS2__
    VioScrollDn((unsigned short) (y1-1), (unsigned short) (x1-1),
                (unsigned short) (y2-1), (unsigned short) (x2-1), 1, cell, 0);
#endif
}

void scroll_left(int x1, int x2, int y)
{
#ifdef __OS2__
    char cell[3] = { ' ', 7, 0 };
#endif
    int x;

    y--;
#ifdef __OS2__
    VioScrollLf((unsigned short) y, (unsigned short) (x1-1), (unsigned short) y,
                (unsigned short) (x2-1), 1, cell, 0);
#endif
    for (x=x1-1; x<x2-1; x++)
        screen[x+y*scrwidth] = screen[x+1+y*scrwidth];
}

void scroll_right(int x1, int x2, int y)
{
#ifdef __OS2__
    char cell[3] = { ' ', 7, 0 };
#endif
    int x;

    y--;
#ifdef __OS2__
    VioScrollRt((unsigned short) y, (unsigned short) (x1-1),
                (unsigned short) y, (unsigned short) (x2-1), 1, cell, 0);
#endif
    for (x=x2-1; x>x1; x--)
        screen[x+y*scrwidth] = screen[x-2+y*scrwidth];
}

static char a_ansitab[8] = {0,4,2,6,1,5,3,7};
static char a_ansistr[256];
static int a_inansi,a_ansipos,a_textattr,a_wrap;
static int a_ypos,a_xpos,a_savex,a_savey;

static void ansi_draw(unsigned char c)
{
    int num, params;
    int paramarr[50];
    char *strp;

    if (a_inansi)
    {
        a_ansistr[a_ansipos] = c;
        a_ansipos++;
        if (a_ansipos == 1)
        {
            if (c != '[') a_inansi = 0;
            return;
        }
        if ((c >= '0' && c <= '9') || c == '?' || c == ';' || c == '=') return;

        /* Get parameters */
        params = 0; paramarr[0] = 0;
        strp = a_ansistr+1;
        while (*strp != '\0')
        {
            if (*strp == ';')
            {
                if (params+1 < sizeof(paramarr)/sizeof(paramarr[0])) params++;
                paramarr[params] = 0;
            }
            else if (*strp >= '0' && *strp <= '9')
                paramarr[params] = paramarr[params]*10 + (*strp-'0');
            else if (*strp != '=' && *strp != '?')
                break;
            strp++;
        }

        switch (c)
        {
            case 'A':
                if (params == 0) num = 1; else num = paramarr[0];
                if (a_ypos-num < 1) a_ypos = 1; else a_ypos -= num;
                wherey = a_ypos;
                break;
            case 'B':
                if (a_ypos < user.ScreenLen)
                {
                    if (params == 0) num = 1; else num = paramarr[0];
                    if (a_ypos+num > user.ScreenLen) a_ypos = user.ScreenLen; else a_ypos += num;
                    wherey = a_ypos;
                }
                break;
            case 'C':
                if (params == 0) num = 1; else num = paramarr[0];
                if (a_xpos+num > scrwidth) a_xpos = scrwidth; else a_xpos += num;
                wherex = a_xpos;
                break;
            case 'D':
                if (params == 0) num = 1; else num = paramarr[0];
                if (a_xpos-num < 1) a_xpos = 1; else a_xpos -= num;
                wherex = a_xpos;
                break;
            case 'H':
            case 'f':
                if (params == 0)
                {
                    wherex = a_xpos = 1;
                    wherey = a_ypos = 1;
                }
                else
                {
                    if (params == 1)
                    {
                        a_ypos = paramarr[0];
                        a_xpos = 1;
                    }
                    else
                    {
                        a_ypos = paramarr[0];
                        a_xpos = paramarr[1];
                    }
                    if (a_xpos < 1) a_xpos = 1;
                    if (a_xpos > user.ScreenWidth) a_xpos = user.ScreenWidth;
                    if (a_ypos < 1) a_ypos = 1;
                    if (a_ypos > user.ScreenLen) a_ypos = user.ScreenLen;
                    wherey = a_ypos;
                    wherex = a_xpos;
                }
                break;
            case 'm':
                for (num=0; num<params; num++)
                {
                    switch (paramarr[num])
                    {
                        case 0:
                            a_textattr = 7;
                            break;
                        case 1:
                            a_textattr |= 8;
                            break;
                        case 5:
                            a_textattr |= 128;
                            break;
                        case 7:
                            a_textattr = 7 << 4;
                            break;
                        case 8:
                            a_textattr = 0;
                            break;
                        case 30:
                        case 31:
                        case 32:
                        case 33:
                        case 34:
                        case 35:
                        case 36:
                        case 37:
                            a_textattr &= 248;
                            a_textattr |= a_ansitab[paramarr[num]-30];
                            break;
                        case 40:
                        case 41:
                        case 42:
                        case 43:
                        case 44:
                        case 45:
                        case 46:
                        case 47:
                            a_textattr &= 15+128;
                            a_textattr |= a_ansitab[paramarr[num]-40] << 4;
                            break;
                    }
                }
                break;
            case 'J':
                if ((params == 1) && (paramarr[0] == 2))
                {
                    /*wherey = a_ypos = 1; wherex = a_xpos = 1;*/
                    tclrscr();
                }
                break;
            case 'K':
                if (a_xpos < scrwidth) tbar(a_xpos,a_ypos,scrwidth,a_ypos,a_textattr);
                break;
            case 'L':
                /* Insert line */
                scroll_down(1,a_ypos,scrwidth,user.ScreenLen);
                tbar(1,a_ypos,scrwidth,a_ypos,a_textattr);
                break;
            case 'M':
                /* Delete line */
                scroll_up(1,a_ypos,scrwidth,user.ScreenLen);
                tbar(1,user.ScreenLen,scrwidth,user.ScreenLen,a_textattr);
                break;
            case '@':
                /* Insert character */
                scroll_right(a_xpos,scrwidth,a_ypos);
                writechr(a_xpos,a_ypos,' ',a_textattr);
                break;
            case 'P':
                /* Delete character */
                scroll_left(a_xpos,scrwidth,a_ypos);
                writechr(scrwidth,a_ypos,' ',a_textattr);
                break;
            case 's':
                a_savex = a_xpos;
                a_savey = a_ypos;
                break;
            case 'u':
                wherex = a_xpos = a_savex;
                wherey = a_ypos = a_savey;
                break;
            case 'h':
                if ((params == 1) && (paramarr[0] == 7)) a_wrap = 1;
                break;
            case 'l':
                if ((params == 1) && (paramarr[0] == 7)) a_wrap = 0;
                break;
        }
        a_inansi = 0;
        return;
    }

    switch (c)
    {
        case 8:
            if (a_xpos > 1)
            {
                a_xpos--;
                wherex = a_xpos;
            }
            break;
        case 13:
            wherex = a_xpos = 1;
            break;
        case 10:
        __10:
            if (a_ypos < user.ScreenLen)
            {
                a_ypos++; wherey = a_ypos;
            }
            else
            {
                scroll_up(1,1,user.ScreenWidth,user.ScreenLen);
                tbar(1,user.ScreenLen,user.ScreenWidth,user.ScreenLen,a_textattr);
                gotoxy(a_xpos,a_ypos);
            }
            break;
        case 27:
            a_inansi = 1;
            a_ansipos = 0;
            break;
        default:
            writechr(a_xpos,a_ypos,c,(char) a_textattr);
            if (a_xpos < user.ScreenWidth)
            {
                a_xpos++; wherex = a_xpos;
            }
            else
            {
                if (a_wrap)
                {
                    wherex = a_xpos = 1;
                    goto __10;
                }
            }
    }
    return;
}

int set_color(char *output)
{
    char ansi_tab[8] = { '0','4','2','6','1','5','3','7' };
    int pos;

    /* Do we need to change color? */
    if (cur_fg == old_fg && cur_bg == old_bg) return 0;

    /* Ascii emulation? */
    if (user.Emulation == USER_EMULATION_ASCII || user.Emulation == 99) return 0;

    strcpy(output, "\x1b[");
    pos = 2;
    if ((old_fg & 8 && (cur_fg & 8) == 0) || (old_bg & 8 && (cur_bg & 8) == 0))
    {
        /* New foreground intensity */
        strcpy(output+pos, "0;");
        old_fg = 7;
        old_bg = 0;
        pos += 2;
    }

    if (cur_fg != old_fg)
    {
        if ((old_fg & 8) == 0 && cur_fg & 8)
        {
            /* New foreground intensity */
            strcpy(output+pos, "1;");
            pos += 2;
        }
        /* New foreground color */
        sprintf(output+pos, "3%c;", ansi_tab[cur_fg & 7]);
        pos += 3;
        old_fg = cur_fg;
    }

    if (cur_bg != old_bg)
    {
        if ((old_bg & 8) == 0 && cur_bg & 8)
        {
            /* New background intensity */
            strcpy(output+pos, "5;");
            pos += 2;
        }
        /* New background color */
        sprintf(output+pos, "4%c;", ansi_tab[cur_bg & 7]);
        pos += 3;
        old_bg = cur_bg;
    }

    output[pos-1] = 'm';
    return pos;
}

/* Convert macros */
char *conv_macros(char *input, char *output)
{
    int inmacro, macropos, num, slen, point, fg, bg;
    char macro[41], *strp, tmp[256], opt;
    struct tm *tim;

    inmacro = 0; macropos = 0;
    while (*input)
    {
        if (inmacro > 0)
        {
            /* @ char found.. */
            if (macropos == 2 && macro[0] == 'X' && *input != '@')
            {
                /* It's a color macro */
                macro[macropos] = *input;
                macropos++;

                bg = -1;
                if (macro[1] >= '0' && macro[1] <= '9')
                {
                    /* 0..9 */
                    bg = macro[1]-'0';
                }
                else if (macro[1] >= 'A' && macro[1] <= 'F')
                {
                    /* 10..15 */
                    bg = macro[1]-'A'+10;
                }

                if (bg > -1)
                {
                    fg = -1;
                    if (macro[2] >= '0' && macro[2] <= '9')
                    {
                        /* 0..9 */
                        fg = macro[2]-'0';
                    }
                    else if (macro[2] >= 'A' && macro[2] <= 'F')
                    {
                        /* 10..15 */
                        fg = macro[2]-'A'+10;
                    }

                    if (fg > -1)
                    {
                        /* Colors ok. */
                        cur_fg = fg;
                        cur_bg = bg;
                        inmacro = 0;
                    }
                }
                input++;
                continue;
            }

            /* Set color */
            output += set_color(output);

            if (*input == '@')
            {
                macro[macropos] = '\0';
                if (macropos == 0)
                {
                    /* @@ -> @ */
                    *output++ = '@';
                    inmacro = 0;
                    input++;
                    continue;
                }
                if (inmacro == 2)
                {
                    /* Macro too long */
                    output += sprintf(output, "%s@", macro);
                    inmacro = 0;
                    input++;
                    continue;
                }

                if (macro[0] == '_' && macro[strlen(macro)-1] == '_')
                {
                    /* Hidden macro */
                    inmacro = 0;
                    input++;
                    continue;
                }

                /* Check if '.' or ':' is found */
                strp = macro; point = 0;
                while (*strp != '\0')
                {
                    if (*strp == '.')
                    {
                        /* '.' found */
                        *(strp++) = '\0';
                        point = 1;
                        break;
                    }
                    if (*strp == ':')
                    {
                        /* ':' found */
                        *(strp++) = '\0';
                        point = 2;
                        break;
                    }
                    strp++;
                }

                opt = '\0';

                for (num=0; num < (int) (sizeof(macro_str)/sizeof(macro_str[0])); num++)
                {
                    if (stricmp(macro_str[num].name,macro) == 0)
                    {
                        switch (macro_str[num].type)
                        {
                            case TYPE_STR:
                                strcpy(tmp,(char *) macro_str[num].ptr);
                                break;
                            case TYPE_STRP:
                                strcpy(tmp,*((char **) macro_str[num].ptr));
                                break;
                            case TYPE_FUNC:
                                ((OUTPUT_FUNC) macro_str[num].ptr)(tmp);
                                break;
                            case TYPE_FUNC2:
                                ((OUTPUT_FUNC2) macro_str[num].ptr)(tmp, strp);
                                strp = NULL; point = 0;
                                break;
                            case TYPE_CHAR:
                                sprintf(tmp,"%c",*((char *) macro_str[num].ptr));
                                break;
                            case TYPE_BYTE:
                                sprintf(tmp,"%u",*((unsigned char *) macro_str[num].ptr));
                                break;
                            case TYPE_SHORT:
                                sprintf(tmp,"%u",*((unsigned short *) macro_str[num].ptr));
                                break;
                            case TYPE_INT:
                                sprintf(tmp,"%u",*((unsigned *) macro_str[num].ptr));
                                break;
                            case TYPE_LONG:
                                sprintf(tmp,"%lu",*((unsigned long *) macro_str[num].ptr));
                                break;
                            case TYPE_DATE:
                                tim = localtime((time_t *) macro_str[num].ptr);
                                sprintf(tmp,"%02d %s %02d  %02d:%02d:%02d",
                                        tim->tm_mday,month[tim->tm_mon],tim->tm_year % 100,
                                        tim->tm_hour,tim->tm_min,tim->tm_sec);
                                break;
                            case TYPE_TIME:
                                tim = localtime((time_t *) macro_str[num].ptr);
                                sprintf(tmp,"%02d:%02d",tim->tm_hour,tim->tm_min);
                                break;
                        }

                        /* If '.' or ':' chars were in macro.. */
                        if (strp != NULL)
                        {
                            num = strlen(strp)-1;
                            if (isdigit(strp[num]) == 0)
                            {
                                opt = strp[num];
                                strp[num] = '\0';
                            }

                            if (sscanf(strp, "%d", &num) == 1)
                            {
                                switch (opt)
                                {
                                    case 'R':
                                        /* Reverse */
                                        slen = strlen(tmp);
                                        if (num > slen)
                                        {
                                            memmove(tmp+num-slen, tmp, slen+1);
                                            memset(tmp, ' ', num-slen);
                                        }
                                        break;
                                    default:
                                        /* Any other */
                                        switch (point)
                                        {
                                            case 1:
                                                tmp[num] = '\0';
                                                break;
                                            case 2:
                                                slen = strlen(tmp);
                                                if (num > slen) memset(tmp+strlen(tmp),' ',num-slen);
                                                tmp[num] = '\0';
                                                break;
                                        }
                                        break;
                                }
                            }
                        }

                        strcpy(output,tmp);
                        output += strlen(tmp);

                        num = -1;
                        break;
                    }
                }

                if (num != -1)
                {
                    /* Macro not found */
                    switch (point)
                    {
                        case 1:
                            *(--strp) = '.';
                            break;
                        case 2:
                            *(--strp) = ':';
                            break;
                    }
                    if (opt != '\0')
                    {
                        num = strlen(strp);
                        strp[num] = opt;
                        strp[num] = '\0';
                    }
                    output += sprintf(output, "@%s@", macro);
                }
                inmacro = 0;
            }
            else
            {
                if (macropos == sizeof(macro)-1)
                {
                    /* Macro too long */
                    macro[macropos] = '\0';
                    if (inmacro == 2)
                    {
                        strcpy(output, macro);
                        output += strlen(macro);
                    }
                    else
                    {
                        output += sprintf(output, "@%s", macro);
                        inmacro = 2;
                    }
                    macropos = 0;
                }
                macro[macropos] = *input;
                macropos++;
            }
            input++;
            continue;
        }

        if (*input == '@')
        {
            inmacro = 1;
            macropos = 0;
        }
        else
        {
            /* Set new color */
            output += set_color(output);

            /* Here goes normal character.. */
            *output = *input;
            output++;
        }

        input++;
    }

    if (inmacro)
    {
        macro[macropos] = '\0';
        output += sprintf(output, "@%s", macro);
    }

    *output = '\0';
    return output;
}

static char writebuf[500], localbuf[500];
static int bufpos = 0;

#ifdef WRITE_OUTPUT

#define flushbuf() \
    { \
        mdm_dataout(writebuf, bufpos); \
        if (Fout != NULL) fwrite(localbuf, bufpos, 1, Fout); \
        bufpos = 0; \
    }
#else

#define flushbuf() \
    { \
        mdm_dataout(writebuf, bufpos); \
        bufpos = 0; \
    }

#endif

int output_modem_chr(unsigned char chr)
{
    int ret;
    char str[100];

    /* Put character in buffer */
    writebuf[bufpos] = chr_out[chr];
    localbuf[bufpos] = local_out[chr];
    bufpos++;

    if (bufpos == sizeof(writebuf))
        flushbuf();
    if (chr == '\n')
    {
        if (more/* && wherey == user.ScreenLen*/)
        {
            if (lines++ == user.ScreenLen-1)
            {
                flushbuf();
                lines = 1;
                ret = yes_no(lang[LANG_MORE_PROMPT]+3, 1, NULL);
                outchr('\r');
                bufpos = (lang[LANG_MORE_PROMPT][0]-'0')*10 + lang[LANG_MORE_PROMPT][1]-'0';
                memset(str, ' ', bufpos); str[bufpos] = '\r'; str[bufpos+1] = '\0'; bufpos = 0;
                outtext(str);
                if (!ret) return 0;
            }
        }
    }

    return 1;
}

int check_special(char chr)
{
    switch (chr)
    {
        case 1:
            // Wait for enter ..
            return 0;
        case 2:
            /* More on */
            more = 1; lines = 1;
            return 1;
        case 3:
            /* More off */
            more = 0;
            return 1;
        case 4:
            /* Save color */
            col_saved = cur_fg + (cur_bg << 4);
            return 1;
    }

    return 0;
}

/* Output data to screen and modem */
int output(char *format, ...)
{
    char tmp[512],tmp2[256],*strp;
    va_list arglist;

    if (!carrier) return 0;

    /* Put text to 'tmp' */
    va_start(arglist,format);
    vsprintf(tmp,format,arglist);
    va_end(arglist);

    conv_macros(tmp,tmp2);

    strp = tmp2; bufpos = 0;
    while (*strp != '\0')
    {
        if (*strp < 32 && check_special(*strp))
        {
            strp++;
            continue;
        }
        ansi_draw(local_out[(unsigned char) *strp]);
        if (!output_modem_chr((unsigned char) *strp))
        {
            gotoxy(wherex, wherey);
            return 0;
        }
        strp++;
    }
    gotoxy(wherex, wherey);
    if (bufpos > 0) flushbuf();

    return 1;
}

/* Output data to screen */
void loc_output(char *format, ...)
{
#ifdef WRITE_OUTPUT
    char ch;
#endif
    char tmp[512], tmp2[256], *strp;
    int fg, bg, ofg, obg;
    va_list arglist;

    /* Put text to 'tmp' */
    va_start(arglist,format);
    vsprintf(tmp,format,arglist);
    va_end(arglist);

    fg = cur_fg; bg = cur_bg; ofg = old_fg; obg = old_bg;
    conv_macros(tmp,tmp2);
 
    strp = tmp2;
    while (*strp != '\0')
    {
        ansi_draw(local_out[(unsigned char) *strp]);
#ifdef WRITE_OUTPUT
        ch = local_out[(unsigned char) *strp];
        if (Fout != NULL) fwrite(&ch, 1, 1, Fout);
#endif
        strp++;
    }
    gotoxy(wherex, wherey);
    cur_fg = fg; cur_bg = bg; old_fg = ofg; old_bg = obg;
}

void outchr(char chr)
{
    char color[30], *strp;

    if (!carrier) return;
    if (chr < 32 && check_special(chr)) return;

    bufpos = 0;
    color[set_color(color)] = '\0'; strp = color;
    while (*strp != '\0')
    {
        ansi_draw(local_out[(unsigned char) *strp]);
        output_modem_chr((unsigned char) *strp);
        strp++;
    }

    ansi_draw(local_out[(unsigned char) chr]);
    gotoxy(wherex, wherey);
    output_modem_chr((unsigned char) chr);
    if (bufpos > 0) flushbuf();
}

void loc_outchr(char chr)
{
#ifdef WRITE_OUTPUT
    char ch;
#endif

    ansi_draw(local_out[(unsigned char) chr]);
    gotoxy(wherex, wherey);
#ifdef WRITE_OUTPUT
    chr = local_out[(unsigned char) chr];
    if (Fout != NULL) fwrite(&ch, 1, 1, Fout);
#endif
}

int outtext(char *text)
{
    char color[30], *strp;

    if (!carrier) return 0;

    bufpos = 0;
    color[set_color(color)] = '\0'; strp = color;
    while (*strp != '\0')
    {
        ansi_draw(local_out[(unsigned char) *strp]);
        output_modem_chr((unsigned char) *strp);
        strp++;
    }

    strp = text;
    while (*strp != '\0')
    {
        if (*strp < 32 && check_special(*strp))
        {
            strp++;
            continue;
        }
        ansi_draw(local_out[(unsigned char) *strp]);
        if (!output_modem_chr((unsigned char) *strp))
        {
            gotoxy(wherex, wherey);
            return 0;
        }
        strp++;
    }
    gotoxy(wherex, wherey);
    if (bufpos > 0) flushbuf();

    return 1;
}

int init_output(void)
{
#ifdef WRITE_OUTPUT
    char dev[20];
#endif

#ifdef __linux__
    if (!KbdInit()) return 0;
#endif

#ifdef __OS2__
    VIOMODEINFO oldmode;

    oldmode.cb = sizeof(VIOMODEINFO);
    VioGetMode(&oldmode,0);
    scrlength = oldmode.row;
    scrwidth = oldmode.col;
#else
    scrlength = 25; scrwidth = 80;
#endif
    user.Emulation = USER_EMULATION_ASCII;
    user.ScreenLen = (unsigned char) scrlength; user.ScreenWidth = (unsigned char) scrwidth;

#ifndef __DOS__
    screen = (unsigned short *) _malloc(200*100*2);
#endif

    a_textattr = 7; a_inansi = 0; a_wrap = 1;
    wherex = a_xpos = 1;
    wherey = a_ypos = scrlength;

#ifdef WRITE_OUTPUT
    if (ttynum == 0)
        Fout = stdout;
    else
    {
        sprintf(dev, "/dev/tty%d", ttynum);
        Fout = fopen(dev, "w");
        dup2(fileno(Fout), STDOUT_FILENO);
        dup2(fileno(Fout), STDERR_FILENO);
    }
    if (Fout != NULL) setbuf(Fout,NULL);
#endif

    old_fg = -1; cur_fg = -1;
    old_bg = -1; cur_bg = -1;

    load_charset('L');
#ifdef __linux__
    /* Latin-1 charset in Linux */
    memcpy(local_in,chr_in,256); /* Keyboard input: Latin-1 -> IBMPC */
    memcpy(local_out,chr_out,256); /* Screen output: IBMPC -> Latin-1 */
    load_charset('I');
#else
    /* IBM charset with others than Linux */
    memcpy(local_out,chr_in,256); /* Screen output: Latin-1 -> IBMPC */
    load_charset('I');
    memcpy(local_in,chr_in,256); /* Keyboard input: No translation */
#endif
    return 1;
}

void deinit_output(void)
{
#ifndef __DOS__
    if (Fout != NULL) fclose(Fout);
#endif
    if (ttynum != 0)
    {
        freopen("/dev/tty", "w", stdout);
        freopen("/dev/tty", "w", stderr);
    }

#ifdef __linux__
    KbdDeInit();
#endif

#ifndef __DOS__
    _free(screen);
#endif
}

void time_func(char *output)
{
    time_t _time;
    struct tm *tmbuf;

    _time = time(NULL);
    tmbuf = localtime(&_time);
    sprintf(output, "%02d:%02d", tmbuf->tm_hour, tmbuf->tm_min);
}

void msgdate_func(char *output)
{
    struct tm *tmbuf;

    tmbuf = localtime((time_t *) &msg_date);
    sprintf(output,"%02d %s %02d  %02d:%02d:%02d",
            tmbuf->tm_mday,month[tmbuf->tm_mon],tmbuf->tm_year % 100,
            tmbuf->tm_hour,tmbuf->tm_min,tmbuf->tm_sec);
}

void clrscr_func(char *output)
{
    /* Do nothing with ASCII emulation */
    if (user.Emulation == USER_EMULATION_ASCII || user.Emulation == 0xff)
    {
        output[0] = 12;
        output[1] = '\0';
        return;
    }

    strcpy(output, CLR_str);
}

void clreol_func(char *output)
{
    /* Do nothing with ASCII emulation */
    if (user.Emulation == USER_EMULATION_ASCII || user.Emulation == 0xff)
    {
        output[0] = '\0';
        return;
    }

    strcpy(output, CLREOL_str);
}

void goto_func(char *output, char *parms)
{
    char *strp;
    int xpos,ypos;

    output[0] = '\0';

    /* Do nothing with ASCII emulation */
    if (user.Emulation == USER_EMULATION_ASCII || user.Emulation == 0xff) return;

    /* No parameters, quit */
    if (parms == NULL) return;

    /* No plot, quit */
    strp = strchr(parms,',');
    if (strp == NULL) return;
    *strp = ' ';

    if (sscanf(parms,"%d %d",&xpos,&ypos) != 2) return;

    if (ypos < 0 || xpos < 0)
        write_log("goto_func() : xpos = %d, ypos = %d", xpos, ypos);
    sprintf(output,"\x1b[%d;%dH", ypos, xpos);
}

void left_func(char *output, char *parms)
{
    int move;

    /* Do nothing with ASCII emulation */
    if (user.Emulation == USER_EMULATION_ASCII || user.Emulation == 0xff)
    {
        output[0] = '\0';
        return;
    }

    if (parms == NULL || sscanf(parms, "%d", &move) != 1) move = 1;
    if (move == 0) return;
    if (move == 1)
        strcpy(output, "\x1b[D");
    else
        sprintf(output, "\x1b[%dD", move);
}

void right_func(char *output, char *parms)
{
    int move;

    /* Do nothing with ASCII emulation */
    if (user.Emulation == USER_EMULATION_ASCII || user.Emulation == 0xff)
    {
        output[0] = '\0';
        return;
    }

    if (parms == NULL || sscanf(parms, "%d", &move) != 1) move = 1;
    if (move == 0) return;
    if (move == 1)
        strcpy(output, "\x1b[C");
    else
        sprintf(output, "\x1b[%dC", move);
}

void cup_func(char *output, char *parms)
{
    int move;

    /* Do nothing with ASCII emulation */
    if (user.Emulation == USER_EMULATION_ASCII || user.Emulation == 0xff)
    {
        output[0] = '\0';
        return;
    }

    if (parms == NULL || sscanf(parms, "%d", &move) != 1) move = 1;
    if (move == 0) return;
    if (move == 1)
        strcpy(output, "\x1b[A");
    else
        sprintf(output, "\x1b[%dA", move);
}

void cdown_func(char *output, char *parms)
{
    int move;

    /* Do nothing with ASCII emulation */
    if (user.Emulation == USER_EMULATION_ASCII || user.Emulation == 0xff)
    {
        output[0] = '\0';
        return;
    }

    if (parms == NULL || sscanf(parms, "%d", &move) != 1) move = 1;
    if (move == 0) return;
    if (move == 1)
        strcpy(output, "\x1b[B");
    else
        sprintf(output, "\x1b[%dB", move);
}

void scroll_down_func(char *output, char *parms)
{
    int ypos;

    if (user.Emulation != USER_EMULATION_ANSI_X364) return;

    /* No parameters, quit */
    if (parms == NULL) return;

    if (sscanf(parms,"%d",&ypos) != 1) return;

    sprintf(output,"\x1b[%d;1H\x1b[L",ypos);
}

void scroll_up_func(char *output, char *parms)
{
    int ypos;

    if (user.Emulation != USER_EMULATION_ANSI_X364) return;

    /* No parameters, quit */
    if (parms == NULL) return;

    if (sscanf(parms,"%d",&ypos) != 1) return;

    sprintf(output,"\x1b[%d;1H\x1b[M",ypos);
}

void who_doing_func(char *output)
{
    sprintf(output, node_doing[noderec.doing], noderec.data);
}

void inschar_func(char *output, char *parms)
{
    int num;

    if (user.Emulation != USER_EMULATION_ANSI_X364) return;

    if (parms == NULL || sscanf(parms,"%d",&num) != 1 || num == 1)
        strcpy(output,"\x1b[@");
    else
        sprintf(output,"\x1b[%d@",num);
}

void delchar_func(char *output, char *parms)
{
    int num;

    if (user.Emulation != USER_EMULATION_ANSI_X364) return;

    if (parms == NULL || sscanf(parms,"%d",&num) != 1 || num == 1)
        strcpy(output,"\x1b[P");
    else
        sprintf(output,"\x1b[%dP",num);
}

void bar_func(char *output, char *parms)
{
    char *strp;
    int num,pos,max;

    /* No parameters, quit */
    if (parms == NULL) return;

    /* No plot, quit */
    strp = strchr(parms,',');
    if (strp == NULL) return;
    *strp = ' ';

    if (sscanf(parms, "%d %d", &pos, &max) != 2) return;

    conv_macros("@X0C", output);
    output += strlen(output);
    for (num=0; num<max; num++)
    {
        if (num == pos)
        {
            conv_macros("@X08", output);
            output += strlen(output);
        }
        else if (num < pos && num == max/3)
        {
            conv_macros("@X0E", output);
            output += strlen(output);
        }
        else if (num < pos && num == max*2/3)
        {
            conv_macros("@X0A", output);
            output += strlen(output);
        }
        *output++ = '.';
    }
    *output = '\0';
}

void last_done_func(char *output)
{
    strcpy(output,"--------");
    if (lastrec.done_flags & DONE_DOWNLOAD) output[0] = 'D';
    if (lastrec.done_flags & DONE_UPLOAD) output[1] = 'U';
    if (lastrec.done_flags & DONE_READ) output[2] = 'R';
    if (lastrec.done_flags & DONE_WRITE) output[3] = 'W';
    if (lastrec.done_flags & DONE_MSGPKT) output[4] = 'Q';
    if (lastrec.done_flags & DONE_YELL) output[5] = 'Y';
    if (lastrec.done_flags & DONE_NEW_USER) output[6] = 'N';
    if (lastrec.done_flags & DONE_CARRIER_LOST) output[7] = 'C';
}

void load_charset(char key)
{
    int num;

    for (num=0; num<=255; num++) chr_out[num] = (unsigned char) num;
    for (num=0; num<=255; num++) chr_in[num] = (unsigned char) num;

    switch (key)
    {
        case 'I':
            /* IBM */
            chr_out[229] = 134;
            chr_out[228] = 132;
            chr_out[246] = 148;
            chr_out[197] = 143;
            chr_out[196] = 142;
            chr_out[214] = 153;
            break;
        case 'L':
            /* Latin-1 */
            chr_out[134] = 229;
            chr_out[132] = 228;
            chr_out[148] = 246;
            chr_out[143] = 197;
            chr_out[142] = 196;
            chr_out[153] = 214;
            chr_in[229] = 134;
            chr_in[228] = 132;
            chr_in[246] = 148;
            chr_in[197] = 143;
            chr_in[196] = 142;
            chr_in[214] = 153;
            break;
    }
}

void restcol_func(char *output)
{
    char str[5];

    sprintf(str, "@X%02X", col_saved);
    conv_macros(str, output);
}

void vt100_func(char *output)
{
    strcpy(output, (user.Attrib1 & USER_ATTRIB_VT100_KEYB) ? lang[LANG_YES] : lang[LANG_NO]);
}
