/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* quote.c - Quote lines */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "quote.h"
#include "files.h"
#include "msg_jam.h"

#define QUOTE_BBBS      1
#define QUOTE_IDIOT     2
#define QUOTE_GOLDED    3

static char oldline[256],oldquote[80],quotestr[256],output[256];
static char initials[11];
static FILE *Ftext;
static char quote_style;

/* Quoting startup stuff, get initials etc. */
void start_quoting(char *from)
{
    int ok,num,slen;

    if (quote_style == QUOTE_BBBS)
    {
        /* BBBS style quoting */
        strcpy(initials, " > ");
    }
    else
    {
        ok = 1;
        initials[0] = '\0'; slen = 0;

        /* Make initials */
        num = 0;
        while (from[num] != '\0')
        {
            if (ok && from[num] != ' ')
            {
                initials[slen] = (char) toupper(from[num]);
                if (slen < 4) slen++;
                ok = 0;
            }
            else
            {
                if (from[num] == ' ') ok = 1;
            }
            num++;
        }

        if (quote_style == QUOTE_IDIOT)
        {
            /* Idiot style quoting, TS>JS>TS>JS> */
            initials[slen] = '>'; initials[slen+1] = 0;
        }
        else
        {
            /* GoldED style quoting, TS>>>>> */
            initials[slen] = '>'; initials[slen+1] = ' '; initials[slen+2] = 0;
            memmove(initials+1,initials,slen+3);
            initials[0] = ' ';
        }
    }

    oldquote[0] = 0;
    oldline[0] = 0;
    quotestr[0] = 0;
}

/* Flush quote text buffers */
void finish_quoting(void)
{
    if (oldline[0] != 0)
    {
        fprintf(Ftext,"%s%s\r\n",initials,oldline);
    }

    if (quotestr[0] != 0)
    {
        if (quote_style != QUOTE_IDIOT)
        {
            if (oldquote[0] != ' ')
            {
                fprintf(Ftext," %s> %s\r\n",oldquote,quotestr);
            }
            else
            {
                fprintf(Ftext,"%s> %s\r\n",oldquote,quotestr);
            }
        }
        else
        {
            fprintf(Ftext,"%s%s%s\r\n",initials,oldquote,quotestr);
        }
    }
}

/* Quote one line */
void quote_line(char *linestr)
{
    int left,num,slen,addline,last;
    char *pstr;

    slen = strlen(linestr);
    left = QUOTE_MAX; last = -1;

    /* Check if line is already quoted */
    for (num=0; num<slen; num++)
    {
        if (linestr[num] == '\0')
        {
            /* It wasn't */
            break;
        }
        else
        {
            if ((linestr[num] == '"') || (linestr[num] == '\'') || (linestr[num] == '<'))
            {
                /* It wasn't */
                break;
            }
            else
            {
                if (linestr[num] == '>')
                {
                    /* Yeah, quote character found.. */
                    left = QUOTE_MAX;
                    last = num;
                }
                else
                {
                    /* It was quoted line */
                    left--;
                    if (left == 0) break;
                }
            }
        }
    }

    if (last > -1)
    {
        if (oldline[0] != 0)
        {
            fprintf(Ftext,"%s%s\r\n",initials,oldline);
            oldline[0] = 0;
        }

        if (quotestr[0] != 0)
        {
            /* Same quote initials as last line? */
            if (!((strncmp(oldquote,linestr,last+1) == 0) && (oldquote[last+1] == 0)))
            {
                /* Nah. */
                if (quote_style != QUOTE_IDIOT)
                {
                    if (oldquote[0] != ' ')
                    {
                        /* Quote text didn't start with space, so add it
                           there to make quote cleaner! :) */
                        sprintf(output," %s> %s",oldquote,quotestr);
                    }
                    else
                    {
                        sprintf(output,"%s> %s",oldquote,quotestr);
                    }
                }
                else
                {
                    /* Just be idiot and add initials before line */
                    sprintf(output,"%s%s%s",initials,oldquote,quotestr);
                }

                fprintf(Ftext,"%s\r\n",output);
                quotestr[0] = 0;
            }
        }

        strncpy(oldquote,linestr,last+1);
        oldquote[last+1] = 0;

        if (quote_style != QUOTE_IDIOT)
        {
            /* We're not idiot, we won't use idiot quoting... */
            if (oldquote[0] != ' ')
            {
                /* Space before quote.. */
                output[0] = ' ';
                strncpy(output+1,linestr,last+1);
                output[last+2] = 0;
            }
            else
            {
                strncpy(output,linestr,last+1);
                output[last+1] = 0;
            }

            /* Take off all spaces from start of line */
            num = 1;
            while (linestr[last+num] == ' ') num++;

            /* Put text to 'output' */
            if (quotestr[0] != '\0')
            {
                sprintf(output+strlen(output),"> %s %s",quotestr,linestr+last+num);
                quotestr[0] = '\0';
            }
            else
            {
                strcat(strcat(output,"> "),linestr+last+num);
            }
        }
        else
        {
            /* We're idiot and use idiot quoting.. */
            if (quotestr[0] != 0)
            {
                sprintf(output+strlen(output),"%s%s%s %s",initials,oldquote,quotestr,linestr+last+1);
                quotestr[0] = '\0';
            }
            else
            {
                strcat(strcpy(output,initials),linestr);
            }
        }

        for (;;)
        {
            slen = strlen(output);
            if (slen <= 79) break;

            /* Too long text line */
            pstr = output+slen-1;
            for (;;)
            {
                if (*pstr == ' ' && slen <= 79) break;
                pstr--;
                slen--;
                if (slen <= last+num+3)
                {
                    pstr = NULL;
                    break;
                }
            }

            if ((pstr == NULL) || ((int) (pstr-output) <= last+num+3))
            {
                strcpy(quotestr,output+79);
                output[79] = '\0';
                break;
            }

            *pstr = '\0';
            strcpy(quotestr,pstr+1);

            if (strlen(quotestr) <= 79) break;

            fprintf(Ftext,"%s\r\n",output);
            if ((oldquote[0] != ' ') && (quote_style != QUOTE_IDIOT))
            {
                sprintf(output," %s> %s",oldquote,quotestr);
            }
            else
            {
                sprintf(output,"%s> %s",oldquote,quotestr);
            }
        }

        fprintf(Ftext,"%s\r\n",output);
        return;
    }

    if (quotestr[0] != '\0')
    {
        if (quote_style != QUOTE_IDIOT)
        {
            if (oldquote[0] != ' ')
            {
                sprintf(output," %s> %s",oldquote,quotestr);
            }
            else
            {
                sprintf(output,"%s> %s",oldquote,quotestr);
            }
        }
        else
        {
            sprintf(output,"%s%s%s",initials,oldquote,quotestr);
        }

        fprintf(Ftext,"%s\r\n",output);
        quotestr[0] = '\0';
    }
    oldquote[0] = '\0';

    addline = 0;
    if (oldline[0] != 0)
    {
        sprintf(output,"%s%s %s",initials,oldline,linestr);
        oldline[0] = 0;
        if (linestr[0] == 0) addline = 1;
    }
    else
    {
        if (linestr[0] != 0)
        {
            strcat(strcpy(output,initials),linestr);
        }
        else
        {
            output[0] = '\0';
        }
    }

    for (;;)
    {
        slen = strlen(output);
        if (slen <= 79) break;

        pstr = output+slen-1; num = strlen(initials)+1;
        for (;;)
        {
            if (*pstr == ' ' && slen <= 79) break;
            pstr--;
            slen--;
            if (slen == num)
            {
                pstr = NULL;
                break;
            }
        }

        if (pstr == NULL)
        {
            /* No spaces found from line, so just split it .. */
            strcpy(oldline,output+79);
            output[79] = '\0';
            fprintf(Ftext,"%s\r\n",output);
            fprintf(Ftext,"%s%s\r\n",initials,oldline);
            if (addline) fprintf(Ftext,"\r\n");
            oldline[0] = '\0';
            return;
        }

        *pstr = '\0';
        strcpy(oldline,pstr+1);
        if (strlen(oldline) <= (79-strlen(initials))) break;

        fprintf(Ftext,"%s\r\n",output);
        if (addline)
        {
            fprintf(Ftext,"\r\n");
            addline = 0;
        }
        strcat(strcpy(output,initials),oldline);
    }

    fprintf(Ftext,"%s\r\n",output);
    if (addline) fprintf(Ftext,"\r\n");
    return;
}

/* Quote message text */
void quote_message(char *fname)
{
    char tmp[256];
    int slen;
    unsigned long num;

    /* Open message text */
    Ftext = fopen(fname, "w+b");
    if (Ftext == NULL) return;
    chmod(fname, 0660);

    msg_init_msgtext();
    start_quoting(msg_from);

    for (num=1; num<=msg_txtlines; num++)
    {
        /* Get text line */
        strcpy(tmp, msg_read_textline(num));

        /* Remove spaces after text line */
        slen = strlen(tmp);
        while (((tmp[slen-1] == 9) || (tmp[slen-1] == 32)) && (slen > 0)) slen--;
        tmp[slen] = '\0';

        if (strncmp(tmp,"---",3) == 0)
        {
            /* Tear line */
            if ((slen == 3) || (tmp[3] == ' ')) tmp[1] = '!';
        }

        if (strncmp(tmp," * Origin:",10) == 0)
        {
            /* Origin */
            tmp[1] = '!';
        }

        if (tmp[0] == '\x01')
        {
            /* Kludge */
            tmp[0] = '@';
        }
        quote_line(tmp);
    }

    finish_quoting();
    fclose(Ftext);
    msg_deinit_msgtext();
}
