/*
 * TXT2MED: Plain text echo definition to MED.FLD convertion utility
 *
 * Created: 31/Mar/95
 * Updated: 28/Nov/96
 *
 * Written by Pete Kvitek of JV Dialogue 1st BBS (2:5020/6)
 * Copyright (c) 1995-1996 by JV DIALOGUE. All rights reserved.
 *
 * Compilation notes:
 *
 *          - this code has been compiled with Borland's C++ v2.0 and
 *            with MSC 6.0a in a small memory model.
 *
 * Revision history:
 *
 * v1.05    01/Dec/96   -- added level control keys '>', '<', '+'
 *                         Andrey Bordachev 2:5020/102.3
 *
 * v1.04    28/Nov/96   -- added 0.40 FLDT and MA_ definitions
 *
 * v1.03    26/Jan/96   -- FF_ADDMSGID replaced with FF_ADDDOMAIN
 *
 * v1.02    20/Dec/95   -- added 3rd party program flags in folder template
 *
 * v1.01    18/Apr/95   -- ported to os/2 for msc
 *
 * v1.00    31/Mar/85   -- initially written
 *
 */

 // Pick up all the standard headers

 #include <io.h>
 #include <ctype.h>
 #include <fcntl.h>
 #include <share.h>
 #include <stdio.h>
 #include <stdlib.h>
 #include <string.h>
 #include <direct.h>
 #include <sys\types.h>
 #include <sys\stat.h>

/*
 * Module declarations
 */

 // Miscellaneous defines

 #define VERSION       "1.05"   // Software release version

 // Standard typed overtyping

 typedef unsigned short	USHORT;
 typedef unsigned long	ULONG;
 typedef unsigned char	BYTE;
 typedef unsigned int   BOOL;

 // Some useful declarations

 #define loop for (;;)

 #define TRUE	1
 #define FALSE	0

 // Netword address structure

 typedef struct _NETADDR {
   USHORT zone;
   USHORT net;
   USHORT node;
   USHORT point;
 } NETADDR;

  // Folder template structure. Note: folder definition file is
  // an array of the variable length folder template structures

  typedef struct _FLDT { 	/* fldt */
    USHORT  cb;			// Size of this structure
    USHORT  fs;             	// Folder flags (FF_)
    USHORT  iLevel;         	// Folder level index
    USHORT  cchName;		// Folder name length
    USHORT  cchArea;		// Folder area length
    USHORT  cchPath;		// Folder msgbase path length
    USHORT  cchOrig;		// Folder origin string length
    USHORT  cchTemp;		// Folder template path length
    NETADDR netAddr;		// Default originating address
    ULONG   attr;		// Default message attributes (MA_)
    USHORT  boardno;		// Hudson msgbase board number
    BYTE    reserved[26];	// Reserved for future extensions
    USHORT  fs3rdParty;		// Third party program flags (FF3_)
//
//  Following strings are stored without trailing null
//
//  CHAR    achName[cchName];	// Folder name
//  CHAR    achArea[cchArea];	// Folder area
//  CHAR    achPath[cchPath];	// Folder msgbase path
//  CHAR    achOrig[cchOrig];	// Folder origin string
//  CHAR    achTemp[cchTemp];	// Folder template path
  } FLDT, * PFLDT;

  // Folder flags in fldt.fs

#define FF_XMSGBASE  	 0x000f	// Msg base type mask
#define FF_MSGSDM   	 0x0000	// FTCS-001 *.MSG message base
#define FF_MSGSQUISH	 0x0001	// Squish *.SQ? message base
#define FF_MSGJAM   	 0x0002	// JAM *.J?? message base
#define FF_MSGHUDSON	 0x0003	// Hudson message base

#define FF_XFLDTYPE  	 0x0070	// Folder type mask
#define FF_NETMAIL       0x0010	// Netmail folder
#define FF_ECHOMAIL      0x0020	// Echomail folder
#define FF_GROUPMAIL     0x0040	// Group mail folder
#define FF_LOCALMAIL     0x0000	// Local mail folder

#define FF_COLLAPSED     0x0080 // Subfolders collapsed
#define FF_TWITFOLDER  	 0x0100	// Twit folder
#define FF_ADDINTL   	 0x0200	// Add ^aINTL kludge
#define FF_ADDDOMAIN 	 0x0400	// Add ^aDOMAIN kludge
#define FF_ADDORIGIN 	 0x0800	// Add origin line
#define FF_RANDORIGIN	 0x1000	// Random origin line
#define FF_SHOWTWITS 	 0x2000	// Always show twit messages
#define FF_NOSENTMAIL	 0x4000	// Don't save in SentMail

  // Third party program flags in fldt.fs3rdParty

#define FF3_SQ2MED_MASK	 0x0003	// Contact Dmitriy Grachjev (2:5020/328)

  // Message base type helpers

#define MSGBASETYPE(fs)  ((fs) & FF_XMSGBASE)

#define IS_MSG(fs)       (MSGBASETYPE(fs) == FF_MSGSDM)
#define IS_SQU(fs)       (MSGBASETYPE(fs) == FF_MSGSQUISH)
#define IS_JAM(fs)       (MSGBASETYPE(fs) == FF_MSGJAM)
#define IS_HUD(fs)       (MSGBASETYPE(fs) == FF_MSGHUDSON)

  // Folder type helpers

#define FOLDERTYPE(fs)   ((fs) & FF_XFLDTYPE)

#define IS_NETMAIL(fs)	 (FOLDERTYPE(fs) == FF_NETMAIL)
#define IS_ECHOMAIL(fs)	 (FOLDERTYPE(fs) == FF_ECHOMAIL)
#define IS_GROUPMAIL(fs) (FOLDERTYPE(fs) == FF_GROUPMAIL)
#define IS_LOCALMAIL(fs) (FOLDERTYPE(fs) == FF_LOCALMAIL)

 // Opus style message attributes

 #define MSGPRIVATE 0x0001
 #define MSGCRASH   0x0002
 #define MSGREAD    0x0004
 #define MSGSENT    0x0008
 #define MSGFILE    0x0010
 #define MSGFWD     0x0020
 #define MSGORPHAN  0x0040
 #define MSGKILL    0x0080
 #define MSGLOCAL   0x0100
 #define MSGHOLD    0x0200
 #define MSGXX2     0x0400
 #define MSGFRQ     0x0800
 #define MSGRRQ     0x1000
 #define MSGCPT     0x2000
 #define MSGARQ     0x4000
 #define MSGURQ     0x8000

  // MadMED message attributes in fldt.attr

#define MA_PRVT	 MSGPRIVATE	// Private                 +
#define MA_CRSH	 MSGCRASH       // Crash                   | These
#define MA_READ	 MSGREAD        // Read                    | flags are
#define MA_SENT	 MSGSENT        // Sent                    | standard for
#define MA_FILE	 MSGFILE        // File attach             | FidoNet(tm)
#define MA_TRAN	 MSGFWD         // Transit                 | message attr
#define MA_ORPH	 MSGORPHAN      // Orphan                  |
#define MA_KILL	 MSGKILL        // Kill/sent               |
#define MA_LOCL	 MSGLOCAL       // Local                   |
#define MA_HOLD	 MSGHOLD        // Hold                    |
#define MA_RSV2	 MSGXX2         // Reserved                |
#define MA_FREQ	 MSGFRQ         // File request            |
#define MA_RREQ	 MSGRRQ         // Return receipt request  |
#define MA_RRCP	 MSGCPT         // Confirmation receipt msg|
#define MA_AREQ	 MSGARQ         // Audit request           |
#define MA_UREQ	 MSGURQ         // File update request     +
#define MA_SCAN	 MSGSCANNED     // Squish scanned          - *.SQ? only
#define MA_VUID	 MSGUID         // Valid xmsg.umsgid       - *.SQ? only???
#define MA_FZON  0x80000000l 	// Zonegate route          +
#define MA_FHUB  0x40000000l 	// Hub/host route          | These flags are
#define MA_FIMM  0x20000000l 	// Immediate               | defined in the
#define MA_FDIR  0x10000000l 	// Direct                  | ^aFLAGS kludge
#define MA_FKFS  0x08000000l 	// Kill file sent          | proposed by
#define MA_FTFS  0x04000000l 	// Truncate file sent      | FSC-0053
#define MA_FRRQ  0x02000000l 	// Mail unpack receipt req |
#define MA_FCFM  0x01000000l 	// Mail read receipt req   |
#define MA_FLOK  0x00800000l 	// Locked message          |
#define MA_FAXS  0x00400000l 	// Archive/sent            |
#define MA_FXMA  0x00200000l 	// Xmail                   +
#define MA_FXXX  0xfff00000l	// Mask for the ^aFLAGS attributes

 // Module variables

 static char achTxtFile[256] = "NEWECHO.LST"; // Echo area definition file
 static char achFldFile[256] = "NEWECHO.FLD"; // MadMED folder file
 static char achLine[256];		      // Echo area definition line

/*
 * This subroutine displays a logo screen
 */

 static void DoShowLogo(void)
 {
   fprintf(stderr,
   "\n"
   "Text Echo Definition File to MadMED Folder Convertor v"VERSION"\n"
   "Written by Pete Kvitek of JV Dialogue 1st BBS, 2:5020/6\n"
   "Copyright (C) 1995-1996 by JV Dialogue. All rights reserved.\n"
   "\n"
   "Usage:    TXT2MED [newecho.lst] [newecho.fld]\n"
   "\n"
   );
 }

/*
 * This subroutine processes the command line parameters
 */

 static void DoProcCmdLine(short cArg, char * apszArg[])
 {
   // Display the logo screen

   DoShowLogo();

   // Get the text file name override if any

   if (cArg > 1) {
     strncpy(achTxtFile, apszArg[1], sizeof(achTxtFile) - 1);
     strupr(achTxtFile);
   }

   // Get the file name override if any

   if (cArg > 2) {
     strncpy(achFldFile, apszArg[2], sizeof(achFldFile) - 1);
     strupr(achFldFile);
   }
 }

/***************************************************************************
* This subroutine skips over the leading spaces and returns first nonspace
*/

  static char DoSkipSpaces(char ** ppch)
  {
    while (isspace(**ppch)) (*ppch)++;
    return **ppch;
  }

/*
 * Main...
 */

 void main(int argc, char *argv[])
 {
   char *pch, *pszArea, *pszPath, *pszName;
   USHORT iLevel = 0;
   BOOL fCollapsed;
   FILE *pfileTxt;
   FILE *pfileFld;
   FLDT fldt;

   // Scan in the command line

   DoProcCmdLine(argc, argv);

   // Clear the folder template structure so that the reserved bytes
   // will not mess up with future extensions if left uninitialized

   memset(&fldt, 0, sizeof(fldt));

   // Open the echo definition file for binary reading and check if ok

   if ((pfileTxt = fopen(achTxtFile, "rt")) == NULL) {
     fprintf(stderr, "Can't open: %s\n", achTxtFile);
     exit(EXIT_FAILURE);
   }

   // Open the folder definition file for binary writing and check if ok
   // Note that since MadMED folder files do not have header can be easily
   // appended to each other so you might want to have "ab" mode here...

   if ((pfileFld = fopen(achFldFile, "wb")) == NULL) {
     fprintf(stderr, "Can't open: %s\n", achFldFile);
     exit(EXIT_FAILURE);
   }

   // Loop reading all the echo area definitions

   while (fgets(achLine, sizeof(achLine) - 1, pfileTxt) != NULL) {

     // Remove trailing newline if any

     if ((pch = strchr(achLine, '\n')) != NULL) *pch = '\0';

     // Skip over leading spaces and check if there is anything to process

     pch = achLine;
     if (!DoSkipSpaces(&pch)) continue;

     // Check for the collapsed folder specification

     if (pch[0] == '+') {
       fCollapsed = TRUE;
       pch++;
       if (!DoSkipSpaces(&pch)) continue;
     } else
       fCollapsed = FALSE;

     // Check for the folder level specification

     switch (pch[0]) {
       case '<' : --iLevel;   goto DoLvl;
       case '>' : ++iLevel; //goto DoLvl;
DoLvl:		  pch++;
		  if (!DoSkipSpaces(&pch)) continue;
     }

     // Reset folder area, path and name pointers

     pszArea = pch; pszPath = NULL; pszName = NULL;

     // Skip over the area tag, fix its end and check if there is more

     while (*pch && !isspace(*pch)) pch++;
     if (!*pch)
       goto DoIt;
     else
       *pch++ = '\0';

     // Check if there is more and set the area path beginning

     if (!DoSkipSpaces(&pch))
       goto DoIt;
     else
       pszPath = pch;

     // Skip over the path, fix its end and check if there is more

     while (*pch && !isspace(*pch)) pch++;
     if (!*pch)
       goto DoIt;
     else
       *pch++ = '\0';

     // Check if there is more and set the area description beginning

     if (!DoSkipSpaces(&pch))
       goto DoIt;
     else
       pszName = pch;

     // Check if there is only area tag specification and treat it
     // as area description

DoIt: if (pszArea && !pszPath && !pszName) {
	pszName = pszArea; pszArea = NULL;
      }

     // Remove trailing spaces from the area description

     if (pszName)
       for (pch = strchr(achLine, 0) - 1; pch > pszName && isspace(*pch);
	    pch--) *pch = '\0';

     // Uppercase areatag and path specification

     if (pszArea) strupr(pszArea);
     if (pszPath) strupr(pszPath);

     // Initialize the folder template structure

     memset(&fldt, 0, sizeof(fldt));

     // Make up folder level and collapsed state

     if (fCollapsed) fldt.fs|= FF_COLLAPSED;
     if (iLevel) fldt.iLevel = iLevel;

     // Check for the message base type specification in folder path,
     // if none specified assume fido *.msg style folder

     if (pszPath != NULL)
       switch (pszPath[0]) {
	 case '$': fldt.fs|= FF_MSGSQUISH; pszPath++; break;
	 case '!': fldt.fs|= FF_MSGJAM;    pszPath++; break;
	 case '#': fldt.fs|= FF_MSGHUDSON; pszPath++;
		   fldt.boardno = atoi(pszPath);
		   while (isdigit(*pszPath)) pszPath++;
		   break;
       }

     // Check if area tag specification contains NETMAIL substring and
     // force netmail folder if so, otherwise assume echomail folder

     if (pszArea)
       if (strstr(pszArea, "NETMAIL"))
	 fldt.fs|= FF_NETMAIL;
       else
	 fldt.fs|= FF_ECHOMAIL;

     // Initialize the rest of the folder template structure

     fldt.cb = sizeof(FLDT);
     fldt.cb+= fldt.cchName = (pszName == NULL) ? 0 : strlen(pszName);
     fldt.cb+= fldt.cchArea = (pszArea == NULL) ? 0 : strlen(pszArea);
     fldt.cb+= fldt.cchPath = (pszPath == NULL) ? 0 : strlen(pszPath);
     fldt.cb+= fldt.cchOrig = 0;

     // Make up the folder default options and message attributes
     // according to the folder type

     if (fldt.fs & FF_ECHOMAIL) {
       fldt.fs|= FF_ADDORIGIN;
       fldt.attr = MA_LOCL;
     } else
     if (fldt.fs & FF_NETMAIL) {
       fldt.fs|= FF_ADDORIGIN;
       fldt.attr = MA_LOCL | MA_KILL | MA_PRVT;
     } else {
       fldt.attr = MA_LOCL | MA_PRVT;
     }

     // Write out fixed portion of the folder template
     // and all the variable length strings past it

     fwrite(&fldt, sizeof(FLDT), 1, pfileFld);
     fwrite(pszName, fldt.cchName, 1, pfileFld);
     fwrite(pszArea, fldt.cchArea, 1, pfileFld);
     fwrite(pszPath, fldt.cchPath, 1, pfileFld);

     // Check if ok

     if (ferror(pfileFld)) {
       fprintf(stderr, "Can't write: %s\n", achFldFile);
       exit(EXIT_FAILURE);
     }
   }

   // Check if we read all of the echo definition file

   if (!feof(pfileTxt)) {
     fprintf(stderr, "Can't read: %s\n", achTxtFile);
     exit(EXIT_FAILURE);
   }

   // Close the areas defintion file and exit

   fclose(pfileTxt); fclose(pfileFld);
   fprintf(stderr, "\nDone...\n");

   exit(EXIT_SUCCESS);
 }

/*
 * End of TXT2MED.C
 */
