/*
 FileNav, File Library Navigator
 Copyright (C) 1995 by Branislav L. Slantchev

 This file is part of FileNav.

 FileNav is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; version 2.

 FileNav is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with FileNav; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "gdlist.h"

/*
	memory allocation debugging library, define MEM_DEBUG to get the
	debugging code, otherwise very little overhead. In main(), call
	mem_init() and at the end, call mem_term(). Only for non-PEXes!
	You can leave mem_init() and mem_term() calls. They will be removed
	in PB_SDK mode by the preprocessor defines below.
*/
#if !defined( PB_SDK )
	#include "memdbg.h"
	#include <string.h>
	#include <stdarg.h>
	#include <assert.h>
#else
	#define mem_init
	#define mem_term
	#define mem_malloc   malloc
	#define mem_free     free
#endif

/*
 * internal data type shorthand for the comparison function
*/
typedef int (*_fcmp_t)(const void *, const void *);

/*
 * Creates a new list.
*/
	dlist_t*
dlist_new( void )
{
	dlist_t *list;

	/* see if the memdbg package has been initialized, abort if not */
#if !defined( PB_SDK )
	assert( 0 != mem_inited );
#endif

	list = mem_malloc( sizeof(dlist_t) );
	if( NULL != list ){
		list->count = 0;
		list->head = list->tail = NULL;
	}
	return list;
}


/*
 * Frees the whole list
*/
	void
dlist_free( dlist_t *list, void (*freeNode)(void *) )
{
	dlatom_t *ptr, *tmp;

	/* make sure we have a valid list pointer here */
	if( NULL != list ){

		/* traverse the list feeing each node on the way */
		for( ptr = list->head; NULL != ptr; ptr = tmp ){
			tmp = ptr->next;
			if( NULL != freeNode ) freeNode( _dlndata(ptr) );
			mem_free( (void *)ptr );
		}

		/* get rid of the list itself */
		mem_free( (void *)list );
	}
}


/*
 * Links a node into the list
*/
	int
dlist_link( dlist_t *list, const void *data, size_t width, int where, ... )
{
	dlatom_t *node, *nptr;

	/* make sure we have a valid list pointer here */
	if( NULL == list ) return -1;

	/* initialize the node: create header and copy data from source */
	node = mem_malloc( sizeof(dlatom_t) + width );
	if( NULL == node ) return -1;
	memcpy( _dlndata(node), data, width );
	node->prev = node->next = NULL;

	/* if this is the first element, just add it to the list and return */
	if( 0 == list->count ){
		list->head = list->tail = node;
		list->count++;
		return 0;
	}

	/* see the linking strategy and decide where we want the node to go */
	/* note that at that point we have at least one element in the list */

	if( GDL_SORT == where ){

		/* retrieve comparison function from the variable argument list */
		va_list varg;
		_fcmp_t fcmp;

		va_start(varg,where);
		fcmp = va_arg(varg, _fcmp_t);
		va_end(varg);

		/* traverse the list until we find a node that is larger than */
		/* the one we need to insert, insert the new node before the  */
		/* one we found. If nothing is found, append the node to list */
		/* The comparison routine should return a value larger than 0 */
		/* if the list node is greater than the new node (first arg)  */

		for( nptr = list->head; NULL != nptr; nptr = nptr->next ){

			if( 0 < fcmp(_dlndata(node), _dlndata(nptr)) ){

				/* special case for the head, go to GDL_START mode */
				if( nptr == list->head ) goto _dlLinkHead;

				/* regular link in between two nodes */
				node->prev = nptr->prev;
				node->next = nptr;
				nptr->prev->next = node;
				nptr->prev = node;
				break;
			}
		}

		/* special case for the tail, go to GDL_END mode */
		if( NULL == nptr ) goto _dlLinkTail;
	}
	else if( GDL_START == where ){
_dlLinkHead: /* we might also jump here from GDL_SORT if necessary */

		node->next = list->head;
		list->head->prev = node;
		list->head = node;
	}
	else{    /* an unknown 'where' value will default to GDL_END linkage */
_dlLinkTail: /* we might also jump here from GDL_SORT if necessary */

		node->prev = list->tail;
		list->tail->next = node;
		list->tail = node;
	}

	list->count++;
	return 0;
}


/*
 * Unlink a node from the list.
*/
	void
dlist_unlink( dlist_t *list, void *curNode, void (*freeNode)(void *) )
{
	dlatom_t *node = _dlnhead( curNode );

	/* make sure we have valid pointers here */
	if( NULL == curNode || NULL == list ) return;

	/* special cases for the head and tail nodes */
	if( list->head == node ) list->head = node->next;
	if( list->tail == node ) list->tail = node->prev;

	/* regular inside unlink between two nodes: the head and tail nodes */
	/* have prev and next set to NULL respectively, so the following    */
	/* will set the new nodes' respective pointers to NULL if necessary */

	if( NULL != node->prev ) node->prev->next = node->next;
	if( NULL != node->next ) node->next->prev = node->prev;

	/* see if we need to free user-defined records */
	if( NULL != freeNode ) freeNode( curNode );

	/* get rid of the node itself, decrement node count and get out */
	mem_free( (void *)node );
	list->count--;
}

/*
 * These functions are simply wrappers around the macros. They are
 * useful of you need to pass pointers to them to other functions.
*/
	void*
__dlist_first( dlist_t *list )
{
	return dlist_first(list);
}

	void*
__dlist_last( dlist_t *list )
{
	return dlist_last(list);
}

	void*
__dlist_prev( void *curNode )
{
	return dlist_prev(curNode);
}

	void*
__dlist_next( void *curNode )
{
	return dlist_next(curNode);
}

	word
__dlist_count( dlist_t *list )
{
	return dlist_count(list);
}

