//
// PbFlgen - Files list generator for Proboard
// Copyright (c) 1994  David L Nugent
//
//			  Unique Computing Pty Limited
//	  S-Mail: PO Box 352, Doveton, Victoria, Australia 3177
//	 FidoNet: 3:632/348@fidonet
//	InterNet: davidn@csource.oz.au, davidn@csource.pronet.com
//
// SYNOPSIS
//
//	This is a crude but effective all files list generator which
//	is dependant on ProBoard v2.0. It does nothing particuarly
//	fancy and was in fact designed to do exactly what it does, no
//	frills, no unnecessary features, no fancy display, but purely
//	to be fast and no-nonsense.
//
//	By default (no command line switches, no configuration files
//	apart from those used by Proboard) this program does exactly
//	what I wanted, nothing more or less. It will generate two files
//	lists - 'all' and 'new' files named ALLFILES.LST and NEWFILES.LST
//	by default, excluding CDROM areas (since these are already in
//	a separate list and don't need to be repeatedly regenerated) for
//	all areas with a security level of 60 or below.
//
//	A list of additional features of this program which may not be
//	so obvious to non-programmer types:
//
//	 - PbFlgen is fully multiuser compatible; it won't fail nor
//	   suffer a hernia if there's somone on-line at the time the
//	   list is generated,
//	 - PbFlgen simply skips areas it cannot process
//	 - Areas which contain no files, or contain no new files (in
//	   the case of the newfiles list) do not cause the generation
//	   of useless, space-wasting area headers in the listing
//	 - You can select which files are placed into the list based on:
//		 -> Standard listing (no CDROM areas)
//		 -> All areas, including CDROM
//		 -> Only CDROM areas (no newfiles list is generated)
//	 - User defined header file can be inserted into the top of
//	   the lists; either separte headers for all and new lists, or
//	   the same one for both.
//	 - The age of files considered new is command-line settable
//	 - PbFlGen uses NO CONFIGURATION file whatsoever.
//	 - An empty list is automatically removed (it therefore won't
//	   replace an existing file in an archive if you choose to
//	   archive the list by batch file etc.
//
// LICENSE
//
//	Here is the source - if you can use it, please feel welcome.
//	While the source code is copyrighted, you may do pretty much
//	what you want with it as you please EXCEPT TO DERIVE SHAREWARE
//	OR OTHER COMMERCIAL SOFTWARE FROM IT. This is a gift to you -
//	if you like it and want to use it in your software even without
//	mention, you're quite welcome, but being a gift it would be nice
//	to see that this contributed to a gift you yourself will make
//	to the BBS community.
//
//	If you enhance and re-release this program it is required only
//	that you:
//	   a) Release the source code to your program
//	   b) Leave this copyright and notice intact
//	Obviously this is completely unenforceable legally so the
//	obligation is only a moral one.
//
// NOTES
//
//	This was compiled with Borland C/C++ v2.0. It will probably work
//	for all Borland compilers, even TurboC since apart from fsopen()
//	(which is implemented in most other MSDOS compilers anyway) and
//	the findfirst etc. it is pretty much ANSI-C.
//	Requires the file "pb_200.h" from the Proboard 2.00 structures.
//	the untyped FILECFG.PRO structure has been typedef'ed PbFileCfg.
//



# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <fcntl.h>
# include <share.h>
# include <string.h>
# include <io.h>
# include <dir.h>
# include <sys/types.h>
# include <sys/stat.h>
# include <errno.h>
# include <ctype.h>
# include <time.h>

# include "pb_200.h"

# define DCHR	'\\'
# define MYNAME "PbFlGen"

unsigned _stklen = 32766;		// Nice large stack

// Statics

static char listpath[MAXPATH];
static char newlpath[MAXPATH];
static char hdrpath[MAXPATH];
static char nhdrpath[MAXPATH];
static uint maxlev = 60;
static uint newday = 30;
static int cdrom = 0;
static struct tm T;
static ulong today;


static char const * mons[] =
{
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

long
date2long (int year, int mon, int day)
{
	static int monofs[] =
	{
		0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334
	};

	year %= 100;
	if (year < 80)
		year += 100;
	return	((long)year * 365L) + (long)((year - 1) / 4) +
			(long)monofs[mon - 1] + (long)day + ((year % 4) && (mon > 2));
}


char *
errstr (int en)
{
	char * p;
	static char buf[128];
	strcpy (buf, strerror (en));
	p = strchr (buf, '\n');
	if (p)
		*p = 0;
	return buf;
}

	// fopen with sharing - comment out for everyone
	// except Borland.
	// Note: derived from BinkleyTerm 2.50 sources
	//		 (c) Bit Bucket Software

FILE *
fsopen (char const * path, char * mode, int shflag)
{
	int fd, open_mode;
	char c;
	char * p;
	int sflag = S_IREAD | S_IWRITE;
	FILE *stream;

	static int share_flags[] =
	{
		SH_COMPAT, SH_DENYNO, SH_DENYRD, SH_DENYRW, SH_DENYWR
	};

	p = mode;
	c = *p++;
	if (c == 'w')
		open_mode = O_CREAT | O_RDWR;
	else if (c == 'r')
		open_mode = O_RDONLY;
	else if (c == 'a')
		open_mode = O_CREAT | O_RDWR | O_APPEND;
	else return 0;
	while (*p)
	{
		c = *p++;
		if (c == 't')
			open_mode |= O_TEXT;
		if (c == 'b')
			open_mode |= O_BINARY;
		if (c == '+')
		{
			if (open_mode & O_RDONLY)
				open_mode = (open_mode & ~O_RDONLY) | O_RDWR;
			else
				if (!(open_mode & O_APPEND))
					open_mode |= O_TRUNC;
		}
	}
	fd = open (path, open_mode | share_flags[shflag], sflag);
	if (fd == -1)
		stream = 0;
	else
	{
		if ((stream = fdopen (fd, mode)) == 0)
			close (fd);
	}
	return stream;
}

	//
	// This is effectively a "list object"
	//

typedef struct FList
{
	FILE * listfp;
	int firstarea;
	int firstfile;
	uint files;
	long space;
	int type;
} FList;

void
listreset (FList * L)
{
	if (L)
	{
		L->firstfile = 1;
		L->files = 0;
		L->space = 0L;
	}
}

void
listopen (FList * L, char const * path, char const * hdr, int typ)
{
	L->listfp = fsopen (path, "w+t", 3);
	if (!L->listfp)
	{
		fprintf (stderr, MYNAME ": Fatal error creating %s - %s\n", path, errstr(errno));
		exit (2);
	}
	L->firstarea = 1;
	listreset(L);
	L->type = typ;
	hdr = hdr;		// Should insert header here
}

void
listhdr (FList * L, PbFileCfg * F)
{
	if (L)
	{
		int l;
		char title[1024];
		L->firstfile = 0;
		if (L->firstarea)
		{
			FILE * fp = fsopen ((L->type) ? nhdrpath : hdrpath, "r", 1);
			if (fp)
			{
				char * cdate = asctime (&T);
				char * which = (!cdrom) ? "Non-CDROM" : (cdrom == 1) ? "All Areas" : "CDROM Only";
				char * listt = (L->type) ? "New Files" : "";
				*strchr (cdate, '\n') = 0;
				while (fgets (title, 1024, fp) != NULL)
					fprintf (L->listfp, title, cdate, which, listt);
				fclose (fp);
			}

			L->firstarea = 0;
		}
		memset (title, '', 79);
		l = sprintf (title, "[ %s ]", F->name);
		if (l < 79)
			title[l] = '-';
		title[79] = 0;
		fprintf (L->listfp, "\n%s\n\n", title);
	}
}

void
listclose (FList * L)
{
	if (L && L->listfp)
	{
		if (!L->firstarea)
			fputs ("\nGenerated by " MYNAME " (c) DLN 1994\n", L->listfp);
		fclose (L->listfp);
		if (L->firstarea)		// remove an empty list
			remove ((L->type) ? newlpath : listpath);
		L->listfp = 0;
	}
}

void
listputln (FList * L, PbFileCfg * F, char * str)
{
	if (L)
	{
		if (L->firstfile)
			listhdr (L, F);
		fprintf (L->listfp, "%s\n", str);
	}
}

void
listtotal (FList *L, PbFileCfg *F)
{
	if (L)
	{
		if (!L->firstfile)
			fprintf (L->listfp, "\n ** Area totals ** [%s]\n %8u files\n%8luk bytes\n\n",
					 F->name, L->files, (long)(L->space / 1024L));
		listreset(L);
	}
}

void
listadd (FList * L, PbFileCfg * F, char * fname, long size, int day, int mon, int year, char *desc)
{
	if (L)
	{
		char fileline[2048];
		char fileinfo[64];

		++L->files;
		strupr (fname);
		if (size == -1L)
			strcpy (fileinfo, "Off-Line");
		else
		{
			sprintf(fileinfo, "%5luk %02.2d-%3s-%02.2d", (long)(size / 1024L), day, mons[mon-1], year-1900);
			L->space += size;
		}
		sprintf (fileline, "%-14s%16s %s", fname, fileinfo, desc);
		listputln (L, F, fileline);
	}
}

#define YEAR(ts)	(((ts >> 9) & 0x7f) + 1980)
#define MONTH(ts)	(((ts >> 5) & 0x0f))
#define DAY(ts) 	(ts & 0x1f)

void
listprocess (FList * L, FList * N, PbFileCfg * F, char * str)
{
	if (*str && !isspace(*str))
	{
		char ch;
		int day =0, mon =0, year =0;
		long size = -1L;
		char * sav, * p = str;
		while (*p && !isspace(*p))
			++p;
		if (!(ch = *(sav = p)))
			goto iscomment;
		*p++ = 0;
		while (*p && isspace(*p))
			++p;
		if (F->type)	// CDROM area
		{
			size = 0L;
			while (*p && isdigit(*p))
				size = (size * 10L) + (long)(*p++ - '0');
			while (*p && isspace(*p))
				++p;
			if (!*p)
			{
				*sav = ch;
				goto iscomment;
			}
			else
			{
				mon = atoi(p); p += 3;
				day = atoi(p); p += 3;
				year = atoi(p) + 1900; p += 2;
				while (*p && isspace (*p))
					++p;
				listadd (L, F, str, size, day, mon, year, p);
			}
		}
		else
		{
			int j, isnew = 0;
			char * s;
			char fpath[MAXPATH];
			struct ffblk ff;
			strcpy (fpath, F->filepath);
			s = fpath + strlen(fpath);
			if (s > fpath && *(s - 1) != DCHR)
				*s++ = DCHR;
			strcpy (s, str);
			j = findfirst (fpath, &ff, 0);
			if (j == 0)
			{
				day = DAY(ff.ff_fdate);
				mon = MONTH(ff.ff_fdate);
				year = YEAR(ff.ff_fdate);
				size = ff.ff_fsize;
				if (date2long(year, mon, day) >= (today - (long)newday))
					++isnew;
			}
			else if (!isalnum(*str))
			{
				*sav = ch;
				goto iscomment;
			}
			listadd (L, F, str, size, day, mon, year, p);
			if (isnew && N)
				listadd (N, F, str, size, day, mon, year, p);
		}
	}
	else
	{
	  iscomment:
		listputln (L, F, str);
	}
}


int
main (int argc, char * argv[])
{
	int carg, area;
	long now;
	FILE * files;
	FList allfiles;
	FList newfiles;
	PbFileCfg F;
	char * tmp, * p;
	char * pbenv = getenv ("PROBOARD");
	char pbpath[MAXPATH];

	tzset();
	now = time(0);
	T = *localtime(&now);
	today = date2long(T.tm_year, T.tm_mon + 1, T.tm_mday);
	if (pbenv)
	{
		strcpy (pbpath, pbenv);
		tmp = pbpath + strlen (pbpath);
	}
	else
	{
		strcpy (pbpath, argv[0]);
		tmp = strrchr (pbpath, DCHR);
		if (!tmp)
			tmp = pbpath;
		*tmp = 0;
	}

		// Examine command line

	for (carg = 0; ++carg < argc; )
	{
		char * argp = argv[carg];
		if (*argp == '-' || *argp == '/')
		{
			++argp;
			switch (tolower(*argp))
			{
			case 'o':           // include only CDROM areas
				++cdrom;
			case 'c':           // include CDROM areas
				++cdrom;
				break;

			case 'p':           // Proboard path
				if (!*++argp && ++carg < argc)
					argp = argv[carg];
				strcpy (pbpath, argp);
				break;

			case 'n':           // New days
				if (!*++argp)
				{
					if (++carg < argc && isdigit(argv[carg][0]))
						argp = argv[carg];
					else
						--carg;
				}
				newday = atoi (argp);
				if (newday == 0)
					newday = 14;
				break;

			case 's':           // Max security
				if (!*++argp)
				{
					if (++carg < argc && isdigit(argv[carg][0]))
						argp = argv[carg];
					else
						--carg;
				}
				maxlev = atoi (argp);
				if (maxlev == 0)
					maxlev = 65535U;
				break;

			case 'h':
				{
					int tp = 0;
					++argp;
					switch (tolower(*argp))
					{
					case 'a':
						++argp;
						tp |= 1;
						break;
					case 'n':
						++argp;
						tp |= 2;
						break;
					default:
						tp = 3;
						break;
					}
					if (!*argp)
					{
						if (++carg < argc)
							argp = argv[carg];
						else
							--carg;
					}
					if (tp & 1)
						strcpy (hdrpath, argp);
					if (tp & 2)
						strcpy (nhdrpath, argp);
				}
				break;

			default:
				fprintf(stderr, MYNAME ": Invalid command line argument - '%c'\n", *argp);
			case '?':
				fprintf(stderr, MYNAME " v1.0; Files List Generator for Proboard v2.0\n"
								"Copyright (C) 1994 David L Nugent & Unique Computing Pty Ltd\n"
								"Usage:\n"
								"  " MYNAME " [-p path] [-n X] [-s X] { FILELIST } { NEWLIST }\n"
								"Where:\n"
								"  -p path   Sets PROBOARD path to 'path' (default=%PROBOARD%)\n"
								"  -n X      Sets age of new files to 'X' (default=30)\n"
								"  -s X      Sets maximum securty level to 'X' (default=60)\n"
								"  -h path   Inserts file into both lists (default=" MYNAME ".hdr)\n"
								"  -ha path  Inserts file into allfiles list (default=" MYNAME ".hdr)\n"
								"  -hn path  Inserts file into newfiles list (default=" MYNAME ".hdr)\n"
								"  -c        Include CDROM areas in listing (default=Skip CDROMs)\n"
								"  -o        List only CDROM areas in listing (default=Non-CDROM areas)\n"
								"  FILELIST  Full path to list file (default=ALLFILES.LST)\n"
								"  NEWLIST   Full path to new files (default=NEWFILES.LST)\n");
				exit(1);
			}
		}
		else
		{
			if (*listpath)
				strcpy (newlpath, argp);
			else
				strcpy (listpath, argp);
		}
	}

	if (!*listpath)
		strcpy (listpath, "ALLFILES.LST");
	if (!*newlpath)
		strcpy (newlpath, "NEWFILES.LST");
	if (!*hdrpath)
		strcpy (hdrpath,  MYNAME ".HDR");
	if (!*nhdrpath)
		strcpy (nhdrpath, MYNAME ".HDR");

	for (p = pbpath; *p ; ++p)
	{
		if (*p == '/')
			*p = DCHR;
		else
			*p = (char)toupper(*p);
	}

	if (tmp > pbpath && *(tmp - 1) != DCHR)
		*tmp++ = DCHR;
	strcpy (tmp, "FILECFG.PRO");

	files = fsopen (pbpath, "rb", 1);
	if (!files)
	{
		fprintf (stderr, MYNAME ": can't access %s - %s\n", pbpath, errstr (errno));
		return 2;
	}

	listopen (&allfiles, listpath, 0, 0);
	if (cdrom != 2)
		listopen (&newfiles, newlpath, 0, 1);

	area = 0;
	printf(MYNAME ": Processing file areas... (use -? switch for help)\n");
	while (fread(&F, sizeof F, 1, files) == 1)
	{
		++area;
		if (*F.name && *F.filepath && F.level <= maxlev &&
			((F.type == 1 && cdrom) || (F.type == 0 && cdrom != 2)))
		{
			FILE * fp;
			printf ("%04u: %s", area, F.name);
			fp = fsopen (F.listpath, "r", 1);
			if (!fp)
				fprintf(stderr, " (skipped)\n  Can't access list '%s' %s\n", F.listpath, errstr (errno));
			else
			{
				char buf[1024];
				while (fgets(buf, 1024, fp) != 0)
				{
					char * p = buf + strlen(buf);
					if (p > buf && *--p == '\n')
						*p = 0;
					listprocess (&allfiles, (cdrom == 2) ? 0 : &newfiles, &F, buf);
				}
				fclose (fp);
				printf (", %u files, %u new.\n", allfiles.files, (cdrom == 2) ? 0 : newfiles.files);
			}
			listtotal (&allfiles, &F);
			if (cdrom != 2)
				listtotal (&newfiles, &F);
		}
	}
	if (cdrom != 2)
		listclose (&newfiles);
	listclose (&allfiles);

	return 0;
}

