/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.reflect;

import org.ckkloverdos.util.ClassUtil;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;

/**
 * Reflection-based utilities.
 * 
 * @author Christos KK Loverdos
 */
public final class ReflectUtil
{
    private ReflectUtil() {}

    /**
     * Obtains the accessor of javabean property. First it will try to locate a getter method,
     * then it will try to locate the field (property) itself.
     * 
     * @param c
     * @param name
     */
    public static IReflectiveAccessor getPropertyAccessor(Class c, String name)
    {
        String mname = "get" + name.substring(0, 1).toUpperCase() + name.substring(1);
        Method method = getMethodNoArgs(c, mname);
        if(null != method)
        {
            return new MethodReflectiveAccessor(method);
        }
        else
        {
            mname = "is" + name.substring(0, 1).toUpperCase() + name.substring(1);
            method = getMethodNoArgs(c, mname);
            if(null != method)
            {
                return new MethodReflectiveAccessor(method);
            }
            else
            {
                Field field = getField(c, name);
                if(null != field)
                {
                    return new FieldReflectiveAccessor(field);
                }
                else
                {
                    return null;
                }
            }
        }
    }

    /**
     * Returns a method with the given <code>name</code> and no arguments of class <code>clazz</code>.
     * @param clazz
     * @param name
     */
    public static Method getMethodNoArgs(Class clazz, String name)
    {
        return getMethod(clazz, name, new Class[]{});
    }

    /**
     * This is a wrapper around {@link Class#getMethod(String, Class[])} but with no
     * exceptions thrown.
     *
     * @param clazz
     * @param name
     * @param types
     * @return the method with the given <code>name</code> and parameters or <code>null</code>
     * if not found.
     */
    public static Method getMethod(Class clazz, String name, Class[] types)
    {
        Method m = null;

        try
        {
            m = clazz.getMethod(name, types);
        }
        catch(NoSuchMethodException e)
        {
        }
        catch(SecurityException e)
        {
        }

        return m;
    }

    /**
     * This is a wrapper around {@link Class#getField(String)}, but with no exceptions
     * thrown.
     *
     * @param clazz
     * @param name
     * @return the field of class <code>clazz</code> with the given <code>name</code>
     * or <code>null</code> if not found.
     */
    public static Field getField(Class clazz, String name)
    {
        Field f = null;
        try
        {
            f = clazz.getField(name);
        }
        catch(NoSuchFieldException e)
        {
        }
        return f;
    }

    /**
     * Gets a {@link java.lang.reflect.Constructor} without throwing any exceptions.
     *
     * @return the <tt>Constructor</tt> object iff no exception is thrown by {@link java.lang.Class#getConstructor(Class[])}
     *         and the constructor has the requested modifiers.
     */
    public static Constructor getConstructor(Class clazz, Class[] types, int yesModifiers)
    {
        return getConstructor(clazz, types, yesModifiers, 0x0);
    }

    /**
     * Gets a {@link java.lang.reflect.Constructor} without throwing any exceptions.
     *
     * @return the <tt>Constructor</tt> object iff no exception is thrown by {@link java.lang.Class#getConstructor(Class[])}
     *         and the constructor has the requested modifiers.
     */
    public static Constructor getConstructor(Class clazz, Class[] types, int yesMod, int noMod)
    {
        Constructor c = null;

        try
        {
            c = clazz.getDeclaredConstructor(types);

            if((c.getModifiers() & yesMod & ~noMod) == 0)
            {
                c = null;
            }
        }
        catch(NoSuchMethodException e)
        {
        }
        catch(SecurityException e)
        {
        }

        return c;
    }

    /**
     * Creates and returns a new instance for the class with name <code>className</code>.
     * It returns <code>null</code> if the instance cannot be created.
     * No exceptions are thrown.
     * @param className
     */
    public static Object newInstance(String className)
    {
    	Class _class = ClassUtil.forName(className, null);
    	if(_class == null)
    	{
    		return null;
    	}
    	return newInstance(_class);
    }

    /**
     * Creates and returns a new instance for the class <code>clazz</code>.
     * It returns <code>null</code> if the instance cannot be created.
     * No exceptions are thrown.
     *
     * @param clazz
     */
    public static Object newInstance(Class clazz)
    {
    	Object instance = null;
    	try
		{
			instance = clazz.newInstance();
		}
		catch(InstantiationException e)
		{
		}
		catch(IllegalAccessException e)
		{
		}

		return instance;
    }

    public static IReflectiveAccessor getAccesorForSelect(Class c, String propertyName, boolean isMethodCall)
    {
        if(isMethodCall)
        {
            return new MethodReflectiveAccessor(getMethodNoArgs(c, propertyName));
        }
        else
        {
            return getPropertyAccessor(c, propertyName);
        }
    }
}
