package zinger.io.file;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import java.io.*;
import java.util.prefs.*;

import javax.swing.*;

public class Console extends Frame implements FileListener
{
    protected class PreferencesDialog extends Dialog
    {
        protected TextField delayField;
        protected TextField fontSizeField;
        protected TextField maxCharsDisplayedField;
        protected Button cancelButton;
        protected Button okayButton;

        protected long delay;
        protected int fontSize;
        protected int maxCharsDisplayed;

        protected Boolean accepted = null;

        public PreferencesDialog()
        {
            super(Console.this, "Preferences", true);
            this.setResizable(false);
        }

        protected void windowInit()
        {
            this.fieldsInit();
            this.buttonsInit();

            Panel centerPanel = new Panel();
            centerPanel.setLayout(new GridLayout(3, 2));
            centerPanel.add(new Label("Delay"));
            centerPanel.add(this.delayField);
            centerPanel.add(new Label("Font Size"));
            centerPanel.add(this.fontSizeField);
            centerPanel.add(new Label("Max Chars Back"));
            centerPanel.add(this.maxCharsDisplayedField);

            Panel southPanel = new Panel();
            southPanel.setLayout(new FlowLayout());
            southPanel.add(this.okayButton);
            southPanel.add(this.cancelButton);

            this.setLayout(new BorderLayout());
            this.add(southPanel, BorderLayout.SOUTH);
            this.add(centerPanel, BorderLayout.CENTER);

            this.pack();
        }

        protected void fieldsInit()
        {
            this.delayField = new TextField();
            this.fontSizeField = new TextField();
            this.maxCharsDisplayedField = new TextField();
        }

        protected void buttonsInit()
        {
            this.cancelButton = new Button("Cancel");
            this.cancelButton.addActionListener(new ActionListener()
            {
                public void actionPerformed(ActionEvent event)
                {
                    Console.PreferencesDialog.this.cancel();
                }
            });

            this.okayButton = new Button("Okay");
            this.okayButton.addActionListener(new ActionListener()
            {
                public void actionPerformed(ActionEvent event)
                {
                    Console.PreferencesDialog.this.accept();
                }
            });
        }

        public void cancel()
        {
            this.accepted = Boolean.FALSE;
            this.hide();
        }

        public void accept()
        {
            try
            {
                this.delay = Long.parseLong(this.delayField.getText()) * 1000;
                this.fontSize = Integer.parseInt(this.fontSizeField.getText());
                this.maxCharsDisplayed = Integer.parseInt(this.maxCharsDisplayedField.getText());
                this.accepted = Boolean.TRUE;
            }
            catch(NumberFormatException ex)
            {
                this.accepted = Boolean.FALSE;
            }
            finally
            {
                this.hide();
            }
        }

        public void show()
        {
            if(this.accepted == null)
            {
                this.windowInit();
            }
            this.accepted = Boolean.FALSE;

            this.delayField.setText(String.valueOf(Console.FILE_WATCHER.getDelay() / 1000));
            this.fontSize = Console.this.getFontSize();
            this.fontSizeField.setText(String.valueOf(this.fontSize));
            this.maxCharsDisplayedField.setText(String.valueOf(Console.this.maxCharsDisplayed));

            super.show();
        }

        public boolean wasAccepted()
        {
            return this.accepted.booleanValue();
        }

        public long getDelay()
        {
            return this.delay;
        }

        public int getFontSize()
        {
            return this.fontSize;
        }

        public int getMaxCharsDisplayed()
        {
			return this.maxCharsDisplayed;
		}
    }

    protected static final String NO_FILE_TITLE = "File Watcher";

    protected static final FileWatcher FILE_WATCHER = new FileWatcher();

    protected CheckboxMenuItem pauseMenuItem;

    protected final ActionListener openHandler = new ActionListener()
        {
            public void actionPerformed(ActionEvent event)
            {
                if(Console.this.file != null)
                {
                    Console.this.fileDialog.setDirectory(Console.this.file.getParent());
                }
                Console.this.fileDialog.show();
                Console.this.openFile(Console.this.fileDialog.getDirectory(), Console.this.fileDialog.getFile());
            }
        };

    protected final ActionListener newWindowHandler = new ActionListener()
        {
            public void actionPerformed(ActionEvent event)
            {
                (new Console()).showFrame();
            }
        };

	protected final ActionListener exitHandler = new ActionListener()
        {
            public void actionPerformed(ActionEvent event)
            {
                System.exit(0);
            }
        };

    protected final ActionListener copyHandler = new ActionListener()
        {
            public void actionPerformed(ActionEvent event)
            {
                Clipboard clipboard = Console.this.outputArea.getToolkit().getSystemClipboard();
                StringSelection selection = new StringSelection(Console.this.outputArea.getSelectedText());
                clipboard.setContents(selection, selection);
            }
        };

    protected final ActionListener selectAllHandler = new ActionListener()
        {
            public void actionPerformed(ActionEvent event)
            {
                Console.this.outputArea.selectAll();
            }
        };

	protected final ActionListener clearHandler = new ActionListener()
        {
            public void actionPerformed(ActionEvent event)
            {
                Console.this.outputArea.setText("");
            }
        };

	protected final ActionListener tailHandler = new ActionListener()
        {
            public void actionPerformed(ActionEvent event)
            {
                Console.this.refresh(true);
                Console.this.outputArea.setCaretPosition(Integer.MAX_VALUE);
            }
        };

	protected final ActionListener refreshHandler = new ActionListener()
        {
            public void actionPerformed(ActionEvent event)
            {
                Console.this.refresh(false);
            }
        };

	protected final ItemListener pauseHandler = new ItemListener()
        {
            public void itemStateChanged(ItemEvent event)
            {
                if(Console.this.pauseMenuItem.getState())
                {
                    Console.FILE_WATCHER.removeFileListener(Console.this);
                }
                else
                {
                    Console.FILE_WATCHER.addFileListener(Console.this);
                }
            }
        };

	protected final ActionListener propertiesHandler = new ActionListener()
        {
            public void actionPerformed(ActionEvent event)
            {
                synchronized(Console.this.preferencesDialog)
                {
                    Console.this.preferencesDialog.show();
                    if(Console.this.preferencesDialog.wasAccepted())
                    {
                        Console.FILE_WATCHER.setDelay(Console.this.preferencesDialog.getDelay());
						Console.this.setFontSize(Console.this.preferencesDialog.getFontSize());
                        Console.this.maxCharsDisplayed = Console.this.preferencesDialog.getMaxCharsDisplayed();
                    }
                }
            }
        };

    private static int counter = 0;

    protected File file;

    protected TextArea outputArea;
    protected FileDialog fileDialog;
    protected PreferencesDialog preferencesDialog;

    private final StringBuffer stringBuffer = new StringBuffer();
    private final char[] charBuffer = new char[1 << 7];
    private long textLength = 0;
    private boolean canAppend = false;

	private final Preferences userPref = Preferences.userNodeForPackage(this.getClass());
    private int maxCharsDisplayed = userPref.getInt("maxCharsDisplayed", 5 << 10);
	
    public Console()
    {
        super(Console.NO_FILE_TITLE);
		
        this.frameInit();
    }
	
    protected void frameInit()
    {
        this.addWindowListener(new WindowAdapter()
        {
            public void windowClosing(WindowEvent ev)
            {
                if(ev.getWindow() == Console.this)
                {
                    Console.this.dispose();
                }
            }
        });

        this.fileDialogInit();

        this.preferencesDialogInit();

        this.outputAreaInit();
		
		this.setFontSize(this.userPref.getInt("fontSize", 12));

        this.menuInit();
        this.add(outputArea);
    }
	
	protected int getFontSize()
	{
		return this.outputArea.getFont().getSize();
	}
	
	protected void setFontSize(int fontSize)
	{
		this.userPref.putInt("fontSize", fontSize);
		this.outputArea.setFont(new Font("Monospaced", 0, fontSize));
	}

    protected void fileDialogInit()
    {
        this.fileDialog = new FileDialog(this);
    }

    protected void preferencesDialogInit()
    {
        this.preferencesDialog = this.new PreferencesDialog();
    }

    protected void outputAreaInit()
    {
        this.outputArea = new TextArea("", 0, 0, TextArea.SCROLLBARS_VERTICAL_ONLY);
        this.outputArea.setEditable(false);
    }

    protected void menuInit()
    {
        MenuItem openMenuItem = new MenuItem("Open", new MenuShortcut(KeyEvent.VK_O));
        openMenuItem.addActionListener(this.openHandler);

        MenuItem newWindowMenuItem = new MenuItem("New Window", new MenuShortcut(KeyEvent.VK_N));
        newWindowMenuItem.addActionListener(this.newWindowHandler);

        MenuItem exitMenuItem = new MenuItem("Exit", new MenuShortcut(KeyEvent.VK_Q));
        exitMenuItem.addActionListener(this.exitHandler);

        Menu fileMenu = new Menu("File");
        fileMenu.add(openMenuItem);
        fileMenu.add(newWindowMenuItem);
        fileMenu.add(new MenuItem("-"));
        fileMenu.add(exitMenuItem);

        MenuItem copyMenuItem = new MenuItem("Copy", new MenuShortcut(KeyEvent.VK_C));
        copyMenuItem.addActionListener(this.copyHandler);

        MenuItem selectAllMenuItem = new MenuItem("Select All", new MenuShortcut(KeyEvent.VK_A));
        selectAllMenuItem.addActionListener(this.selectAllHandler);

        Menu editMenu = new Menu("Edit");
        editMenu.add(copyMenuItem);
        editMenu.add(selectAllMenuItem);

        MenuItem clearMenuItem = new MenuItem("Clear", new MenuShortcut(KeyEvent.VK_L));
        clearMenuItem.addActionListener(this.clearHandler);

        MenuItem tailMenuItem = new MenuItem("Tail", new MenuShortcut(KeyEvent.VK_T));
        tailMenuItem.addActionListener(this.tailHandler);

        MenuItem refreshMenuItem = new MenuItem("Refresh", new MenuShortcut(KeyEvent.VK_R));
        refreshMenuItem.addActionListener(this.refreshHandler);

        this.pauseMenuItem = new CheckboxMenuItem("Pause", false);
        this.pauseMenuItem.setShortcut(new MenuShortcut(KeyEvent.VK_SPACE));
        pauseMenuItem.addItemListener(this.pauseHandler);

        MenuItem propertiesMenuItem = new MenuItem("Properties", new MenuShortcut(KeyEvent.VK_P));
        propertiesMenuItem.addActionListener(this.propertiesHandler);

        Menu viewMenu = new Menu("View");
        viewMenu.add(clearMenuItem);
        viewMenu.add(tailMenuItem);
        viewMenu.add(refreshMenuItem);
        viewMenu.add(pauseMenuItem);
        viewMenu.add(new MenuItem("-"));
        viewMenu.add(propertiesMenuItem);

        MenuBar menuBar = new MenuBar();
        menuBar.add(fileMenu);
        menuBar.add(editMenu);
        menuBar.add(viewMenu);
        this.setMenuBar(menuBar);
    }

    public synchronized void hide()
    {
        if(this.isVisible())
        {
            --Console.counter;
            if(Console.counter == 0)
            {
                System.exit(0);
            }
            super.hide();
        }
    }

    public synchronized void show()
    {
        if(!this.isVisible())
        {
            ++Console.counter;
            super.show();
        }
    }

    public Dimension getPreferredSize()
    {
        return new Dimension(600, 400);
    }

    public void openFile(String directory, String file)
    {
        this.openFile(file == null ? null : new File(directory, file));
    }

    public synchronized void openFile(File file)
    {
        if(file != null)
        {
            if(this.file == null)
            {
                Console.FILE_WATCHER.addFileListener(this);
            }
            this.file = file;
			this.outputArea.setText("-= " + this.file + " =-\n\n");
			this.canAppend = true;
			this.textLength = Math.max(0, this.file.length() - this.maxCharsDisplayed);
            Console.FILE_WATCHER.addFile(this.file);
            try
            {
                this.setTitle(file.getCanonicalPath());
            }
            catch(IOException ex)
            {
                ex.printStackTrace();
                this.setTitle(Console.NO_FILE_TITLE);
            }
        }
    }

    public synchronized void refresh(boolean append)
    {
        if(this.file != null && this.file.exists())
        {
            try
            {
				int toRemove = this.outputArea.getText().length() - this.maxCharsDisplayed;
				if(toRemove > 0)
				{
					this.outputArea.replaceRange("", 0, toRemove);
				}

                Reader reader = new FileReader(this.file);
                if(append)
                {
                    reader.skip(this.textLength);
                }
                else
                {
                    this.textLength = 0;
                }
                this.stringBuffer.setLength(0);
                for(int bytesRead = reader.read(this.charBuffer); bytesRead > 0; bytesRead = reader.read(this.charBuffer))
                {
                    this.textLength += bytesRead;
                    this.stringBuffer.append(charBuffer, 0, bytesRead);
                }
                reader.close();
                /*
                for(int i = this.stringBuffer.indexOf("\r"); i >= 0; i = this.stringBuffer.indexOf("\r"))
                {
                    this.stringBuffer.deleteCharAt(i);
                }
                */
                String text = this.stringBuffer.toString();
                if(append)
                {
                    //this.textLength += text.length();
                    this.outputArea.append(text);
                }
                else
                {
                    //this.textLength = text.length();
                    this.outputArea.setText(text);
                }
            }
            catch(FileNotFoundException ex)
            {
                ex.printStackTrace();
                this.outputArea.setText(ex.getMessage());
            }
            catch(IOException ex)
            {
                ex.printStackTrace();
                this.outputArea.setText(ex.getMessage());
            }
        }
        else
        {
            this.outputArea.setText("");
        }
    }

    public void handleFileEvent(FileEvent event)
    {
        if(this.file != null && this.file.equals(event.getFile()))
        {
            switch(event.getType())
            {
                case FileEvent.MODIFIED:
                break;

                case FileEvent.DELETED:
                this.canAppend = false;
                break;

                case FileEvent.CREATED:
                this.canAppend = false;
                break;
            }
            this.refresh(this.canAppend);
            this.canAppend = true;
        }
    }

    public void showFrame()
    {
        this.pack();
        this.show();
    }

    public static void main(String[] args)
    {
        String delayProperty = System.getProperty("delay");
        try
        {
            if(delayProperty != null)
            {
                Console.FILE_WATCHER.setDelay(Long.parseLong(delayProperty));
            }
        }
        catch(NumberFormatException ex)
        {
            System.err.println("Delay property not understood.");
        }
        Console console = null;
        for(int i = 0; i < args.length; i++)
        {
            console = new Console();
            console.openFile(new File(args[i]));
            console.showFrame();
        }
        if(console == null)
        {
            console = new Console();
            console.showFrame();
        }
    }
}
