/*
 * $Id:AbstractPaletteRuntime.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.mandelbrot.extensions.fractal.outcolouring;

import net.sf.jame.core.util.Palette;
import net.sf.jame.mandelbrot.palette.PaletteRendererRuntimeElement;
import net.sf.jame.mandelbrot.renderer.RenderedPoint;

/**
 * @author Andrea Medeghini
 */
public abstract class AbstractPaletteRuntime<T extends AbstractPaletteConfig> extends AbstractOutcolouringFormulaRuntime<T> {
	private PaletteRendererRuntimeElement rendererElement;
	protected int[] colorTable;
	protected Palette palette;
	protected int iterations;

	/**
	 * @see net.sf.jame.mandelbrot.fractal.outcolouring.extension.OutcolouringFormulaExtensionRuntime#prepareForRendering(int)
	 */
	@Override
	public void prepareForRendering(final int iterations) {
		if (rendererElement.getRendererRuntime() != null) {
			rendererElement.getRendererRuntime().prepareForRendering();
			final Palette newPalette = rendererElement.getRendererRuntime().renderPalette();
			if ((this.iterations != iterations) || (colorTable == null) || (palette != newPalette)) {
				this.colorTable = newPalette.renderTable(iterations);
			}
			this.iterations = iterations;
			this.palette = newPalette;
		}
		else {
			this.iterations = 0;
			this.palette = null;
			this.colorTable = new int[] { 0xFF000000 };
		}
	}

	/**
	 * @param cp
	 * @return the color.
	 */
	@Override
	public int renderColor(final RenderedPoint cp) {
		if (this.colorTable.length == 0) {
			return 0;
		}
		return this.colorTable[this.renderIndex(cp) % this.colorTable.length];
	}

	/**
	 * @param cp
	 * @param shift
	 * @return the color.
	 */
	@Override
	public int renderColor(final RenderedPoint cp, final int shift) {
		if (shift != 0) {
			if (this.colorTable.length == 0) {
				return 0;
			}
			int index = this.renderIndex(cp) + shift;
			if (index < 0) {
				index = this.colorTable.length - (-index) % this.colorTable.length;
			}
			return this.colorTable[index % this.colorTable.length];
		}
		else {
			return this.renderColor(cp);
		}
	}

	/**
	 * @param cp
	 * @return the index.
	 */
	protected abstract int renderIndex(RenderedPoint cp);

	/**
	 * @see net.sf.jame.core.extension.ConfigurableExtensionRuntime#isChanged()
	 */
	@Override
	public boolean isChanged() {
		final boolean rendererChanged = (rendererElement != null) && rendererElement.isChanged();
		return super.isChanged() || rendererChanged;
	}

	/**
	 * @see net.sf.jame.core.extension.ConfigurableExtensionRuntime#configReloaded()
	 */
	@Override
	public void configReloaded() {
		super.configReloaded();
		rendererElement = new PaletteRendererRuntimeElement(getConfig().getPaletteRendererElement());
	}
}
