/*
 * $Id:DistributedServiceProcessor.java 491 2008-01-28 21:59:31Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.networking.spool;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.LinkedList;
import java.util.List;

import net.sf.jame.networking.EventMessage;
import net.sf.jame.networking.RequestMessage;
import net.sf.jame.networking.ResponseMessage;
import net.sf.jame.networking.ServiceException;
import net.sf.jame.networking.ServiceMessage;
import net.sf.jame.networking.ServiceProcessor;
import net.sf.jame.networking.ServiceSession;
import net.sf.jame.networking.SessionHandler;
import net.sf.jame.service.spool.DistributedJobInterface;
import net.sf.jame.service.spool.JobEvent;
import net.sf.jame.service.spool.JobInterface;
import net.sf.jame.service.spool.JobListener;
import net.sf.jame.service.spool.JobService;
import net.sf.jame.service.spool.JobStatus;
import net.sf.jame.service.spool.impl.DistributedJobDecoder;

import org.apache.log4j.Logger;

/**
 * @author Andrea Medeghini
 */
public class DistributedServiceProcessor implements ServiceProcessor {
	private static final Logger logger = Logger.getLogger(DistributedServiceProcessor.class);
	private final List<ExecutorTask> tasks = new LinkedList<ExecutorTask>();
	private final Object monitor = new Object();
	private final JobService<? extends DistributedJobInterface> jobService;
	private final int maxJobCount;
	private Thread thread;
	private boolean running;
	private boolean dirty;

	/**
	 * @param jobService
	 * @param maxJobCount
	 */
	public DistributedServiceProcessor(final JobService<? extends DistributedJobInterface> jobService, final int maxJobCount) {
		this.jobService = jobService;
		this.maxJobCount = maxJobCount;
	}

	/**
	 * @see net.sf.jame.networking.ServiceProcessor#start()
	 */
	public void start() {
		jobService.start();
		if (thread == null) {
			thread = new Thread(new ExecutorHandler(), "SpoolServiceProcessor Executor Thread");
			thread.setDaemon(true);
			running = true;
			thread.start();
		}
	}

	/**
	 * @see net.sf.jame.networking.ServiceProcessor#stop()
	 */
	public void stop() {
		jobService.stop();
		if (thread != null) {
			running = false;
			thread.interrupt();
			try {
				thread.join();
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}
	}

	/**
	 * @see net.sf.jame.networking.ServiceProcessor#createSessionHandler()
	 */
	public SessionHandler createSessionHandler() {
		return new SpoolSessionHandler();
	}

	private class ExecutorHandler implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			final long pollingTime = 60 * 1000L;
			final List<ExecutorTask> tasksToRun = new LinkedList<ExecutorTask>();
			try {
				while (running) {
					try {
						synchronized (tasks) {
							tasksToRun.addAll(tasks);
							tasks.clear();
						}
						for (final ExecutorTask task : tasksToRun) {
							task.run();
						}
						tasksToRun.clear();
					}
					catch (final Exception e) {
						e.printStackTrace();
					}
					synchronized (monitor) {
						if (!dirty) {
							monitor.wait(pollingTime);
						}
						dirty = false;
					}
				}
			}
			catch (final InterruptedException e) {
			}
		}
	}

	private class ExecutorTask implements Runnable {
		private final ServiceSession session;
		private final ServiceMessage message;

		/**
		 * @param session
		 * @param message
		 */
		public ExecutorTask(final ServiceSession session, final ServiceMessage message) {
			this.session = session;
			this.message = message;
		}

		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			try {
				logger.debug("Ready to send message " + message + " on session " + session.getSessionId());
				if (!session.isExpired()) {
					session.sendMessage(message);
				}
				else {
					logger.error("Failed to send the message " + message + " on session " + session.getSessionId());
					logger.error("Invalidate session " + session.getSessionId());
					session.invalidate();
				}
			}
			catch (final Exception e) {
				logger.error("Failed to send the message " + message + " on session " + session.getSessionId());
				logger.error("Invalidate session " + session.getSessionId());
				session.invalidate();
				e.printStackTrace();
			}
		}
	}

	private class SpoolSessionHandler implements SessionHandler {
		private ServiceSession session;

		/**
		 * @see net.sf.jame.networking.SessionHandler#dispose()
		 */
		public void dispose() {
			if (session != null) {
				session.dispose();
				session = null;
			}
		}

		/**
		 * @see net.sf.jame.networking.SessionHandler#setSession(net.sf.jame.networking.ServiceSession)
		 */
		public void setSession(final ServiceSession session) {
			this.session = session;
		}

		/**
		 * @see net.sf.jame.networking.SessionHandler#getSession()
		 */
		public ServiceSession getSession() {
			return session;
		}

		/**
		 * @see net.sf.jame.networking.SessionHandler#isExpired()
		 */
		public boolean isExpired() {
			return session == null || session.isExpired();
		}

		/**
		 * @see net.sf.jame.networking.SessionHandler#onMessage(net.sf.jame.networking.ServiceMessage)
		 */
		public void onMessage(final ServiceMessage message) throws ServiceException {
			if (isExpired()) {
				return;
			}
			try {
				switch (message.getMessageType()) {
					case ServiceMessage.MESSAGE_TYPE_REQUEST: {
						final RequestMessage request = (RequestMessage) message;
						switch (request.getRequestType()) {
							case RequestMessage.TYPE_HELLO: {
								if (jobService.getJobCount() < maxJobCount) {
									final String jobId = jobService.createJob(new SpoolJobListener(session));
									final DistributedJobInterface job = jobService.getJob(jobId);
									synchronized (tasks) {
										final ResponseMessage response = createHelloResponse(request, job);
										tasks.add(new ExecutorTask(session, response));
									}
								}
								else {
									synchronized (tasks) {
										final ResponseMessage response = createHelloResponse(request, null);
										tasks.add(new ExecutorTask(session, response));
									}
								}
								break;
							}
							case RequestMessage.TYPE_PUT: {
								final String jobId = (String) ((Object[]) request.getUserData())[0];
								final int frameNumber = (Integer) ((Object[]) request.getUserData())[1];
								final byte[] data = (byte[]) ((Object[]) request.getUserData())[2];
								final DistributedJobInterface job = jobService.getJob(jobId);
								final DistributedJobDecoder decoder = new DistributedJobDecoder(data);
								job.setJobData(decoder.getJobData());
								job.setJob(decoder.getJob());
								job.setClip(decoder.getClip());
								job.setFirstFrameNumber(frameNumber);
								jobService.runJob(jobId);
								synchronized (tasks) {
									final ResponseMessage response = createPutResponse(request, job);
									tasks.add(new ExecutorTask(session, response));
								}
								break;
							}
							case RequestMessage.TYPE_GET: {
								final String jobId = (String) ((Object[]) request.getUserData())[0];
								final int frameNumber = (Integer) ((Object[]) request.getUserData())[1];
								final DistributedJobInterface job = jobService.getJob(jobId);
								synchronized (tasks) {
									final ResponseMessage response = createGetResponse(request, job, frameNumber);
									tasks.add(new ExecutorTask(session, response));
								}
								break;
							}
							case RequestMessage.TYPE_ABORT: {
								final String jobId = (String) request.getUserData();
								final DistributedJobInterface job = jobService.getJob(jobId);
								jobService.abortJob(jobId);
								synchronized (tasks) {
									final ResponseMessage response = createAbortResponse(request, job);
									tasks.add(new ExecutorTask(session, response));
								}
								break;
							}
							case RequestMessage.TYPE_DELETE: {
								final String jobId = (String) request.getUserData();
								final DistributedJobInterface job = jobService.getJob(jobId);
								jobService.deleteJob(jobId);
								synchronized (tasks) {
									final ResponseMessage response = createDeleteResponse(request, job);
									tasks.add(new ExecutorTask(session, response));
								}
								break;
							}
							default: {
								break;
							}
						}
						break;
					}
					case ServiceMessage.MESSAGE_TYPE_RESPONSE: {
						break;
					}
					case ServiceMessage.MESSAGE_TYPE_FAILURE: {
						break;
					}
					case ServiceMessage.MESSAGE_TYPE_EVENT: {
						break;
					}
					case ServiceMessage.MESSAGE_TYPE_KEEPALIVE: {
						break;
					}
					default: {
						break;
					}
				}
			}
			catch (final Exception e) {
				e.printStackTrace();
			}
			synchronized (monitor) {
				dirty = true;
				monitor.notify();
			}
		}

		private ResponseMessage createHelloResponse(final RequestMessage request, final DistributedJobInterface job) {
			final ResponseMessage message = new ResponseMessage();
			message.setRequestId(request.getRequestId());
			message.setResponseType(RequestMessage.TYPE_HELLO);
			if (job != null) {
				message.setUserData(job.getJobId());
				message.setReturnCode(0);
			}
			else {
				message.setReturnCode(1);
			}
			return message;
		}

		private ResponseMessage createPutResponse(final RequestMessage request, final DistributedJobInterface job) throws Exception {
			final ResponseMessage message = new ResponseMessage();
			message.setRequestId(request.getRequestId());
			message.setResponseType(RequestMessage.TYPE_PUT);
			if (job != null) {
				message.setUserData(job.getJobId());
				message.setReturnCode(0);
			}
			else {
				message.setReturnCode(1);
			}
			return message;
		}

		private ResponseMessage createGetResponse(final RequestMessage request, final DistributedJobInterface job, final int frameNumber) throws Exception {
			final ResponseMessage message = new ResponseMessage();
			message.setRequestId(request.getRequestId());
			message.setResponseType(RequestMessage.TYPE_GET);
			if (job != null) {
				try {
					message.setUserData(new Object[] { job.getJobId(), frameNumber, getJobData(job, frameNumber) });
					message.setReturnCode(0);
				}
				catch (final IOException e) {
					message.setReturnCode(1);
					e.printStackTrace();
				}
			}
			else {
				message.setReturnCode(1);
			}
			return message;
		}

		private byte[] getJobData(final DistributedJobInterface job, final int frameNumber) throws IOException {
			final int tw = job.getJob().getTileWidth();
			final int th = job.getJob().getTileHeight();
			final int bw = job.getJob().getBorderWidth();
			final int bh = job.getJob().getBorderHeight();
			final int sw = tw + 2 * bw;
			final int sh = th + 2 * bh;
			final byte[] data = new byte[sw * sh * 4];
			RandomAccessFile raf = null;
			try {
				if (job.getFirstFrameNumber() > 0) {
					final long pos = (frameNumber - job.getFirstFrameNumber() - 1) * sw * sh * 4;
					raf = job.getRAF();
					raf.seek(pos);
					raf.readFully(data);
				}
				else {
					final long pos = frameNumber * sw * sh * 4;
					raf = job.getRAF();
					raf.seek(pos);
					raf.readFully(data);
				}
			}
			catch (final IOException e) {
				// System.out.println("firstFrame = " + job.getFirstFrameNumber() + ", frame = " + frameNumber + ", length = " + raf.length() / (sw * sh * 4));
				e.printStackTrace();
			}
			finally {
				if (raf != null) {
					try {
						raf.close();
					}
					catch (final IOException e) {
					}
				}
			}
			return data;
		}

		private ResponseMessage createAbortResponse(final RequestMessage request, final DistributedJobInterface job) throws Exception {
			final ResponseMessage message = new ResponseMessage();
			message.setRequestId(request.getRequestId());
			message.setResponseType(RequestMessage.TYPE_ABORT);
			if (job != null) {
				message.setUserData(job.getJobId());
				message.setReturnCode(0);
			}
			else {
				message.setReturnCode(1);
			}
			return message;
		}

		private ResponseMessage createDeleteResponse(final RequestMessage request, final JobInterface job) throws Exception {
			final ResponseMessage message = new ResponseMessage();
			message.setRequestId(request.getRequestId());
			message.setResponseType(RequestMessage.TYPE_DELETE);
			if (job != null) {
				message.setUserData(job.getJobId());
				message.setReturnCode(0);
			}
			else {
				message.setReturnCode(1);
			}
			return message;
		}
	}

	private class SpoolJobListener implements JobListener {
		private final ServiceSession session;

		/**
		 * @param session
		 */
		public SpoolJobListener(final ServiceSession session) {
			this.session = session;
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#stateChanged(net.sf.jame.service.spool.Job)
		 */
		public void stateChanged(final JobInterface job) {
			if (session.isExpired()) {
				return;
			}
			synchronized (tasks) {
				try {
					final EventMessage message = createStatusMessage(job);
					tasks.add(new ExecutorTask(session, message));
				}
				catch (final Exception e) {
					e.printStackTrace();
				}
			}
			synchronized (monitor) {
				dirty = true;
				monitor.notify();
			}
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#started(net.sf.jame.service.spool.Job)
		 */
		public void started(final JobInterface job) {
			if (session.isExpired()) {
				return;
			}
			synchronized (tasks) {
				try {
					final EventMessage message = createStartMessage(job);
					tasks.add(new ExecutorTask(session, message));
				}
				catch (final Exception e) {
					e.printStackTrace();
				}
			}
			synchronized (monitor) {
				dirty = true;
				monitor.notify();
			}
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#stopped(net.sf.jame.service.spool.Job)
		 */
		public void stopped(final JobInterface job) {
			if (session.isExpired()) {
				return;
			}
			synchronized (tasks) {
				try {
					final EventMessage message = createStopMessage(job);
					tasks.add(new ExecutorTask(session, message));
				}
				catch (final Exception e) {
					e.printStackTrace();
				}
			}
			synchronized (monitor) {
				dirty = true;
				monitor.notify();
			}
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#terminated(net.sf.jame.service.spool.JobInterface)
		 */
		public void terminated(final JobInterface job) {
		}

		private EventMessage createStatusMessage(final JobInterface job) throws Exception {
			final EventMessage message = new EventMessage();
			message.setUserData(new JobEvent(JobEvent.EVENT_TYPE_STATUS, new JobStatus(job.getJobId(), job.getFrameNumber())));
			return message;
		}

		private EventMessage createStartMessage(final JobInterface job) throws Exception {
			final EventMessage message = new EventMessage();
			message.setUserData(new JobEvent(JobEvent.EVENT_TYPE_START, new JobStatus(job.getJobId(), job.getFrameNumber())));
			return message;
		}

		private EventMessage createStopMessage(final JobInterface job) throws Exception {
			final EventMessage message = new EventMessage();
			message.setUserData(new JobEvent(JobEvent.EVENT_TYPE_STOP, new JobStatus(job.getJobId(), job.getFrameNumber())));
			return message;
		}
	}
}
