/*
 * $Id:JXTANetworkService.java 485 2008-01-27 09:19:41Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.networking.jxta;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import net.jxta.discovery.DiscoveryEvent;
import net.jxta.discovery.DiscoveryListener;
import net.jxta.discovery.DiscoveryService;
import net.jxta.document.Advertisement;
import net.jxta.document.AdvertisementFactory;
import net.jxta.document.MimeMediaType;
import net.jxta.document.StructuredTextDocument;
import net.jxta.endpoint.ByteArrayMessageElement;
import net.jxta.endpoint.Message;
import net.jxta.endpoint.MessageElement;
import net.jxta.endpoint.StringMessageElement;
import net.jxta.id.IDFactory;
import net.jxta.peergroup.PeerGroup;
import net.jxta.peergroup.PeerGroupID;
import net.jxta.pipe.PipeMsgEvent;
import net.jxta.pipe.PipeMsgListener;
import net.jxta.pipe.PipeService;
import net.jxta.platform.ModuleClassID;
import net.jxta.platform.NetworkManager;
import net.jxta.protocol.DiscoveryResponseMsg;
import net.jxta.protocol.ModuleClassAdvertisement;
import net.jxta.protocol.ModuleSpecAdvertisement;
import net.jxta.protocol.PipeAdvertisement;
import net.jxta.util.JxtaBiDiPipe;
import net.jxta.util.JxtaServerPipe;
import net.jxta.util.PipeEventListener;
import net.jxta.util.PipeStateListener;
import net.sf.jame.core.util.Files;
import net.sf.jame.networking.ServiceConsumer;
import net.sf.jame.networking.ServiceException;
import net.sf.jame.networking.ServiceListener;
import net.sf.jame.networking.ServiceMessage;
import net.sf.jame.networking.ServiceProcessor;
import net.sf.jame.networking.ServiceProducer;
import net.sf.jame.networking.ServiceSession;
import net.sf.jame.networking.SessionHandler;
import net.sf.jame.networking.SessionIDFactory;

import org.apache.log4j.Logger;

/**
 * @author Andrea Medeghini
 */
public class JXTANetworkService {
	private static final Logger logger = Logger.getLogger(JXTANetworkService.class);
	private DiscoveryService discovery;
	private NetworkManager manager;
	private JxtaServerPipe serverPipe;
	private PipeAdvertisement pipeadv;
	private final List<ModuleSpecAdvertisement> avertisements = new ArrayList<ModuleSpecAdvertisement>();
	private final HashMap<String, ServiceSession> sessions = new HashMap<String, ServiceSession>();
	private final ServiceProcessor processor;
	private final String serviceURI;
	private final String serviceName;
	private final String serviceAuthor;
	private final String serviceVersion;
	private final File workdir;
	private Thread thread;
	private boolean running;
	private boolean dirty;
	private final Object monitor = new Object();

	/**
	 * @param workdir
	 * @param serviceURI
	 * @param serviceName
	 * @param serviceAuthor
	 * @param serviceVersion
	 * @param processor
	 */
	public JXTANetworkService(final File workdir, final String serviceURI, final String serviceName, final String serviceAuthor, final String serviceVersion, final ServiceProcessor processor) {
		this.workdir = workdir;
		this.serviceURI = serviceURI;
		this.serviceName = serviceName;
		this.serviceAuthor = serviceAuthor;
		this.serviceVersion = serviceVersion;
		this.processor = processor;
	}

	/**
	 * 
	 */
	public void start() {
		processor.start();
		if (thread == null) {
			thread = new Thread(new NetworkHandler(), "JXTANetworkService Thread");
			thread.setDaemon(true);
			running = true;
			thread.start();
		}
	}

	/**
	 * 
	 */
	public void stop() {
		processor.stop();
		if (thread != null) {
			running = false;
			thread.interrupt();
			try {
				thread.join();
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}
	}

	/**
	 * @return
	 */
	public List<ModuleSpecAdvertisement> getAdvertisements() {
		synchronized (avertisements) {
			return new LinkedList<ModuleSpecAdvertisement>(avertisements);
		}
	}

	/**
	 * @param pipeadv
	 * @param listener
	 * @return
	 * @throws ServiceException
	 */
	public ServiceSession createSession(final PipeAdvertisement pipeadv, final ServiceListener listener) throws ServiceException {
		try {
			final JxtaBiDiPipe pipe = new JxtaBiDiPipe();
			pipe.setMaxRetryTimeout(10000);
			pipe.setRetryTimeout(10000);
			pipe.setReliable(true);
			final JXTAConnectionHandler handler = new JXTAConnectionHandler(pipe);
			pipe.connect(manager.getNetPeerGroup(), null, pipeadv, 10000, handler, true);
			final String sessionId = SessionIDFactory.newSessionId();
			handler.setSessionId(sessionId);
			final ServiceSession session = new JXTAClientServiceSession(sessionId, new JXTAClientServiceConsumer(handler, listener), new JXTAClientServiceProducer(handler));
			logger.info("Client session created " + session.getSessionId());
			synchronized (sessions) {
				sessions.put(session.getSessionId(), session);
			}
			return session;
		}
		catch (final Exception e) {
			throw new ServiceException(e);
		}
	}

	@SuppressWarnings("unchecked")
	private static void printAdvertisement(final Advertisement adv) {
		try {
			final StructuredTextDocument doc = (StructuredTextDocument) adv.getDocument(MimeMediaType.XMLUTF8);
			final StringWriter out = new StringWriter();
			doc.sendToWriter(out);
			logger.debug(out.toString());
			out.close();
		}
		catch (final Exception e) {
		}
	}

	private static PipeAdvertisement createPipeAdvertisement(final String name) {
		final PipeAdvertisement advertisement = (PipeAdvertisement) AdvertisementFactory.newAdvertisement(PipeAdvertisement.getAdvertisementType());
		advertisement.setPipeID(IDFactory.newPipeID(PeerGroupID.defaultNetPeerGroupID, String.valueOf(System.currentTimeMillis()).getBytes()));
		advertisement.setType(PipeService.UnicastType);
		advertisement.setName(name);
		return advertisement;
	}

	private class NetworkHandler implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			try {
				final File cache = new File(workdir, "JXTACache");
				cache.mkdirs();
				Files.deleteFiles(cache);
				manager = new NetworkManager(NetworkManager.ConfigMode.EDGE, serviceName, cache.toURI());
				manager.setPeerID(IDFactory.newPeerID(PeerGroupID.defaultNetPeerGroupID, String.valueOf(System.currentTimeMillis()).getBytes()));
				manager.startNetwork();
				manager.waitForRendezvousConnection(5000);
				final PeerGroup netPeerGroup = manager.getNetPeerGroup();
				discovery = netPeerGroup.getDiscoveryService();
				pipeadv = createPipeAdvertisement(serviceName);
				final Thread discoveryThread = new Thread(new DiscoveryHandler(), "JXTANetworkService Discovery Thread");
				discoveryThread.setDaemon(true);
				discoveryThread.start();
				final Thread publishThread = new Thread(new PublishHandler(), "JXTANetworkService Publish Thread");
				publishThread.setDaemon(true);
				publishThread.start();
				final Thread cleanupThread = new Thread(new CleanupHandler(), "JXTANetworkService Cleanup Thread");
				cleanupThread.setDaemon(true);
				cleanupThread.start();
				serverPipe = new JxtaServerPipe(netPeerGroup, pipeadv);
				serverPipe.setPipeTimeout(10 * 60 * 1000);
				while (running) {
					try {
						final JxtaBiDiPipe pipe = serverPipe.accept();
						if (pipe != null) {
							final JXTAConnectionHandler handler = new JXTAConnectionHandler(pipe);
							final String sessionId = SessionIDFactory.newSessionId();
							handler.setSessionId(sessionId);
							final SessionHandler sessionHandler = processor.createSessionHandler();
							final ServiceSession session = new JXTAServerServiceSession(sessionId, new JXTAServerServiceConsumer(handler, sessionHandler), new JXTAServerServiceProducer(handler));
							sessionHandler.setSession(session);
							logger.info("Server session created " + session.getSessionId());
							synchronized (sessions) {
								sessions.put(session.getSessionId(), session);
							}
						}
					}
					catch (final Exception e) {
						e.printStackTrace();
					}
				}
				discoveryThread.interrupt();
				publishThread.interrupt();
				cleanupThread.interrupt();
				discoveryThread.join();
				publishThread.join();
				cleanupThread.join();
			}
			catch (final InterruptedException e) {
			}
			catch (final Exception e) {
				e.printStackTrace();
			}
			synchronized (avertisements) {
				avertisements.clear();
			}
			manager.stopNetwork();
		}
	}

	private class CleanupHandler implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			final long pollingTime = 30 * 1000L;
			final List<ServiceSession> activeSessions = new LinkedList<ServiceSession>();
			try {
				while (running) {
					try {
						synchronized (sessions) {
							final Iterator<ServiceSession> sessionIterator = sessions.values().iterator();
							while (sessionIterator.hasNext()) {
								final ServiceSession session = sessionIterator.next();
								if (session.isExpired()) {
									logger.info("Dispose expired session " + session.getSessionId());
									sessionIterator.remove();
									session.dispose();
								}
								else {
									activeSessions.add(session);
								}
							}
						}
						for (final ServiceSession session : activeSessions) {
							try {
								session.consumeMessages();
							}
							catch (final Exception e) {
								e.printStackTrace();
								session.invalidate();
							}
						}
						activeSessions.clear();
					}
					catch (final Exception e) {
						e.printStackTrace();
					}
					synchronized (monitor) {
						if (!dirty) {
							monitor.wait(pollingTime);
						}
						dirty = false;
					}
				}
			}
			catch (final InterruptedException e) {
			}
			synchronized (sessions) {
				for (final ServiceSession session : sessions.values()) {
					logger.info("Dispose session " + session.getSessionId());
					session.dispose();
				}
				sessions.clear();
			}
		}
	}

	private class PublishHandler implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			final long lifetime = 120 * 1000L;
			final long expiration = 120 * 1000L;
			final long pollingTime = lifetime + 10 * 1000L;
			final ModuleClassAdvertisement mcadv = (ModuleClassAdvertisement) AdvertisementFactory.newAdvertisement(ModuleClassAdvertisement.getAdvertisementType());
			mcadv.setName("JXTAMOD:" + serviceName);
			mcadv.setDescription(serviceName);
			final ModuleClassID mcID = IDFactory.newModuleClassID();
			mcadv.setModuleClassID(mcID);
			final ModuleSpecAdvertisement msadv = (ModuleSpecAdvertisement) AdvertisementFactory.newAdvertisement(ModuleSpecAdvertisement.getAdvertisementType());
			msadv.setName("JXTASPEC:" + serviceName);
			msadv.setCreator(serviceAuthor);
			msadv.setVersion(serviceVersion);
			msadv.setModuleSpecID(IDFactory.newModuleSpecID(mcID));
			msadv.setSpecURI(serviceURI);
			msadv.setPipeAdvertisement(pipeadv);
			try {
				while (running) {
					try {
						logger.debug("Publish module class advertisement");
						printAdvertisement(mcadv);
						discovery.publish(mcadv, lifetime, expiration);
						discovery.remotePublish(mcadv, expiration);
						logger.debug("Publish module spec advertisement");
						printAdvertisement(msadv);
						discovery.publish(msadv, lifetime, expiration);
						discovery.remotePublish(msadv, expiration);
					}
					catch (final Exception e) {
						e.printStackTrace();
					}
					Thread.sleep(pollingTime);
				}
			}
			catch (final InterruptedException e) {
			}
		}
	}

	private class DiscoveryHandler implements Runnable, DiscoveryListener {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			final long pollingTime = 60 * 1000L;
			discovery.addDiscoveryListener(this);
			try {
				while (running) {
					try {
						synchronized (avertisements) {
							avertisements.clear();
						}
						discovery.getRemoteAdvertisements(null, DiscoveryService.ADV, "Name", "JXTASPEC:" + serviceName, 1, null);
					}
					catch (final Exception e) {
						e.printStackTrace();
					}
					Thread.sleep(pollingTime);
				}
			}
			catch (final InterruptedException e) {
			}
			discovery.removeDiscoveryListener(this);
		}

		/**
		 * @see net.jxta.discovery.DiscoveryListener#discoveryEvent(net.jxta.discovery.DiscoveryEvent)
		 */
		public void discoveryEvent(final DiscoveryEvent e) {
			final DiscoveryResponseMsg response = e.getResponse();
			logger.debug("Got a discovery response [" + response.getResponseCount() + " elements] from peer " + e.getSource());
			final Enumeration<Advertisement> advEnum = response.getAdvertisements();
			if (advEnum != null) {
				while (advEnum.hasMoreElements()) {
					final Advertisement adv = advEnum.nextElement();
					if (adv.getAdvType().equals(ModuleSpecAdvertisement.getAdvertisementType())) {
						final ModuleSpecAdvertisement msadv = (ModuleSpecAdvertisement) adv;
						logger.debug("Got a module spec advertisement");
						printAdvertisement(msadv);
						synchronized (avertisements) {
							avertisements.add(msadv);
						}
					}
				}
			}
		}
	}

	private class JXTAConnectionHandler implements PipeMsgListener, PipeStateListener, PipeEventListener {
	    private final MimeMediaType GZIP_MEDIA_TYPE = new MimeMediaType("application/gzip").intern();
		private static final int MAX_LENGTH = 32 * 1024; 
		private static final int TIMEOUT = 10000; 
		private static final int TIMEOUT_COUNT = 3; 
		private ByteArrayOutputStream baos = new ByteArrayOutputStream();
		private final List<Message> newMessages = new ArrayList<Message>();
		private final List<Message> messages = new ArrayList<Message>();
		private ServiceListener listener;
		private JxtaBiDiPipe pipe;
		private long lastReceivedTime;
		private String lastCorrelationId;
		private int lastPartSent;
		private int lastPartReceived;
		private int lastResult;
		private final Object lock = new Object();
		private byte[] buffer = new byte[65536];
		private String sessionId;

		/**
		 * @param pipe
		 */
		public JXTAConnectionHandler(final JxtaBiDiPipe pipe) {
			this.pipe = pipe;
			pipe.setMessageListener(this);
			pipe.setPipeEventListener(this);
			pipe.setPipeStateListener(this);
			lastReceivedTime = System.currentTimeMillis();
		}

		/**
		 * @param sessionId
		 */
		public void setSessionId(String sessionId) {
			this.sessionId = sessionId;
		}

		/**
		 * 
		 */
		public void dispose() {
			if (pipe != null) {
				try {
					if (pipe.isBound()) {
						pipe.close();
					}
				}
				catch (final IOException e) {
				}
				pipe = null;
			}
			newMessages.clear();
			messages.clear();
			listener = null;
		}

		/**
		 * @param session
		 */
		public void setServiceListener(final ServiceListener listener) {
			this.listener = listener;
		}

		/**
		 * @return
		 */
		public boolean isDisposed() {
			return pipe == null;
		}

		/**
		 * @return
		 */
		public boolean isConnected() {
			return pipe != null && pipe.isBound();
		}

		/**
		 * @return
		 */
		public boolean isTimeout() {
			return (System.currentTimeMillis() - lastReceivedTime) > 120 * 1000L;
		}

		/**
		 * @param message
		 * @throws Exception
		 */
		public synchronized void sendMessage(final ServiceMessage message) throws ServiceException {
			ByteArrayOutputStream baos = null;
			GZIPOutputStream zos = null;
			ObjectOutputStream oos = null;
			try {
				baos = new ByteArrayOutputStream();
				zos = new GZIPOutputStream(baos);
				oos = new ObjectOutputStream(zos);
				oos.writeObject(message);
				oos.close();
				zos.finish();
				zos.close();
				baos.close();
				byte[] data = baos.toByteArray();
				int maxLength = MAX_LENGTH;
				int offset = 0;
				int parts = 1;
				int part = 1;
				if (logger.isDebugEnabled()) {
					logger.debug("[sessionId = " + sessionId + "] Sending message " + message + " (dataLength = " + data.length + " bytes)");
				}
				if (data.length > maxLength) {
					parts += data.length / maxLength;
					for (int i = 1; i < parts; i++) {
						if (!sendDataMessage(message.getMessageId(), parts, part, data, offset, maxLength)) {
							throw new Exception("[sessionId = " + sessionId + "] Failed to send message " + message.getMessageId() + " (part = " + part + " of " + parts + ")");
						}
						offset += maxLength;
						part += 1;
					}
					if (!sendDataMessage(message.getMessageId(), parts, part, data, offset, data.length % maxLength)) {
						throw new Exception("[sessionId = " + sessionId + "] Failed to send message " + message.getMessageId() + " (part = " + part + " of " + parts + ")");
					}
				}
				else {
					if (!sendDataMessage(message.getMessageId(), parts, part, data, 0, data.length)) {
						throw new Exception("[sessionId = " + sessionId + "] Failed to send message " + message.getMessageId() + " (part = " + part + " of " + parts + ")");
					}
				}
			}
			catch (final Exception e) {
				throw new ServiceException(e);
			}
			finally {
				if (oos != null) {
					try {
						oos.close();
					}
					catch (Exception e) {
					}
				}
				if (zos != null) {
					try {
						zos.close();
					}
					catch (Exception e) {
					}
				}
				if (baos != null) {
					try {
						baos.close();
					}
					catch (Exception e) {
					}
				}
			}
		}

		private boolean sendDataMessage(String correlationId, int parts, int part, byte[] data, int offset, int length) throws ServiceException {
			try {
				boolean messageSent = false;
				synchronized (lock) {
					lastResult = 0;
					lastPartSent = part;
					for (int i = 0; i < TIMEOUT_COUNT; i++) {
						final Message msg = new Message();
						msg.addMessageElement(new StringMessageElement("correlationId", correlationId, null));
						msg.addMessageElement(new StringMessageElement("messageParts", String.valueOf(parts), null));
						msg.addMessageElement(new StringMessageElement("messagePart", String.valueOf(part), null));
						msg.addMessageElement(new StringMessageElement("messageType", "data", null));
						msg.addMessageElement(new ByteArrayMessageElement("messageBody", GZIP_MEDIA_TYPE, data, offset, length, null));
						pipe.sendMessage(msg);
						if (logger.isDebugEnabled()) {
							logger.debug("[sessionId = " + sessionId + "] Sending data message (correlationId = " + correlationId.toString() + ", part = " + part + " of " + parts + ", size = " + msg.getByteLength() + " bytes)");
						}
						lock.wait(TIMEOUT);
						if (lastResult == 1) {
							messageSent = true;
							break;
						}
						if (lastResult == 2) {
							break;
						}
					}
				}
				return messageSent;
			}
			catch (final Exception e) {
				throw new ServiceException(e);
			}
		}

		private void sendAckMessage(String correlationId, int part) throws ServiceException {
			try {
				final Message msg = new Message();
				msg.addMessageElement(new StringMessageElement("correlationId", correlationId, null));
				msg.addMessageElement(new StringMessageElement("messagePart", String.valueOf(part), null));
				msg.addMessageElement(new StringMessageElement("messageType", "ack", null));
				pipe.sendMessage(msg);
				if (logger.isDebugEnabled()) {
					logger.debug("[sessionId = " + sessionId + "] Sending ack message (correlationId = " + correlationId.toString() + ", part = " + part + ", size = " + msg.getByteLength() + " bytes)");
				}
			}
			catch (final Exception e) {
				throw new ServiceException(e);
			}
		}

		private void sendErrorMessage(String correlationId, int part) throws ServiceException {
			try {
				final Message msg = new Message();
				msg.addMessageElement(new StringMessageElement("correlationId", correlationId, null));
				msg.addMessageElement(new StringMessageElement("messagePart", String.valueOf(part), null));
				msg.addMessageElement(new StringMessageElement("messageType", "error", null));
				pipe.sendMessage(msg);
				if (logger.isDebugEnabled()) {
					logger.debug("[sessionId = " + sessionId + "] Sending error message (correlationId = " + correlationId.toString() + ", part = " + part + ", size = " + msg.getByteLength() + " bytes)");
				}
			}
			catch (final Exception e) {
				throw new ServiceException(e);
			}
		}

		/**
		 * @throws Exception
		 */
		public void consumeMessages() throws ServiceException {
			try {
				synchronized (newMessages) {
					messages.addAll(newMessages);
					newMessages.clear();
				}
				if (messages.size() > 0) {
					for (final Message message : messages) {
						final MessageElement correlationId = message.getMessageElement("correlationId");
						final MessageElement messageParts = message.getMessageElement("messageParts");
						final MessageElement messagePart = message.getMessageElement("messagePart");
						final MessageElement messageType = message.getMessageElement("messageType");
						final MessageElement messageBody = message.getMessageElement("messageBody");
						if ("keepalive".equals(messageType.toString())) {
							logger.debug("[sessionId = " + sessionId + "] Received keepalive message (size = " + message.getByteLength() + " bytes)");
						}
						else if ("data".equals(messageType.toString())) {
							int parts = Integer.parseInt(messageParts.toString());
							int part = Integer.parseInt(messagePart.toString());
							if (lastCorrelationId != null && !lastCorrelationId.equals(correlationId.toString())) {
								sendErrorMessage(correlationId.toString(), part);
								lastCorrelationId = null;
								lastPartReceived = 0;
								baos.reset();
								throw new Exception("[sessionId = " + sessionId + "] Invalid correlationId " + correlationId.toString());
							}
							if (lastCorrelationId == null || lastCorrelationId.equals(correlationId.toString())) {
								if (logger.isDebugEnabled()) {
									logger.debug("[sessionId = " + sessionId + "] Received data message (correlationId = " + correlationId.toString() + ", part = " + part + " of " + parts + ", size = " + message.getByteLength() + " bytes)");
								}
								lastCorrelationId = correlationId.toString();
								sendAckMessage(lastCorrelationId, part);
								if (part > lastPartReceived) {
									lastPartReceived = part;
									int length = 0;
									InputStream is = messageBody.getStream();
									while ((length = is.read(buffer)) > 0) {
										baos.write(buffer, 0, length);
									}
									if (lastPartReceived == parts) {
										byte[] data = baos.toByteArray();
										lastCorrelationId = null;
										lastPartReceived = 0;
										baos.reset();
										if (listener != null) {
											ByteArrayInputStream bais = null;
											ObjectInputStream ois = null;
											GZIPInputStream zis = null;
											try {
												bais = new ByteArrayInputStream(data);
												zis = new GZIPInputStream(bais);
												ois = new ObjectInputStream(zis);
												final ServiceMessage msg = (ServiceMessage) ois.readObject();
												if (logger.isDebugEnabled()) {
													logger.debug("[sessionId = " + sessionId + "] Received message " + msg + " (dataLength = " + data.length + " bytes)");
												}
												listener.onMessage(msg);
											}
											finally {
												if (ois != null) {
													try {
														ois.close();
													}
													catch (Exception e) {
													}
												}
												if (zis != null) {
													try {
														zis.close();
													}
													catch (Exception e) {
													}
												}
												if (bais != null) {
													try {
														bais.close();
													}
													catch (Exception e) {
													}
												}
											}
										}
									}
								}
							}
						}
						else if ("ack".equals(messageType.toString())) {
							synchronized (lock) {
								int part = Integer.parseInt(messagePart.toString());
								if (logger.isDebugEnabled()) {
									logger.debug("[sessionId = " + sessionId + "] Received ack message (correlationId = " + correlationId.toString() + ", part = " + part + ", size = " + message.getByteLength() + " bytes)");
								}
								if (part == lastPartSent) {
									lastResult = 1;
									lock.notify();
								}
							}
						}
						else if ("error".equals(messageType.toString())) {
							synchronized (lock) {
								int part = Integer.parseInt(messagePart.toString());
								if (logger.isDebugEnabled()) {
									logger.debug("[sessionId = " + sessionId + "] Received error message (correlationId = " + correlationId.toString() + ", part = " + part + ", size = " + message.getByteLength() + " bytes)");
								}
								lastResult = 2;
								lock.notify();
							}
						}
					}
				}
				messages.clear();
			}
			catch (final Exception e) {
				throw new ServiceException(e);
			}
		}

		/**
		 * @param message
		 * @return
		 */
		private boolean isMessageValid(final Message message) {
			final MessageElement correlationId = message.getMessageElement("correlationId");
			final MessageElement messageParts = message.getMessageElement("messageParts");
			final MessageElement messagePart = message.getMessageElement("messagePart");
			final MessageElement messageType = message.getMessageElement("messageType");
			final MessageElement messageBody = message.getMessageElement("messageBody");
			if ("data".equals(messageType.toString())) {
				return correlationId != null && messageParts != null && messagePart != null && messageType != null && messageBody != null;
			}
			else if ("ack".equals(messageType.toString())) {
				return correlationId != null && messagePart != null && messageType != null;
			}
			else if ("error".equals(messageType.toString())) {
				return correlationId != null && messagePart != null && messageType != null;
			}
			else if ("keepalive".equals(messageType.toString())) {
				return true;
			}
			return false;
		}

		/**
		 * @see net.jxta.pipe.PipeMsgListener#pipeMsgEvent(net.jxta.pipe.PipeMsgEvent)
		 */
		public void pipeMsgEvent(final PipeMsgEvent e) {
			final Message message = e.getMessage();
			if (message != null) {
				if (isMessageValid(message)) {
					synchronized (newMessages) {
						lastReceivedTime = System.currentTimeMillis();
						newMessages.add(message);
					}
				}
			}
			synchronized (monitor) {
				dirty = true;
				monitor.notify();
			}
		}

		/**
		 * @see net.jxta.util.PipeStateListener#stateEvent(java.lang.Object, int)
		 */
		public void stateEvent(final Object source, final int event) {
		}

		/**
		 * @see net.jxta.util.PipeEventListener#pipeEvent(int)
		 */
		public void pipeEvent(final int event) {
		}

		/**
		 * @param message
		 * @throws ServiceException
		 */
		public void consumeMessage(ServiceMessage message) throws ServiceException {
			lastReceivedTime = System.currentTimeMillis();
			if (listener != null) {
				listener.onMessage(message);
			}
		}

		/**
		 * @throws ServiceException 
		 */
		public void sendKeepAliveMessage() throws ServiceException {
			try {
				final Message msg = new Message();
				msg.addMessageElement(new StringMessageElement("messageType", "keepalive", null));
				pipe.sendMessage(msg);
				if (logger.isDebugEnabled()) {
					logger.debug("[sessionId = " + sessionId + "] Sending keepalive message (size = " + msg.getByteLength() + " bytes)");
				}
			}
			catch (final Exception e) {
				throw new ServiceException(e);
			}
		}
	}

	private class JXTAClientServiceProducer implements ServiceProducer {
		private JXTAConnectionHandler handler;

		/**
		 * @param handler
		 */
		public JXTAClientServiceProducer(final JXTAConnectionHandler handler) {
			this.handler = handler;
		}

		/**
		 * @see net.sf.jame.p2p.ServiceProducer#sendMessage(net.sf.jame.p2p.ServiceMessage)
		 */
		public void sendMessage(final ServiceMessage message) throws ServiceException {
			if (handler != null) {
				handler.sendMessage(message);
			}
		}

		/**
		 * @see net.sf.jame.p2p.ServiceProducer#dispose()
		 */
		public void dispose() {
			if (handler != null) {
				handler.dispose();
				handler = null;
			}
		}

		/**
		 * @see net.sf.jame.networking.ServiceProducer#sendKeepAliveMessage()
		 */
		public void sendKeepAliveMessage() throws ServiceException {
			if (handler != null) {
				handler.sendKeepAliveMessage();
			}
		}
	}

	private class JXTAServerServiceProducer implements ServiceProducer {
		private JXTAConnectionHandler handler;

		/**
		 * @param handler
		 */
		public JXTAServerServiceProducer(final JXTAConnectionHandler handler) {
			this.handler = handler;
		}

		/**
		 * @see net.sf.jame.p2p.ServiceProducer#sendMessage(net.sf.jame.p2p.ServiceMessage)
		 */
		public void sendMessage(final ServiceMessage message) throws ServiceException {
			if (handler != null) {
				handler.sendMessage(message);
			}
		}

		/**
		 * @see net.sf.jame.p2p.ServiceProducer#dispose()
		 */
		public void dispose() {
			if (handler != null) {
				handler.dispose();
				handler = null;
			}
		}

		/**
		 * @see net.sf.jame.networking.ServiceProducer#sendKeepAliveMessage()
		 */
		public void sendKeepAliveMessage() throws ServiceException {
			if (handler != null) {
				handler.sendKeepAliveMessage();
			}
		}
	}

	private class JXTAClientServiceConsumer implements ServiceConsumer {
		private JXTAConnectionHandler handler;
		private ServiceListener listener;

		/**
		 * @param handler
		 * @param listener
		 */
		public JXTAClientServiceConsumer(final JXTAConnectionHandler handler, final ServiceListener listener) {
			this.handler = handler;
			this.listener = listener;
			handler.setServiceListener(this);
		}

		/**
		 * @see net.sf.jame.p2p.ServiceConsumer#onMessage(net.sf.jame.p2p.ServiceMessage)
		 */
		public void onMessage(final ServiceMessage message) throws ServiceException {
			if (listener != null) {
				listener.onMessage(message);
			}
		}

		/**
		 * @see net.sf.jame.p2p.ServiceConsumer#consumeMessages()
		 */
		public void consumeMessages() throws ServiceException {
			if (handler != null) {
				handler.consumeMessages();
			}
		}

		/**
		 * @see net.sf.jame.p2p.ServiceConsumer#isTimeout()
		 */
		public boolean isTimeout() {
			if (handler != null) {
				return handler.isTimeout();
			}
			return true;
		}

		/**
		 * @see net.sf.jame.p2p.ServiceConsumer#dispose()
		 */
		public void dispose() {
			if (handler != null) {
				handler.dispose();
				handler = null;
			}
			listener = null;
		}

		/**
		 * @see net.sf.jame.networking.ServiceConsumer#consumeMessage(net.sf.jame.networking.ServiceMessage)
		 */
		public void consumeMessage(ServiceMessage message) throws ServiceException {
			if (handler != null) {
				handler.consumeMessage(message);
			}
		}
	}

	private class JXTAServerServiceConsumer implements ServiceConsumer {
		private JXTAConnectionHandler handler;
		private ServiceListener listener;

		/**
		 * @param handler
		 * @param listener
		 */
		public JXTAServerServiceConsumer(final JXTAConnectionHandler handler, final ServiceListener listener) {
			this.handler = handler;
			this.listener = listener;
			handler.setServiceListener(this);
		}

		/**
		 * @see net.sf.jame.p2p.ServiceListener#onMessage(net.sf.jame.p2p.ServiceMessage)
		 */
		public void onMessage(final ServiceMessage message) throws ServiceException {
			if (listener != null) {
				listener.onMessage(message);
			}
		}

		/**
		 * @see net.sf.jame.p2p.ServiceConsumer#consumeMessages()
		 */
		public void consumeMessages() throws ServiceException {
			if (handler != null) {
				handler.consumeMessages();
			}
		}

		/**
		 * @see net.sf.jame.p2p.ServiceConsumer#isTimeout()
		 */
		public boolean isTimeout() {
			if (handler != null) {
				return handler.isTimeout();
			}
			return true;
		}

		/**
		 * @see net.sf.jame.p2p.ServiceConsumer#dispose()
		 */
		public void dispose() {
			if (handler != null) {
				handler.dispose();
				handler = null;
			}
			listener = null;
		}

		/**
		 * @see net.sf.jame.networking.ServiceConsumer#consumeMessage(net.sf.jame.networking.ServiceMessage)
		 */
		public void consumeMessage(ServiceMessage message) throws ServiceException {
			if (handler != null) {
				handler.consumeMessage(message);
			}
		}
	}

	private class JXTAClientServiceSession extends ServiceSession {
		/**
		 * @param sessionId
		 * @param consumer
		 * @param producer
		 * @throws Exception
		 */
		public JXTAClientServiceSession(final String sessionId, final JXTAClientServiceConsumer consumer, final JXTAClientServiceProducer producer) throws Exception {
			super(sessionId, consumer, producer);
		}

		/**
		 * @see net.sf.jame.networking.ServiceSession#dispose()
		 */
		@Override
		public void dispose() {
			logger.debug("Disposed session " + getSessionId());
			super.dispose();
		}

		/**
		 * @see net.sf.jame.networking.ServiceSession#isLocalSession()
		 */
		@Override
		public boolean isLocalSession() {
			return false;
		}
	}

	private class JXTAServerServiceSession extends ServiceSession {
		/**
		 * @param sessionId
		 * @param consumer
		 * @param producer
		 * @throws Exception
		 */
		public JXTAServerServiceSession(final String sessionId, final JXTAServerServiceConsumer consumer, final JXTAServerServiceProducer producer) throws Exception {
			super(sessionId, consumer, producer);
		}

		/**
		 * @see net.sf.jame.networking.ServiceSession#dispose()
		 */
		@Override
		public void dispose() {
			logger.debug("Disposed session " + getSessionId());
			super.dispose();
		}

		/**
		 * @see net.sf.jame.networking.ServiceSession#isLocalSession()
		 */
		@Override
		public boolean isLocalSession() {
			return false;
		}
	}
}
