/*
 * $Id:DummyJobService.java 488 2008-01-27 09:21:39Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.networking.test;

import java.util.HashMap;
import java.util.Iterator;

import net.sf.jame.service.spool.JobFactory;
import net.sf.jame.service.spool.JobIDFactory;
import net.sf.jame.service.spool.JobInterface;
import net.sf.jame.service.spool.JobListener;
import net.sf.jame.service.spool.JobService;
import net.sf.jame.service.spool.JobServiceListener;

import org.apache.log4j.Logger;

/**
 * @author Andrea Medeghini
 */
public class DummyJobService<T extends DummyJob> implements JobService<T> {
	private static final Logger logger = Logger.getLogger(DummyJobService.class);
	private final HashMap<String, ScheduledJob> scheduledJobs = new HashMap<String, ScheduledJob>();
	private final HashMap<String, T> terminatedJobs = new HashMap<String, T>();
	private final HashMap<String, T> spooledJobs = new HashMap<String, T>();
	private final JobFactory<T> jobFactory;
	private final Object monitor = new Object();
	private Thread thread;
	private boolean running;
	private boolean dirty;

	/**
	 * @param jobFactory
	 */
	public DummyJobService(final JobFactory<T> jobFactory) {
		this.jobFactory = jobFactory;
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#start()
	 */
	public void start() {
		if (thread == null) {
			thread = new Thread(new ServiceHandler(), "DummyJobService Thread");
			thread.setDaemon(true);
			running = true;
			thread.start();
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#stop()
	 */
	public void stop() {
		if (thread != null) {
			running = false;
			thread.interrupt();
			try {
				thread.join();
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#getJobCount()
	 */
	public int getJobCount() {
		synchronized (spooledJobs) {
			return spooledJobs.size();
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#deleteJob(java.lang.String)
	 */
	public void deleteJob(final String jobId) {
		synchronized (terminatedJobs) {
			terminatedJobs.remove(jobId);
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#abortJob(java.lang.String)
	 */
	public void abortJob(final String jobId) {
		synchronized (spooledJobs) {
			final T job = spooledJobs.get(jobId);
			if (job != null) {
				synchronized (scheduledJobs) {
					final ScheduledJob scheduledJob = scheduledJobs.get(jobId);
					if (scheduledJob != null) {
						scheduledJob.stop();
					}
				}
			}
		}
		synchronized (monitor) {
			dirty = true;
			monitor.notify();
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#createJob(net.sf.jame.service.spool.JobListener)
	 */
	public String createJob(final JobListener listener) {
		synchronized (spooledJobs) {
			final T job = jobFactory.createJob(JobIDFactory.newJobId(), listener);
			spooledJobs.put(job.getJobId(), job);
			return job.getJobId();
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#getJob(java.lang.String)
	 */
	public T getJob(final String jobId) {
		synchronized (spooledJobs) {
			T job = spooledJobs.get(jobId);
			if (job == null) {
				synchronized (terminatedJobs) {
					job = terminatedJobs.get(jobId);
				}
			}
			return job;
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#runJob(java.lang.String)
	 */
	public void runJob(final String jobId) {
		synchronized (spooledJobs) {
			final T job = spooledJobs.get(jobId);
			if (job != null) {
				synchronized (scheduledJobs) {
					scheduledJobs.put(jobId, new ScheduledJob(job, System.currentTimeMillis() + 10 * 1000L));
				}
			}
		}
		synchronized (monitor) {
			dirty = true;
			monitor.notify();
		}
	}

	private class ServiceHandler implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			try {
				while (running) {
					synchronized (scheduledJobs) {
						synchronized (terminatedJobs) {
							final Iterator<ScheduledJob> jobIterator = scheduledJobs.values().iterator();
							while (jobIterator.hasNext()) {
								final ScheduledJob scheduledJob = jobIterator.next();
								if (!scheduledJob.getJob().isStarted()) {
									scheduledJob.start();
									logger.info("Job started " + scheduledJob.getJob());
								}
								else if (scheduledJob.getJob().isStarted() && System.currentTimeMillis() > scheduledJob.time) {
									scheduledJob.setFrameNumber(0);
									logger.info("Job updated " + scheduledJob.getJob());
								}
								else if (scheduledJob.getJob().isStarted() && System.currentTimeMillis() > scheduledJob.time + 30 * 1000L) {
									scheduledJob.stop();
									logger.info("Job stopped " + scheduledJob.getJob());
								}
								if (scheduledJob.getJob().isTerminated()) {
									jobIterator.remove();
									terminatedJobs.put(scheduledJob.getJob().getJobId(), scheduledJob.getJob());
									logger.info("Job terminated " + scheduledJob.getJob());
									scheduledJob.dispose();
								}
								else if ((System.currentTimeMillis() - scheduledJob.getJob().getLastUpdate()) > 60 * 1000L) {
									jobIterator.remove();
									terminatedJobs.put(scheduledJob.getJob().getJobId(), scheduledJob.getJob());
									logger.info("Job terminated " + scheduledJob.getJob());
									scheduledJob.stop();
									scheduledJob.dispose();
								}
							}
						}
					}
					synchronized (spooledJobs) {
						synchronized (terminatedJobs) {
							final Iterator<T> jobIterator = terminatedJobs.values().iterator();
							while (jobIterator.hasNext()) {
								final JobInterface job = jobIterator.next();
								if ((System.currentTimeMillis() - job.getLastUpdate()) > 120 * 1000L) {
									logger.info("Remove terminated job " + job);
									spooledJobs.remove(job.getJobId());
									jobIterator.remove();
									job.stop();
									job.dispose();
								}
							}
						}
					}
					synchronized (monitor) {
						if (!dirty) {
							monitor.wait(10000);
						}
						dirty = false;
					}
				}
			}
			catch (final InterruptedException e) {
			}
		}
	}

	private class ScheduledJob {
		private final T job;
		private long time;

		/**
		 * @param job
		 * @param time
		 */
		public ScheduledJob(final T job, final long time) {
			this.job = job;
			this.time = time;
		}

		/**
		 * @param frameNumber
		 */
		public void setFrameNumber(final int frameNumber) {
			job.setFrameNumber(frameNumber);
		}

		/**
		 * 
		 */
		public void start() {
			job.start();
		}

		/**
		 * 
		 */
		public void stop() {
			time = 0;
			job.stop();
		}

		/**
		 * @return the job
		 */
		public T getJob() {
			return job;
		}

		/**
		 * @return the time
		 */
		public long getTime() {
			return time;
		}

		/**
		 * 
		 */
		public void dispose() {
		}
	}

	public void addServiceListener(JobServiceListener listener) {
		// TODO Auto-generated method stub
		
	}

	public void removeServiceListener(JobServiceListener listener) {
		// TODO Auto-generated method stub
		
	}
}
