/*
 * $Id:Service.java 479 2008-01-21 22:09:44Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.List;

import net.sf.jame.core.xml.XML;
import net.sf.jame.core.xml.XMLNodeBuilder;
import net.sf.jame.service.clip.MovieClipDataRow;
import net.sf.jame.service.clip.MovieClipService;
import net.sf.jame.service.encoder.EncoderContext;
import net.sf.jame.service.encoder.EncoderException;
import net.sf.jame.service.encoder.RAFEncoderContext;
import net.sf.jame.service.encoder.extension.EncoderExtensionRuntime;
import net.sf.jame.service.job.RenderJob;
import net.sf.jame.service.job.RenderJobDataRow;
import net.sf.jame.service.job.RenderJobService;
import net.sf.jame.service.profile.RenderProfileDataRow;
import net.sf.jame.service.profile.RenderProfileService;
import net.sf.jame.twister.TwisterClip;
import net.sf.jame.twister.TwisterClipController;
import net.sf.jame.twister.TwisterClipXMLExporter;
import net.sf.jame.twister.TwisterClipXMLImporter;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * @author Andrea Medeghini
 */
public class Service {
	private static final Logger logger = Logger.getLogger(Service.class);
	public static final int TILE_WIDTH = 256;
	public static final int TILE_HEIGHT = 256;
	private final List<JobStatusListener> statusListeners = new ArrayList<JobStatusListener>();
	private final List<ServiceListener> listeners = new ArrayList<ServiceListener>();
	private final File workdir;
	private MovieClipService clipService;
	private RenderProfileService profileService;
	private RenderJobService jobService;
	private Session session;

	/**
	 * @param session
	 * @param workdir
	 * @throws ServiceException
	 */
	public Service(final Session session, final File workdir) throws ServiceException {
		try {
			clipService = new MovieClipService(workdir);
			profileService = new RenderProfileService(workdir);
			jobService = new RenderJobService(workdir);
			this.workdir = workdir;
			this.session = session;
			clipService.init(session);
			profileService.init(session);
			jobService.init(session);
		}
		catch (final Exception e) {
			throw new ServiceException(e);
		}
	}

	/**
	 * @return
	 * @throws ServiceException
	 */
	public List<MovieClipDataRow> loadClips() throws ServiceException {
		try {
			session.openTransaction();
			final List<MovieClipDataRow> clips = clipService.loadAll(session);
			session.closeTransaction();
			return clips;
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param clip
	 * @throws ServiceException
	 */
	public void createClip(final MovieClipDataRow clip) throws ServiceException {
		try {
			session.openTransaction();
			clipService.create(session, clip);
			session.closeTransaction();
			fireClipCreated(clip);
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param clip
	 * @throws ServiceException
	 */
	public void saveClip(final MovieClipDataRow clip) throws ServiceException {
		try {
			session.openTransaction();
			clipService.save(session, clip);
			session.closeTransaction();
			fireClipUpdated(clip);
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param clip
	 * @throws ServiceException
	 */
	public void deleteClip(final MovieClipDataRow clip) throws ServiceException {
		try {
			session.openTransaction();
			final List<RenderProfileDataRow> profiles = profileService.loadAll(session, clip.getClipId());
			for (final RenderProfileDataRow profile : profiles) {
				profileService.delete(session, profile);
			}
			clipService.delete(session, clip);
			session.closeTransaction();
			for (final RenderProfileDataRow profile : profiles) {
				fireProfileDeleted(profile);
			}
			fireClipDeleted(clip);
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param clipId
	 * @return
	 * @throws ServiceException
	 */
	public MovieClipDataRow getClip(final int clipId) throws ServiceException {
		try {
			session.openTransaction();
			final MovieClipDataRow clip = clipService.getClip(session, clipId);
			session.closeTransaction();
			return clip;
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param clipId
	 * @return
	 * @throws ServiceException
	 */
	public List<RenderProfileDataRow> loadProfiles(final int clipId) throws ServiceException {
		try {
			session.openTransaction();
			final List<RenderProfileDataRow> profiles = profileService.loadAll(session, clipId);
			session.closeTransaction();
			return profiles;
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param clipId
	 * @return
	 */
	public List<RenderProfileDataRow> resetProfiles(final int clipId) throws ServiceException {
		try {
			session.openTransaction();
			final List<RenderProfileDataRow> profiles = profileService.loadAll(session, clipId);
			for (final RenderProfileDataRow profile : profiles) {
				profile.setJobCreated(0);
				profile.setJobStored(0);
				profileService.saveJobs(session, profile);
				profileService.clean(session, profile);
			}
			session.closeTransaction();
			for (final RenderProfileDataRow profile : profiles) {
				fireProfileUpdated(profile);
			}
			return profiles;
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param profile
	 * @throws ServiceException
	 */
	public void createProfile(final RenderProfileDataRow profile) throws ServiceException {
		try {
			session.openTransaction();
			profileService.create(session, profile);
			session.closeTransaction();
			fireProfileCreated(profile);
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param profile
	 * @throws ServiceException
	 */
	public void saveProfile(final RenderProfileDataRow profile) throws ServiceException {
		try {
			session.openTransaction();
			profileService.save(session, profile);
			session.closeTransaction();
			fireProfileUpdated(profile);
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param profile
	 * @throws ServiceException
	 */
	public void deleteProfile(final RenderProfileDataRow profile) throws ServiceException {
		try {
			session.openTransaction();
			final List<RenderJobDataRow> jobs = jobService.loadAll(session, profile.getProfileId());
			for (final RenderJobDataRow job : jobs) {
				jobService.delete(session, job);
			}
			profileService.delete(session, profile);
			session.closeTransaction();
			for (final RenderJobDataRow job : jobs) {
				fireJobDeleted(job);
			}
			fireProfileDeleted(profile);
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param profileId
	 * @return
	 * @throws ServiceException
	 */
	public RenderProfileDataRow getProfile(final int profileId) throws ServiceException {
		try {
			session.openTransaction();
			final RenderProfileDataRow profile = profileService.getProfile(session, profileId);
			session.closeTransaction();
			return profile;
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	// /**
	// * @param profileId
	// * @return
	// * @throws ServiceException
	// */
	// public List<RenderJobDataRow> loadJobs(final int profileId) throws ServiceException {
	// try {
	// session.openTransaction();
	// final List<RenderJobDataRow> jobs = jobService.loadAll(session, profileId);
	// session.closeTransaction();
	// return jobs;
	// }
	// catch (final TransactionException e) {
	// throw new ServiceException(e);
	// }
	// catch (final Exception e) {
	// try {
	// session.abortTransaction();
	// }
	// catch (final TransactionException x) {
	// x.printStackTrace();
	// }
	// throw new ServiceException(e);
	// }
	// }
	/**
	 * @return
	 * @throws ServiceException
	 */
	public List<RenderJobDataRow> loadJobs() throws ServiceException {
		try {
			session.openTransaction();
			final List<RenderJobDataRow> jobs = jobService.loadAll(session);
			session.closeTransaction();
			return jobs;
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param job
	 * @throws ServiceException
	 */
	public void createJob(final RenderJobDataRow job) throws ServiceException {
		try {
			session.openTransaction();
			final RenderProfileDataRow profile = profileService.getProfile(session, job.getProfileId());
			final MovieClipDataRow clip = clipService.getClip(session, profile.getClipId());
			clip.setStatus(clip.getStatus() + 1);
			clipService.saveStatus(session, clip);
			profile.setStatus(profile.getStatus() + 1);
			profileService.saveStatus(session, profile);
			jobService.create(session, job);
			startJob(job);
			session.closeTransaction();
			fireProfileUpdated(profile);
			fireClipUpdated(clip);
			fireJobCreated(job);
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param job
	 * @throws ServiceException
	 */
	public void saveJob(final RenderJobDataRow job) throws ServiceException {
		try {
			session.openTransaction();
			jobService.save(session, job);
			session.closeTransaction();
			fireJobUpdated(job);
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param job
	 * @throws ServiceException
	 */
	public void deleteJob(final RenderJobDataRow job) throws ServiceException {
		try {
			session.openTransaction();
			final RenderProfileDataRow profile = profileService.getProfile(session, job.getProfileId());
			final MovieClipDataRow clip = clipService.getClip(session, profile.getClipId());
			clip.setStatus(clip.getStatus() - 1);
			clipService.saveStatus(session, clip);
			profile.setStatus(profile.getStatus() - 1);
			profileService.saveStatus(session, profile);
			stopJob(job);
			jobService.delete(session, job);
			session.closeTransaction();
			fireProfileUpdated(profile);
			fireClipUpdated(clip);
			fireJobAborted(job);
			fireJobStopped(job);
			fireJobDeleted(job);
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param jobId
	 * @return
	 * @throws ServiceException
	 */
	public RenderJobDataRow getJob(final int jobId) throws ServiceException {
		try {
			session.openTransaction();
			final RenderJobDataRow job = jobService.getJob(session, jobId);
			session.closeTransaction();
			return job;
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	public void saveJobStatus(final RenderJobDataRow job) throws ServiceException {
		try {
			session.openTransaction();
			jobService.saveStatus(session, job);
			session.closeTransaction();
			fireJobStatus(job);
		}
		catch (final Exception e) {
			throw new ServiceException(e);
		}
	}

	private void startJob(RenderJobDataRow job) throws ServiceException {
		try {
			job.setStatus(1);
			jobService.saveStatus(session, job);
			job = jobService.getJob(session, job.getJobId());
			fireJobStatus(job);
		}
		catch (final Exception e) {
			throw new ServiceException(e);
		}
	}

	private void stopJob(RenderJobDataRow job) throws ServiceException {
		try {
			job.setStatus(0);
			jobService.saveStatus(session, job);
			job = jobService.getJob(session, job.getJobId());
			fireJobStatus(job);
		}
		catch (final Exception e) {
			throw new ServiceException(e);
		}
	}

	/**
	 * @throws ServiceException
	 */
	public void startJobs() throws ServiceException {
		try {
			session.openTransaction();
			final List<RenderJobDataRow> jobs = jobService.loadAll(session);
			for (final RenderJobDataRow job : jobs) {
				stopJob(job);
				startJob(job);
			}
			session.closeTransaction();
			for (final RenderJobDataRow job : jobs) {
				fireJobStarted(job);
			}
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @throws ServiceException
	 */
	public void stopJobs() throws ServiceException {
		try {
			session.openTransaction();
			final List<RenderJobDataRow> jobs = jobService.loadAll(session);
			for (final RenderJobDataRow job : jobs) {
				stopJob(job);
			}
			session.closeTransaction();
			for (final RenderJobDataRow job : jobs) {
				fireJobAborted(job);
			}
			for (final RenderJobDataRow job : jobs) {
				fireJobStopped(job);
			}
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @throws ServiceException
	 */
	public void deleteJobs() throws ServiceException {
		try {
			session.openTransaction();
			final List<RenderJobDataRow> jobs = jobService.loadAll(session);
			for (final RenderJobDataRow job : jobs) {
				stopJob(job);
				jobService.delete(session, job);
			}
			final List<RenderProfileDataRow> profiles = profileService.loadAll(session);
			for (final RenderProfileDataRow profile : profiles) {
				profile.setStatus(0);
				profileService.saveStatus(session, profile);
			}
			final List<MovieClipDataRow> clips = clipService.loadAll(session);
			for (final MovieClipDataRow clip : clips) {
				clip.setStatus(0);
				clipService.saveStatus(session, clip);
			}
			session.closeTransaction();
			for (final RenderProfileDataRow profile : profiles) {
				fireProfileUpdated(profile);
			}
			for (final MovieClipDataRow clip : clips) {
				fireClipUpdated(clip);
			}
			for (final RenderJobDataRow job : jobs) {
				fireJobAborted(job);
			}
			for (final RenderJobDataRow job : jobs) {
				fireJobStopped(job);
			}
			for (final RenderJobDataRow job : jobs) {
				fireJobDeleted(job);
			}
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param profileId
	 */
	public void startJobs(final int profileId) throws ServiceException {
		try {
			session.openTransaction();
			final List<RenderJobDataRow> jobs = jobService.loadAll(session, profileId);
			for (final RenderJobDataRow job : jobs) {
				stopJob(job);
				startJob(job);
			}
			session.closeTransaction();
			for (final RenderJobDataRow job : jobs) {
				fireJobAborted(job);
			}
			for (final RenderJobDataRow job : jobs) {
				fireJobStopped(job);
			}
			for (final RenderJobDataRow job : jobs) {
				fireJobStarted(job);
			}
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param profileId
	 */
	public void stopJobs(final int profileId) throws ServiceException {
		try {
			session.openTransaction();
			final List<RenderJobDataRow> jobs = jobService.loadAll(session, profileId);
			for (final RenderJobDataRow job : jobs) {
				stopJob(job);
			}
			session.closeTransaction();
			for (final RenderJobDataRow job : jobs) {
				fireJobAborted(job);
			}
			for (final RenderJobDataRow job : jobs) {
				fireJobStopped(job);
			}
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @throws ServiceException
	 */
	public void resumeJobs() throws ServiceException {
		try {
			session.openTransaction();
			final List<RenderJobDataRow> jobs = jobService.loadAll(session);
//			for (final RenderJobDataRow job : jobs) {
//				startJob(job);
//			}
			session.closeTransaction();
			for (final RenderJobDataRow job : jobs) {
				fireJobResumed(job);
			}
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param profileId
	 * @throws ServiceException
	 */
	public void deleteJobs(final int profileId) throws ServiceException {
		try {
			session.openTransaction();
			final List<RenderJobDataRow> jobs = jobService.loadAll(session, profileId);
			final RenderProfileDataRow profile = profileService.getProfile(session, profileId);
			final MovieClipDataRow clip = clipService.getClip(session, profile.getClipId());
			for (final RenderJobDataRow job : jobs) {
				stopJob(job);
				jobService.delete(session, job);
			}
			clip.setStatus(clip.getStatus() - profile.getStatus());
			clipService.saveStatus(session, clip);
			profile.setStatus(0);
			profileService.saveStatus(session, profile);
			session.closeTransaction();
			fireProfileUpdated(profile);
			fireClipUpdated(clip);
			for (final RenderJobDataRow job : jobs) {
				fireJobAborted(job);
			}
			for (final RenderJobDataRow job : jobs) {
				fireJobStopped(job);
			}
			for (final RenderJobDataRow job : jobs) {
				fireJobDeleted(job);
			}
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param profileId
	 * @throws ServiceException
	 */
	public void createJobs(final int profileId) throws ServiceException {
		try {
			session.openTransaction();
			final RenderProfileDataRow profile = profileService.getProfile(session, profileId);
			final MovieClipDataRow clip = clipService.getClip(session, profile.getClipId());
			profile.setTotalFrames((profile.getStopTime() - profile.getStartTime()) * profile.getFrameRate());
			profile.setJobFrame(0);
			profileService.saveFrames(session, profile);
			final List<RenderJobDataRow> jobs = createJobs(clip, profile);
			session.closeTransaction();
			fireProfileUpdated(profile);
			fireClipUpdated(clip);
			if (Service.logger.isInfoEnabled()) {
				Service.logger.info("Created " + profile.getJobCreated() + " jobs");
			}
			for (final RenderJobDataRow job : jobs) {
				fireJobCreated(job);
			}
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	private List<RenderJobDataRow> createJobs(final MovieClipDataRow clip, final RenderProfileDataRow profile) throws SessionException, DataTableException, ServiceException {
		final List<RenderJobDataRow> jobs = new ArrayList<RenderJobDataRow>();
		final int nx = profile.getImageWidth() / Service.TILE_WIDTH;
		final int ny = profile.getImageHeight() / Service.TILE_HEIGHT;
		final int rx = profile.getImageWidth() - nx * Service.TILE_WIDTH;
		final int ry = profile.getImageHeight() - ny * Service.TILE_HEIGHT;
		if ((nx > 0) && (ny > 0)) {
			for (int i = 0; i < nx; i++) {
				for (int j = 0; j < ny; j++) {
					final RenderJobDataRow job = new RenderJobDataRow(new RenderJob());
					job.setClipName(profile.getClipName());
					job.setProfileName(profile.getProfileName());
					job.setClipId(profile.getClipId());
					job.setProfileId(profile.getProfileId());
					job.setFrameNumber(profile.getJobFrame());
					job.setImageWidth(profile.getImageWidth());
					job.setImageHeight(profile.getImageHeight());
					job.setTileWidth(Service.TILE_WIDTH);
					job.setTileHeight(Service.TILE_HEIGHT);
					job.setBorderWidth(24);
					job.setBorderHeight(24);
					job.setTileOffsetX(Service.TILE_WIDTH * i);
					job.setTileOffsetY(Service.TILE_HEIGHT * j);
					job.setQuality(profile.getQuality());
					job.setFrameRate(profile.getFrameRate());
					jobs.add(job);
				}
			}
		}
		if (rx > 0) {
			for (int j = 0; j < ny; j++) {
				final RenderJobDataRow job = new RenderJobDataRow(new RenderJob());
				job.setClipName(profile.getClipName());
				job.setProfileName(profile.getProfileName());
				job.setClipId(profile.getClipId());
				job.setProfileId(profile.getProfileId());
				job.setFrameNumber(profile.getJobFrame());
				job.setImageWidth(profile.getImageWidth());
				job.setImageHeight(profile.getImageHeight());
				job.setTileWidth(rx);
				job.setTileHeight(Service.TILE_HEIGHT);
				job.setBorderWidth(24);
				job.setBorderHeight(24);
				job.setTileOffsetX(nx * Service.TILE_WIDTH);
				job.setTileOffsetY(Service.TILE_HEIGHT * j);
				job.setQuality(profile.getQuality());
				job.setFrameRate(profile.getFrameRate());
				jobs.add(job);
			}
		}
		if (ry > 0) {
			for (int i = 0; i < nx; i++) {
				final RenderJobDataRow job = new RenderJobDataRow(new RenderJob());
				job.setClipName(profile.getClipName());
				job.setProfileName(profile.getProfileName());
				job.setClipId(profile.getClipId());
				job.setProfileId(profile.getProfileId());
				job.setFrameNumber(profile.getJobFrame());
				job.setImageWidth(profile.getImageWidth());
				job.setImageHeight(profile.getImageHeight());
				job.setTileWidth(Service.TILE_WIDTH);
				job.setTileHeight(ry);
				job.setBorderWidth(24);
				job.setBorderHeight(24);
				job.setTileOffsetX(Service.TILE_WIDTH * i);
				job.setTileOffsetY(ny * Service.TILE_HEIGHT);
				job.setQuality(profile.getQuality());
				job.setFrameRate(profile.getFrameRate());
				jobs.add(job);
			}
		}
		if ((rx > 0) && (ry > 0)) {
			final RenderJobDataRow job = new RenderJobDataRow(new RenderJob());
			job.setClipName(profile.getClipName());
			job.setProfileName(profile.getProfileName());
			job.setClipId(profile.getClipId());
			job.setProfileId(profile.getProfileId());
			job.setFrameNumber(profile.getJobFrame());
			job.setImageWidth(profile.getImageWidth());
			job.setImageHeight(profile.getImageHeight());
			job.setTileWidth(rx);
			job.setTileHeight(ry);
			job.setBorderWidth(24);
			job.setBorderHeight(24);
			job.setTileOffsetX(nx * Service.TILE_WIDTH);
			job.setTileOffsetY(ny * Service.TILE_HEIGHT);
			job.setQuality(profile.getQuality());
			job.setFrameRate(profile.getFrameRate());
			jobs.add(job);
		}
		profile.setJobCreated(jobs.size());
		profile.setJobStored(0);
		profileService.saveJobs(session, profile);
		for (final RenderJobDataRow job : jobs) {
			clip.setStatus(clip.getStatus() + 1);
			clipService.saveStatus(session, clip);
			profile.setStatus(profile.getStatus() + 1);
			profileService.saveStatus(session, profile);
			jobService.create(session, job);
			startJob(job);
		}
		return jobs;
	}

	private List<RenderJobDataRow> createPostProcessJob(final MovieClipDataRow clip, final RenderProfileDataRow profile) throws SessionException, DataTableException, ServiceException {
		final List<RenderJobDataRow> jobs = new ArrayList<RenderJobDataRow>();
		final RenderJobDataRow job = new RenderJobDataRow(new RenderJob());
		job.setClipName(profile.getClipName());
		job.setProfileName(profile.getProfileName());
		job.setClipId(profile.getClipId());
		job.setProfileId(profile.getProfileId());
		job.setFrameNumber(0);
		job.setFrameRate(profile.getFrameRate());
		job.setStartTime(profile.getStartTime());
		job.setStopTime(profile.getStopTime());
		job.setTileWidth(profile.getImageWidth());
		job.setTileHeight(profile.getImageHeight());
		job.setImageWidth(profile.getImageWidth());
		job.setImageHeight(profile.getImageHeight());
		job.setBorderWidth(0);
		job.setBorderHeight(0);
		job.setTileOffsetX(0);
		job.setTileOffsetY(0);
		job.setType(2);
		jobs.add(job);
		clip.setStatus(clip.getStatus() + 1);
		clipService.saveStatus(session, clip);
		profile.setStatus(profile.getStatus() + 1);
		profileService.saveStatus(session, profile);
		jobService.create(session, job);
		return jobs;
	}

	private List<RenderJobDataRow> createCopyProcessJob(final MovieClipDataRow clip, final RenderProfileDataRow profile, final RenderJobDataRow processJob) throws SessionException, DataTableException, ServiceException {
		final List<RenderJobDataRow> jobs = new ArrayList<RenderJobDataRow>();
		final RenderJobDataRow job = new RenderJobDataRow(new RenderJob());
		job.setClipName(processJob.getClipName());
		job.setProfileName(processJob.getProfileName());
		job.setClipId(processJob.getClipId());
		job.setProfileId(processJob.getProfileId());
		job.setFrameNumber(0);
		job.setFrameRate(processJob.getFrameRate());
		job.setStartTime(processJob.getStartTime());
		job.setStopTime(processJob.getStopTime());
		job.setTileWidth(processJob.getTileWidth());
		job.setTileHeight(processJob.getTileHeight());
		job.setImageWidth(processJob.getImageWidth());
		job.setImageHeight(processJob.getImageHeight());
		job.setBorderWidth(processJob.getBorderWidth());
		job.setBorderHeight(processJob.getBorderHeight());
		job.setTileOffsetX(processJob.getTileOffsetX());
		job.setTileOffsetY(processJob.getTileOffsetY());
		job.setType(1);
		job.setJobId(processJob.getJobId());
		jobs.add(job);
		clip.setStatus(clip.getStatus() + 1);
		clipService.saveStatus(session, clip);
		profile.setStatus(profile.getStatus() + 1);
		profileService.saveStatus(session, profile);
		jobService.save(session, job);
		return jobs;
	}

	private List<RenderJobDataRow> createProcessJob(final MovieClipDataRow clip, final RenderProfileDataRow profile, final RenderJobDataRow processJob) throws SessionException, DataTableException, ServiceException {
		final List<RenderJobDataRow> jobs = new ArrayList<RenderJobDataRow>();
		final RenderJobDataRow job = new RenderJobDataRow(new RenderJob());
		job.setClipName(processJob.getClipName());
		job.setProfileName(processJob.getProfileName());
		job.setClipId(processJob.getClipId());
		job.setProfileId(processJob.getProfileId());
		job.setFrameNumber(processJob.getFrameNumber());
		job.setFrameRate(processJob.getFrameRate());
		job.setStartTime(processJob.getStartTime());
		job.setStopTime(processJob.getStopTime());
		job.setTileWidth(processJob.getTileWidth());
		job.setTileHeight(processJob.getTileHeight());
		job.setImageWidth(processJob.getImageWidth());
		job.setImageHeight(processJob.getImageHeight());
		job.setBorderWidth(processJob.getBorderWidth());
		job.setBorderHeight(processJob.getBorderHeight());
		job.setTileOffsetX(processJob.getTileOffsetX());
		job.setTileOffsetY(processJob.getTileOffsetY());
		job.setType(0);
		job.setJobId(processJob.getJobId());
		jobs.add(job);
		clip.setStatus(clip.getStatus() + 1);
		clipService.saveStatus(session, clip);
		profile.setStatus(profile.getStatus() + 1);
		profileService.saveStatus(session, profile);
		jobService.save(session, job);
		return jobs;
	}

	/**
	 * @param job
	 * @throws ServiceException
	 */
	public void jobCompleted(final RenderJobDataRow job) throws ServiceException {
		try {
			synchronized (session) {
				session.openTransaction();
				final MovieClipDataRow clip = clipService.getClip(session, job.getClipId());
				final RenderProfileDataRow profile = profileService.getProfile(session, job.getProfileId());
				clip.setStatus(clip.getStatus() - 1);
				clipService.saveStatus(session, clip);
				profile.setStatus(profile.getStatus() - 1);
				profileService.saveStatus(session, profile);
				job.setStatus(0);
				jobService.saveStatus(session, job);
				List<RenderJobDataRow> jobs = null;
				profile.setJobFrame(job.getFrameNumber());
				profileService.saveFrames(session, profile);
				if (job.isCopyProcess()) {
					profile.setJobStored(profile.getJobStored() + 1);
					profileService.saveJobs(session, profile);
					jobService.delete(session, job);
				}
				if (job.isCopyProcess() && profile.getJobStored() == profile.getJobCreated()) {
					jobs = createPostProcessJob(clip, profile);
					for (final RenderJobDataRow createdJob : jobs) {
						startJob(createdJob);
					}
				}
				else if (job.isProcess() && (job.getFrameRate() == 0 || job.getFrameNumber() == (profile.getTotalFrames() - 1))) {
					jobs = createCopyProcessJob(clip, profile, job);
					for (final RenderJobDataRow createdJob : jobs) {
						startJob(createdJob);
					}
				}
				else if (job.isProcess()) {
					jobs = createProcessJob(clip, profile, job);
					for (final RenderJobDataRow createdJob : jobs) {
						startJob(createdJob);
					}
				}
				else if (job.isPostProcess()) {
					jobService.delete(session, job);
				}
				session.closeTransaction();
				fireProfileUpdated(profile);
				fireClipUpdated(clip);
				if (job.isCopyProcess() || job.isPostProcess()) {
					fireJobDeleted(job);
				}
				if (jobs != null) {
					for (final RenderJobDataRow createdJob : jobs) {
						fireJobCreated(createdJob);
						fireJobStarted(createdJob);
					}
				}
			}
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @throws ServiceException
	 */
	public void deleteAll() throws ServiceException {
		try {
			session.openTransaction();
			jobService.deleteAll(session);
			profileService.deleteAll(session);
			clipService.deleteAll(session);
			session.closeTransaction();
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param dataRow
	 * @param file
	 * @throws ServiceException
	 */
	public void importClip(final MovieClipDataRow dataRow, final File file) throws ServiceException {
		try {
			session.openTransaction();
			final TwisterClipXMLImporter importer = new TwisterClipXMLImporter();
			final InputStream is = new FileInputStream(file);
			Document doc = XML.loadDocument(is, "twister-clip.xml");
			final TwisterClip clip = importer.importFromElement(doc.getDocumentElement());
			is.close();
			// MovieClipDataRow clipData = clipService.getClip(session, dataRow.getClipId());
			final TwisterClipController controller = new TwisterClipController(clip);
			controller.init();
			dataRow.setDuration(controller.getDuration());
			clipService.save(session, dataRow);
			final TwisterClipXMLExporter exporter = new TwisterClipXMLExporter();
			doc = XML.createDocument();
			final XMLNodeBuilder builder = XML.createDefaultXMLNodeBuilder(doc);
			final Element element = exporter.exportToElement(clip, builder);
			doc.appendChild(element);
			final OutputStream os = clipService.getOutputStream(dataRow.getClipId());
			XML.saveDocument(os, "twister-clip.xml", doc);
			os.close();
			session.closeTransaction();
			fireClipUpdated(dataRow);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param dataRow
	 * @param file
	 * @throws ServiceException
	 */
	public void exportClip(final MovieClipDataRow dataRow, final File file) throws ServiceException {
		try {
			final TwisterClipXMLImporter importer = new TwisterClipXMLImporter();
			final InputStream is = clipService.getInputStream(dataRow.getClipId());
			Document doc = XML.loadDocument(is, "twister-clip.xml");
			final TwisterClip clip = importer.importFromElement(doc.getDocumentElement());
			is.close();
			final TwisterClipXMLExporter exporter = new TwisterClipXMLExporter();
			doc = XML.createDocument();
			final XMLNodeBuilder builder = XML.createDefaultXMLNodeBuilder(doc);
			final Element element = exporter.exportToElement(clip, builder);
			doc.appendChild(element);
			final OutputStream os = new FileOutputStream(file);
			XML.saveDocument(os, "twister-clip.xml", doc);
			os.close();
		}
		catch (final Exception e) {
			throw new ServiceException(e);
		}
	}

	/**
	 * @param profile
	 * @param encoder
	 * @param path
	 * @throws ServiceException
	 */
	public void exportProfile(final RenderProfileDataRow profile, final EncoderExtensionRuntime<?> encoder, final File path) throws ServiceException {
		try {
			final EncoderContext context = new DefaultEncoderContext(profile);
			encoder.encode(context, path);
		}
		catch (final EncoderException e) {
			throw new ServiceException(e);
		}
		catch (final IOException e) {
			throw new ServiceException(e);
		}
	}

	/**
	 * @param listener
	 */
	public void addServiceListener(final ServiceListener listener) {
		listeners.add(listener);
	}

	/**
	 * @param listener
	 */
	public void removeServiceListener(final ServiceListener listener) {
		listeners.remove(listener);
	}

	/**
	 * @param listener
	 */
	public void addJobStatusListener(final JobStatusListener listener) {
		statusListeners.add(listener);
	}

	/**
	 * @param listener
	 */
	public void removeJobStatusListener(final JobStatusListener listener) {
		statusListeners.remove(listener);
	}

	/**
	 * @param clip
	 */
	protected void fireClipCreated(final MovieClipDataRow clip) {
		for (final ServiceListener listener : listeners) {
			listener.clipCreated(clip);
		}
	}

	/**
	 * @param clip
	 */
	protected void fireClipDeleted(final MovieClipDataRow clip) {
		for (final ServiceListener listener : listeners) {
			listener.clipDeleted(clip);
		}
	}

	/**
	 * @param clip
	 */
	protected void fireClipUpdated(final MovieClipDataRow clip) {
		for (final ServiceListener listener : listeners) {
			listener.clipUpdated(clip);
		}
	}

	/**
	 * @param profile
	 */
	protected void fireProfileCreated(final RenderProfileDataRow profile) {
		for (final ServiceListener listener : listeners) {
			listener.profileCreated(profile);
		}
	}

	/**
	 * @param profile
	 */
	protected void fireProfileDeleted(final RenderProfileDataRow profile) {
		for (final ServiceListener listener : listeners) {
			listener.profileDeleted(profile);
		}
	}

	/**
	 * @param profile
	 */
	protected void fireProfileUpdated(final RenderProfileDataRow profile) {
		for (final ServiceListener listener : listeners) {
			listener.profileUpdated(profile);
		}
	}

	/**
	 * @param job
	 */
	protected void fireJobCreated(final RenderJobDataRow job) {
		for (final ServiceListener listener : listeners) {
			listener.jobCreated(job);
		}
	}

	/**
	 * @param job
	 */
	protected void fireJobDeleted(final RenderJobDataRow job) {
		for (final ServiceListener listener : listeners) {
			listener.jobDeleted(job);
		}
	}

	/**
	 * @param job
	 */
	protected void fireJobUpdated(final RenderJobDataRow job) {
		for (final ServiceListener listener : listeners) {
			listener.jobUpdated(job);
		}
	}

	/**
	 * @param job
	 */
	protected void fireJobStarted(final RenderJobDataRow job) {
		for (final ServiceListener listener : listeners) {
			listener.jobStarted(job);
		}
	}

	/**
	 * @param job
	 */
	protected void fireJobAborted(final RenderJobDataRow job) {
		for (final ServiceListener listener : listeners) {
			listener.jobAborted(job);
		}
	}

	/**
	 * @param job
	 */
	protected void fireJobStopped(final RenderJobDataRow job) {
		for (final ServiceListener listener : listeners) {
			listener.jobStopped(job);
		}
	}

	/**
	 * @param job
	 */
	protected void fireJobResumed(final RenderJobDataRow job) {
		for (final ServiceListener listener : listeners) {
			listener.jobResumed(job);
		}
	}

	/**
	 * @param job
	 */
	protected void fireJobStatus(final RenderJobDataRow job) {
		for (final JobStatusListener listener : statusListeners) {
			listener.jobStatusChanged(job);
		}
	}

	/**
	 * @param clipId
	 * @return
	 * @throws FileNotFoundException
	 */
	public OutputStream getClipOutputStream(final int clipId) throws FileNotFoundException {
		return clipService.getOutputStream(clipId);
	}

	/**
	 * @param profileId
	 * @return
	 * @throws FileNotFoundException
	 */
	public OutputStream getProfileOutputStream(final int profileId) throws FileNotFoundException {
		return profileService.getOutputStream(profileId);
	}

	/**
	 * @param jobId
	 * @return
	 * @throws FileNotFoundException
	 */
	public OutputStream getJobOutputStream(final int jobId) throws FileNotFoundException {
		return jobService.getOutputStream(jobId);
	}

	/**
	 * @param jobId
	 * @param append
	 * @return
	 * @throws FileNotFoundException
	 */
	public OutputStream getJobOutputStream(final int jobId, final boolean append) throws FileNotFoundException {
		return jobService.getOutputStream(jobId, append);
	}

	/**
	 * @param clipId
	 * @return
	 * @throws FileNotFoundException
	 */
	public InputStream getClipInputStream(final int clipId) throws FileNotFoundException {
		return clipService.getInputStream(clipId);
	}

	/**
	 * @param profileId
	 * @return
	 * @throws FileNotFoundException
	 */
	public InputStream getProfileInputStream(final int profileId) throws FileNotFoundException {
		return profileService.getInputStream(profileId);
	}

	/**
	 * @param jobId
	 * @return
	 * @throws FileNotFoundException
	 */
	public InputStream getJobInputStream(final int jobId) throws FileNotFoundException {
		return jobService.getInputStream(jobId);
	}

	/**
	 * @param clip
	 * @throws ServiceException
	 */
	public void cleanClip(final MovieClipDataRow clip) throws ServiceException {
		try {
			session.openTransaction();
			clipService.clean(session, clip);
			session.closeTransaction();
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param profile
	 * @throws ServiceException
	 */
	public void cleanProfile(final RenderProfileDataRow profile) throws ServiceException {
		try {
			session.openTransaction();
			profile.setJobCreated(0);
			profile.setJobStored(0);
			profileService.saveJobs(session, profile);
			profileService.clean(session, profile);
			session.closeTransaction();
			fireProfileUpdated(profile);
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	/**
	 * @param job
	 * @throws ServiceException
	 */
	public void cleanJob(final RenderJobDataRow job) throws ServiceException {
		try {
			session.openTransaction();
			jobService.clean(session, job);
			session.closeTransaction();
		}
		catch (final TransactionException e) {
			throw new ServiceException(e);
		}
		catch (final Exception e) {
			try {
				session.abortTransaction();
			}
			catch (final TransactionException x) {
				x.printStackTrace();
			}
			throw new ServiceException(e);
		}
	}

	private class DefaultEncoderContext extends RAFEncoderContext {
		/**
		 * @param profile
		 * @throws IOException
		 */
		public DefaultEncoderContext(final RenderProfileDataRow profile) throws IOException {
			super(profileService.getRandomAccessFile(profile.getProfileId()), profile.getImageWidth(), profile.getImageHeight(), profile.getFrameRate(), (profile.getStopTime() - profile.getStartTime()) * profile.getFrameRate());
		}
	}

	public RandomAccessFile getClipRandomAccessFile(final int clipId) throws FileNotFoundException {
		return clipService.getRandomAccessFile(clipId);
	}

	public RandomAccessFile getProfileRandomAccessFile(final int profileId) throws FileNotFoundException {
		return profileService.getRandomAccessFile(profileId);
	}

	public RandomAccessFile getJobRandomAccessFile(final int jobId) throws FileNotFoundException {
		return jobService.getRandomAccessFile(jobId);
	}

	/**
	 * @return
	 */
	public File getWorkspace() {
		return workdir;
	}
}
