/*
 * $Id:JMFAbstractEncoderRuntime.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service.extensions.encoder;

import java.awt.Dimension;
import java.io.File;
import java.io.IOException;

import javax.media.Buffer;
import javax.media.ConfigureCompleteEvent;
import javax.media.Controller;
import javax.media.ControllerEvent;
import javax.media.ControllerListener;
import javax.media.DataSink;
import javax.media.Duration;
import javax.media.EndOfMediaEvent;
import javax.media.Format;
import javax.media.Manager;
import javax.media.MediaLocator;
import javax.media.NoDataSinkException;
import javax.media.NoProcessorException;
import javax.media.NotRealizedError;
import javax.media.PrefetchCompleteEvent;
import javax.media.Processor;
import javax.media.RealizeCompleteEvent;
import javax.media.ResourceUnavailableEvent;
import javax.media.Time;
import javax.media.control.TrackControl;
import javax.media.datasink.DataSinkErrorEvent;
import javax.media.datasink.DataSinkEvent;
import javax.media.datasink.DataSinkListener;
import javax.media.datasink.EndOfStreamEvent;
import javax.media.format.RGBFormat;
import javax.media.protocol.ContentDescriptor;
import javax.media.protocol.DataSource;
import javax.media.protocol.PullBufferDataSource;
import javax.media.protocol.PullBufferStream;

import net.sf.jame.service.encoder.EncoderContext;
import net.sf.jame.service.encoder.EncoderException;
import net.sf.jame.service.encoder.EncoderHook;
import net.sf.jame.service.encoder.extension.EncoderExtensionConfig;

import org.apache.log4j.Logger;

/**
 * @author Andrea Medeghini
 */
public abstract class JMFAbstractEncoderRuntime<T extends EncoderExtensionConfig> extends AbstractEncoderExtensionRuntime<T> implements DataSinkListener, ControllerListener {
	private static final Logger logger = Logger.getLogger(JMFAbstractEncoderRuntime.class);
	private final Object waitSync = new Object();
	private final Object waitFileSync = new Object();
	private boolean stateTransitionOK = true;
	private boolean fileDone = false;
	private boolean fileSuccess = true;
	private EncoderHook hook;

	/**
	 * @see net.sf.jame.service.encoder.extension.EncoderExtensionRuntime#setInterruptHook(net.sf.jame.service.encoder.EncoderHook)
	 */
	@Override
	public void setInterruptHook(final EncoderHook hook) {
		this.hook = hook;
	}

	/**
	 * @see net.sf.jame.service.encoder.extension.EncoderExtensionRuntime#encode(net.sf.jame.service.encoder.EncoderContext, java.io.File)
	 */
	@Override
	public void encode(final EncoderContext context, final File path) throws EncoderException {
		fireStatusChanged(0);
		if (isMovieSupported()) {
			try {
				long time = System.currentTimeMillis();
				final DataSource source = new ImageSource(context);
				final MediaLocator dest = new MediaLocator("file://" + path.getAbsolutePath());
				final Processor processor = Manager.createProcessor(source);
				processor.addControllerListener(this);
				processor.configure();
				if (!this.waitForState(processor, Processor.Configured)) {
					throw new EncoderException("Failed to configure the processor.");
				}
				processor.setContentDescriptor(new ContentDescriptor(this.getFormatName()));
				processor.realize();
				final TrackControl[] trs = processor.getTrackControls();
				JMFAbstractEncoderRuntime.logger.info("Video format: " + trs[0].getFormat());
				if (!this.waitForState(processor, Controller.Realized)) {
					throw new EncoderException("Failed to realize the processor.");
				}
				final DataSink filewriter = Manager.createDataSink(processor.getDataOutput(), dest);
				if (filewriter == null) {
					throw new EncoderException("Failed to create a DataSink for the given output MediaLocator: " + dest);
				}
				if (JMFAbstractEncoderRuntime.logger.isDebugEnabled()) {
					JMFAbstractEncoderRuntime.logger.debug("start processing...");
				}
				filewriter.addDataSinkListener(this);
				this.fileDone = false;
				try {
					filewriter.open();
					filewriter.start();
					processor.start();
					this.waitForFileDone();
					filewriter.stop();
					filewriter.close();
				}
				catch (final IOException e) {
					throw new EncoderException("IO error during processing", e);
				}
				if (JMFAbstractEncoderRuntime.logger.isDebugEnabled()) {
					JMFAbstractEncoderRuntime.logger.debug("...done processing.");
				}
				time = System.currentTimeMillis() - time;
				if (JMFAbstractEncoderRuntime.logger.isInfoEnabled()) {
					JMFAbstractEncoderRuntime.logger.info("Profile exported: elapsed time " + String.format("%3.2f", time / 1000.0d) + "s");
				}
			}
			catch (final NoProcessorException e) {
				throw new EncoderException(e);
			}
			catch (final NoDataSinkException e) {
				throw new EncoderException(e);
			}
			catch (final NotRealizedError e) {
				throw new EncoderException(e);
			}
			catch (final IOException e) {
				throw new EncoderException(e);
			}
		}
		else {
			throw new EncoderException("Can't encode the movie");
		}
	}

	/**
	 * @return
	 */
	protected abstract String getFormatName();

	private boolean waitForState(final Processor p, final int state) {
		synchronized (this.waitSync) {
			try {
				while ((p.getState() < state) && this.stateTransitionOK) {
					this.waitSync.wait();
				}
			}
			catch (final Exception e) {
			}
		}
		return this.stateTransitionOK;
	}

	private boolean waitForFileDone() {
		synchronized (this.waitFileSync) {
			try {
				while (!this.fileDone) {
					this.waitFileSync.wait();
				}
			}
			catch (final Exception e) {
			}
		}
		return this.fileSuccess;
	}

	/**
	 * @see javax.media.ControllerListener#controllerUpdate(javax.media.ControllerEvent)
	 */
	public void controllerUpdate(final ControllerEvent evt) {
		if ((evt instanceof ConfigureCompleteEvent) || (evt instanceof RealizeCompleteEvent) || (evt instanceof PrefetchCompleteEvent)) {
			synchronized (this.waitSync) {
				this.stateTransitionOK = true;
				this.waitSync.notifyAll();
			}
		}
		else if (evt instanceof ResourceUnavailableEvent) {
			synchronized (this.waitSync) {
				this.stateTransitionOK = false;
				this.waitSync.notifyAll();
			}
		}
		else if (evt instanceof EndOfMediaEvent) {
			evt.getSourceController().stop();
			evt.getSourceController().close();
		}
	}

	/**
	 * @see javax.media.datasink.DataSinkListener#dataSinkUpdate(javax.media.datasink.DataSinkEvent)
	 */
	public void dataSinkUpdate(final DataSinkEvent evt) {
		if (evt instanceof EndOfStreamEvent) {
			synchronized (this.waitFileSync) {
				this.fileDone = true;
				this.waitFileSync.notifyAll();
			}
		}
		else if (evt instanceof DataSinkErrorEvent) {
			synchronized (this.waitFileSync) {
				this.fileDone = true;
				this.fileSuccess = false;
				this.waitFileSync.notifyAll();
			}
		}
	}

	private class ImageSource extends PullBufferDataSource {
		private final EncoderContext context;

		public ImageSource(final EncoderContext context) {
			this.context = context;
		}

		@Override
		public PullBufferStream[] getStreams() {
			return new PullBufferStream[] { new ImageStream(this.context) };
		}

		@Override
		public void connect() throws IOException {
		}

		@Override
		public void disconnect() {
		}

		@Override
		public String getContentType() {
			return ContentDescriptor.RAW;
		}

		@Override
		public Object getControl(final String control) {
			return null;
		}

		@Override
		public Object[] getControls() {
			return new Object[0];
		}

		@Override
		public Time getDuration() {
			return Duration.DURATION_UNKNOWN;
		}

		@Override
		public void start() throws IOException {
		}

		@Override
		public void stop() throws IOException {
		}
	}

	private class ImageStream implements PullBufferStream {
		private final EncoderContext context;
		private final RGBFormat format;
		private final int length;
		private int frame;
		private final long duration;
		private long timestamp;

		public ImageStream(final EncoderContext context) {
			this.context = context;
			this.duration = 1000000000 / context.getFrameRate();
			this.length = context.getImageWidth() * context.getImageHeight() * 4;
			this.format = new RGBFormat(new Dimension(context.getImageWidth(), context.getImageHeight()), this.length, Format.byteArray, context.getFrameRate(), 32, 1, 2, 3);
		}

		public Format getFormat() {
			return this.format;
		}

		public boolean willReadBlock() {
			return false;
		}

		public boolean endOfStream() {
			return this.frame == this.context.getFrameCount();
		}

		public ContentDescriptor getContentDescriptor() {
			return new ContentDescriptor(ContentDescriptor.RAW);
		}

		public long getContentLength() {
			return this.length * this.context.getFrameCount();
		}

		public Object getControl(final String control) {
			return null;
		}

		public Object[] getControls() {
			return new Object[0];
		}

		public void read(final Buffer buffer) throws IOException {
			try {
				if (this.frame < this.context.getFrameCount()) {
					final byte[] data = this.context.getTileAsByteArray(this.frame, 0, 0, this.context.getImageWidth(), this.context.getImageHeight(), 4);
					buffer.setData(data);
					buffer.setOffset(0);
					buffer.setLength(this.length);
					buffer.setFormat(this.format);
					buffer.setDuration(this.duration);
					buffer.setTimeStamp(this.timestamp);
					buffer.setFlags(buffer.getFlags() | Buffer.FLAG_KEY_FRAME);
					this.timestamp += this.duration;
					this.frame += 1;
					if (JMFAbstractEncoderRuntime.this.hook != null && JMFAbstractEncoderRuntime.this.hook.isInterrupted()) {
						buffer.setEOM(true);
						buffer.setOffset(0);
						buffer.setLength(0);
						buffer.setDuration(0);
						buffer.setData(new byte[0]);
						buffer.setTimeStamp(this.timestamp);
					}
					fireStatusChanged((int) Math.rint(this.frame / (float) this.context.getFrameCount() * 100));
				}
				else {
					buffer.setEOM(true);
					buffer.setOffset(0);
					buffer.setLength(0);
					buffer.setDuration(0);
					buffer.setData(new byte[0]);
					buffer.setTimeStamp(this.timestamp);
				}
			}
			catch (final IOException e) {
				e.printStackTrace();
				throw e;
			}
		}
	}
}
