/*
 * $Id:ServiceFrame.java 495 2008-01-29 22:04:56Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.twister.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.FontMetrics;
import java.awt.GraphicsEnvironment;
import java.awt.HeadlessException;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.Semaphore;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.tree.DefaultMutableTreeNode;

import net.sf.jame.core.extension.ConfigurableExtension;
import net.sf.jame.core.extension.ExtensionException;
import net.sf.jame.core.swing.extension.ConfigurableExtensionComboBoxModel;
import net.sf.jame.core.swing.extension.ExtensionListCellRenderer;
import net.sf.jame.core.swing.util.GUIFactory;
import net.sf.jame.core.xml.XML;
import net.sf.jame.core.xml.XMLNodeBuilder;
import net.sf.jame.service.AsyncService;
import net.sf.jame.service.ServiceException;
import net.sf.jame.service.ServiceRegistry;
import net.sf.jame.service.AsyncService.ServiceCallback;
import net.sf.jame.service.AsyncService.ServiceVoidCallback;
import net.sf.jame.service.clip.MovieClip;
import net.sf.jame.service.clip.MovieClipDataRow;
import net.sf.jame.service.profile.RenderProfile;
import net.sf.jame.service.profile.RenderProfileDataRow;
import net.sf.jame.service.spool.JobServiceListener;
import net.sf.jame.service.spool.extension.SpoolExtensionConfig;
import net.sf.jame.service.spool.extension.SpoolExtensionRuntime;
import net.sf.jame.service.swing.AlternateTableCellRenderer;
import net.sf.jame.service.swing.BundleTreeCellRenderer;
import net.sf.jame.service.swing.BundleTreeModel;
import net.sf.jame.service.swing.EditClipDialog;
import net.sf.jame.service.swing.EditProfileDialog;
import net.sf.jame.service.swing.EncoderDialog;
import net.sf.jame.service.swing.IExtensionPointTreeCellRenderer;
import net.sf.jame.service.swing.IExtensionPointTreeModel;
import net.sf.jame.service.swing.MovieClipTableModel;
import net.sf.jame.service.swing.RenderJobTableModel;
import net.sf.jame.service.swing.RenderProfileTableModel;
import net.sf.jame.service.swing.ServiceAdapter;
import net.sf.jame.service.swing.ServiceContext;
import net.sf.jame.service.swing.util.ZIPFileFilter;
import net.sf.jame.twister.Rectangle;
import net.sf.jame.twister.TwisterClip;
import net.sf.jame.twister.TwisterClipXMLExporter;
import net.sf.jame.twister.TwisterClipXMLImporter;
import net.sf.jame.twister.TwisterConfig;
import net.sf.jame.twister.TwisterConfigBuilder;
import net.sf.jame.twister.TwisterSequence;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class ServiceFrame extends JFrame {
	private static final long serialVersionUID = 1L;
	private static final String SERVICE_FRAME_TITLE = "serviceFrame.title";
	private static final String SERVICE_FRAME_WIDTH = "serviceFrame.width";
	private static final String SERVICE_FRAME_HEIGHT = "serviceFrame.height";
	private static final String SERVICE_FRAME_ICON = "serviceFrame.icon";
	private final ServicePanel servicePanel;
	private final ServiceContext context;

	/**
	 * @param context
	 * @param service
	 * @param clipModel
	 * @param profileModel
	 * @param jobModel
	 * @throws HeadlessException
	 */
	public ServiceFrame(final ServiceContext context, final AsyncService service, final MovieClipTableModel clipModel, final RenderProfileTableModel profileModel, final RenderJobTableModel jobModel) throws HeadlessException {
		servicePanel = new ServicePanel(context, service, clipModel, profileModel, jobModel);
		getContentPane().add(servicePanel);
		setDefaultCloseOperation(WindowConstants.HIDE_ON_CLOSE);
		final int defaultWidth = Integer.parseInt(TwisterSwingResources.getInstance().getString(ServiceFrame.SERVICE_FRAME_WIDTH));
		final int defaultHeight = Integer.parseInt(TwisterSwingResources.getInstance().getString(ServiceFrame.SERVICE_FRAME_HEIGHT));
		final int width = Integer.getInteger(ServiceFrame.SERVICE_FRAME_WIDTH, defaultWidth);
		final int height = Integer.getInteger(ServiceFrame.SERVICE_FRAME_HEIGHT, defaultHeight);
		setTitle(TwisterSwingResources.getInstance().getString(ServiceFrame.SERVICE_FRAME_TITLE));
		final URL resource = ServiceFrame.class.getClassLoader().getResource(TwisterSwingResources.getInstance().getString(ServiceFrame.SERVICE_FRAME_ICON));
		if (resource != null) {
			setIconImage(getToolkit().createImage(resource));
		}
		setSize(new Dimension(width, height));
		final Point p = GraphicsEnvironment.getLocalGraphicsEnvironment().getCenterPoint();
		p.x -= getWidth() / 2;
		p.y -= getHeight() / 2;
		this.setLocation(p);
		addWindowListener(new ServiceWindowListener(servicePanel));
		this.context = context;
	}

	private class ServiceWindowListener extends WindowAdapter {
		private final ServicePanel servicePanel;

		/**
		 * @param canvas
		 */
		public ServiceWindowListener(final ServicePanel servicePanel) {
			this.servicePanel = servicePanel;
		}

		/**
		 * @see java.awt.event.WindowAdapter#windowOpened(java.awt.event.WindowEvent)
		 */
		@Override
		public void windowOpened(final WindowEvent e) {
			context.addFrame(ServiceFrame.this);
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					if (servicePanel != null) {
						try {
							if (!servicePanel.isStarted()) {
								servicePanel.start();
							}
						}
						catch (final ExtensionException x) {
							x.printStackTrace();
						}
					}
				}
			});
		}

		/**
		 * @see java.awt.event.WindowAdapter#windowClosing(java.awt.event.WindowEvent)
		 */
		@Override
		public void windowClosing(final WindowEvent e) {
			if (servicePanel != null) {
				if (servicePanel.isStarted()) {
					servicePanel.stop();
				}
			}
			dispose();
			context.removeFrame(ServiceFrame.this);
			if (context.getFrameCount() == 0) {
				context.exit();
			}
		}

		/**
		 * @see java.awt.event.WindowAdapter#windowDeiconified(java.awt.event.WindowEvent)
		 */
		@Override
		public void windowDeiconified(final WindowEvent e) {
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					if (servicePanel != null) {
						try {
							if (!servicePanel.isStarted()) {
								servicePanel.start();
							}
						}
						catch (final ExtensionException x) {
							x.printStackTrace();
						}
					}
				}
			});
		}

		/**
		 * @see java.awt.event.WindowAdapter#windowIconified(java.awt.event.WindowEvent)
		 */
		@Override
		public void windowIconified(final WindowEvent e) {
			if (servicePanel != null) {
				if (servicePanel.isStarted()) {
					servicePanel.stop();
				}
			}
		}

		/**
		 * @see java.awt.event.WindowAdapter#windowClosing(java.awt.event.WindowEvent)
		 */
		@Override
		public void windowClosed(final WindowEvent e) {
		}
	}

	public static class ServicePanel extends JPanel {
		private static final Logger logger = Logger.getLogger(ServicePanel.class);
		private static final long serialVersionUID = 1L;
		private static final String STRING_FRAME_TAB_CLIPS = "tab.clips";
		private static final String STRING_FRAME_TAB_JOBS = "tab.jobs";
		private static final String STRING_FRAME_TAB_BUNDLES = "tab.bundles";
		private static final String STRING_FRAME_TAB_EXTENSIONPOINTS = "tab.extensionPoints";
		private static final String STRING_FRAME_TREE_BUNDLES = "tree.bundles";
		private static final String STRING_FRAME_TREE_EXTENSIONPOINTS = "tree.extensionPoints";
		private final JButton clipOpenButton = GUIFactory.createSmallButton(new ClipOpenAction(), TwisterSwingResources.getInstance().getString("tooltip.openClip"));
		private final JButton clipCreateButton = GUIFactory.createSmallButton(new ClipCreateAction(), TwisterSwingResources.getInstance().getString("tooltip.createClip"));
		private final JButton clipDeleteButton = GUIFactory.createSmallButton(new ClipDeleteAction(), TwisterSwingResources.getInstance().getString("tooltip.deleteClip"));
		private final JButton clipModifyButton = GUIFactory.createSmallButton(new ClipModifyAction(), TwisterSwingResources.getInstance().getString("tooltip.modifyClip"));
		private final JButton clipImportButton = GUIFactory.createSmallButton(new ClipImportAction(), TwisterSwingResources.getInstance().getString("tooltip.importClip"));
		private final JButton clipExportButton = GUIFactory.createSmallButton(new ClipExportAction(), TwisterSwingResources.getInstance().getString("tooltip.exportClip"));
		private final JButton profileCreateButton = GUIFactory.createSmallButton(new ProfileCreateAction(), TwisterSwingResources.getInstance().getString("tooltip.createProfile"));
		private final JButton profileDeleteButton = GUIFactory.createSmallButton(new ProfileDeleteAction(), TwisterSwingResources.getInstance().getString("tooltip.deleteProfile"));
		private final JButton profileModifyButton = GUIFactory.createSmallButton(new ProfileModifyAction(), TwisterSwingResources.getInstance().getString("tooltip.modifyProfile"));
		private final JButton profileRenderButton = GUIFactory.createSmallButton(new ProfileRenderAction(), TwisterSwingResources.getInstance().getString("tooltip.renderProfile"));
		private final JButton profileCleanButton = GUIFactory.createSmallButton(new ProfileCleanAction(), TwisterSwingResources.getInstance().getString("tooltip.cleanProfile"));
		private final JButton profileAbortButton = GUIFactory.createSmallButton(new ProfileAbortAction(), TwisterSwingResources.getInstance().getString("tooltip.abortProfile"));
		private final JButton profileStartButton = GUIFactory.createSmallButton(new ProfileStartAction(), TwisterSwingResources.getInstance().getString("tooltip.restartProfile"));
		private final JButton profileStopButton = GUIFactory.createSmallButton(new ProfileStopAction(), TwisterSwingResources.getInstance().getString("tooltip.stopProfile"));
		private final JButton profileExportButton = GUIFactory.createSmallButton(new ProfileExportAction(), TwisterSwingResources.getInstance().getString("tooltip.exportProfile"));
		// private final JButton jobSuspendButton = GUIFactory.createSmallButton(new JobsSuspendAction(), TwisterSwingResources.getInstance().getString("tooltip.suspendJobs"));
		// private final JButton jobResumeButton = GUIFactory.createSmallButton(new JobsResumeAction(), TwisterSwingResources.getInstance().getString("tooltip.resumeJobs"));
		// private final JButton jobDeleteButton = GUIFactory.createSmallButton(new JobsDeleteAction(), TwisterSwingResources.getInstance().getString("tooltip.deleteJobs"));
		private final JButton checkUpdateButton = GUIFactory.createSmallButton(new CheckUpdateAction(), TwisterSwingResources.getInstance().getString("tooltip.checkUpdate"));
		private final JButton showVersionButton = GUIFactory.createSmallButton(new ShowVersionAction(), TwisterSwingResources.getInstance().getString("tooltip.showVersion"));
		private final JButton showSystemInfoButton = GUIFactory.createSmallButton(new ShowSystemInfoAction(), TwisterSwingResources.getInstance().getString("tooltip.showSystemInfo"));
		private final JButton showAboutButton = GUIFactory.createSmallButton(new ShowAboutAction(), TwisterSwingResources.getInstance().getString("tooltip.showAbout"));
		private final JButton changeWorkspaceButton = GUIFactory.createSmallButton(new ChangeWorkspaceAction(), TwisterSwingResources.getInstance().getString("tooltip.changeWorkspace"));
		private final JFileChooser clipChooser = new JFileChooser(System.getProperty("user.home"));
		private final JFileChooser fileChooser = new JFileChooser(System.getProperty("user.home"));
		private final JLabel spoolStatusLabel;
		private final JComboBox extensionComboBox;
		// private JComboBox clipListCombobox;
		// private JComboBox profileListCombobox;
		// private JTextField nameFilter = GUIFactory.createTextField();
		private final EncoderDialog encoderDialog;
		private final PreviewCanvas preview;
		private final ServiceTable clipTable;
		private final ServiceTable profileTable;
		private final ServiceTable jobTable;
		private final MovieClipTableModel clipModel;
		private final RenderProfileTableModel profileModel;
		private final RenderJobTableModel jobModel;
		private final Semaphore semaphore = new Semaphore(0, true);
		private final ServiceContext context;
		private final AsyncService service;
		private final JobServiceListener listener; 

		/**
		 * @param context
		 * @param service
		 * @param clipModel
		 * @param profileModel
		 * @param jobModel
		 */
		@SuppressWarnings("unchecked")
		public ServicePanel(final ServiceContext context, final AsyncService service, final MovieClipTableModel clipModel, final RenderProfileTableModel profileModel, final RenderJobTableModel jobModel) {
			this.context = context;
			encoderDialog = new EncoderDialog(new JFrame(), service);
			setLayout(new BorderLayout());
			clipChooser.setFileFilter(new ZIPFileFilter());
			clipChooser.setMultiSelectionEnabled(false);
			fileChooser.setDialogTitle(TwisterSwingResources.getInstance().getString("label.workspace"));
			fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
			fileChooser.setMultiSelectionEnabled(false);
			final JPanel panel = new JPanel(new BorderLayout());
			clipTable = new ServiceTable(clipModel);
			// clipTable.setAutoCreateRowSorter(true);
			final JPanel clipTablePanel = new JPanel(new BorderLayout());
			final JPanel clipButtonPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
			clipButtonPanel.add(clipCreateButton);
			clipButtonPanel.add(clipModifyButton);
			clipButtonPanel.add(clipDeleteButton);
			clipButtonPanel.add(clipImportButton);
			clipButtonPanel.add(clipExportButton);
			clipButtonPanel.add(clipOpenButton);
			clipTablePanel.add(new JScrollPane(clipTable), BorderLayout.CENTER);
			clipTablePanel.add(clipButtonPanel, BorderLayout.SOUTH);
			profileTable = new ServiceTable(profileModel);
			// profileTable.setAutoCreateRowSorter(true);
			final JPanel profileTablePanel = new JPanel(new BorderLayout());
			final JPanel profileButtonPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
			profileButtonPanel.add(profileCreateButton);
			profileButtonPanel.add(profileModifyButton);
			profileButtonPanel.add(profileDeleteButton);
			profileButtonPanel.add(profileRenderButton);
			profileButtonPanel.add(profileCleanButton);
			profileButtonPanel.add(profileAbortButton);
			profileButtonPanel.add(profileStartButton);
			profileButtonPanel.add(profileStopButton);
			profileButtonPanel.add(profileExportButton);
			profileTablePanel.add(new JScrollPane(profileTable), BorderLayout.CENTER);
			profileTablePanel.add(profileButtonPanel, BorderLayout.SOUTH);
			jobTable = new ServiceTable(jobModel);
			// jobTable.setAutoCreateRowSorter(true);
			jobTable.setRowSelectionAllowed(false);
			final JPanel jobTablePanel = new JPanel(new BorderLayout());
			final Box jobButtonPanel = Box.createHorizontalBox();
			// jobButtonPanel.add(jobSuspendButton);
			// jobButtonPanel.add(jobResumeButton);
			// jobButtonPanel.add(jobDeleteButton);
			final ConfigurableExtensionComboBoxModel model = new ConfigurableExtensionComboBoxModel(ServiceRegistry.getInstance().getSpoolRegistry(), false);
			spoolStatusLabel = GUIFactory.createLabel("", JLabel.LEFT);
			spoolStatusLabel.setPreferredSize(new Dimension(300, GUIFactory.DEFAULT_HEIGHT));
			extensionComboBox = GUIFactory.createSmallComboBox(model, TwisterSwingResources.getInstance().getString("tooltip.spool"));
			listener = new JobServiceListener() {
				public void stateChanged(String message) {
					spoolStatusLabel.setText(message);
				}
			}; 
			try {
				model.setSelectedItemByExtensionId("service.spool.local");
				if (service.getJobService() != null) {
					service.getJobService().removeServiceListener(listener);
				}
				service.setJobService((((ConfigurableExtension<SpoolExtensionRuntime<?>, SpoolExtensionConfig>) ((ConfigurableExtensionComboBoxModel) extensionComboBox.getModel()).getSelectedItem()).createExtensionRuntime().getJobService(service.getService())));
				spoolStatusLabel.setText("");
				if (service.getJobService() != null) {
					service.getJobService().addServiceListener(listener);
				}
			}
			catch (final ExtensionException x) {
				x.printStackTrace();
			}
			extensionComboBox.setPreferredSize(new Dimension(160, GUIFactory.DEFAULT_HEIGHT));
			extensionComboBox.setRenderer(new ExtensionListCellRenderer());
			extensionComboBox.setOpaque(false);
			jobButtonPanel.add(extensionComboBox);
			jobButtonPanel.add(spoolStatusLabel);
			jobButtonPanel.add(Box.createHorizontalGlue());
			jobButtonPanel.add(changeWorkspaceButton);
			jobTablePanel.add(new JScrollPane(jobTable), BorderLayout.CENTER);
			jobTablePanel.add(jobButtonPanel, BorderLayout.SOUTH);
			final JPanel clipPreview = new JPanel(new BorderLayout());
			preview = new PreviewCanvas();
			clipPreview.add(preview);
			clipPreview.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.preview"));
			clipTablePanel.setOpaque(false);
			profileTablePanel.setOpaque(false);
			jobTablePanel.setOpaque(false);
			final JSplitPane clipsTableSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, true, clipTablePanel, clipPreview);
			clipsTableSplitPane.setOneTouchExpandable(true);
			clipsTableSplitPane.setDividerLocation(550);
			clipsTableSplitPane.setDividerSize(10);
			final JSplitPane clipsSplitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, true, clipsTableSplitPane, profileTablePanel);
			clipsSplitPane.setOneTouchExpandable(true);
			clipsSplitPane.setDividerLocation(200);
			clipsSplitPane.setDividerSize(10);
			// final JSplitPane splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, true, topPanel, bottomPanel);
			// splitPane.setDividerLocation(300);
			final FontMetrics clipTableFM = getFontMetrics(clipTable.getTableHeader().getFont());
			for (int i = 0; i < clipTable.getColumnCount(); i++) {
				clipTable.getColumnModel().getColumn(i).setCellRenderer(new AlternateTableCellRenderer());
				clipTable.getColumnModel().getColumn(i).setPreferredWidth(SwingUtilities.computeStringWidth(clipTableFM, (((String) clipTable.getColumnModel().getColumn(i).getHeaderValue()))));
			}
			clipTable.getColumnModel().getColumn(MovieClipTableModel.NAME).setPreferredWidth(100);
			final FontMetrics profileTableFM = getFontMetrics(profileTable.getTableHeader().getFont());
			for (int i = 0; i < profileTable.getColumnCount(); i++) {
				profileTable.getColumnModel().getColumn(i).setCellRenderer(new AlternateTableCellRenderer());
				profileTable.getColumnModel().getColumn(i).setPreferredWidth(SwingUtilities.computeStringWidth(profileTableFM, (((String) profileTable.getColumnModel().getColumn(i).getHeaderValue()))));
			}
			profileTable.getColumnModel().getColumn(RenderProfileTableModel.NAME).setPreferredWidth(60);
			profileTable.getColumnModel().getColumn(RenderProfileTableModel.STATUS).setPreferredWidth(50);
			final FontMetrics jobTableFM = getFontMetrics(jobTable.getTableHeader().getFont());
			for (int i = 0; i < jobTable.getColumnCount(); i++) {
				jobTable.getColumnModel().getColumn(i).setCellRenderer(new AlternateTableCellRenderer());
				jobTable.getColumnModel().getColumn(i).setPreferredWidth(SwingUtilities.computeStringWidth(jobTableFM, (((String) jobTable.getColumnModel().getColumn(i).getHeaderValue()))));
			}
			jobTable.getColumnModel().getColumn(RenderJobTableModel.NAME).setPreferredWidth(100);
			jobTable.getColumnModel().getColumn(RenderJobTableModel.TYPE).setPreferredWidth(80);
			clipPreview.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createTitledBorder(BorderFactory.createEmptyBorder(), TwisterSwingResources.getInstance().getString("label.preview"), TitledBorder.CENTER, TitledBorder.BELOW_TOP), BorderFactory.createLineBorder(Color.BLACK)));
			clipTablePanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEmptyBorder(), TwisterSwingResources.getInstance().getString("label.clipPanel"), TitledBorder.LEFT, TitledBorder.BELOW_TOP));
			profileTablePanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEmptyBorder(), TwisterSwingResources.getInstance().getString("label.profilePanel"), TitledBorder.LEFT, TitledBorder.BELOW_TOP));
			clipsTableSplitPane.setBorder(BorderFactory.createEmptyBorder());
			clipsSplitPane.setBorder(BorderFactory.createEmptyBorder());
			jobTablePanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEmptyBorder(), TwisterSwingResources.getInstance().getString("label.jobPanel"), TitledBorder.LEFT, TitledBorder.BELOW_TOP));
			this.add(panel, BorderLayout.CENTER);
			final JPanel clipsPanel = new JPanel(new BorderLayout());
			final JPanel jobsPanel = new JPanel(new BorderLayout());
			clipsPanel.setOpaque(false);
			jobsPanel.setOpaque(false);
			jobTablePanel.setOpaque(true);
			clipsPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(4, 4, 4, 4), BorderFactory.createEtchedBorder(EtchedBorder.LOWERED)));
			jobsPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(4, 4, 4, 4), BorderFactory.createEtchedBorder(EtchedBorder.LOWERED)));
			clipsPanel.add(clipsSplitPane);
			// JPanel jobTableFilter = new JPanel(new FlowLayout(FlowLayout.LEFT));
			// clipListCombobox = GUIFactory.createComboBox(new DefaultComboBoxModel());
			// clipListCombobox.setRenderer(new MovieClipListCellRenderer());
			// profileListCombobox = GUIFactory.createComboBox(new DefaultComboBoxModel());
			// profileListCombobox.setRenderer(new RenderProfileListCellRenderer());
			// JPanel clipListComboboxPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			// JPanel profileListComboboxPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
			// clipListComboboxPanel.add(clipListCombobox);
			// profileListComboboxPanel.add(profileListCombobox);
			// clipListComboboxPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEmptyBorder(), TwisterSwingResources.getInstance().getString("label.clipFilter"), TitledBorder.CENTER, TitledBorder.BELOW_TOP));
			// profileListComboboxPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory.createEmptyBorder(), TwisterSwingResources.getInstance().getString("label.profileFilter"), TitledBorder.CENTER, TitledBorder.BELOW_TOP));
			// jobTableFilter.add(clipListComboboxPanel);
			// jobTableFilter.add(profileListComboboxPanel);
			// jobTableFilter.add(GUIFactory.createLabel("Filter"));
			// jobTableFilter.add(nameFilter);
			// jobsPanel.add(jobTableFilter, BorderLayout.NORTH);
			jobsPanel.add(jobTablePanel, BorderLayout.CENTER);
			final JTree bundleTree = createBundleTree();
			final JTree extensionPointTree = createExtensionPointTree();
			final JTabbedPane tabbedPane = new JTabbedPane();
			final JPanel bundlePanel = new JPanel(new BorderLayout());
			final JPanel bundleButtons = new JPanel(new FlowLayout(FlowLayout.CENTER));
			bundleButtons.add(checkUpdateButton);
			bundleButtons.add(showVersionButton);
			bundleButtons.add(showSystemInfoButton);
			bundleButtons.add(showAboutButton);
			final JPanel bundleTreePanel = new JPanel(new BorderLayout());
			bundleTreePanel.add(new JScrollPane(bundleTree), BorderLayout.CENTER);
			bundleTreePanel.add(bundleButtons, BorderLayout.SOUTH);
			bundleTreePanel.setBorder(BorderFactory.createEmptyBorder(4, 4, 4, 4));
			bundlePanel.add(bundleTreePanel, BorderLayout.CENTER);
			bundlePanel.setOpaque(false);
			bundlePanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(4, 4, 4, 4), BorderFactory.createEtchedBorder(EtchedBorder.LOWERED)));
			final JPanel extensionPointTreePanel = new JPanel(new BorderLayout());
			extensionPointTreePanel.add(new JScrollPane(extensionPointTree), BorderLayout.CENTER);
			extensionPointTreePanel.setBorder(BorderFactory.createEmptyBorder(4, 4, 4, 4));
			final JPanel extensionPointPanel = new JPanel(new BorderLayout());
			extensionPointPanel.add(extensionPointTreePanel);
			extensionPointPanel.setOpaque(false);
			extensionPointPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(4, 4, 4, 4), BorderFactory.createEtchedBorder(EtchedBorder.LOWERED)));
			tabbedPane.addTab(TwisterSwingResources.getInstance().getString(ServicePanel.STRING_FRAME_TAB_CLIPS), clipsPanel);
			tabbedPane.addTab(TwisterSwingResources.getInstance().getString(ServicePanel.STRING_FRAME_TAB_JOBS), jobsPanel);
			tabbedPane.addTab(TwisterSwingResources.getInstance().getString(ServicePanel.STRING_FRAME_TAB_BUNDLES), bundlePanel);
			tabbedPane.addTab(TwisterSwingResources.getInstance().getString(ServicePanel.STRING_FRAME_TAB_EXTENSIONPOINTS), extensionPointPanel);
			panel.add(tabbedPane);
			tabbedPane.addChangeListener(new ChangeListener() {
				public void stateChanged(final ChangeEvent e) {
					if (tabbedPane.getSelectedIndex() == 0) {
						if (clipTable.getSelectedRowCount() == 1) {
							profileModel.clear();
							updateButtons();
							final int clipId = clipModel.getClip(clipTable.convertRowIndexToModel(clipTable.getSelectedRow())).getClipId();
							service.getClip(new ServiceCallback<MovieClipDataRow>() {
								public void executed(final MovieClipDataRow profile) {
									profileModel.reload(clipId);
									try {
										final TwisterClipXMLImporter importer = new TwisterClipXMLImporter();
										final InputStream is = service.getService().getClipInputStream(clipId);
										final Document doc = XML.loadDocument(is, "twister-clip.xml");
										is.close();
										final TwisterClip clip = importer.importFromElement(doc.getDocumentElement());
										preview.stopRenderers();
										preview.stop();
										preview.start(clip);
										preview.startRenderers();
										preview.setArea(null);
									}
									catch (final Exception x) {
										ServicePanel.logger.error("Can't load the clip " + clipId, x);
										x.printStackTrace();
										JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.loadClip"), TwisterSwingResources.getInstance().getString("label.loadClip"), JOptionPane.ERROR_MESSAGE);
									}
								}

								public void failed(final Throwable throwable) {
									ServicePanel.logger.error("Can't get the clip " + clipId);
									throwable.printStackTrace();
								}
							}, clipId);
						}
						else {
							profileModel.clear();
							updateButtons();
						}
					}
					else {
						preview.stopRenderers();
						preview.stop();
					}
				}
			});
			// nameFilter.setPreferredSize(new Dimension(400, 20));
			// final TableRowSorter<RenderJobTableModel> sorter = new TableRowSorter<RenderJobTableModel>();
			// jobTable.setRowSorter(sorter);
			// nameFilter.getDocument().addDocumentListener(new DocumentListener() {
			// public void changedUpdate(DocumentEvent e) {
			// sorter.setRowFilter(RowFilter.regexFilter(nameFilter.getText(), 0));
			// }
			//
			// public void insertUpdate(DocumentEvent e) {
			// // sorter.setRowFilter(RowFilter.regexFilter(nameFilter.getText(), 0));
			// }
			//
			// public void removeUpdate(DocumentEvent e) {
			// // sorter.setRowFilter(RowFilter.regexFilter(nameFilter.getText(), 0));
			// }
			// });
			clipTable.getSelectionModel().addListSelectionListener(new ClipListSelectionListener());
			profileTable.getSelectionModel().addListSelectionListener(new ProfileListSelectionListener());
			this.clipModel = clipModel;
			this.profileModel = profileModel;
			this.jobModel = jobModel;
			this.service = service;
			// clipModel.addTableModelListener(new TableModelListener() {
			// public void tableChanged(TableModelEvent e) {
			// switch (e.getType()) {
			// case TableModelEvent.INSERT: {
			// for (int i = e.getFirstRow(); i <= e.getLastRow(); i++) {
			// logger.debug("insert clip " + i);
			// ((DefaultComboBoxModel) clipListCombobox.getModel()).addElement(clipModel.getClip(i));
			// }
			// break;
			// }
			// case TableModelEvent.DELETE: {
			// for (int i = e.getLastRow(); i >= e.getFirstRow(); i--) {
			// logger.debug("delete clip " + i);
			// ((DefaultComboBoxModel) clipListCombobox.getModel()).removeElementAt(i);
			// }
			// break;
			// }
			// default: {
			// break;
			// }
			// }
			// }
			// });
			// profileModel.addTableModelListener(new TableModelListener() {
			// public void tableChanged(TableModelEvent e) {
			// logger.debug(e);
			// switch (e.getType()) {
			// case TableModelEvent.INSERT: {
			// for (int i = e.getFirstRow(); i <= e.getLastRow(); i++) {
			// logger.debug("insert profile " + i);
			// ((DefaultComboBoxModel) profileListCombobox.getModel()).addElement(profileModel.getProfile(i));
			// }
			// break;
			// }
			// case TableModelEvent.DELETE: {
			// for (int i = e.getLastRow(); i >= e.getFirstRow(); i--) {
			// logger.debug("delete profile " + i);
			// ((DefaultComboBoxModel) profileListCombobox.getModel()).removeElementAt(i);
			// }
			// break;
			// }
			// default: {
			// break;
			// }
			// }
			// }
			// });
			extensionComboBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					try {
						if (service.getJobService() != null) {
							service.getJobService().removeServiceListener(listener);
						}
						service.setJobService((((ConfigurableExtension<SpoolExtensionRuntime<?>, SpoolExtensionConfig>) ((ConfigurableExtensionComboBoxModel) extensionComboBox.getModel()).getSelectedItem()).createExtensionRuntime().getJobService(service.getService())));
						spoolStatusLabel.setText("");
						if (service.getJobService() != null) {
							service.getJobService().addServiceListener(listener);
						}
					}
					catch (final ExtensionException x) {
						x.printStackTrace();
					}
				}
			});
			final ServicePanelListener listener = new ServicePanelListener();
			profileModel.addTableModelListener(new TableListener());
			clipModel.addTableModelListener(new TableListener());
			jobModel.addTableModelListener(new TableListener());
			this.service.addServiceListener(new ServiceVoidCallback() {
				/**
				 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
				 */
				public void executed() {
				}

				/**
				 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
				 */
				public void failed(final Throwable throwable) {
					ServicePanel.logger.error("Can't register the listener", throwable);
				}
			}, listener);
			this.service.resumeJobs(new ServiceVoidCallback() {
				/**
				 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
				 */
				public void executed() {
					jobModel.reload();
				}

				/**
				 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
				 */
				public void failed(final Throwable throwable) {
					ServicePanel.logger.error("Can't resume the jobs", throwable);
				}
			});
		}

		/**
		 * @return
		 */
		public boolean isStarted() {
			return preview.isStarted();
		}

		/**
		 * @throws ExtensionException
		 */
		public void start() throws ExtensionException {
			preview.start();
		}

		/**
		 * 
		 */
		public void stop() {
			preview.stop();
		}

		private void updateButtons() {
			clipCreateButton.setEnabled(true);
			// clipDeleteButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (clipModel.getClip(clipTable.convertRowIndexToModel(clipTable.getSelectedRow())).getStatus() == 0));
			// clipModifyButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (clipModel.getClip(clipTable.convertRowIndexToModel(clipTable.getSelectedRow())).getStatus() == 0));
			// clipImportButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (clipModel.getClip(clipTable.convertRowIndexToModel(clipTable.getSelectedRow())).getStatus() == 0));
			// clipExportButton.setEnabled(clipTable.getSelectedRowCount() == 1);
			// clipOpenButton.setEnabled(clipTable.getSelectedRowCount() == 1);
			// profileCreateButton.setEnabled(clipTable.getSelectedRowCount() == 1);
			// profileDeleteButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() == 1) && (profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getStatus() == 0));
			// profileModifyButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() == 1) && (profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getStatus() == 0));
			// profileRenderButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() == 1) && (profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getStatus() == 0));
			// profileCleanButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() == 1) && (profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getStatus() == 0));
			// profileAbortButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() == 1) && (profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getStatus() != 0));
			// profileStartButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() == 1) && (profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getStatus() != 0));
			// profileStopButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() == 1) && (profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getStatus() != 0));
			// profileExportButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() == 1) && (profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getStatus() == 0)
			// && (profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getJobCreated() == profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getJobStored())
			// && (profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getJobStored() == 1));
			clipDeleteButton.setEnabled((clipTable.getSelectedRowCount() > 0) && clipStatusEquals(0));
			clipModifyButton.setEnabled((clipTable.getSelectedRowCount() == 1) && clipStatusEquals(0));
			clipImportButton.setEnabled((clipTable.getSelectedRowCount() == 1) && clipStatusEquals(0));
			clipExportButton.setEnabled(clipTable.getSelectedRowCount() > 0);
			clipOpenButton.setEnabled(clipTable.getSelectedRowCount() == 1);
			profileCreateButton.setEnabled(clipTable.getSelectedRowCount() == 1);
			profileDeleteButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() > 0) && profileStatusEquals(0));
			profileModifyButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() == 1) && profileStatusEquals(0));
			profileRenderButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() > 0) && profileStatusEquals(0));
			profileCleanButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() > 0) && profileStatusEquals(0));
			profileAbortButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() > 0) && profileStatusNotEquals(0));
			profileStartButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() > 0) && profileStatusNotEquals(0));
			profileStopButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() > 0) && profileStatusNotEquals(0));
			profileExportButton.setEnabled((clipTable.getSelectedRowCount() == 1) && (profileTable.getSelectedRowCount() == 1) && (profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getStatus() == 0) && (profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getJobCreated() == profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getJobStored())
					&& profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getJobStored() > 0);
			// jobSuspendButton.setEnabled(!isSuspended);
			// jobResumeButton.setEnabled(isSuspended);
			// jobDeleteButton.setEnabled(true);
			changeWorkspaceButton.setEnabled(jobTable.getRowCount() == 0);
			extensionComboBox.setEnabled(jobTable.getRowCount() == 0);
		}

		private boolean clipStatusEquals(final int status) {
			if (clipTable.getSelectedRowCount() == 0) {
				return false;
			}
			final int[] rows = clipTable.getSelectedRows();
			for (int i = 0; i < rows.length; i++) {
				if (clipModel.getClip(clipTable.convertRowIndexToModel(rows[i])).getStatus() != status) {
					return false;
				}
			}
			return true;
		}

		private boolean profileStatusEquals(final int status) {
			if (profileTable.getSelectedRowCount() == 0) {
				return false;
			}
			final int[] rows = profileTable.getSelectedRows();
			for (int i = 0; i < rows.length; i++) {
				if (profileModel.getProfile(profileTable.convertRowIndexToModel(rows[i])).getStatus() != status) {
					return false;
				}
			}
			return true;
		}

		private boolean profileStatusNotEquals(final int status) {
			if (profileTable.getSelectedRowCount() == 0) {
				return false;
			}
			final int[] rows = profileTable.getSelectedRows();
			for (int i = 0; i < rows.length; i++) {
				if (profileModel.getProfile(profileTable.convertRowIndexToModel(rows[i])).getStatus() == status) {
					return false;
				}
			}
			return true;
		}

		private MovieClipDataRow createDefaultMovieClip() {
			final MovieClipDataRow clip = new MovieClipDataRow(new MovieClip());
			clip.setClipName("New Clip");
			clip.setDescription("Default Configuration");
			return clip;
		}

		private RenderProfileDataRow createDefaultRenderProfile(final int clipId) {
			final RenderProfileDataRow profile = new RenderProfileDataRow(new RenderProfile());
			profile.setClipId(clipId);
			profile.setProfileName("New Profile");
			profile.setImageWidth(640);
			profile.setImageHeight(480);
			profile.setQuality(100);
			return profile;
		}

		private void showEditClipWindow(final MovieClipDataRow clip) {
			final EditClipDialog editClipDialog = new EditClipDialog(service, clip);
			editClipDialog.setVisible(true);
		}

		private void showEditProfileWindow(final RenderProfileDataRow profile) {
			final EditProfileDialog editProfileDialog = new EditProfileDialog(service, profile);
			editProfileDialog.setVisible(true);
			updateButtons();
		}

		private JTree createBundleTree() {
			final BundleTreeModel treeModel = new BundleTreeModel(new DefaultMutableTreeNode(TwisterSwingResources.getInstance().getString(ServicePanel.STRING_FRAME_TREE_BUNDLES)));
			final JTree tree = new JTree(treeModel);
			tree.setFont(GUIFactory.NORMAL_FONT);
			tree.setShowsRootHandles(true);
			tree.setCellRenderer(new BundleTreeCellRenderer());
			return tree;
		}

		private JTree createExtensionPointTree() {
			final IExtensionPointTreeModel treeModel = new IExtensionPointTreeModel(new DefaultMutableTreeNode(TwisterSwingResources.getInstance().getString(ServicePanel.STRING_FRAME_TREE_EXTENSIONPOINTS)));
			final JTree tree = new JTree(treeModel);
			tree.setFont(GUIFactory.NORMAL_FONT);
			tree.setShowsRootHandles(true);
			tree.setCellRenderer(new IExtensionPointTreeCellRenderer());
			return tree;
		}

		private class ClipListSelectionListener implements ListSelectionListener {
			public void valueChanged(final ListSelectionEvent e) {
				if (!e.getValueIsAdjusting()) {
					if (clipTable.getSelectedRowCount() == 1) {
						profileModel.clear();
						updateButtons();
						final int clipId = clipModel.getClip(clipTable.convertRowIndexToModel(clipTable.getSelectedRow())).getClipId();
						service.getClip(new ServiceCallback<MovieClipDataRow>() {
							public void executed(final MovieClipDataRow profile) {
								profileModel.reload(clipId);
								try {
									final TwisterClipXMLImporter importer = new TwisterClipXMLImporter();
									final InputStream is = service.getService().getClipInputStream(clipId);
									final Document doc = XML.loadDocument(is, "twister-clip.xml");
									is.close();
									final TwisterClip clip = importer.importFromElement(doc.getDocumentElement());
									preview.stopRenderers();
									preview.stop();
									preview.start(clip);
									preview.startRenderers();
									preview.setArea(null);
								}
								catch (final Exception x) {
									ServicePanel.logger.error("Can't load the clip " + clipId, x);
									x.printStackTrace();
									JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.loadClip"), TwisterSwingResources.getInstance().getString("label.loadClip"), JOptionPane.ERROR_MESSAGE);
								}
							}

							public void failed(final Throwable throwable) {
								ServicePanel.logger.error("Can't get the clip " + clipId);
								throwable.printStackTrace();
							}
						}, clipId);
					}
					else {
						profileModel.clear();
						updateButtons();
					}
				}
			}
			// public void openClip(TwisterClip clip) {
			// try {
			// if (clip != null) {
			// try {
			// canvas.stopRenderers();
			// canvas.stop();
			// clipController = new TwisterClipController(clip);
			// clipController.addControllerListener(new ControllerListener() {
			// public void actionRedone(NodeAction action) {
			// // logger.debug("Redo: " + action);
			// }
			//
			// public void actionUndone(NodeAction action) {
			// // logger.debug("Undo: " + action);
			// }
			//
			// public void configChanged() {
			// canvas.stopRenderers();
			// // logger.debug("Config changed");
			// canvas.startRenderers();
			// }
			// });
			// clipController.init();
			// playConfig = clipController.getConfig();
			// if (clipController.getDuration() > 0) {
			// playConfig.setContext(new NullConfigContext());
			// canvas.setListener(new RenderListener() {
			// public void frameRendered() {
			// if (clipController != null) {
			// if (!clipController.redoAction(1000 / canvas.getFrameRate(), true)) {
			// clipController.init();
			// }
			// }
			// }
			// });
			// canvas.setSymbol(TwisterCanvas.SYMBOL_PLAY);
			// canvas.start(playConfig, TwisterCanvas.STATE_PLAY);
			// }
			// else {
			// canvas.start(playConfig, TwisterCanvas.STATE_EDIT);
			// }
			// canvas.startRenderers();
			// }
			// catch (ExtensionException x) {
			// x.printStackTrace();
			// }
			// }
			// panel.updateButtons();
			// }
			// catch (HeadlessException e) {
			// e.printStackTrace();
			// }
			// }
		}

		private class ProfileListSelectionListener implements ListSelectionListener {
			public void valueChanged(final ListSelectionEvent e) {
				if (!e.getValueIsAdjusting()) {
					updateButtons();
					// if (profileTable.getSelectedRowCount() == 1) {
					// jobModel.clear();
					// updateButtons();
					// final int profileId = profileModel.getProfile(profileTable.convertRowIndexToModel(profileTable.getSelectedRow())).getProfileId();
					// jobModel.reload(profileId);
					// service.getProfile(new ServiceCallback<RenderProfileDataRow>() {
					// public void executed(final RenderProfileDataRow profile) {
					// preview.setArea(new Rectangle(profile.getOffsetX(), profile.getOffsetY(), profile.getImageWidth(), profile.getImageHeight()));
					// }
					//
					// public void failed(final Throwable throwable) {
					// ServicePanel.logger.error("Can't get the profile " + profileId);
					// throwable.printStackTrace();
					// }
					// }, profileId);
					// }
					// else {
					// jobModel.clear();
					// updateButtons();
					// }
				}
			}
		}

		private class ClipOpenAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ClipOpenAction() {
				super(TwisterSwingResources.getInstance().getString("action.open"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (clipTable.getSelectedRowCount() == 1) {
					final int row = clipTable.getSelectedRow();
					final MovieClipDataRow dataRow = clipModel.getClip(clipTable.convertRowIndexToModel(row));
					service.getClip(new ServiceCallback<MovieClipDataRow>() {
						public void executed(final MovieClipDataRow value) {
							try {
								final InputStream is = service.getService().getClipInputStream(dataRow.getClipId());
								final Document doc = XML.loadDocument(is, "twister-clip.xml");
								final TwisterClipXMLImporter importer = new TwisterClipXMLImporter();
								final TwisterClip clip = importer.importFromElement(doc.getDocumentElement());
								is.close();
								context.openClip(clip);
								semaphore.release();
							}
							catch (final Exception x) {
								ServicePanel.logger.error("Can't load the clip", x);
								semaphore.release();
								JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.loadClip"), TwisterSwingResources.getInstance().getString("label.openClip"), JOptionPane.ERROR_MESSAGE);
							}
						}

						public void failed(final Throwable throwable) {
							ServicePanel.logger.error("Can't open the clip", throwable);
							semaphore.release();
							JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.openClip"), TwisterSwingResources.getInstance().getString("label.openClip"), JOptionPane.ERROR_MESSAGE);
						}
					}, dataRow.getClipId());
					try {
						semaphore.acquire();
					}
					catch (final InterruptedException x) {
						x.printStackTrace();
					}
				}
			}
		}

		private class ClipCreateAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ClipCreateAction() {
				super(TwisterSwingResources.getInstance().getString("action.create"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				final MovieClipDataRow dataRow = createDefaultMovieClip();
				service.createClip(new ServiceVoidCallback() {
					/**
					 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
					 */
					public void executed() {
						final TwisterConfigBuilder configBuilder = new TwisterConfigBuilder();
						try {
							final TwisterClip clip = new TwisterClip();
							final TwisterSequence sequence = new TwisterSequence();
							clip.addSequence(sequence);
							final TwisterConfig config = configBuilder.createDefaultConfig();
							sequence.setFinalConfig(config);
							final TwisterClipXMLExporter exporter = new TwisterClipXMLExporter();
							final Document doc = XML.createDocument();
							final XMLNodeBuilder builder = XML.createDefaultXMLNodeBuilder(doc);
							final Element element = exporter.exportToElement(clip, builder);
							doc.appendChild(element);
							final OutputStream os = service.getService().getClipOutputStream(dataRow.getClipId());
							XML.saveDocument(os, "twister-clip.xml", doc);
							os.close();
							semaphore.release();
						}
						catch (final Exception x) {
							ServicePanel.logger.error("Can't save the clip", x);
							semaphore.release();
							JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.saveClip"), TwisterSwingResources.getInstance().getString("label.createClip"), JOptionPane.ERROR_MESSAGE);
						}
					}

					/**
					 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
					 */
					public void failed(final Throwable throwable) {
						ServicePanel.logger.error("Can't create the clip", throwable);
						semaphore.release();
						JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.createClip"), TwisterSwingResources.getInstance().getString("label.createClip"), JOptionPane.ERROR_MESSAGE);
					}
				}, dataRow);
				try {
					semaphore.acquire();
				}
				catch (final InterruptedException x) {
					x.printStackTrace();
				}
			}
		}

		private class ClipDeleteAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ClipDeleteAction() {
				super(TwisterSwingResources.getInstance().getString("action.delete"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (clipTable.getSelectedRowCount() > 0) {
					if (JOptionPane.showConfirmDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("message.confirmDeleteClips"), TwisterSwingResources.getInstance().getString("label.deleteClips"), JOptionPane.YES_NO_OPTION) == JOptionPane.OK_OPTION) {
						final int[] rows = clipTable.getSelectedRows();
						final MovieClipDataRow[] clips = new MovieClipDataRow[rows.length];
						final boolean[] error = new boolean[] { false };
						for (int i = 0; i < rows.length; i++) {
							clips[i] = clipModel.getClip(clipTable.convertRowIndexToModel(rows[i]));
						}
						for (int i = 0; i < rows.length; i++) {
							service.deleteClip(new ServiceVoidCallback() {
								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
								 */
								public void executed() {
									semaphore.release();
								}

								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
								 */
								public void failed(final Throwable throwable) {
									ServicePanel.logger.error("Can't delete the clip", throwable);
									error[0] = true;
									semaphore.release();
									JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.deleteClip"), TwisterSwingResources.getInstance().getString("label.deleteClip"), JOptionPane.ERROR_MESSAGE);
								}
							}, clips[i]);
							try {
								semaphore.acquire();
							}
							catch (final InterruptedException x) {
								x.printStackTrace();
							}
							if (error[0]) {
								break;
							}
						}
						clipTable.clearSelection();
					}
				}
			}
		}

		private class ClipModifyAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ClipModifyAction() {
				super(TwisterSwingResources.getInstance().getString("action.modify"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (clipTable.getSelectedRowCount() == 1) {
					final int row = clipTable.getSelectedRow();
					final MovieClipDataRow dataRow = clipModel.getClip(clipTable.convertRowIndexToModel(row));
					service.getClip(new ServiceCallback<MovieClipDataRow>() {
						public void executed(final MovieClipDataRow clip) {
							SwingUtilities.invokeLater(new Runnable() {
								public void run() {
									showEditClipWindow(clip);
								}
							});
							semaphore.release();
						}

						public void failed(final Throwable throwable) {
							ServicePanel.logger.error("Can't get the clip " + dataRow.getClipId(), throwable);
							semaphore.release();
						}
					}, dataRow.getClipId());
					try {
						semaphore.acquire();
					}
					catch (final InterruptedException x) {
						x.printStackTrace();
					}
				}
			}
		}

		private class ClipImportAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ClipImportAction() {
				super(TwisterSwingResources.getInstance().getString("action.importClip"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (clipTable.getSelectedRowCount() > 0) {
					if (JOptionPane.showConfirmDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("message.confirmImportClips"), TwisterSwingResources.getInstance().getString("label.importClips"), JOptionPane.YES_NO_OPTION) == JOptionPane.OK_OPTION) {
						final int[] rows = clipTable.getSelectedRows();
						final boolean[] error = new boolean[] { false };
						for (final int row : rows) {
							final MovieClipDataRow clip = clipModel.getClip(clipTable.convertRowIndexToModel(row));
							clipChooser.setDialogTitle(TwisterSwingResources.getInstance().getString("label.importClip"));
							final int returnVal = clipChooser.showOpenDialog(ServicePanel.this);
							if (returnVal == JFileChooser.APPROVE_OPTION) {
								final File file = clipChooser.getSelectedFile();
								service.importClip(new ServiceVoidCallback() {
									/**
									 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
									 */
									public void executed() {
										service.resetProfiles(new ServiceCallback<List<RenderProfileDataRow>>() {
											/**
											 * @param profiles
											 */
											public void executed(final List<RenderProfileDataRow> profiles) {
												semaphore.release();
											}

											/**
											 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
											 */
											public void failed(final Throwable throwable) {
												ServicePanel.logger.error("Can't reset the clip profiles", throwable);
												error[0] = true;
												semaphore.release();
												JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.resetProfiles"), TwisterSwingResources.getInstance().getString("label.importClip"), JOptionPane.ERROR_MESSAGE);
											}
										}, clip.getClipId());
									}

									/**
									 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
									 */
									public void failed(final Throwable throwable) {
										ServicePanel.logger.error("Can't import the clip", throwable);
										error[0] = true;
										semaphore.release();
										JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.importClip"), TwisterSwingResources.getInstance().getString("label.importClip"), JOptionPane.ERROR_MESSAGE);
									}
								}, clip, file);
								try {
									semaphore.acquire();
								}
								catch (final InterruptedException x) {
									x.printStackTrace();
								}
							}
							if (error[0]) {
								break;
							}
						}
					}
				}
				if (clipTable.getSelectedRowCount() > 0) {
					profileModel.clear();
					updateButtons();
					final int clipId = clipModel.getClip(clipTable.convertRowIndexToModel(clipTable.getSelectedRow())).getClipId();
					service.getClip(new ServiceCallback<MovieClipDataRow>() {
						public void executed(final MovieClipDataRow profile) {
							profileModel.reload(clipId);
							try {
								final TwisterClipXMLImporter importer = new TwisterClipXMLImporter();
								final InputStream is = service.getService().getClipInputStream(clipId);
								final Document doc = XML.loadDocument(is, "twister-clip.xml");
								is.close();
								final TwisterClip clip = importer.importFromElement(doc.getDocumentElement());
								preview.stopRenderers();
								preview.stop();
								preview.start(clip);
								preview.startRenderers();
								preview.setArea(null);
							}
							catch (final Exception x) {
								ServicePanel.logger.error("Can't load the clip " + clipId, x);
								x.printStackTrace();
								JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.loadClip"), TwisterSwingResources.getInstance().getString("label.loadClip"), JOptionPane.ERROR_MESSAGE);
							}
						}

						public void failed(final Throwable throwable) {
							ServicePanel.logger.error("Can't get the clip " + clipId);
							throwable.printStackTrace();
						}
					}, clipId);
				}
				else {
					profileModel.clear();
					updateButtons();
				}
			}
		}

		private class ClipExportAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ClipExportAction() {
				super(TwisterSwingResources.getInstance().getString("action.exportClip"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (clipTable.getSelectedRowCount() > 0) {
					final int[] rows = clipTable.getSelectedRows();
					final boolean[] error = new boolean[] { false };
					for (final int row : rows) {
						clipChooser.setDialogTitle(TwisterSwingResources.getInstance().getString("label.exportClip"));
						final int returnVal = clipChooser.showSaveDialog(ServicePanel.this);
						if (returnVal == JFileChooser.APPROVE_OPTION) {
							final File file = clipChooser.getSelectedFile();
							if (file.exists()) {
								if (JOptionPane.showConfirmDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("message.confirmOverwrite"), TwisterSwingResources.getInstance().getString("label.exportClips"), JOptionPane.WARNING_MESSAGE) == JOptionPane.OK_OPTION) {
									service.exportClip(new ServiceVoidCallback() {
										/**
										 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
										 */
										public void executed() {
											semaphore.release();
										}

										/**
										 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
										 */
										public void failed(final Throwable throwable) {
											ServicePanel.logger.error("Can't export the clip", throwable);
											error[0] = true;
											semaphore.release();
											JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.exportClip"), TwisterSwingResources.getInstance().getString("label.exportClip"), JOptionPane.ERROR_MESSAGE);
										}
									}, clipModel.getClip(clipTable.convertRowIndexToModel(row)), file);
									try {
										semaphore.acquire();
									}
									catch (final InterruptedException x) {
										x.printStackTrace();
									}
								}
							}
							else {
								service.exportClip(new ServiceVoidCallback() {
									/**
									 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
									 */
									public void executed() {
										semaphore.release();
									}

									/**
									 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
									 */
									public void failed(final Throwable throwable) {
										ServicePanel.logger.error("Can't export the clip", throwable);
										error[0] = true;
										semaphore.release();
										JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.exportClip"), TwisterSwingResources.getInstance().getString("label.exportClip"), JOptionPane.ERROR_MESSAGE);
									}
								}, clipModel.getClip(clipTable.convertRowIndexToModel(row)), file);
								try {
									semaphore.acquire();
								}
								catch (final InterruptedException x) {
									x.printStackTrace();
								}
							}
						}
						if (error[0]) {
							break;
						}
					}
				}
			}
		}

		private class ProfileCreateAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ProfileCreateAction() {
				super(TwisterSwingResources.getInstance().getString("action.create"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (clipTable.getSelectedRowCount() == 1) {
					final int clipId = clipModel.getClip(clipTable.convertRowIndexToModel(clipTable.getSelectedRow())).getClipId();
					service.createProfile(new ServiceVoidCallback() {
						/**
						 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
						 */
						public void executed() {
							semaphore.release();
						}

						/**
						 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
						 */
						public void failed(final Throwable throwable) {
							ServicePanel.logger.error("Can't create the profile", throwable);
							semaphore.release();
							JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.createProfile"), TwisterSwingResources.getInstance().getString("label.createProfile"), JOptionPane.ERROR_MESSAGE);
						}
					}, createDefaultRenderProfile(clipId));
					try {
						semaphore.acquire();
					}
					catch (final InterruptedException x) {
						x.printStackTrace();
					}
				}
			}
		}

		private class ProfileDeleteAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ProfileDeleteAction() {
				super(TwisterSwingResources.getInstance().getString("action.delete"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (profileTable.getSelectedRowCount() > 0) {
					if (JOptionPane.showConfirmDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("message.confirmDeleteProfiles"), TwisterSwingResources.getInstance().getString("label.deleteProfiles"), JOptionPane.YES_NO_OPTION) == JOptionPane.OK_OPTION) {
						final int[] rows = profileTable.getSelectedRows();
						final boolean[] error = new boolean[] { false };
						final RenderProfileDataRow[] profiles = new RenderProfileDataRow[rows.length];
						for (int i = 0; i < rows.length; i++) {
							profiles[i] = profileModel.getProfile(profileTable.convertRowIndexToModel(rows[i]));
						}
						for (int i = 0; i < rows.length; i++) {
							service.deleteProfile(new ServiceVoidCallback() {
								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
								 */
								public void executed() {
									semaphore.release();
								}

								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
								 */
								public void failed(final Throwable throwable) {
									ServicePanel.logger.error("Can't delete the profile", throwable);
									error[0] = true;
									semaphore.release();
									JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.deleteProfile"), TwisterSwingResources.getInstance().getString("label.deleteProfile"), JOptionPane.ERROR_MESSAGE);
								}
							}, profiles[i]);
							try {
								semaphore.acquire();
							}
							catch (final InterruptedException x) {
								x.printStackTrace();
							}
							if (error[0]) {
								break;
							}
						}
						profileTable.clearSelection();
					}
				}
			}
		}

		private class ProfileModifyAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ProfileModifyAction() {
				super(TwisterSwingResources.getInstance().getString("action.modify"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (profileTable.getSelectedRowCount() == 1) {
					final int row = profileTable.getSelectedRow();
					final RenderProfileDataRow dataRow = profileModel.getProfile(profileTable.convertRowIndexToModel(row));
					service.getProfile(new ServiceCallback<RenderProfileDataRow>() {
						public void executed(final RenderProfileDataRow profile) {
							SwingUtilities.invokeLater(new Runnable() {
								public void run() {
									showEditProfileWindow(profile);
								}
							});
							semaphore.release();
						}

						public void failed(final Throwable throwable) {
							ServicePanel.logger.error("Can't get the profile " + dataRow.getProfileId(), throwable);
							semaphore.release();
						}
					}, dataRow.getProfileId());
					try {
						semaphore.acquire();
					}
					catch (final InterruptedException x) {
						x.printStackTrace();
					}
				}
			}
		}

		private class ProfileRenderAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ProfileRenderAction() {
				super(TwisterSwingResources.getInstance().getString("action.render"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (profileTable.getSelectedRowCount() > 0) {
					if (JOptionPane.showConfirmDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("message.confirmRenderProfiles"), TwisterSwingResources.getInstance().getString("label.renderProfiles"), JOptionPane.YES_NO_OPTION) == JOptionPane.OK_OPTION) {
						final int[] rows = profileTable.getSelectedRows();
						final boolean[] error = new boolean[] { false };
						for (final int row : rows) {
							final RenderProfileDataRow profile = profileModel.getProfile(profileTable.convertRowIndexToModel(row));
							service.deleteJobs(new ServiceVoidCallback() {
								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
								 */
								public void executed() {
									try {
										service.getService().createJobs(profile.getProfileId());
										service.getService().startJobs(profile.getProfileId());
										semaphore.release();
									}
									catch (final ServiceException e) {
										ServicePanel.logger.error("Can't render the profile", e);
										error[0] = true;
										semaphore.release();
										JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.createJobs"), TwisterSwingResources.getInstance().getString("label.renderProfile"), JOptionPane.ERROR_MESSAGE);
									}
								}

								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
								 */
								public void failed(final Throwable throwable) {
									ServicePanel.logger.error("Can't render the profile", throwable);
									error[0] = true;
									semaphore.release();
									JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.deleteJobs"), TwisterSwingResources.getInstance().getString("label.renderProfile"), JOptionPane.ERROR_MESSAGE);
								}
							}, profile.getProfileId());
							try {
								semaphore.acquire();
							}
							catch (final InterruptedException x) {
								x.printStackTrace();
							}
							if (error[0]) {
								break;
							}
						}
						updateButtons();
					}
				}
			}
		}

		private class ProfileAbortAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ProfileAbortAction() {
				super(TwisterSwingResources.getInstance().getString("action.abort"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (profileTable.getSelectedRowCount() > 0) {
					if (JOptionPane.showConfirmDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("message.confirmAbortProfiles"), TwisterSwingResources.getInstance().getString("label.abortProfiles"), JOptionPane.YES_NO_OPTION) == JOptionPane.OK_OPTION) {
						final int[] rows = profileTable.getSelectedRows();
						final boolean[] error = new boolean[] { false };
						for (final int row : rows) {
							final RenderProfileDataRow profile = profileModel.getProfile(profileTable.convertRowIndexToModel(row));
							service.stopJobs(new ServiceVoidCallback() {
								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
								 */
								public void executed() {
									service.deleteJobs(new ServiceVoidCallback() {
										/**
										 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
										 */
										public void executed() {
											semaphore.release();
										}

										/**
										 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
										 */
										public void failed(final Throwable throwable) {
											ServicePanel.logger.error("Can't abort the profile", throwable);
											error[0] = true;
											semaphore.release();
											JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.deleteJobs"), TwisterSwingResources.getInstance().getString("label.abortProfile"), JOptionPane.ERROR_MESSAGE);
										}
									}, profile.getProfileId());
								}

								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
								 */
								public void failed(final Throwable throwable) {
									ServicePanel.logger.error("Can't abort the profile", throwable);
									error[0] = true;
									semaphore.release();
									JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.deleteJobs"), TwisterSwingResources.getInstance().getString("label.abortProfile"), JOptionPane.ERROR_MESSAGE);
								}
							}, profile.getProfileId());
							try {
								semaphore.acquire();
							}
							catch (final InterruptedException x) {
								x.printStackTrace();
							}
							if (error[0]) {
								break;
							}
						}
						updateButtons();
					}
				}
			}
		}

		private class ProfileCleanAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ProfileCleanAction() {
				super(TwisterSwingResources.getInstance().getString("action.clean"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (profileTable.getSelectedRowCount() > 0) {
					if (JOptionPane.showConfirmDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("message.confirmCleanProfiles"), TwisterSwingResources.getInstance().getString("label.cleanProfiles"), JOptionPane.YES_NO_OPTION) == JOptionPane.OK_OPTION) {
						final int[] rows = profileTable.getSelectedRows();
						final boolean[] error = new boolean[] { false };
						for (final int row : rows) {
							final RenderProfileDataRow profile = profileModel.getProfile(profileTable.convertRowIndexToModel(row));
							service.cleanProfile(new ServiceVoidCallback() {
								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
								 */
								public void executed() {
									semaphore.release();
								}

								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
								 */
								public void failed(final Throwable throwable) {
									ServicePanel.logger.error("Can't clean the profile", throwable);
									error[0] = true;
									semaphore.release();
									JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.cleanProfile"), TwisterSwingResources.getInstance().getString("label.cleanProfile"), JOptionPane.ERROR_MESSAGE);
								}
							}, profile);
							try {
								semaphore.acquire();
							}
							catch (final InterruptedException x) {
								x.printStackTrace();
							}
							if (error[0]) {
								break;
							}
						}
						updateButtons();
					}
				}
			}
		}

		private class ProfileStartAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ProfileStartAction() {
				super(TwisterSwingResources.getInstance().getString("action.restart"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (profileTable.getSelectedRowCount() > 0) {
					final int[] rows = profileTable.getSelectedRows();
					final boolean[] error = new boolean[] { false };
					for (final int row : rows) {
						final RenderProfileDataRow profile = profileModel.getProfile(profileTable.convertRowIndexToModel(row));
						service.startJobs(new ServiceVoidCallback() {
							/**
							 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
							 */
							public void executed() {
								semaphore.release();
							}

							/**
							 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
							 */
							public void failed(final Throwable throwable) {
								ServicePanel.logger.error("Can't restart the profile", throwable);
								error[0] = true;
								semaphore.release();
								JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.restartJobs"), TwisterSwingResources.getInstance().getString("label.restartProfile"), JOptionPane.ERROR_MESSAGE);
							}
						}, profile.getProfileId());
						try {
							semaphore.acquire();
						}
						catch (final InterruptedException x) {
							x.printStackTrace();
						}
						if (error[0]) {
							break;
						}
					}
					updateButtons();
				}
			}
		}

		private class ProfileStopAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ProfileStopAction() {
				super(TwisterSwingResources.getInstance().getString("action.stop"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (profileTable.getSelectedRowCount() > 0) {
					if (JOptionPane.showConfirmDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("message.confirmStopProfiles"), TwisterSwingResources.getInstance().getString("label.stopProfiles"), JOptionPane.YES_NO_OPTION) == JOptionPane.OK_OPTION) {
						final int[] rows = profileTable.getSelectedRows();
						final boolean[] error = new boolean[] { false };
						for (final int row : rows) {
							final RenderProfileDataRow profile = profileModel.getProfile(profileTable.convertRowIndexToModel(row));
							service.stopJobs(new ServiceVoidCallback() {
								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
								 */
								public void executed() {
									semaphore.release();
								}

								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
								 */
								public void failed(final Throwable throwable) {
									ServicePanel.logger.error("Can't stop the profile", throwable);
									error[0] = true;
									semaphore.release();
									JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.stopJobs"), TwisterSwingResources.getInstance().getString("label.stopProfile"), JOptionPane.ERROR_MESSAGE);
								}
							}, profile.getProfileId());
							try {
								semaphore.acquire();
							}
							catch (final InterruptedException x) {
								x.printStackTrace();
							}
							if (error[0]) {
								break;
							}
						}
						updateButtons();
					}
				}
			}
		}

		private class ProfileExportAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ProfileExportAction() {
				super(TwisterSwingResources.getInstance().getString("action.export"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (profileTable.getSelectedRowCount() > 0) {
					final int[] rows = profileTable.getSelectedRows();
					encoderDialog.setModal(true);
					for (final int row : rows) {
						final RenderProfileDataRow profile = profileModel.getProfile(profileTable.convertRowIndexToModel(row));
						encoderDialog.setProfile(profile);
						encoderDialog.setVisible(true);
					}
					updateButtons();
				}
			}
		}

		// private class JobsSuspendAction extends AbstractAction {
		// private static final long serialVersionUID = 1L;
		//
		// /**
		// *
		// */
		// public JobsSuspendAction() {
		// super(TwisterSwingResources.getInstance().getString("action.suspend"));
		// }
		//
		// /**
		// * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		// */
		// public void actionPerformed(final ActionEvent e) {
		// if (JOptionPane.showConfirmDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("message.confirmSuspendJobs"), TwisterSwingResources.getInstance().getString("label.suspendJobs"), JOptionPane.YES_NO_OPTION) == JOptionPane.OK_OPTION) {
		// service.suspendJobs(new ServiceVoidCallback() {
		// /**
		// * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
		// */
		// public void executed() {
		// semaphore.release();
		// }
		//
		// /**
		// * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
		// */
		// public void failed(final Throwable throwable) {
		// ServicePanel.logger.error("Can't suspend the jobs", throwable);
		// semaphore.release();
		// JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.suspendJobs"), TwisterSwingResources.getInstance().getString("label.suspendJobs"), JOptionPane.ERROR_MESSAGE);
		// }
		// });
		// try {
		// semaphore.acquire();
		// }
		// catch (final InterruptedException x) {
		// x.printStackTrace();
		// }
		// updateButtons();
		// }
		// }
		// }
		//
		// private class JobsResumeAction extends AbstractAction {
		// private static final long serialVersionUID = 1L;
		//
		// /**
		// *
		// */
		// public JobsResumeAction() {
		// super(TwisterSwingResources.getInstance().getString("action.resume"));
		// }
		//
		// /**
		// * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		// */
		// public void actionPerformed(final ActionEvent e) {
		// service.resumeJobs(new ServiceVoidCallback() {
		// /**
		// * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
		// */
		// public void executed() {
		// semaphore.release();
		// }
		//
		// /**
		// * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
		// */
		// public void failed(final Throwable throwable) {
		// ServicePanel.logger.error("Can't resume the jobs", throwable);
		// semaphore.release();
		// JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.resumeJobs"), TwisterSwingResources.getInstance().getString("label.resumeJobs"), JOptionPane.ERROR_MESSAGE);
		// }
		// });
		// try {
		// semaphore.acquire();
		// }
		// catch (final InterruptedException x) {
		// x.printStackTrace();
		// }
		// updateButtons();
		// }
		// }
		//
		// private class JobsDeleteAction extends AbstractAction {
		// private static final long serialVersionUID = 1L;
		//
		// /**
		// *
		// */
		// public JobsDeleteAction() {
		// super(TwisterSwingResources.getInstance().getString("action.delete"));
		// }
		//
		// /**
		// * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		// */
		// public void actionPerformed(final ActionEvent e) {
		// if (JOptionPane.showConfirmDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("message.confirmDeleteJobs"), TwisterSwingResources.getInstance().getString("label.deleteJobs"), JOptionPane.YES_NO_OPTION) == JOptionPane.OK_OPTION) {
		// service.deleteJobs(new ServiceVoidCallback() {
		// /**
		// * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
		// */
		// public void executed() {
		// semaphore.release();
		// }
		//
		// /**
		// * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
		// */
		// public void failed(final Throwable throwable) {
		// ServicePanel.logger.error("Can't delete the jobs", throwable);
		// semaphore.release();
		// JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.deleteJobs"), TwisterSwingResources.getInstance().getString("label.deleteJobs"), JOptionPane.ERROR_MESSAGE);
		// }
		// });
		// try {
		// semaphore.acquire();
		// }
		// catch (final InterruptedException x) {
		// x.printStackTrace();
		// }
		// updateButtons();
		// }
		// }
		// }
		private class CheckUpdateAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public CheckUpdateAction() {
				super(TwisterSwingResources.getInstance().getString("action.checkUpdate"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				BufferedReader reader = null;
				try {
					final URL url = new URL(TwisterSwingResources.getInstance().getString("url.releases"));
					final URLConnection connection = url.openConnection();
					reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));
					String newRelease = null;
					String release = null;
					String line = null;
					while ((line = reader.readLine()) != null) {
						if (line.length() > 0) {
							release = line;
							if (release.equals(TwisterSwingResources.getInstance().getString("release"))) {
								break;
							}
						}
					}
					while ((line = reader.readLine()) != null) {
						if (line.length() > 0) {
							release = line;
							if (!release.equals(TwisterSwingResources.getInstance().getString("release"))) {
								if (TwisterSwingResources.getInstance().getString("acceptRC").equals("true") || (release.indexOf("RC") == -1)) {
									newRelease = release;
									break;
								}
							}
						}
					}
					if (newRelease != null) {
						JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("message.newReleaseAvailable"), TwisterSwingResources.getInstance().getString("action.checkUpdate"), JOptionPane.PLAIN_MESSAGE);
						ServiceDesktop.browse(new URI(TwisterSwingResources.getInstance().getString("url.download")));
					}
					else {
						JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("message.newReleaseNotAvailable"), TwisterSwingResources.getInstance().getString("action.checkUpdate"), JOptionPane.PLAIN_MESSAGE);
					}
				}
				catch (final Exception x) {
					JOptionPane.showMessageDialog(ServicePanel.this, TwisterSwingResources.getInstance().getString("error.checkUpdate"), TwisterSwingResources.getInstance().getString("action.checkUpdate"), JOptionPane.WARNING_MESSAGE);
				}
				finally {
					if (reader != null) {
						try {
							reader.close();
						}
						catch (final IOException x) {
						}
					}
				}
			}
		}

		private class ShowVersionAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ShowVersionAction() {
				super(TwisterSwingResources.getInstance().getString("action.showVersion"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				JOptionPane.showMessageDialog(ServicePanel.this, "Version " + TwisterSwingResources.getInstance().getString("release"), TwisterSwingResources.getInstance().getString("label.version"), JOptionPane.PLAIN_MESSAGE);
			}
		}

		private class ShowSystemInfoAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ShowSystemInfoAction() {
				super(TwisterSwingResources.getInstance().getString("action.showSystemInfo"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				final StringBuilder builder = new StringBuilder();
				builder.append("JRE VER ");
				builder.append(System.getProperty("java.version"));
				builder.append(", ");
				builder.append(System.getProperty("java.vendor"));
				builder.append("\n");
				builder.append("OS NAME ");
				builder.append(System.getProperty("os.name"));
				builder.append(", VER ");
				builder.append(System.getProperty("os.version"));
				builder.append(", ARCH ");
				builder.append(System.getProperty("os.arch"));
				builder.append(", CPUs ");
				builder.append(Runtime.getRuntime().availableProcessors());
				builder.append("\n");
				builder.append("MEM MAX ");
				builder.append(Runtime.getRuntime().maxMemory() / 1024);
				builder.append("kb, USED ");
				builder.append(Runtime.getRuntime().totalMemory() / 1024);
				builder.append("kb, FREE ");
				builder.append(Runtime.getRuntime().freeMemory() / 1024);
				builder.append("kb\n");
				JOptionPane.showMessageDialog(ServicePanel.this, builder.toString(), TwisterSwingResources.getInstance().getString("label.systemInfo"), JOptionPane.PLAIN_MESSAGE);
			}
		}

		private class ShowAboutAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ShowAboutAction() {
				super(TwisterSwingResources.getInstance().getString("action.showAbout"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				BufferedReader reader = null;
				try {
					reader = new BufferedReader(new InputStreamReader(getClass().getResourceAsStream("/about.txt")));
					String line = null;
					final StringBuilder builder = new StringBuilder();
					while ((line = reader.readLine()) != null) {
						builder.append(line);
						builder.append("\n");
					}
					JOptionPane.showMessageDialog(ServicePanel.this, builder.toString(), TwisterSwingResources.getInstance().getString("label.about"), JOptionPane.PLAIN_MESSAGE);
				}
				catch (final Exception x) {
					x.printStackTrace();
				}
				finally {
					if (reader != null) {
						try {
							reader.close();
						}
						catch (final IOException x) {
						}
					}
				}
			}
		}

		private class ChangeWorkspaceAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			/**
			 * 
			 */
			public ChangeWorkspaceAction() {
				super(TwisterSwingResources.getInstance().getString("action.changeWorkspace"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				final Properties properties = new Properties();
				File workspace = null;
				try {
					properties.load(new FileInputStream(System.getProperty("user.home") + "/JAME.properties"));
					workspace = new File((String) properties.get("workspace"));
					if (workspace == null) {
						workspace = new File(System.getProperty("user.home") + "/" + System.getProperty("jame.workspace", "JAME-workspace"));
					}
				}
				catch (final Exception x) {
					x.printStackTrace();
				}
				// fileChooser.setCurrentDirectory(workspace.getParentFile());
				fileChooser.setCurrentDirectory(workspace);
				// fileChooser.setSelectedFile(workspace);
				final int returnVal = fileChooser.showSaveDialog(new JFrame());
				if (returnVal == JFileChooser.APPROVE_OPTION) {
					workspace = fileChooser.getSelectedFile().getAbsoluteFile().getAbsoluteFile();
					if (workspace != null) {
						properties.put("workspace", workspace.getAbsolutePath());
						try {
							properties.store(new FileOutputStream(System.getProperty("user.home") + "/JAME.properties"), null);
							context.restart();
						}
						catch (final Exception x) {
							x.printStackTrace();
						}
					}
				}
			}
		}

		private class ServicePanelListener extends ServiceAdapter {
			// /**
			// * @see net.sf.jame.service.swing.ServiceAdapter#jobsResumed()
			// */
			// @Override
			// public void jobsResumed() {
			// // isSuspended = false;
			// // updateButtons();
			// }
			//
			// /**
			// * @see net.sf.jame.service.swing.ServiceAdapter#jobsSuspended()
			// */
			// @Override
			// public void jobsSuspended() {
			// // isSuspended = true;
			// // updateButtons();
			// }
			/**
			 * @see net.sf.jame.service.swing.ServiceAdapter#clipCreated(net.sf.jame.service.clip.MovieClipDataRow)
			 */
			@Override
			public void clipCreated(final MovieClipDataRow clip) {
			}

			/**
			 * @see net.sf.jame.service.swing.ServiceAdapter#clipDeleted(net.sf.jame.service.clip.MovieClipDataRow)
			 */
			@Override
			public void clipDeleted(final MovieClipDataRow clip) {
			}

			/**
			 * @see net.sf.jame.service.swing.ServiceAdapter#clipUpdated(net.sf.jame.service.clip.MovieClipDataRow)
			 */
			@Override
			public void clipUpdated(final MovieClipDataRow clip) {
			}

			/**
			 * @see net.sf.jame.service.swing.ServiceAdapter#profileCreated(net.sf.jame.service.profile.RenderProfileDataRow)
			 */
			@Override
			public void profileCreated(final RenderProfileDataRow profile) {
			}

			/**
			 * @see net.sf.jame.service.swing.ServiceAdapter#profileDeleted(net.sf.jame.service.profile.RenderProfileDataRow)
			 */
			@Override
			public void profileDeleted(final RenderProfileDataRow profile) {
			}

			/**
			 * @see net.sf.jame.service.swing.ServiceAdapter#profileUpdated(net.sf.jame.service.profile.RenderProfileDataRow)
			 */
			@Override
			public void profileUpdated(final RenderProfileDataRow profile) {
				preview.setArea(new Rectangle(profile.getOffsetX(), profile.getOffsetY(), profile.getImageWidth(), profile.getImageHeight()));
			}
		}

		private class TableListener implements TableModelListener {
			/**
			 * @see javax.swing.event.TableModelListener#tableChanged(javax.swing.event.TableModelEvent)
			 */
			public void tableChanged(final TableModelEvent e) {
				updateButtons();
			}
		}
		// private class MovieClipListCellRenderer extends DefaultListCellRenderer {
		// private static final long serialVersionUID = 1L;
		//
		// /**
		// * @see javax.swing.DefaultListCellRenderer#getListCellRendererComponent(javax.swing.JList, java.lang.Object, int, boolean, boolean)
		// */
		// @Override
		// public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
		// if (value != null) {
		// return super.getListCellRendererComponent(list, ((MovieClipDataRow) value).getClipName(), index, isSelected, cellHasFocus);
		// }
		// else {
		// return super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
		// }
		// }
		// }
		//
		// private class RenderProfileListCellRenderer extends DefaultListCellRenderer {
		// private static final long serialVersionUID = 1L;
		//
		// /**
		// * @see javax.swing.DefaultListCellRenderer#getListCellRendererComponent(javax.swing.JList, java.lang.Object, int, boolean, boolean)
		// */
		// @Override
		// public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
		// if (value != null) {
		// return super.getListCellRendererComponent(list, ((RenderProfileDataRow) value).getProfileName(), index, isSelected, cellHasFocus);
		// }
		// else {
		// return super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
		// }
		// }
		// }
	}
}
