/*
 * $Id:TwisterFrame.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.twister.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.DisplayMode;
import java.awt.FlowLayout;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.HeadlessException;
import java.awt.Image;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Semaphore;

import javax.imageio.ImageIO;
import javax.swing.AbstractAction;
import javax.swing.DefaultSingleSelectionModel;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;
import javax.swing.border.LineBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import net.sf.jame.core.config.DefaultConfigContext;
import net.sf.jame.core.extension.ExtensionException;
import net.sf.jame.core.swing.IconButton;
import net.sf.jame.core.swing.util.GUIFactory;
import net.sf.jame.core.tree.DefaultNodeSession;
import net.sf.jame.core.tree.NodeAction;
import net.sf.jame.core.tree.NodeActionValue;
import net.sf.jame.core.xml.XML;
import net.sf.jame.core.xml.XMLNodeBuilder;
import net.sf.jame.service.AsyncService;
import net.sf.jame.service.AsyncService.ServiceVoidCallback;
import net.sf.jame.service.clip.MovieClip;
import net.sf.jame.service.clip.MovieClipDataRow;
import net.sf.jame.service.swing.MovieClipTableModel;
import net.sf.jame.service.swing.RenderJobTableModel;
import net.sf.jame.service.swing.RenderProfileTableModel;
import net.sf.jame.service.swing.ServiceContext;
import net.sf.jame.twister.TwisterBookmark;
import net.sf.jame.twister.TwisterClip;
import net.sf.jame.twister.TwisterClipController;
import net.sf.jame.twister.TwisterClipXMLExporter;
import net.sf.jame.twister.TwisterConfig;
import net.sf.jame.twister.TwisterConfigBuilder;
import net.sf.jame.twister.TwisterConfigNodeBuilder;
import net.sf.jame.twister.TwisterRuntime;
import net.sf.jame.twister.TwisterSequence;
import net.sf.jame.twister.TwisterSessionController;
import net.sf.jame.twister.TwisterTree;
import net.sf.jame.twister.renderer.DefaultTwisterRenderer;
import net.sf.jame.twister.renderer.TwisterRenderingHints;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * @author Andrea Medeghini
 */
public class TwisterFrame extends JFrame {
	private static final long serialVersionUID = 1L;
	private static final String TWISTER_FRAME_TITLE = "twisterFrame.title";
	private static final String TWISTER_FRAME_WIDTH = "twisterFrame.width";
	private static final String TWISTER_FRAME_HEIGHT = "twisterFrame.height";
	private static final String TWISTER_FRAME_ICON = "twisterFrame.icon";
	private static final Logger logger = Logger.getLogger(TwisterFrame.class);
	private final Semaphore semaphore = new Semaphore(0, true);
	private final TwisterSessionController sessionController;
	private final DefaultSingleSelectionModel model;
	private final TwisterTree twisterTree;
	private final AsyncService service;
	private JFrame configFrame;
	private static ServiceFrame serviceFrame;
	private final TwisterConfig config;
	private final TwisterCanvas canvas;
	private TwisterClip clip;
	private TwisterClip tmpClip;
	private TwisterClip playClip;
	private TwisterTree tree;
	private long clipDuration;
	private long sequenceStopTime;
	private long sequenceStartTime;
	private TwisterSequence sequence;
	private final TwisterPanel panel;
	private final TwisterContext context;

	// private int vcells;
	// private int hcells;
	/**
	 * @param context
	 * @param service
	 * @param config
	 * @param hcells
	 * @param vcells
	 * @throws HeadlessException
	 * @throws ExtensionException
	 */
	public TwisterFrame(final TwisterContext context, final AsyncService service, final TwisterConfig config, final int hcells, final int vcells) throws HeadlessException, ExtensionException {
		// this.hcells = hcells;
		// this.vcells = vcells;
		this.context = context;
		final int defaultWidth = Integer.parseInt(TwisterSwingResources.getInstance().getString(TwisterFrame.TWISTER_FRAME_WIDTH));
		final int defaultHeight = Integer.parseInt(TwisterSwingResources.getInstance().getString(TwisterFrame.TWISTER_FRAME_HEIGHT));
		final int width = Integer.getInteger(TwisterFrame.TWISTER_FRAME_WIDTH, defaultWidth);
		final int height = Integer.getInteger(TwisterFrame.TWISTER_FRAME_HEIGHT, defaultHeight);
		model = new DefaultSingleSelectionModel();
		canvas = new TwisterCanvas(hcells, vcells, model);
		panel = new TwisterPanel(canvas);
		getContentPane().add(panel);
		setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
		setTitle(TwisterSwingResources.getInstance().getString(TwisterFrame.TWISTER_FRAME_TITLE));
		final URL resource = TwisterFrame.class.getClassLoader().getResource(TwisterSwingResources.getInstance().getString(TwisterFrame.TWISTER_FRAME_ICON));
		if (resource != null) {
			setIconImage(getToolkit().createImage(resource));
		}
		this.setSize(new Dimension(width, height));
		final Point p = GraphicsEnvironment.getLocalGraphicsEnvironment().getCenterPoint();
		p.x -= getWidth() / 2;
		p.y -= getHeight() / 2;
		this.setLocation(p);
		this.service = service;
		this.config = config;
		sessionController = new TwisterSessionController("options", config);
		sessionController.init();
		sessionController.setRenderContext(canvas);
		twisterTree = new TwisterTree();
		final TwisterConfigNodeBuilder nodeBuilder = new TwisterConfigNodeBuilder(config);
		twisterTree.getRootNode().setContext(config.getContext());
		twisterTree.getRootNode().setSession(sessionController);
		nodeBuilder.createNodes(twisterTree.getRootNode());
		addWindowListener(new TwisterWindowListener(canvas));
		canvas.setShowBookmarkIcons(true);
		canvas.addChangeListener(new ChangeListener() {
			public void stateChanged(final ChangeEvent e) {
				clip = canvas.getClip();
				tmpClip = canvas.getClip();
				playClip = tmpClip;
				sequence = null;
				tree = null;
				panel.updateButtons();
			}
		});
		context.addFrame(this);
		if (TwisterFrame.serviceFrame == null) {
			TwisterFrame.serviceFrame = new ServiceFrame(new DefaulServiceContext(context, service, hcells, vcells), service, new MovieClipTableModel(service), new RenderProfileTableModel(service), new RenderJobTableModel(service));
		}
	}

	/**
	 * @see java.lang.Object#finalize()
	 */
	@Override
	protected void finalize() throws Throwable {
		TwisterFrame.logger.debug("Frame finalized");
		super.finalize();
	}

	/**
	 * @param clip
	 */
	public void loadClip(final TwisterClip clip) {
		try {
			if (clip != null) {
				try {
					final TwisterClipController controller = new TwisterClipController(clip);
					controller.init();
					if (controller.getDuration() > 0) {
						canvas.stopRenderers();
						canvas.stop();
						this.clip = clip;
						tmpClip = clip;
						playClip = tmpClip;
						sequence = null;
						tree = null;
						canvas.start(clip);
						canvas.startRenderers();
						canvas.refresh();
					}
					else {
						canvas.stopRenderers();
						sequence = null;
						tree = null;
						config.setFrameConfigElement(controller.getConfig().getFrameConfigElement().clone());
						config.setEffectConfigElement(controller.getConfig().getEffectConfigElement().clone());
						config.setBackground(controller.getConfig().getBackground());
						canvas.startRenderers();
						canvas.refresh();
					}
				}
				catch (final ExtensionException x) {
					x.printStackTrace();
				}
			}
			panel.updateButtons();
		}
		catch (final HeadlessException e) {
			e.printStackTrace();
		}
	}

	private void openConfigWindow(final TwisterConfig config, final TwisterCanvas canvas) {
		if (configFrame == null) {
			configFrame = new ConfigFrame(config, twisterTree, canvas, sessionController);
			// if (!Boolean.getBoolean("twisterFrame.useConfigFrame")) {
			// configFrame = new NavigatorFrame(twisterTree, canvas, sessionController);
			// }
			// else {
			// configFrame = new ConfigFrame(config, twisterTree, canvas, sessionController);
			// }
		}
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				configFrame.setVisible(true);
				configFrame.toFront();
			}
		});
	}

	private void openServiceWindow(final AsyncService service) {
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				TwisterFrame.serviceFrame.setVisible(true);
				TwisterFrame.serviceFrame.toFront();
			}
		});
	}

	// private void openPluginsWindow() {
	// if (pluginsFrame == null) {
	// pluginsFrame = new PluginsFrame();
	// }
	// SwingUtilities.invokeLater(new Runnable() {
	// public void run() {
	// pluginsFrame.setVisible(true);
	// pluginsFrame.toFront();
	// }
	// });
	// }
	private TwisterBookmark createBookmark() throws Exception {
		final TwisterBookmark bookmark = new TwisterBookmark();
		bookmark.setConfig(config.clone());
		final TwisterRuntime runtime = new TwisterRuntime(bookmark.getConfig());
		final DefaultTwisterRenderer renderer = new DefaultTwisterRenderer(runtime);
		final Map<Object, Object> hints = new HashMap<Object, Object>();
		hints.put(TwisterRenderingHints.KEY_MEMORY, TwisterRenderingHints.MEMORY_LOW);
		renderer.setRenderingHints(hints);
		bookmark.setRenderer(renderer);
		return bookmark;
	}

	private class TwisterWindowListener extends WindowAdapter {
		private final TwisterCanvas canvas;

		/**
		 * @param canvas
		 */
		public TwisterWindowListener(final TwisterCanvas canvas) {
			this.canvas = canvas;
		}

		/**
		 * @see java.awt.event.WindowAdapter#windowOpened(java.awt.event.WindowEvent)
		 */
		@Override
		public void windowOpened(final WindowEvent e) {
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					if (canvas != null) {
						try {
							if (!canvas.isStarted()) {
								canvas.start(config);
								canvas.startRenderers();
								canvas.refresh();
								panel.updateButtons();
							}
						}
						catch (final ExtensionException x) {
							x.printStackTrace();
						}
					}
				}
			});
		}

		/**
		 * @see java.awt.event.WindowAdapter#windowClosing(java.awt.event.WindowEvent)
		 */
		@Override
		public void windowClosing(final WindowEvent e) {
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					if (canvas != null) {
						if (canvas.isStarted()) {
							canvas.stopRenderers();
							canvas.stop();
							canvas.refresh();
							panel.updateButtons();
						}
					}
					dispose();
					if (configFrame != null) {
						configFrame.dispose();
						configFrame = null;
					}
					context.removeFrame(TwisterFrame.this);
					if (context.getFrameCount() == 0) {
						context.exit();
					}
				}
			});
		}

		/**
		 * @see java.awt.event.WindowAdapter#windowDeiconified(java.awt.event.WindowEvent)
		 */
		@Override
		public void windowDeiconified(final WindowEvent e) {
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					if (canvas != null) {
						try {
							if (!canvas.isStarted()) {
								canvas.start(config);
								canvas.startRenderers();
								canvas.refresh();
								panel.updateButtons();
							}
						}
						catch (final ExtensionException x) {
							x.printStackTrace();
						}
					}
				}
			});
		}

		/**
		 * @see java.awt.event.WindowAdapter#windowIconified(java.awt.event.WindowEvent)
		 */
		@Override
		public void windowIconified(final WindowEvent e) {
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					if (canvas != null) {
						if (canvas.isStarted()) {
							canvas.stopRenderers();
							canvas.stop();
							canvas.refresh();
							panel.updateButtons();
						}
					}
				}
			});
		}

		/**
		 * @see java.awt.event.WindowAdapter#windowClosing(java.awt.event.WindowEvent)
		 */
		@Override
		public void windowClosed(final WindowEvent e) {
		}
	}

	private MovieClipDataRow createDefaultMovieClip(final long duration) {
		final MovieClipDataRow clip = new MovieClipDataRow(new MovieClip());
		clip.setClipName("New Clip");
		clip.setDescription("");
		clip.setDuration(duration);
		return clip;
	}

	public class TwisterPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private JButton editConfigButton = GUIFactory.createButton(new EditConfigAction(), TwisterSwingResources.getInstance().getString("tooltip.editConfiguration"));
		private JButton showServiceButton = GUIFactory.createButton(new ShowServiceAction(), TwisterSwingResources.getInstance().getString("tooltip.showService"));
		private JButton playSessionButton = GUIFactory.createButton(new PlaySessionAction(), TwisterSwingResources.getInstance().getString("tooltip.playSession"));
		private JButton startSessionButton = GUIFactory.createButton(new StartSessionAction(), TwisterSwingResources.getInstance().getString("tooltip.startSession"));
		private JButton stopSessionButton = GUIFactory.createButton(new StopSessionAction(), TwisterSwingResources.getInstance().getString("tooltip.stopSession"));
		private JButton suspendSessionButton = GUIFactory.createButton(new SuspendSessionAction(), TwisterSwingResources.getInstance().getString("tooltip.suspendSession"));
		private JButton resumeSessionButton = GUIFactory.createButton(new ResumeSessionAction(), TwisterSwingResources.getInstance().getString("tooltip.resumeSession"));
		private JButton storePhotoButton = GUIFactory.createButton(new StorePhotoAction(), TwisterSwingResources.getInstance().getString("tooltip.storePhoto"));
		private JButton storeMovieButton = GUIFactory.createButton(new StoreMovieAction(), TwisterSwingResources.getInstance().getString("tooltip.storeMovie"));
		private JButton undoButton = GUIFactory.createButton(new UndoAction(), TwisterSwingResources.getInstance().getString("tooltip.undo"));
		private JButton redoButton = GUIFactory.createButton(new RedoAction(), TwisterSwingResources.getInstance().getString("tooltip.redo"));
		private JButton addBookmarkButton = GUIFactory.createButton(new AddBookmarkAction(), TwisterSwingResources.getInstance().getString("tooltip.addBookmark"));
		private JButton fullScreenButton = GUIFactory.createButton(new FullScreenAction(), TwisterSwingResources.getInstance().getString("tooltip.fullScreen"));

		/**
		 * @param canvas
		 */
		public TwisterPanel(final TwisterCanvas canvas) {
			setLayout(new BorderLayout());
			final JPanel panelButtons = new JPanel(new FlowLayout(FlowLayout.CENTER));
			final JPanel panelCanvas = new JPanel(new BorderLayout());
			panelButtons.setBackground(new Color(0x2f2f2f));
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/options-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/options-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/options-pressed-button.png"));
				editConfigButton = new IconButton(normalImage, pressedImage, focusedImage, new EditConfigAction());
				editConfigButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.editConfiguration"));
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/clips-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/clips-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/clips-pressed-button.png"));
				showServiceButton = new IconButton(normalImage, pressedImage, focusedImage, new ShowServiceAction());
				showServiceButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.showService"));
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/record-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/record-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/record-pressed-button.png"));
				startSessionButton = new IconButton(normalImage, pressedImage, focusedImage, new StartSessionAction());
				startSessionButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.startSession"));
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/stop-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/stop-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/stop-pressed-button.png"));
				stopSessionButton = new IconButton(normalImage, pressedImage, focusedImage, new StopSessionAction());
				stopSessionButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.stopSession"));
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/pause-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/pause-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/pause-pressed-button.png"));
				suspendSessionButton = new IconButton(normalImage, pressedImage, focusedImage, new SuspendSessionAction());
				suspendSessionButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.suspendSession"));
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/play-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/play-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/play-pressed-button.png"));
				playSessionButton = new IconButton(normalImage, pressedImage, focusedImage, new PlaySessionAction());
				playSessionButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.playSession"));
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/record-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/record-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/record-pressed-button.png"));
				resumeSessionButton = new IconButton(normalImage, pressedImage, focusedImage, new ResumeSessionAction());
				resumeSessionButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.resumeSession"));
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/photo-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/photo-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/photo-pressed-button.png"));
				storePhotoButton = new IconButton(normalImage, pressedImage, focusedImage, new StorePhotoAction());
				storePhotoButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.storePhoto"));
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/movie-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/movie-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/movie-pressed-button.png"));
				storeMovieButton = new IconButton(normalImage, pressedImage, focusedImage, new StoreMovieAction());
				storeMovieButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.storeMovie"));
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/add-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/add-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/add-pressed-button.png"));
				addBookmarkButton = new IconButton(normalImage, pressedImage, focusedImage, new AddBookmarkAction());
				addBookmarkButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.addBookmark"));
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/left-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/left-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/left-pressed-button.png"));
				undoButton = new IconButton(normalImage, pressedImage, focusedImage, new UndoAction());
				undoButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.undo"));
				((IconButton) undoButton).setRepeatTime(50);
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/right-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/right-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/right-pressed-button.png"));
				redoButton = new IconButton(normalImage, pressedImage, focusedImage, new RedoAction());
				redoButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.redo"));
				((IconButton) redoButton).setRepeatTime(50);
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			try {
				final Image normalImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/up-normal-button.png"));
				final Image focusedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/up-focused-button.png"));
				final Image pressedImage = ImageIO.read(TwisterFrame.class.getResourceAsStream("/icons/up-pressed-button.png"));
				fullScreenButton = new IconButton(normalImage, pressedImage, focusedImage, new FullScreenAction());
				fullScreenButton.setToolTipText(TwisterSwingResources.getInstance().getString("tooltip.fullScreen"));
			}
			catch (final IOException e) {
				e.printStackTrace();
			}
			panelButtons.add(editConfigButton);
			panelButtons.add(showServiceButton);
			panelButtons.add(startSessionButton);
			panelButtons.add(stopSessionButton);
			panelButtons.add(suspendSessionButton);
			panelButtons.add(resumeSessionButton);
			panelButtons.add(playSessionButton);
			panelButtons.add(storePhotoButton);
			panelButtons.add(storeMovieButton);
			panelButtons.add(undoButton);
			panelButtons.add(redoButton);
			panelButtons.add(addBookmarkButton);
			// panelButtons.add(fullScreenButton);
			panelCanvas.add(canvas);
			panelCanvas.setBorder(new LineBorder(Color.BLACK));
			this.add(panelButtons, BorderLayout.SOUTH);
			this.add(panelCanvas, BorderLayout.CENTER);
			model.addChangeListener(new BookmarkSelectionListener());
			updateButtons();
		}

		public void updateButtons() {
			startSessionButton.setEnabled(canStartSession());
			startSessionButton.setVisible(canStartSession());
			stopSessionButton.setEnabled(canStopSession());
			stopSessionButton.setVisible(canStopSession());
			resumeSessionButton.setEnabled(canResumeSession());
			resumeSessionButton.setVisible(canResumeSession());
			suspendSessionButton.setEnabled(canSuspendSession());
			suspendSessionButton.setVisible(canSuspendSession());
			playSessionButton.setEnabled(canPlaySession());
			playSessionButton.setVisible(canPlaySession());
			storeMovieButton.setEnabled(canStoreMovie());
			storeMovieButton.setVisible(canStoreMovie());
			storePhotoButton.setEnabled(canStorePhoto());
			storePhotoButton.setVisible(canStorePhoto());
			undoButton.setEnabled(canUndo());
			undoButton.setVisible(canUndo());
			redoButton.setEnabled(canRedo());
			redoButton.setVisible(canRedo());
			addBookmarkButton.setEnabled(canAddBookmark());
			addBookmarkButton.setVisible(canAddBookmark());
			editConfigButton.setEnabled(canEditConfig());
			editConfigButton.setVisible(canEditConfig());
		}

		/**
		 * @return
		 */
		private boolean isPlayState() {
			return canvas.getState() == TwisterCanvas.STATE_PLAY;
		}

		/**
		 * @return
		 */
		private boolean isEditState() {
			return canvas.getState() == TwisterCanvas.STATE_EDIT;
		}

		/**
		 * @return
		 */
		private boolean canStorePhoto() {
			return isEditState() || isPlayState();
		}

		/**
		 * @return
		 */
		private boolean canStoreMovie() {
			return tmpClip != null && isEditState();
		}

		/**
		 * @return
		 */
		private boolean canPlaySession() {
			return (clip == null && tmpClip != null && isEditState()) || (playClip == null && isPlayState());
		}

		/**
		 * @return
		 */
		private boolean canSuspendSession() {
			return (clip != null && sequence != null && isEditState()) || (playClip != null && isPlayState());
		}

		/**
		 * @return
		 */
		private boolean canResumeSession() {
			return clip != null && sequence == null && isEditState();
		}

		/**
		 * @return
		 */
		private boolean canStopSession() {
			return (clip != null && isEditState()) || isPlayState();
		}

		/**
		 * @return
		 */
		private boolean canStartSession() {
			return clip == null && isEditState();
		}

		/**
		 * @return
		 */
		private boolean canUndo() {
			return isEditState();
		}

		/**
		 * @return
		 */
		private boolean canRedo() {
			return isEditState();
		}

		/**
		 * @return
		 */
		private boolean canAddBookmark() {
			return isEditState();
		}

		/**
		 * @return
		 */
		private boolean canEditConfig() {
			return isEditState();
		}

		protected class EditConfigAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			public EditConfigAction() {
				super(TwisterSwingResources.getInstance().getString("action.editConfig"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				openConfigWindow(config, canvas);
			}
		}

		protected class ShowServiceAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			public ShowServiceAction() {
				super(TwisterSwingResources.getInstance().getString("action.showService"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				openServiceWindow(service);
			}
		}

		protected class StartSessionAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			public StartSessionAction() {
				super(TwisterSwingResources.getInstance().getString("action.startSession"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (canStartSession()) {
					tree = new TwisterTree();
					clip = new TwisterClip();
					sequence = new TwisterSequence();
					sequence.setInitialConfig(config.clone());
					final TwisterConfigNodeBuilder builder = new TwisterConfigNodeBuilder(config);
					builder.createNodes(tree.getRootNode());
					tree.getRootNode().setContext(config.getContext());
					tree.getRootNode().setSession(new DefaultNodeSession("clip " + clip.getSequenceCount()));
					canvas.setSymbol(TwisterCanvas.SYMBOL_RECORD);
					sequenceStartTime = System.currentTimeMillis();
					clipDuration = 0;
				}
				updateButtons();
			}
		}

		protected class StopSessionAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			public StopSessionAction() {
				super(TwisterSwingResources.getInstance().getString("action.stopSession"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (isPlayState()) {
					try {
						canvas.stopRenderers();
						canvas.stop();
						playClip = null;
						sequence = null;
						clip = null;
						tree = null;
						config.getContext().updateTimestamp();
						config.setFrameConfigElement(canvas.getConfig().getFrameConfigElement().clone());
						config.setEffectConfigElement(canvas.getConfig().getEffectConfigElement().clone());
						config.setBackground(canvas.getConfig().getBackground());
						canvas.start(config);
						canvas.startRenderers();
						canvas.refresh();
					}
					catch (final ExtensionException x) {
						x.printStackTrace();
					}
				}
				else if (canStopSession()) {
					if (sequence != null) {
						sequenceStopTime = System.currentTimeMillis();
						final List<NodeAction> nodeActions = tree.getRootNode().getSession().getActions();
						sequence.setFinalConfig(config.clone());
						for (final NodeAction nodeAction : nodeActions) {
							final NodeActionValue value = nodeAction.toActionValue();
							value.setTimestamp(value.getTimestamp() - sequenceStartTime);
							final NodeAction action = new NodeAction(value);
							sequence.addAction(action);
							// logger.debug(action);
						}
						sequence.setDuration(sequenceStopTime - sequenceStartTime);
						clipDuration += sequence.getDuration();
						clip.addSequence(sequence);
					}
					tmpClip = clip;
					sequence = null;
					clip = null;
					tree = null;
					canvas.setSymbol(TwisterCanvas.SYMBOL_NONE);
				}
				updateButtons();
			}
		}

		protected class SuspendSessionAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			public SuspendSessionAction() {
				super(TwisterSwingResources.getInstance().getString("action.suspendSession"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (canSuspendSession()) {
					if (isEditState()) {
						sequenceStopTime = System.currentTimeMillis();
						final List<NodeAction> nodeActions = tree.getRootNode().getSession().getActions();
						sequence.setFinalConfig(config.clone());
						for (final NodeAction nodeAction : nodeActions) {
							final NodeActionValue value = nodeAction.toActionValue();
							value.setTimestamp(value.getTimestamp() - sequenceStartTime);
							sequence.addAction(new NodeAction(value));
						}
						sequence.setDuration(sequenceStopTime - sequenceStartTime);
						clipDuration += sequence.getDuration();
						clip.addSequence(sequence);
						sequence = null;
						tree = null;
						canvas.setSymbol(TwisterCanvas.SYMBOL_PAUSE);
					}
					else if (isPlayState()) {
						canvas.setSymbol(TwisterCanvas.SYMBOL_PAUSE);
						canvas.suspend();
						playClip = null;
					}
				}
				updateButtons();
			}
		}

		protected class ResumeSessionAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			public ResumeSessionAction() {
				super(TwisterSwingResources.getInstance().getString("action.resumeSession"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (canResumeSession()) {
					sequence = new TwisterSequence();
					sequence.setInitialConfig(config.clone());
					tree = new TwisterTree();
					final TwisterConfigNodeBuilder builder = new TwisterConfigNodeBuilder(config);
					builder.createNodes(tree.getRootNode());
					tree.getRootNode().setContext(config.getContext());
					tree.getRootNode().setSession(new DefaultNodeSession("clip " + clip.getSequenceCount()));
					canvas.setSymbol(TwisterCanvas.SYMBOL_RECORD);
					sequenceStartTime = System.currentTimeMillis();
				}
				updateButtons();
			}
		}

		protected class PlaySessionAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			public PlaySessionAction() {
				super(TwisterSwingResources.getInstance().getString("action.playSession"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (canPlaySession()) {
					try {
						if (canvas.isSuspended()) {
							playClip = tmpClip;
							canvas.setSymbol(TwisterCanvas.SYMBOL_PLAY);
							canvas.resume();
							canvas.refresh();
						}
						else {
							if (configFrame != null) {
								configFrame.setVisible(false);
							}
							canvas.stopRenderers();
							canvas.stop();
							playClip = tmpClip;
							canvas.start(playClip);
							canvas.startRenderers();
							canvas.refresh();
						}
					}
					catch (final ExtensionException x) {
						x.printStackTrace();
					}
				}
				updateButtons();
			}
		}

		protected class StorePhotoAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			public StorePhotoAction() {
				super(TwisterSwingResources.getInstance().getString("action.storePhoto"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (canStorePhoto()) {
					if (isEditState()) {
						if (JOptionPane.showConfirmDialog(TwisterPanel.this, TwisterSwingResources.getInstance().getString("message.confirmSavePhoto"), TwisterSwingResources.getInstance().getString("label.savePhoto"), JOptionPane.YES_NO_OPTION) == JOptionPane.OK_OPTION) {
							final MovieClipDataRow dataRow = createDefaultMovieClip(0);
							service.createClip(new ServiceVoidCallback() {
								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
								 */
								public void executed() {
									try {
										final TwisterClipXMLExporter clipExporter = new TwisterClipXMLExporter();
										final Document doc = XML.createDocument();
										final XMLNodeBuilder builder = XML.createDefaultXMLNodeBuilder(doc);
										final TwisterSequence sequence = new TwisterSequence();
										sequence.setInitialConfig(config.clone());
										sequence.setFinalConfig(config.clone());
										final TwisterClip clip = new TwisterClip();
										clip.addSequence(sequence);
										final Element element = clipExporter.exportToElement(clip, builder);
										doc.appendChild(element);
										final OutputStream os = service.getService().getClipOutputStream(dataRow.getClipId());
										XML.saveDocument(os, "twister-clip.xml", doc);
										os.close();
									}
									catch (final Exception x) {
										x.printStackTrace();
									}
									semaphore.release();
								}

								/**
								 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
								 */
								public void failed(final Throwable throwable) {
									TwisterFrame.logger.error("Can't create the clip", throwable);
									semaphore.release();
								}
							}, dataRow);
							try {
								semaphore.acquire();
							}
							catch (final InterruptedException x) {
								x.printStackTrace();
							}
						}
					}
					else if (isPlayState()) {
						final MovieClipDataRow dataRow = createDefaultMovieClip(0);
						service.createClip(new ServiceVoidCallback() {
							/**
							 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
							 */
							public void executed() {
								try {
									final TwisterClipXMLExporter clipExporter = new TwisterClipXMLExporter();
									final Document doc = XML.createDocument();
									final XMLNodeBuilder builder = XML.createDefaultXMLNodeBuilder(doc);
									final TwisterSequence sequence = new TwisterSequence();
									final TwisterConfig config = canvas.getConfig();
									sequence.setInitialConfig(config);
									sequence.setFinalConfig(config);
									final TwisterClip clip = new TwisterClip();
									clip.addSequence(sequence);
									final Element element = clipExporter.exportToElement(clip, builder);
									doc.appendChild(element);
									final OutputStream os = service.getService().getClipOutputStream(dataRow.getClipId());
									XML.saveDocument(os, "twister-clip.xml", doc);
									os.close();
								}
								catch (final Exception x) {
									x.printStackTrace();
								}
								semaphore.release();
							}

							/**
							 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
							 */
							public void failed(final Throwable throwable) {
								TwisterFrame.logger.error("Can't create the clip", throwable);
								semaphore.release();
							}
						}, dataRow);
						try {
							semaphore.acquire();
						}
						catch (final InterruptedException x) {
							x.printStackTrace();
						}
					}
				}
			}
		}

		protected class StoreMovieAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			public StoreMovieAction() {
				super(TwisterSwingResources.getInstance().getString("action.storeMovie"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				if (canStoreMovie()) {
					if (JOptionPane.showConfirmDialog(TwisterPanel.this, TwisterSwingResources.getInstance().getString("message.confirmSaveMovie"), TwisterSwingResources.getInstance().getString("label.saveMovie"), JOptionPane.YES_NO_OPTION) == JOptionPane.OK_OPTION) {
						final TwisterClipController controller = new TwisterClipController(tmpClip);
						controller.init();
						logger.debug("Clip duration = " + clipDuration + ", controller duration = " + controller.getDuration());
						final MovieClipDataRow clipDataRow = createDefaultMovieClip(controller.getDuration());
						service.createClip(new ServiceVoidCallback() {
							/**
							 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
							 */
							public void executed() {
								try {
									final TwisterClipXMLExporter clipExporter = new TwisterClipXMLExporter();
									final Document doc = XML.createDocument();
									final XMLNodeBuilder builder = XML.createDefaultXMLNodeBuilder(doc);
									final Element element = clipExporter.exportToElement(tmpClip, builder);
									doc.appendChild(element);
									final OutputStream os = service.getService().getClipOutputStream(clipDataRow.getClipId());
									XML.saveDocument(os, "twister-clip.xml", doc);
									os.close();
								}
								catch (final Exception x) {
									x.printStackTrace();
								}
								semaphore.release();
							}

							/**
							 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
							 */
							public void failed(final Throwable throwable) {
								TwisterFrame.logger.error("Can't create the clip", throwable);
								semaphore.release();
							}
						}, clipDataRow);
						try {
							semaphore.acquire();
						}
						catch (final InterruptedException x) {
							x.printStackTrace();
						}
					}
				}
			}
		}

		protected class UndoAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			public UndoAction() {
				super(TwisterSwingResources.getInstance().getString("action.undo"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				// SwingUtilities.invokeLater(new Runnable() {
				// public void run() {
				canvas.submitCommand(new Runnable() {
					/**
					 * @see java.lang.Runnable#run()
					 */
					public void run() {
						if (sessionController != null) {
							sessionController.undoAction(true);
						}
					}
				});
				// }
				// });
			}
		}

		protected class RedoAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			public RedoAction() {
				super(TwisterSwingResources.getInstance().getString("action.redo"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				// SwingUtilities.invokeLater(new Runnable() {
				// public void run() {
				canvas.submitCommand(new Runnable() {
					/**
					 * @see java.lang.Runnable#run()
					 */
					public void run() {
						if (sessionController != null) {
							sessionController.redoAction(true);
						}
					}
				});
				// }
				// });
			}
		}

		protected class AddBookmarkAction extends AbstractAction {
			private static final long serialVersionUID = 1L;

			public AddBookmarkAction() {
				super(TwisterSwingResources.getInstance().getString("action.addBookmark"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				try {
					canvas.addBookmark(createBookmark());
				}
				catch (final Exception x) {
					x.printStackTrace();
				}
			}
		}

		protected class FullScreenAction extends AbstractAction {
			private static final long serialVersionUID = 1L;
			private boolean isFullScreen;

			public FullScreenAction() {
				super(TwisterSwingResources.getInstance().getString("action.fullScreen"));
			}

			/**
			 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
			 */
			public void actionPerformed(final ActionEvent e) {
				final GraphicsEnvironment environment = GraphicsEnvironment.getLocalGraphicsEnvironment();
				if (!GraphicsEnvironment.isHeadless()) {
					final GraphicsDevice device = environment.getDefaultScreenDevice();
					if (device.getType() == GraphicsDevice.TYPE_RASTER_SCREEN) {
						if (device.isFullScreenSupported()) {
							try {
								canvas.stopRenderers();
								canvas.stop();
								if (!isFullScreen) {
									device.setFullScreenWindow(TwisterFrame.this);
									isFullScreen = true;
									if (device.isDisplayChangeSupported()) {
										device.setDisplayMode(new DisplayMode(1024, 768, 32, DisplayMode.REFRESH_RATE_UNKNOWN));
									}
								}
								else {
									device.setFullScreenWindow(null);
									isFullScreen = false;
								}
								canvas.start(config);
								canvas.startRenderers();
							}
							catch (final ExtensionException x) {
								x.printStackTrace();
							}
						}
					}
				}
			}
		}

		private class BookmarkSelectionListener implements ChangeListener {
			/**
			 * @see javax.swing.event.ChangeListener#stateChanged(javax.swing.event.ChangeEvent)
			 */
			public void stateChanged(final ChangeEvent e) {
				TwisterFrame.logger.debug(model.getSelectedIndex());
				final TwisterBookmark bookmark = canvas.getSelectedBookmark();
				if (bookmark != null) {
					canvas.stopRenderers();
					config.getContext().updateTimestamp();
					config.setFrameConfigElement(bookmark.getConfig().getFrameConfigElement().clone());
					config.setEffectConfigElement(bookmark.getConfig().getEffectConfigElement().clone());
					canvas.startRenderers();
					canvas.refresh();
				}
				updateButtons();
			}
		}
	}

	private class DefaulServiceContext implements ServiceContext {
		private final TwisterContext context;
		private final AsyncService service;
		private final int vcells;
		private final int hcells;

		/**
		 * @param context
		 * @param service
		 * @param vcells
		 * @param hcells
		 */
		public DefaulServiceContext(final TwisterContext context, final AsyncService service, final int vcells, final int hcells) {
			this.context = context;
			this.service = service;
			this.vcells = vcells;
			this.hcells = hcells;
		}

		/**
		 * @see net.sf.jame.service.swing.ServiceContext#openClip(net.sf.jame.twister.TwisterClip)
		 */
		public void openClip(final TwisterClip clip) {
			if (clip != null) {
				try {
					final TwisterConfigBuilder configBuilder = new TwisterConfigBuilder();
					final TwisterConfig config = configBuilder.createDefaultConfig();
					config.setContext(new DefaultConfigContext());
					final TwisterFrame frame = new TwisterFrame(context, service, config, hcells, vcells);
					frame.loadClip(clip);
					frame.setVisible(true);
				}
				catch (final ExtensionException x) {
					x.printStackTrace();
				}
			}
		}

		/**
		 * @see net.sf.jame.service.swing.ServiceContext#removeFrame(javax.swing.JFrame)
		 */
		public void removeFrame(final JFrame frame) {
			context.removeFrame(frame);
		}

		/**
		 * @see net.sf.jame.service.swing.ServiceContext#addFrame(javax.swing.JFrame)
		 */
		public void addFrame(final JFrame frame) {
			context.addFrame(frame);
		}

		/**
		 * @see net.sf.jame.service.swing.ServiceContext#getFrameCount()
		 */
		public int getFrameCount() {
			return context.getFrameCount();
		}

		/**
		 * @see net.sf.jame.service.swing.ServiceContext#exit()
		 */
		public void exit() {
			context.exit();
		}

		/**
		 * @see net.sf.jame.service.swing.ServiceContext#restart()
		 */
		public void restart() {
			context.restart();
		}
	}
}
