/*
 * This file is part of jDiffChaser.
 *
 *  jDiffChaser is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  jDiffChaser is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with jDiffChaser; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package org.jdiffchaser.testing;

import java.io.File;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.jdiffchaser.scenarihandling.Player;
import org.jdiffchaser.scenarihandling.PlayerException;
import org.jdiffchaser.scenarihandling.PlayerServer;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.net.InetAddress;
import javax.swing.JFrame;
import org.jdiffchaser.utils.LogUtilities;

public abstract class RemoteServer {
    
    private static final Logger LOGGER = LogUtilities.getLogger();
    
    protected String[] args;
    
    public abstract Player getPlayer(String fullClassName, JFrame clientWindow, String hostName, 
                                     int jmxPort, String screenshotsDirectory) throws PlayerException;
    
    private Thread restartShutdownHook;
    
    public RemoteServer(final String[] args) {

        this.args = args;
        
        LogUtilities.configureSimpleConsoleLogs(Level.FINER.toString());

        restartShutdownHook = new Thread(){
            public void run(){
                    if(Player.shouldRestart()){
                        
                        boolean windowsRunning = false;
                        String os = System.getProperty("os.name").toLowerCase();
                        System.out.println("os.name is " + os);
                        if(os.indexOf("win")>=0){
                            windowsRunning = true;
                        }
                        
                        System.out.println(" Restarting JMX server to play forthcoming tests");
                        String toolsHome = System.getProperty("tools.home");
                        String javaLibraryPath = System.getProperty("java.library.path");
                        String classpath = System.getProperty("java.class.path");
                        String command = null;
                        if(windowsRunning){ //quotes are needed with Windows
                            String javaWinCommand = '\"' + System.getProperty("java.home") + File.separatorChar + "bin" + File.separator + "java\"";
                            command = javaWinCommand 
                                             + " -Xmx" + ((Runtime.getRuntime().maxMemory() / 1024) / 1024) + "M"
                                             + " -Dtools.home=\"" + toolsHome 
                                             + "\" -Djava.library.path=\"" + javaLibraryPath + "\" -classpath \"" + classpath
                                             + "\" " + RemoteServer.this.getClass().getName();
                        }else{//whereas macos don't need it for example
                            String javaCommand = System.getProperty("java.home") + File.separatorChar + "bin" + File.separator + "java";
                            command = javaCommand 
                                             + " -Xmx" + ((Runtime.getRuntime().maxMemory() / 1024) / 1024) + "M"
                                             + " -Dtools.home=" + toolsHome 
                                             + " -Djava.library.path="+ javaLibraryPath + " -classpath " + classpath
                                             + " " + RemoteServer.this.getClass().getName();
                        }
                        for(int i=0; i<args.length; i++){
                            command += ' ' + args[i];
                        }
                        System.out.println("command used to restart: " + command);
                        Process p = null;
                        try{
                              p = Runtime.getRuntime().exec(command);
                        }catch(Throwable e){
                            System.out.println("Unable to restart JMX Server");
                            e.printStackTrace();
                        }
                    }else{
                        System.out.println(" No restart requested ");
                    }
            }
        };
    }
    
    public void launch(){
        try{
            JFrame testFrame = new JFrame("dummy");
            Runtime.getRuntime().addShutdownHook(restartShutdownHook);
            testFrame.setVisible(false);
            String screenshotsPath = null;
            String fullClassName   = null;
            int port               = -1;
            if(args.length<3){
                System.out.println("Usage: " + this.getClass().getName() 
                                   + " <port> <fullClassName to test> <screenshotsPath> [<app arg1> <app arg2> <...>]");
                System.exit(1);
            }else{
                port = Integer.parseInt(args[0]);  
                fullClassName = args[1];  
                screenshotsPath = args[2];
            }
            
            final PlayerServer pls = new PlayerServer(getPlayer(fullClassName,
                                                                testFrame, 
                                                                InetAddress.getLocalHost().getHostAddress(), 
                                                                port, 
                                                                screenshotsPath));   
            testFrame.addWindowListener(new WindowAdapter(){
                public void windowClosed(WindowEvent we){
                    try{
                        pls.close();
                    }catch(Exception e){
                        e.printStackTrace();
                    }
                }
           });            
        }catch(Throwable t){
            t.printStackTrace();
        }        
    }
        
}
