package jp.co.sra.jun.collections.sequences;

import java.awt.Color;
import java.awt.Point;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;

/**
 * JunDoubleMatrix class
 * 
 *  @author    nisinaka
 *  @created   1998/10/09 (by nisinaka)
 *  @updated   2002/11/01 (by nisinaka)
 *  @updated   2007/05/09 (by m-asada)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDoubleMatrix.java,v 8.14 2008/02/20 06:30:52 nisinaka Exp $
 */
public class JunDoubleMatrix extends JunMatrix {

	protected static StSymbol DefalutInverseCalculationMethod = $("GaussElimination");

	/**
	 * Create a new instance of <code>JunDoubleMatrix</code> and initialize it.
	 * 
	 * @param squareSize int
	 * @category Instance creation
	 */
	public JunDoubleMatrix(int squareSize) {
		super(squareSize);
	}

	/**
	 * Create a new instance of <code>JunDoubleMatrix</code> and initialize it.
	 * 
	 * @param squareSize int
	 * @param collection java.lang.Object[]
	 * @category Instance creation
	 */
	public JunDoubleMatrix(int squareSize, double[] collection) {
		super(squareSize);
		this.collection_(collection);
	}

	/**
	 * Create a new instance of <code>JunDoubleMatrix</code> and initialize it.
	 * 
	 * @param rowSize int
	 * @param columnSize int
	 * @category Instance creation
	 */
	public JunDoubleMatrix(int rowSize, int columnSize) {
		super(rowSize, columnSize);
	}

	/**
	 * Create a new instance of <code>JunDoubleMatrix</code> and initialize it.
	 * 
	 * @param rowSize int
	 * @param columnSize int
	 * @param collection double[]
	 * @category Instance creation
	 */
	public JunDoubleMatrix(int rowSize, int columnSize, double[] collection) {
		super(rowSize, columnSize);
		this.collection_(collection);
	}

	/**
	 * Create a new instance of <code>JunDoubleMatrix</code> and initialize it.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StDisplayable
	 * @category Instance creation
	 */
	public JunDoubleMatrix(StDisplayable anImage) {
		this(anImage, new StBlockClosure() {
			public Object value_(Object color) {
				return new Double(StColorValue._GetLuminance((Color) color));
			}
		});
	}

	/**
	 * Create a new instance of <code>JunDoubleMatrix</code> and initialize it.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StDisplayable
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category Instance creation
	 */
	public JunDoubleMatrix(StDisplayable anImage, StBlockClosure aBlock) {
		this(anImage.bounds().width, anImage.bounds().height);
		StImage theImage = anImage.asImage();
		int width = theImage.width();
		int height = theImage.height();
		for (int y = 0; y < width; y++) {
			for (int x = 0; x < height; x++) {
				Color aColor = new Color(theImage.getPixel(x, y));
				Double aValue = (Double) aBlock.value_(aColor);
				this.row_column_put_(x, y, aValue);
			}
		}
	}

	/**
	 * Create a new unit matrix.
	 * 
	 * @param aClass java.lang.Class
	 * @param squareSize int
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category Instance creation
	 */
	public static JunDoubleMatrix Unit_(Class aClass, int squareSize) {
		JunDoubleMatrix unitMatrix = (JunDoubleMatrix) _New(aClass, new Integer(squareSize));

		for (int i = 0; i < squareSize; i++) {
			for (int j = 0; j < squareSize; j++) {
				unitMatrix._put(i, j, new Double((i == j) ? 1 : 0));
			}
		}

		return unitMatrix;
	}

	/**
	 * Set the default inverse calculation method to 'Gauss Elimination'.
	 * 
	 * @category Inverse calculation
	 */
	public static void CalculateInverseByGaussElimination() {
		DefaultInverseCalculationMethod_($("GaussElimination"));
	}

	/**
	 * Set the default inverse calculation method to 'LU Decomposition'.
	 * 
	 * @category Inverse calculation
	 */
	public static void CalculateInverseByLuDecomposition() {
		DefaultInverseCalculationMethod_($("LuDecomposition"));
	}

	/**
	 * Answer the current default method for calculating inverse.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category Defaults
	 */
	public static StSymbol DefaultInverseCalculationMethod() {
		return DefalutInverseCalculationMethod;
	}

	/**
	 * Set the new default method for calculating inverse.
	 * 
	 * @param newDefalutInverseCalculationMethod jp.co.sra.smalltalk.StSymbol
	 * @category Defaults
	 */
	protected static void DefaultInverseCalculationMethod_(StSymbol newDefalutInverseCalculationMethod) {
		DefalutInverseCalculationMethod = newDefalutInverseCalculationMethod;
	}

	/**
	 * Answer the value at the specified place as double.
	 * 
	 * @param rowNumber int
	 * @param columnNumber int
	 * @return double
	 * @category accessing
	 */
	public double _doubleValueAt(int rowNumber, int columnNumber) {
		return ((Double) this._at(rowNumber, columnNumber)).doubleValue();
	}

	/**
	 * Put a double value at the specified place.
	 * 
	 * @param rowNumber int
	 * @param columnNumber int
	 * @param value double
	 * @category accessing
	 */
	public void _doubleValuePut(int rowNumber, int columnNumber, double value) {
		super._put(rowNumber, columnNumber, new Double(value));
	}

	/**
	 * Answer the array of objects in the specified row.
	 * 
	 * @param index int
	 * @return double[]
	 * @category accessing
	 */
	public double[] _doubleValuesBasicAt_(int index) {
		Object[] values = this.atRow_(index);
		double[] doubleValues = new double[values.length];
		for (int i = 0; i < values.length; i++) {
			doubleValues[i] = ((Double) values[i]).doubleValue();
		}
		return doubleValues;
	}

	/**
	 * Put 'anObject' at the all position.
	 * 
	 * @param anObject java.lang.Object
	 * @see jp.co.sra.jun.collections.sequences.JunAbstractMatrix#atAllPut_(java.lang.Object)
	 * @category accessing
	 */
	public void atAllPut_(double anObject) {
		super.atAllPut_(new Double(anObject));
	}

	/**
	 * Put 'anObject' at the position specified with 'aPoint'.
	 * 
	 * @param aPoint java.awt.Point
	 * @param anObject double
	 * @category accessing
	 */
	public void atPoint_put_(Point aPoint, double anObject) {
		this._doubleValuePut(aPoint.x, aPoint.y, anObject);
	}

	/**
	 * Set the array of objects in the specified row.
	 * 
	 * @param anInteger int
	 * @param aCollection double[]
	 * @category accessing
	 */
	public void basicAt_put_(int index, double[] aCollection) {
		Double[] doubleValues = new Double[aCollection.length];
		for (int i = 0; i < aCollection.length; i++) {
			doubleValues[i] = new Double(aCollection[i]);
		}
		this.atRow_put_(index, doubleValues);
	}

	/**
	 * Set all elements of the matrix from 'collection'.
	 * 
	 * @param collection double[]
	 * @category accessing
	 */
	public void collection_(double[] collection) {
		int index = 0;
		for (int i = 0; i < rowSize; i++) {
			for (int j = 0; j < columnSize; j++) {
				_matrix[i][j] = new Double(collection[index]);
				index++;
			}
		}
	}

	/**
	 * Put 'anObject' at the position specified with 'rowNumber' and 'columnNumber'.
	 * 
	 * @param rowNumber int
	 * @param columnNumber int
	 * @param anObject double
	 * @category accessing
	 */
	public void row_column_put_(int rowNumber, int columnNumber, double anObject) {
		this._doubleValuePut(rowNumber, columnNumber, anObject);
	}

	/**
	 * Convert this matrix as an array of double.
	 * 
	 * @return double[]
	 * @category converting
	 */
	public double[] asArrayOfDouble() {
		double[] anArray = new double[rowSize * columnSize];
		int index = 0;
		for (int i = 0; i < rowSize; i++) {
			for (int j = 0; j < columnSize; j++) {
				anArray[index] = this._doubleValueAt(i, j);
				index++;
			}
		}
		return anArray;
	}

	/**
	 * Convert the receiver to an image as <code>StImage</code>.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category converting
	 */
	public StImage asImage() {
		return (StImage) this.asImageColorBlock_(new StBlockClosure() {
			public Object value_(Object value) {
				return StColorValue.Brightness_(Math.max(0, Math.min(((Number) value).doubleValue(), 1)));
			}
		});
	}

	/**
	 * Convert the receiver to an image as <code>StImage</code> with the specified color block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StImage
	 * @category converting
	 */
	public StImage asImageColorBlock_(final StBlockClosure aBlock) {
		final StImage anImage = new StImage(this.rowSize(), this.columnSize());
		this.doIJ_(new StBlockClosure() {
			public Object value_value_value_(Object value, Object row, Object column) {
				Color colorValue = (Color) aBlock.value_(value);
				anImage.setPixel(((Number) row).intValue(), ((Number) column).intValue(), colorValue.getRGB());
				return null;
			}
		});
		return anImage;
	}

	/**
	 * Answer the sum of this matrix and aMatrix. This method corresponds to
	 * the '+' method in Smalltalk.
	 * 
	 * @param aMatrix jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category matrix functions
	 */
	public JunDoubleMatrix _plus(JunDoubleMatrix aMatrix) {
		JunDoubleMatrix newMatrix = (JunDoubleMatrix) this.copy();
		for (int i = 0; i < rowSize; i++) {
			for (int j = 0; j < columnSize; j++) {
				newMatrix._doubleValuePut(i, j, (newMatrix._doubleValueAt(i, j) + aMatrix._doubleValueAt(i, j)));
			}
		}
		return newMatrix;
	}

	/**
	 * Calculate a product element at i & j with aMatrix.
	 * 
	 * @param i int
	 * @param j int
	 * @param aMatrix jp.co.sra.jun.collections.sequences.JunMatrix
	 * @return java.lang.Object
	 * @see jp.co.sra.jun.collections.sequences.JunMatrix#_calculateProductElementAt(int, int, jp.co.sra.jun.collections.sequences.JunMatrix)
	 * @category matrix functions
	 */
	protected Object _calculateProductElementAt(int i, int j, JunMatrix aMatrix) {
		final int j_ = j;
		final JunDoubleMatrix aMatrix_ = (JunDoubleMatrix) aMatrix;
		final StValueHolder answerHolder_ = new StValueHolder(0);
		StBlockClosure aBlock = new StBlockClosure() {
			public Object value_value_(Object value, Object k) {
				double answer = answerHolder_._doubleValue();
				double myValue = ((Double) value).doubleValue();
				double otherValue = aMatrix_._doubleValueAt(((Integer) k).intValue(), j_);
				answer = answer + (myValue * otherValue);
				answerHolder_.value_(answer);

				return null;
			}
		};
		this.rowIndex_doJ_(i, aBlock);
		return answerHolder_.value();
	}

	/**
	 * Answer the inverse of this matrix.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category matrix functions
	 */
	public JunDoubleMatrix inverse() {
		if (this.inverseCalculationMethod() == $("GaussElimination")) {
			return this.inverseByGaussElimination();
		} else if (this.inverseCalculationMethod() == $("LuDecomposition")) {
			return this.inverseByLuDecomposition();
		}
		return this.inverseByGaussElimination();
	}

	/**
	 * Calculate the inverse of this matrix by Gauss Elimination method.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category matrix functions
	 */
	public JunDoubleMatrix inverseByGaussElimination() {
		if (!this.isSquare()) {
			throw SmalltalkException.Error("the matrix is not square.");
		}

		JunDoubleMatrix copy = (JunDoubleMatrix) this.copy();
		JunDoubleMatrix inverse = Unit_(JunDoubleMatrix.class, rowSize);

		for (int i = 0; i < columnSize; i++) {
			double value = copy._doubleValueAt(i, i);

			if (value == 0) {
				int x = i;
				while (value == 0 && x < rowSize) {
					x = x + 1;
					value = copy._doubleValueAt(x, i);
				}

				if (value != 0) {
					value = 1 / value;
				}

				final int i_ = i;
				final int x_ = x;
				final double value_ = value;
				final JunDoubleMatrix copy_ = copy;
				copy.rowIndex_doJ_(i, new StBlockClosure() {
					public Object value_value_(Object anElement, Object anIndex) {
						double each = ((Double) anElement).doubleValue();
						int j = ((Integer) anIndex).intValue();
						double v = each + (copy_._doubleValueAt(x_, j) * value_);
						copy_._doubleValuePut(i_, j, v);

						return null;
					}
				});

				final JunDoubleMatrix inverse_ = inverse;
				inverse.rowIndex_doJ_(i, new StBlockClosure() {
					public Object value_value_(Object anElement, Object anIndex) {
						double each = ((Double) anElement).doubleValue();
						int j = ((Integer) anIndex).intValue();
						double v = each + (inverse_._doubleValueAt(x_, j) * value_);
						inverse_._doubleValuePut(i_, j, v);

						return null;
					}
				});
			} else {
				final int i_ = i;
				final double value_ = value;
				final JunDoubleMatrix copy_ = copy;
				copy.rowIndex_doJ_(i, new StBlockClosure() {
					public Object value_value_(Object anElement, Object anIndex) {
						double v = ((Double) anElement).doubleValue() / value_;
						copy_._doubleValuePut(i_, ((Integer) anIndex).intValue(), v);

						return null;
					}
				});

				final JunDoubleMatrix inverse_ = inverse;
				inverse.rowIndex_doJ_(i, new StBlockClosure() {
					public Object value_value_(Object anElement, Object anIndex) {
						double v = ((Double) anElement).doubleValue() / value_;
						inverse_._doubleValuePut(i_, ((Integer) anIndex).intValue(), v);

						return null;
					}
				});
			}

			for (int k = 0; k < rowSize; k++) {
				if (k != i) {
					final double value_ = copy._doubleValueAt(k, i);
					final int i_ = i;
					final int k_ = k;
					final JunDoubleMatrix copy_ = copy;
					copy.rowIndex_doJ_(k, new StBlockClosure() {
						public Object value_value_(Object anElement, Object anIndex) {
							double each = ((Double) anElement).doubleValue();
							int j = ((Integer) anIndex).intValue();
							double v = each - (copy_._doubleValueAt(i_, j) * value_);
							copy_._doubleValuePut(k_, j, v);

							return null;
						}
					});

					final JunDoubleMatrix inverse_ = inverse;
					inverse.rowIndex_doJ_(k, new StBlockClosure() {
						public Object value_value_(Object anElement, Object anIndex) {
							double each = ((Double) anElement).doubleValue();
							int j = ((Integer) anIndex).intValue();
							double v = each - (inverse_._doubleValueAt(i_, j) * value_);
							inverse_._doubleValuePut(k_, j, v);

							return null;
						}
					});
				}
			}
		}

		return inverse;
	}

	/**
	 * Calculate the inverse of this matrix by LU Decomposition method.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category matrix functions
	 */
	public JunDoubleMatrix inverseByLuDecomposition() {
		if (this.isSquare() == false) {
			throw SmalltalkException.Error("the matrix is not square.");
		}

		Object[] lu = this.luDecompose();
		if (lu == null) {
			throw SmalltalkException.Error("the matrix is singular.");
		}

		JunDoubleMatrix luMatrix = (JunDoubleMatrix) lu[0];
		int[] pivotSequence = (int[]) lu[1];
		int size = this.rowSize();
		JunDoubleMatrix inverse = new JunDoubleMatrix(size);
		for (int columnIndex = 0; columnIndex < size; columnIndex++) {
			for (int i = 0; i < size; i++) {
				int index = pivotSequence[i];
				double value = (index == columnIndex) ? 1 : 0;
				for (int j = 0; j < i; j++) {
					value -= (luMatrix._doubleValueAt(index, j) * inverse._doubleValueAt(j, columnIndex));
				}
				inverse._doubleValuePut(i, columnIndex, value);
			}
			for (int i = size - 1; i >= 0; i--) {
				int index = pivotSequence[i];
				double value = inverse._doubleValueAt(i, columnIndex);
				for (int j = i + 1; j < size; j++) {
					value -= (luMatrix._doubleValueAt(index, j) * inverse._doubleValueAt(j, columnIndex));
				}
				inverse._doubleValuePut(i, columnIndex, value / luMatrix._doubleValueAt(index, i));
			}
		}

		return inverse;
	}

	/**
	 * Answer the current method for calculating inverse.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category private
	 */
	protected StSymbol inverseCalculationMethod() {
		return DefaultInverseCalculationMethod();
	}

	/**
	 * Calculate the receiver's LU matrix.
	 * 
	 * @return java.lang.Object[]
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category matrix functions
	 */
	protected Object[] luDecompose() {
		if (this.isSquare() == false) {
			throw SmalltalkException.Error("the matrix is not square.");
		}

		double value;
		double maxValue;
		int size = this.rowSize();
		double[] weights = new double[size];
		int[] pivotSequence = new int[size];
		JunDoubleMatrix luMatrix = (JunDoubleMatrix) this.copy();
		double determinant = 1;
		for (int i = 0; i < size; i++) {
			maxValue = 0;
			for (int j = 0; j < size; j++) {
				value = Math.abs(luMatrix._doubleValueAt(i, j));
				if (maxValue < value) {
					maxValue = value;
				}
			}
			if (maxValue == 0) {
				return null;
			}
			weights[i] = 1 / maxValue;
			pivotSequence[i] = i;
		}

		int maxIndex = -1;
		for (int columnIndex = 0; columnIndex < size; columnIndex++) {
			maxValue = -1;
			for (int i = columnIndex; i < size; i++) {
				int index = pivotSequence[i];
				value = Math.abs(luMatrix._doubleValueAt(index, columnIndex)) * weights[index];
				if (value > maxValue) {
					maxValue = value;
					maxIndex = i;
				}
			}
			int pivotIndex = pivotSequence[maxIndex];
			if (maxIndex != columnIndex) {
				pivotSequence[maxIndex] = pivotSequence[columnIndex];
				pivotSequence[columnIndex] = pivotIndex;
				determinant = -determinant;
			}
			double pivot = luMatrix._doubleValueAt(pivotIndex, columnIndex);
			if (pivot == 0) {
				return null;
			}
			determinant *= pivot;
			for (int i = columnIndex + 1; i < size; i++) {
				value = luMatrix._doubleValueAt(pivotSequence[i], columnIndex);
				value /= pivot;
				luMatrix._doubleValuePut(pivotSequence[i], columnIndex, value);
				for (int j = columnIndex + 1; j < size; j++) {
					luMatrix._doubleValuePut(pivotSequence[i], j, luMatrix._doubleValueAt(pivotSequence[i], j) - (value * luMatrix._doubleValueAt(pivotIndex, j)));
				}
			}
		}

		return new Object[] { luMatrix, pivotSequence, new Double(determinant) };
	}

}
