package jp.co.sra.jun.delaunay.twoD;

import java.awt.*;
import java.io.*;
import java.util.*;
import jp.co.sra.jun.geometry.basic.*;
import jp.co.sra.smalltalk.*;

/**
 * Jun2dDelaunayLoop class
 * 
 *  @author    Ryouichi Matsuda
 *  @created   2002/01/14 (by Ryouichi Matsuda)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dDelaunayLoop.java,v 8.11 2008/02/20 06:30:54 nisinaka Exp $
 */
public class Jun2dDelaunayLoop extends Jun2dDelaunayElement {
	protected Jun2dDelaunayHalfEdge halfEdge;

	/**
	 * DOCUMENT ME!
	 * 
	 * @return double
	 */
	public double area() {
		Jun2dDelaunayVertex vertex1;
		Jun2dDelaunayVertex vertex2;
		Jun2dDelaunayHalfEdge hEdge;
		double area;

		vertex1 = halfEdge.vertex();
		vertex2 = halfEdge.next().vertex();
		hEdge = halfEdge.next().next();
		area = 0.0d;

		do {
			area = area + (hEdge.vertex().areaWith_with_(vertex1, vertex2));
			hEdge = hEdge.next();
		} while (hEdge != halfEdge);

		return area;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return java.util.Vector
	 */
	public Vector asArrayOfJun2dPoint() {
		final Vector points = new Vector();

		this.verticesDo_(new StBlockClosure() {
			public Object value_(Object vertex_) {
				Jun2dDelaunayVertex vertex = (Jun2dDelaunayVertex) vertex_;

				points.addElement(vertex.asJun2dPoint());

				return null;
			}
		});

		return points;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return java.util.Vector
	 */
	public Vector asArrayOfJun3dPoint() {
		final Vector points = new Vector();

		this.verticesDo_(new StBlockClosure() {
			public Object value_(Object vertex_) {
				Jun2dDelaunayVertex vertex = (Jun2dDelaunayVertex) vertex_;

				points.addElement(vertex.asJun3dPoint());

				return null;
			}
		});

		return points;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 */
	public Jun2dDelaunayVertex center() {
		final StValueHolder x = new StValueHolder(); // double
		final StValueHolder y = new StValueHolder(); // double
		final StValueHolder num = new StValueHolder(); // int

		x.value_(0.0d);
		y.value_(0.0d);
		num.value_(0);
		this.verticesDo_(new StBlockClosure() {
			public Object value_(final Object vertex_) {
				Jun2dDelaunayVertex vertex = (Jun2dDelaunayVertex) vertex_;

				x.value_(x._doubleValue() + vertex.x());
				y.value_(y._doubleValue() + vertex.y());
				num.value_(num._intValue() + 1);

				return null;
			}
		});

		return new Jun2dDelaunayVertex(x._doubleValue() / num._doubleValue(), y._doubleValue() / num._doubleValue());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return boolean
	 */
	public boolean contains_(Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		return this.containsX_y_(aJun2dDelaunayVertex.x(), aJun2dDelaunayVertex.y());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * 
	 * @return boolean
	 */
	public boolean containsX_y_(final double xNumber, final double yNumber) {
		if (this.area() > DECIMAL_12) {
			final StValueHolder minX = new StValueHolder(); // double
			final StValueHolder maxX = new StValueHolder(); // double
			final StValueHolder minY = new StValueHolder(); // double
			final StValueHolder maxY = new StValueHolder(); // double

			minX.value_(halfEdge.vertex().x());
			maxX.value_(halfEdge.vertex().x());
			minY.value_(halfEdge.vertex().y());
			maxY.value_(halfEdge.vertex().y());
			this.verticesDo_(new StBlockClosure() {
				public Object value_(final Object vertex_) {
					Jun2dDelaunayVertex vertex = (Jun2dDelaunayVertex) vertex_;
					double vx;
					double vy;

					vx = vertex.x();
					vy = vertex.y();

					if (vx < minX._doubleValue()) {
						minX.value_(vx);
					}

					if (maxX._doubleValue() < vx) {
						maxX.value_(vx);
					}

					if (vy < minY._doubleValue()) {
						minY.value_(vy);
					}

					if (maxY._doubleValue() < vy) {
						maxY.value_(vy);
					}

					return null;
				}
			});

			if (((minX._doubleValue() - DECIMAL_12) <= xNumber) && (xNumber <= (maxX._doubleValue() + DECIMAL_12)) && ((minY._doubleValue() - DECIMAL_12) <= yNumber) && (yNumber <= (maxY._doubleValue() + DECIMAL_12))) {
				Object returnObject = this.halfEdgesDo_(new StBlockClosure() {
					public Object value_(final Object hEdge_) {
						Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;
						Jun2dDelaunayVertex v1;
						Jun2dDelaunayVertex v2;

						v1 = hEdge.pair().vertex();
						v2 = hEdge.vertex();

						if ((((v1.x() - xNumber) * (v2.y() - yNumber)) - ((v1.y() - yNumber) * (v2.x() - xNumber))) < 0) {
							return Boolean.FALSE;
						}

						return null;
					}
				});

				return (returnObject == Boolean.FALSE) ? false : true;
			}
		}

		return false;
	}

	/**
	 * Display the receiver on aGraphics.
	 * 
	 * @param aGraphicsContext java.awt.Graphics
	 */
	public void displayOn_(final Graphics aGraphicsContext) {
		this.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				hEdge.displayOn_(aGraphicsContext);

				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object edgesDo_(final StBlockClosure aBlock) {
		return halfEdge.do_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				Object returnObject = aBlock.value_(hEdge.edge());

				if (returnObject != null) {
					return returnObject; // continuation exit
				}

				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	public Jun2dDelaunayHalfEdge farestHalfEdgeFromVertex_(Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		Jun2dDelaunayHalfEdge farest;
		double maxSquaredDistance;
		Jun2dDelaunayHalfEdge hEdge;

		farest = halfEdge;
		maxSquaredDistance = halfEdge.vertex().squaredDistance_(aJun2dDelaunayVertex);
		hEdge = halfEdge.next();

		do {
			double squaredDistance;

			squaredDistance = hEdge.vertex().squaredDistance_(aJun2dDelaunayVertex);

			if (maxSquaredDistance < squaredDistance) {
				maxSquaredDistance = squaredDistance;
				farest = hEdge;
			}

			hEdge = hEdge.next();
		} while (hEdge != halfEdge);

		return farest;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	public Jun2dDelaunayHalfEdge halfEdge() {
		return halfEdge;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayHalfEdge
	 *        jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	public void halfEdge_(Jun2dDelaunayHalfEdge aJun2dDelaunayHalfEdge) {
		halfEdge = aJun2dDelaunayHalfEdge;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object halfEdgesDo_(final StBlockClosure aBlock) {
		return halfEdge.do_(aBlock);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return boolean
	 */
	public boolean hasBoundaryVertex() {
		Object returnObject = this.verticesDo_(new StBlockClosure() {
			public Object value_(Object vertex_) {
				Jun2dDelaunayVertex vertex = (Jun2dDelaunayVertex) vertex_;

				if (vertex.isBoundaryVertex()) {
					return Boolean.TRUE;
				}

				return null;
			}
		});

		return (returnObject == Boolean.TRUE) ? true : false;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex1 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param aJun2dDelaunayVertex2 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return boolean
	 */
	public boolean intersectsWithLineSegmentFrom_to_(final Jun2dDelaunayVertex aJun2dDelaunayVertex1, final Jun2dDelaunayVertex aJun2dDelaunayVertex2) {
		Object returnObject = halfEdge.do_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				if (hEdge.intersectsWithLineSegmentFrom_to_(aJun2dDelaunayVertex1, aJun2dDelaunayVertex2)) {
					return Boolean.TRUE;
				}

				return null;
			}
		});

		return (returnObject == Boolean.TRUE) ? true : false;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return boolean
	 */
	public boolean isTriangle() {
		return (halfEdge.next().next().next() == halfEdge);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	public Jun2dDelaunayHalfEdge nearestHalfEdgeFromVertex_(Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		Jun2dDelaunayHalfEdge nearest;
		double minSquaredDistance;
		Jun2dDelaunayHalfEdge hEdge;

		nearest = halfEdge;
		minSquaredDistance = halfEdge.vertex().squaredDistance_(aJun2dDelaunayVertex);
		hEdge = halfEdge.next();

		do {
			double squaredDistance;

			squaredDistance = hEdge.vertex().squaredDistance_(aJun2dDelaunayVertex);

			if (squaredDistance < minSquaredDistance) {
				minSquaredDistance = squaredDistance;
				nearest = hEdge;
			}

			hEdge = hEdge.next();
		} while (hEdge != halfEdge);

		return nearest;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object neighborsDo_(final StBlockClosure aBlock) {
		return halfEdge.do_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;
				Jun2dDelaunayLoop l;

				l = hEdge.pair().loop();

				if (l != null) {
					Object returnObject = aBlock.value_(l);

					if (returnObject != null) {
						return returnObject; // continuation exit
					}
				}

				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 */
	public JunPoint point1() {
		return halfEdge.vertex().point();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 */
	public JunPoint point2() {
		return halfEdge.next().vertex().point();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 */
	public JunPoint point3() {
		return halfEdge.next().next().vertex().point();
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws IOException DOCUMENT ME!
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(final Writer aWriter) throws IOException {
		aWriter.write("Loop(");
		this.verticesDo_(new StBlockClosure() {
			public Object value_(Object vertex_) {
				Jun2dDelaunayVertex vertex = (Jun2dDelaunayVertex) vertex_;

				try {
					aWriter.write(Double.toString(vertex.x()));
					aWriter.write("@");
					aWriter.write(Double.toString(vertex.y()));
					aWriter.write(" ");
				} catch (IOException e) {
					//
				}

				return null;
			}
		});
		aWriter.write(")");
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object verticesDo_(final StBlockClosure aBlock) {
		return halfEdge.do_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				Object returnObject = aBlock.value_(hEdge.vertex());

				if (returnObject != null) {
					return returnObject; // continuation exit
				}

				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	protected Jun2dDelaunayHalfEdge nearestHalfEdgeFromX_y_(double xNumber, double yNumber) {
		Jun2dDelaunayHalfEdge nearest;
		double minSquaredDistance;
		Jun2dDelaunayHalfEdge hEdge;

		nearest = halfEdge;
		minSquaredDistance = halfEdge.edge().squaredDistanceFromX_y_(xNumber, yNumber);
		hEdge = halfEdge.next();

		do {
			double squaredDistance;

			squaredDistance = hEdge.edge().squaredDistanceFromX_y_(xNumber, yNumber);

			if (squaredDistance < minSquaredDistance) {
				minSquaredDistance = squaredDistance;
				nearest = hEdge;
			}

			hEdge = hEdge.next();
		} while (hEdge != halfEdge);

		return nearest;
	}
}
