package jp.co.sra.jun.dxf.support;

import java.awt.Color;
import java.io.File;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.TreeSet;

import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StCheckBoxMenuItem;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayView;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjector;

/**
 * JunCADModel class
 * 
 *  @author    nisinaka
 *  @created   2002/07/11 (by nisinaka)
 *  @updated   2003/05/15 (by nisinaka)
 *  @updated   2005/03/02 (by nisinaka)
 *  @updated   2007/08/27 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun651 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunCADModel.java,v 8.15 2008/02/20 06:30:54 nisinaka Exp $
 */
public class JunCADModel extends JunOpenGLDisplayModel {

	protected HashMap layerNames;
	protected ArrayList hiddenObjects;
	protected StMenu _layersMenu;

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		layerNames = null;
		hiddenObjects = null;
		_layersMenu = null;
	}

	/**
	 * Answer the current layer names.
	 * 
	 * @return java.util.HashMap
	 * @category accessing
	 */
	public HashMap layerNames() {
		return layerNames;
	}

	/**
	 * Set the new layer names.
	 * 
	 * @param newLayerNames java.util.HashMap
	 * @category accessing
	 */
	public void layerNames_(HashMap newLayerNames) {
		layerNames = newLayerNames;
	}

	/**
	 * Answer the names of the layers as an array of String.
	 * 
	 * @return java.lang.String[]
	 * @category accessing
	 */
	public String[] layerNameStrings() {
		TreeSet aTreeSet = new TreeSet(new Comparator() {
			public int compare(Object arg0, Object arg1) {
				return ((String) arg0).compareTo((String) arg1);
			}
		});
		aTreeSet.addAll(this.layerNames().keySet());
		return (String[]) aTreeSet.toArray(new String[aTreeSet.size()]);
	}

	/**
	 * Cache the layers which have been hidden, in case they get turned on again.
	 * 
	 * @return java.util.ArrayList
	 * @category accessing
	 */
	protected ArrayList hiddenObjects() {
		if (hiddenObjects == null) {
			hiddenObjects = new ArrayList();
		}
		return hiddenObjects;
	}

	/**
	 * Answer my display projector.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#displayProjector()
	 * @category accessing
	 */
	public JunOpenGLProjector displayProjector() {
		if (openGLProjector == null) {
			openGLProjector = new JunOpenGLProjector();
			openGLProjector.projection_(openGLProjector.projection().asParallelProjection());
			openGLProjector.eyePoint_(this.defaultEyePoint());
			openGLProjector.sightPoint_(this.defaultSightPoint());
			openGLProjector.upVector_(this.defaultUpVector());
			openGLProjector.viewFactor_(this.defaultViewFactor());
		}

		return openGLProjector;
	}

	/**
	 * Answer the default eye point. Assume 2d data, looking down Z axis.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultEyePoint()
	 * @category defaults
	 */
	public Jun3dPoint defaultEyePoint() {
		if (this.openGL3dObject() == null) {
			return new Jun3dPoint(0, 0, 10000);
		}

		Jun3dPoint defaultEyePoint = (Jun3dPoint) this.defaultProjectionTable().get($("eyePoint"));
		if (defaultEyePoint == null) {
			Jun3dBoundingBox box = this.boundingBox();
			double distance = box.depth();
			distance = Math.max(distance * 2, 10000);
			defaultEyePoint = this.defaultSightPoint().plus_(new Jun3dPoint(0, 0, distance));
		}

		return defaultEyePoint;
	}

	/**
	 * Answer the default up vector. Assume 2d data, looking down Z axis.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultUpVector()
	 * @category defaults
	 */
	public Jun3dPoint defaultUpVector() {
		Jun3dPoint defaultUpVector = (Jun3dPoint) this.defaultProjectionTable().get($("upVector"));
		if (defaultUpVector == null) {
			defaultUpVector = new Jun3dPoint(0, 1, 0);
		}
		return defaultUpVector;
	}

	/**
	 * Set the layer names. Keep track of which layers are displayed. boolean
	 * values are not currently used...
	 * 
	 * @param names java.util.Enumeration
	 * @category layers
	 */
	protected void setLayerNames_(Enumeration names) {
		HashMap table = new HashMap();
		while (names.hasMoreElements()) {
			table.put(names.nextElement(), Boolean.TRUE);
		}
		this.layerNames_(table);
	}

	/**
	 * Set the layer objects.
	 * 
	 * @param objects java.util.Enumeration
	 * @category layers
	 */
	protected void setLayerObjects_(Enumeration objects) {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		while (objects.hasMoreElements()) {
			compoundObject.add_((JunOpenGL3dObject) objects.nextElement());
		}
		this.displayObject_(compoundObject);
	}

	/**
	 * Find object in the layers list.
	 * 
	 * @param aString java.lang.String
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category layers
	 */
	protected JunOpenGL3dObject layerAt_(String aString) {
		JunOpenGL3dObject[] obs = ((JunOpenGL3dCompoundObject) this.displayObject()).components();
		for (int i = 0; i < obs.length; i++) {
			if (obs[i].name().equals(aString)) {
				return obs[i];
			}
		}
		return null;
	}

	/**
	 * Find object in the cache.
	 * 
	 * @param aString java.lang.String
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category layers
	 */
	protected JunOpenGL3dObject hiddenLayerAt_(String aString) {
		ArrayList obs = this.hiddenObjects();
		for (int i = 0; i < obs.size(); i++) {
			JunOpenGL3dObject x = (JunOpenGL3dObject) obs.get(i);
			if (x.name().equals(aString)) {
				return x;
			}
		}
		return null;
	}

	/**
	 * Make the named layer visible again.
	 * 
	 * @param aString java.lang.String
	 * @category layers
	 */
	protected void toggleLayerOn_(String aString) {
		JunOpenGL3dObject layer = this.hiddenLayerAt_(aString);
		if (layer != null) {
			this.hiddenObjects().remove(layer);
			((JunOpenGL3dCompoundObject) this.displayObject()).add_(layer);
			this.toggleLayerMenu_(aString, true);
		}
	}

	/**
	 * Hide the named layer by removing it from the display object and caching it.
	 * 
	 * @param aString java.lang.String
	 * @category layers
	 */
	protected void toggleLayerOff_(String aString) {
		JunOpenGL3dObject layer = this.layerAt_(aString);
		if (layer != null) {
			this.hiddenObjects().add(layer);
			((JunOpenGL3dCompoundObject) this.displayObject()).remove_(layer);
			this.toggleLayerMenu_(aString, false);
		}
	}

	/**
	 * Change the check mark in the Layers menu.
	 * 
	 * @param aString java.lang.String
	 * @param beOn boolean
	 * @category layers
	 */
	protected void toggleLayerMenu_(String aString, boolean beOn) {
		StMenu layersMenu = (StMenu) this._menuBar().atNameKey_($("layersMenu"));
		if (layersMenu == null) {
			return;
		}

		StCheckBoxMenuItem mItem = (StCheckBoxMenuItem) layersMenu.atNameKey_($(aString));
		if (mItem != null) {
			mItem.beSelected(beOn);
		}
	}

	/**
	 * Show all layers.
	 * 
	 * @category layers
	 */
	protected void toggleAllLayersOn() {
		String[] names = this.layerNameStrings();
		for (int i = 0; i < names.length; i++) {
			this.toggleLayerOn_(names[i]);
		}
	}

	/**
	 * Hide all layers.
	 * 
	 * @category layers
	 */
	protected void toggleAllLayersOff() {
		String[] names = this.layerNameStrings();
		for (int i = 0; i < names.length; i++) {
			this.toggleLayerOff_(names[i]);
		}
	}

	/**
	 * Read the DXF file and create layers.
	 * 
	 * @param aFile java.io.File
	 * @return java.util.Hashtable
	 * @category reading
	 */
	protected Hashtable readFromDXF_(File aFile) {
		if (aFile == null) {
			return null;
		}

		Hashtable layers = null;
		JunDXFParser parser = JunDXFParser.OnSmart_(aFile);
		if (parser.parsedOk()) {
			layers = parser.layers();
		}
		return layers;
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		JunOpenGLDisplayView defaultView = (JunOpenGLDisplayView) super.defaultView();
		defaultView.getOpenGLDrawable().toComponent().setBackground(Color.black);
		return defaultView;
	}

	/**
	 * Answer the window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("DXF Viewer");
	}

	/**
	 * Set the 'Layers' menu to be check-marked lines with layer names, so the
	 * user can turn individual layers on and off.
	 * 
	 * @category interface opening
	 */
	protected void addLayerMenuItems() {
		StMenu layersSubMenu = new StMenu("Layers");
		String[] layerNameStrings = this.layerNameStrings();
		for (int i = 0; i < layerNameStrings.length; i++) {
			String ss = layerNameStrings[i];
			StCheckBoxMenuItem theMenuItem = new StCheckBoxMenuItem(ss, true, new MenuPerformer(this, "toggleLayer_", ss));
			layersSubMenu.add(theMenuItem);
		}

		((StMenu) this._menuBar().atNameKey_($("layersMenu"))).add(layersSubMenu);
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createFileMenu());
			_menuBar.add(this._createEditMenu());
			_menuBar.add(this._createViewMenu());
			_menuBar.add(this._createLightMenu());
			_menuBar.add(this._createLayersMenu());
			_menuBar.add(this._createMiscMenu());
		}
		return _menuBar;
	}

	/**
	 * Create a "Layers" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createLayersMenu() {
		StMenu layersMenu = new StMenu($String("Layer", "Layers"), $("layersMenu"));
		layersMenu.add(new StMenuItem($String("Show all"), new MenuPerformer(this, "allLayersShow")));
		layersMenu.add(new StMenuItem($String("Hide all"), new MenuPerformer(this, "allLayersHide")));
		return layersMenu;
	}

	/**
	 * Menu message: Open an LST file.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#openLST()
	 * @category menu messages
	 */
	public void openLST() {
		this.openDXF();
	}

	/**
	 * Menu message: open a DXF file.
	 * 
	 * @category menu messages
	 */
	public void openDXF() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "DXF"), new String[] { "*.dxf", "*.DXF" }) };
		File aFile = JunFileRequesterDialog.RequestFile($String("Select a <1p> file.", null, "DXF"), fileTypes, fileTypes[0]);
		if (aFile == null) {
			return;
		}
		Hashtable layers = this.readFromDXF_(aFile);
		if (layers == null) {
			return;
		}

		this.setLayerNames_(layers.keys());
		this.setLayerObjects_(layers.elements());
		this.addLayerMenuItems();
		this.resetView();
		if (this.showModel() != null) {
			this.showModel().resetView();
		}
	}

	/**
	 * Show the layers of the parser.
	 * 
	 * @param aParser jp.co.sra.jun.dxf.JunDXFParser
	 * @category menu messages
	 */
	public void openParser_(JunDXFParser aParser) {
		Hashtable layers = aParser.layers();
		if (layers == null) {
			return;
		}

		this.setLayerNames_(layers.keys());
		this.setLayerObjects_(layers.elements());
		this.addLayerMenuItems();
		this.resetView();
		if (this.showModel() != null) {
			this.showModel().resetView();
		}
	}

	/**
	 * Menu message: show all layers.
	 * 
	 * @category menu messages
	 */
	public void allLayersShow() {
		this.toggleAllLayersOn();
		this.changed_($("object"));
	}

	/**
	 * Menu message: hide all layers.
	 * 
	 * @category menu messages
	 */
	public void allLayersHide() {
		this.toggleAllLayersOff();
		this.changed_($("object"));
	}

	/**
	 * Toggle the layer on and off.
	 * 
	 * @param aString java.lang.String
	 * @category menu messages
	 */
	protected void toggleLayer_(String aString) {
		JunOpenGL3dObject layer = this.layerAt_(aString);
		if (layer == null) {
			this.toggleLayerOn_(aString);
		} else {
			this.toggleLayerOff_(aString);
		}
		this.changed_($("object"));
	}

}
