package jp.co.sra.jun.geometry.basic;

import java.awt.Point;
import java.io.IOException;
import java.io.Writer;

import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.curves.Jun2dPolyline;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;
import jp.co.sra.jun.geometry.transformations.Jun2dTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dVertex;

/**
 * Jun2dPoint class
 * 
 *  @author    nisinaka
 *  @created   1998/09/29 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2004/10/19 (by Mitsuhiro Asada)
 *  @updated   2004/12/06 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun682 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dPoint.java,v 8.26 2008/02/20 06:30:55 nisinaka Exp $
 */
public class Jun2dPoint extends JunPoint {
	/** The <i>x</i> coordinate. */
	protected double x;

	/** The <i>y</i> coordinate. */
	protected double y;

	/**
	 * Answer the new Jun3dPoint which is the same kind of point as the
	 * receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Coercing
	 */
	public static Jun2dPoint Coerce_(JunPoint aPoint) {
		return new Jun2dPoint(aPoint.x(), aPoint.y());
	}

	/**
	 * Answer the new Jun2dPoint which is the same kind of point as the
	 * receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aNumber double
	 * @category Coercing
	 */
	public static Jun2dPoint Coerce_(double aNumber) {
		return new Jun2dPoint(aNumber, aNumber);
	}

	/**
	 * Answer a new Jun2dPoint with unit values.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Constants access
	 */
	public static Jun2dPoint Unity() {
		return new Jun2dPoint(1.0d, 1.0d);
	}

	/**
	 * Answer a new Jun2dPoint with zero values.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Constants access
	 */
	public static Jun2dPoint Zero() {
		return new Jun2dPoint(0.0d, 0.0d);
	}

	/**
	 * Create a new Jun2dPoint.
	 * 
	 * @param newX double
	 * @param newY double
	 * @category Instance creation
	 */
	public Jun2dPoint(double newX, double newY) {
		super();
		x = newX;
		y = newY;
	}

	/**
	 * Create a new Jun2dPoint.
	 * 
	 * @param aPoint java.awt.Point
	 * @category Instance creation
	 */
	public Jun2dPoint(Point aPoint) {
		super();
		x = aPoint.x;
		y = aPoint.y;
	}

	/**
	 * Create a new instance of Jun2dPoint and initialize it with two Numbers.
	 * 
	 * @param numberX java.lang.Number
	 * @param numberY java.lang.Number
	 * @category Instance creation
	 */
	public Jun2dPoint(Number numberX, Number numberY) {
		this(numberX.doubleValue(), numberY.doubleValue());
	}

	/**
	 * Create a new instance of Jun2dPoint and initialize it with another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public Jun2dPoint(JunPoint aPoint) {
		this(aPoint.x(), aPoint.y());
	}

	/**
	 * Create a new instance of Jun2dPoint and initialize it with an array of double.
	 * 
	 * @param anArray double[]
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Instance creation
	 */
	public static Jun2dPoint FromArray_(double[] anArray) {
		return new Jun2dPoint(anArray[0], anArray[1]);
	}

	/**
	 * Answer a new Jun2dPoint created as a copy of another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @deprecated since jun499, use constructor
	 * @category Instance creation
	 */
	public static Jun2dPoint FromPoint_(JunPoint aPoint) {
		return new Jun2dPoint(aPoint);
	}

	/**
	 * Create a new Jun2dPoint with rho and theta.
	 * 
	 * @param rho double
	 * @param theta double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Instance creation
	 */
	public static Jun2dPoint Rho_theta_(double rho, double theta) {
		return new Jun2dPoint(rho * Math.cos(theta), rho * Math.sin(theta));
	}

	/**
	 * Answer my bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category accessing
	 */
	public Jun2dBoundingBox boundingBox() {
		return this.asBoundingBox();
	}

	/**
	 * Answer the receiver's normal unit vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint normalUnitVector() {
		return this.unitVector();
	}

	/**
	 * Answer the receiver's normal vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint normalVector() {
		return this;
	}

	/**
	 * Answer the x coordinate.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#x()
	 * @category accessing
	 */
	public double x() {
		return this.x;
	}

	/**
	 * Answer the y coordinate.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#y()
	 * @category accessing
	 */
	public double y() {
		return this.y;
	}

	/**
	 * Answer the z coordinate.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double z() {
		return 0;
	}

	/**
	 * Answer the absolute Point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category arithmetic
	 */
	public Jun2dPoint abs() {
		return new Jun2dPoint(Math.abs(x), Math.abs(y));
	}

	/**
	 * Answer the result of dividing this point by delta. This method
	 * corresponds to the method '/' in Smalltalk.
	 * 
	 * @param delta double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category arithmetic
	 */
	public Jun2dPoint dividedBy_(double delta) {
		return new Jun2dPoint(x / delta, y / delta);
	}

	/**
	 * Answer the result of dividing this point by aPoint. This method
	 * corresponds to the method '/' in Smalltalk.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category arithmetic
	 */
	public Jun2dPoint dividedBy_(Jun2dPoint aPoint) {
		return new Jun2dPoint(x / aPoint.x, y / aPoint.y);
	}

	/**
	 * Answer the difference between this point and delta.
	 * 
	 * @param delta double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category arithmetic
	 */
	public Jun2dPoint minus_(double delta) {
		return new Jun2dPoint(x - delta, y - delta);
	}

	/**
	 * Answer the difference between this point and aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category arithmetic
	 */
	public Jun2dPoint minus_(Jun2dPoint aPoint) {
		return new Jun2dPoint(x - aPoint.x, y - aPoint.y);
	}

	/**
	 * Answer the result of multiplying this point by aPoint.
	 * 
	 * @param delta double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category arithmetic
	 */
	public Jun2dPoint multipliedBy_(double delta) {
		return new Jun2dPoint(x * delta, y * delta);
	}

	/**
	 * Answer the result of multiplying this point by aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category arithmetic
	 */
	public Jun2dPoint multipliedBy_(Jun2dPoint aPoint) {
		return new Jun2dPoint(x * aPoint.x, y * aPoint.y);
	}

	/**
	 * Answer the negation of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category arithmetic
	 */
	public Jun2dPoint negated() {
		return Jun2dPoint.Zero().minus_(this);
	}

	/**
	 * Answer the sum of this point and delta.
	 * 
	 * @param delta double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category arithmetic
	 */
	public Jun2dPoint plus_(double delta) {
		return new Jun2dPoint(x + delta, y + delta);
	}

	/**
	 * Answer the sum of this point and aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#plus_(jp.co.sra.jun.geometry.basic.JunPoint)
	 * @category arithmetic
	 */
	public JunPoint plus_(JunPoint aPoint) {
		if (aPoint instanceof Jun2dPoint) {
			return this.plus_((Jun2dPoint) aPoint);
		}

		return aPoint.plus_(this);
	}

	/**
	 * Answer the sum of this point and aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category arithmetic
	 */
	public Jun2dPoint plus_(Jun2dPoint aPoint) {
		return new Jun2dPoint(x + aPoint.x, y + aPoint.y);
	}

	/**
	 * Answer 1 divided by the receiver.
	 * Provide an error notification if the receiver is 0.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category arithmetic
	 */
	public Jun2dPoint reciprocal() {
		return Unity().dividedBy_(this);
	}

	/**
	 * Answer the number representing the ordering of the receiver in the
	 * generality hierarchy.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#generality()
	 * @category coercing
	 */
	public int generality() {
		return 200;
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || anObject.getClass() != this.getClass()) {
			return false;
		}

		Jun2dPoint aPoint = (Jun2dPoint) anObject;
		return this.isEqualNumber_to_(x, aPoint.x) && this.isEqualNumber_to_(y, aPoint.y);
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || anObject.getClass() != this.getClass()) {
			return false;
		}

		Jun2dPoint aPoint = (Jun2dPoint) anObject;
		return (x == aPoint.x) && (y == aPoint.y);
	}

	/**
	 * Computes a hash code for this object.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#hashCode()
	 * @category comparing
	 */
	public int hashCode() {
		long bitsX = Double.doubleToLongBits(x);
		int xHash = (int) (bitsX ^ (bitsX >> 32));
		long bitsY = Double.doubleToLongBits(y);
		int yHash = (int) (bitsY ^ (bitsY >> 32));
		return ((xHash >> 2) | yHash);
	}

	/**
	 * Answer true if the receiver is greater than the point.
	 * This method corresponds to the method "&gt;" in smalltalk.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category comparing
	 */
	public boolean isGreaterThan_(Jun2dPoint aPoint) {
		return (x > aPoint.x) && (y > aPoint.y);
	}

	/**
	 * Answer true if the receiver is greater than or equal to the point.
	 * This method corresponds to the method "&gt;=" in smalltalk.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category comparing
	 */
	public boolean isGreaterThanOrEqualTo_(Jun2dPoint aPoint) {
		return (x >= aPoint.x) && (y >= aPoint.y);
	}

	/**
	 * Answer true if the receiver is less than aPoint.
	 * This method corresponds to the method "&lt;" in Smalltalk.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category comparing
	 */
	public boolean isLessThan_(Jun2dPoint aPoint) {
		return (x < aPoint.x) && (y < aPoint.y);
	}

	/**
	 * Answer true if the receiver is less than or equal to aPoint.
	 * This method corresponds to the method "&lt;=" in Smalltalk.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category comparing
	 */
	public boolean isLessThanOrEqualTo_(Jun2dPoint aPoint) {
		return (x <= aPoint.x) && (y <= aPoint.y);
	}

	/**
	 * Answer the maximum point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category comparing
	 */
	public Jun2dPoint max_(Jun2dPoint aPoint) {
		return new Jun2dPoint(Math.max(this.x, aPoint.x), Math.max(this.y, aPoint.y));
	}

	/**
	 * Answer the minimum point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category comparing
	 */
	public Jun2dPoint min_(Jun2dPoint aPoint) {
		return new Jun2dPoint(Math.min(this.x, aPoint.x), Math.min(this.y, aPoint.y));
	}

	/**
	 * Answer the bounding box with a point as the extent.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @category compatibilities
	 */
	public Jun2dBoundingBox extent_(Jun2dPoint aPoint) {
		return Jun2dBoundingBox.Origin_extent_(this, aPoint);
	}

	/**
	 * Answer the receiver as a java.awt.point.
	 * 
	 * @return java.awt.Point
	 * @category converting
	 */
	public Point _toPoint() {
		Jun2dPoint aPoint = this.rounded();
		return new Point((int) aPoint.x, (int) aPoint.y);
	}

	/**
	 * Answer the receiver as Jun2dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#as2dPoint()
	 * @category converting
	 */
	public Jun2dPoint as2dPoint() {
		return this;
	}

	/**
	 * Answer the receiver as a Jun3dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#as3dPoint()
	 * @category converting
	 */
	public Jun3dPoint as3dPoint() {
		return new Jun3dPoint(this.x(), this.y(), 0);
	}

	/**
	 * Answer the receiver as an array of double.
	 * 
	 * @return double[]
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#asArray()
	 * @category converting
	 */
	public double[] asArray() {
		return new double[] { x, y };
	}

	/**
	 * Convert to Jun2dBoundingBox.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @category converting
	 */
	public Jun2dBoundingBox asBoundingBox() {
		return Jun2dBoundingBox.Origin_corner_(this, this);
	}

	/**
	 * Convert to a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dVertex aVertex = new JunOpenGL3dVertex(new Jun3dPoint(x, y, 0));
		aVertex.size_(5);
		aVertex.paint_(this.defaultColor());
		return aVertex;
	}

	/**
	 * Answer the point which difference from this point to another point. 
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category double dispatching
	 */
	public Jun2dPoint differenceFromPoint_(Jun2dPoint aPoint) {
		return new Jun2dPoint(aPoint.x - x, aPoint.y - y);
	}

	/**
	 * Answer true if the point is equal to the receiver, otherwise false.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category double dispatching
	 */
	public boolean equalFromPoint_(Jun2dPoint aPoint) {
		return aPoint.x == x && aPoint.y == y;
	}

	/**
	 * Answer the product of the Jun2dPoint and the receiver.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category double dispatching
	 */
	public Jun2dPoint productFromPoint_(Jun2dPoint aPoint) {
		return new Jun2dPoint(aPoint.x * x, aPoint.y * y);
	}

	/**
	 * Answer the quo point from integer.
	 * 
	 * @param anInteger int
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category double dispatching
	 */
	public Jun2dPoint quoFromInteger_(int anInteger) {
		return new Jun2dPoint(anInteger / x, anInteger / y).truncated();
	}

	/**
	 * Answer the quotient point from another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category double dispatching
	 */
	public Jun2dPoint quotientFromPoint_(Jun2dPoint aPoint) {
		return new Jun2dPoint(aPoint.x / x, aPoint.y / y);
	}

	/**
	 * Answer the sum point from another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category double dispatching
	 */
	public Jun2dPoint sumFromPoint_(Jun2dPoint aPoint) {
		return new Jun2dPoint(aPoint.x + x, aPoint.y + y);
	}

	/**
	 * Answer an angle made with three points (including me).
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category functions
	 */
	public JunAngle angleBetween_and_(Jun2dPoint aPoint1, Jun2dPoint aPoint2) {
		Jun2dPoint v1 = aPoint1.minus_(this);
		Jun2dPoint v2 = aPoint2.minus_(this);
		double r1 = v1.rho();
		double r2 = v2.rho();

		if ((r1 < ACCURACY) || (r2 < ACCURACY)) {
			return JunAngle.Zero();
		} else {
			double sin = (((y + aPoint1.y) * (x - aPoint1.x)) + ((aPoint1.y + aPoint2.y) * (aPoint1.x - aPoint2.x)) + (aPoint2.y + y) * (aPoint2.x - x)) / r1 / r2;
			sin = Math.min(Math.max(sin, -1.0d), 1.0d);

			if (v1.dotProduct_(v2) >= 0) {
				return JunAngle.FromRad_(Math.asin(sin));
			} else if (Math.abs(sin) < ACCURACY) {
				return JunAngle.FromRad_(Math.PI);
			} else {
				int sign = (sin < 0) ? (-1) : 1;

				return JunAngle.FromRad_((Math.PI - Math.abs(Math.asin(sin))) * sign);
			}
		}
	}

	/**
	 * Answer the bounding ball object with raidus value.
	 * 
	 * @param radiusValue double
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category functions
	 */
	public Jun2dBoundingBall ball_(double radiusValue) {
		return new Jun2dBoundingBall(this, radiusValue);
	}

	/**
	 * Answer the receiver's bisector with the another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category functions
	 */
	public Jun2dLine bisector_(Jun2dPoint aPoint) {
		return Jun2dLine.Between_and_(this, aPoint);
	}

	/**
	 * Answer the bounding box object with another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category functions
	 */
	public Jun2dBoundingBox box_(Jun2dPoint aPoint) {
		return Jun2dBoundingBox.Vertex_vertex_(this, aPoint);
	}

	/**
	 * Answer the center point between this point and another point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun2dPoint center_(Jun2dPoint aPoint) {
		Jun2dPoint thePoint = Coerce_(aPoint);
		return this.plus_(thePoint.minus_(this).dividedBy_(2));
	}

	/**
	 * Answer the bounding box with a point as the corner.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public Jun2dBoundingBox corner_(Jun2dPoint aPoint) {
		return Jun2dBoundingBox.Origin_corner_(this, aPoint);
	}

	/**
	 * Answer the distance from aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return double
	 * @category functions
	 */
	public double distance_(Jun2dPoint aPoint) {
		return this.minus_(aPoint).rho();
	}

	/**
	 * Answer the result of a dot product with aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return double
	 * @category functions
	 */
	public double dotProduct_(Jun2dPoint aPoint) {
		Jun2dPoint newPoint = this.multipliedBy_(aPoint);

		return newPoint.x + newPoint.y;
	}

	/**
	 * Answer the line from another point to this.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category functions
	 */
	public Jun2dLine from_(JunPoint aPoint) {
		return new Jun2dLine(aPoint, this);
	}

	/**
	 * Answer the result of the grid with aPoint.
	 * 
	 * @param aPoint
	 * @return Jun2dPoint
	 * @category functions
	 */
	public Jun2dPoint grid_(Jun2dPoint aPoint) {
		double newX;
		double newY;
		if (aPoint.x == 0.0d) {
			newX = 0.0d;
		} else {
			newX = (double) Math.round(x / aPoint.x) * aPoint.x;
		}
		if (aPoint.y == 0.0d) {
			newY = 0.0d;
		} else {
			newY = (double) Math.round(y / aPoint.y) * aPoint.y;
		}
		return new Jun2dPoint(newX, newY);
	}

	/**
	 * Answer the result of the inner product with aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return double
	 * @category functions
	 */
	public double innerProduct_(Jun2dPoint aPoint) {
		return this.dotProduct_(aPoint);
	}

	/**
	 * Set the length from the origin.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public Jun2dPoint length_(double aNumber) {
		double length = this.length();

		if (length <= JunPoint.ACCURACY) {
			return this;
		}

		return this.multipliedBy_(aNumber / length);
	}

	/**
	 * Answer the polyline with the specified two points.
	 * 
	 * @param firstPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param secondPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @category functions
	 */
	public Jun2dPolyline polyline_with_(Jun2dPoint firstPoint, Jun2dPoint secondPoint) {
		return new Jun2dPolyline(new Jun2dPoint[] { this, firstPoint, secondPoint });
	}

	/**
	 * Answer the polyline with the specified three points.
	 * 
	 * @param firstPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param secondPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param thirdPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @category functions
	 */
	public Jun2dPolyline polyline_with_with_(Jun2dPoint firstPoint, Jun2dPoint secondPoint, Jun2dPoint thirdPoint) {
		return new Jun2dPolyline(new Jun2dPoint[] { this, firstPoint, secondPoint, thirdPoint });
	}

	/**
	 * Answer the polyline with the specified four points.
	 * 
	 * @param firstPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param secondPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param thirdPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param forthPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @category functions
	 */
	public Jun2dPolyline polyline_with_with_(Jun2dPoint firstPoint, Jun2dPoint secondPoint, Jun2dPoint thirdPoint, Jun2dPoint forthPoint) {
		return new Jun2dPolyline(new Jun2dPoint[] { this, firstPoint, secondPoint, thirdPoint, forthPoint });
	}

	/**
	 * Answer the product of the Jun2dPoint and the receiver.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return double
	 * @category functions
	 */
	public double product_(Jun2dPoint aPoint) {
		return (x * aPoint.y) - (y * aPoint.x);
	}

	/**
	 * Answer the line from this to another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category functions
	 */
	public Jun2dLine to_(JunPoint aPoint) {
		return new Jun2dLine(this, aPoint);
	}

	/**
	 * Answer the triangle from this to another points.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @category functions
	 */
	public Jun2dTriangle triangle_and_(JunPoint aPoint1, JunPoint aPoint2) {
		return Jun2dTriangle.On_on_on_(this, aPoint1, aPoint2);
	}

	/**
	 * Answer the unit vector derived from this point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public Jun2dPoint unitVector() {
		return this.length_(1.0d);
	}

	/**
	 * Answer the result of a vector product with aPoint.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public Jun2dPoint vectorProduct_(Jun2dPoint aJun2dPoint) {
		Jun3dPoint aPoint = Jun3dPoint.Coerce_(this).vectorProduct_(Jun3dPoint.Coerce_(aJun2dPoint));
		return new Jun2dPoint(aPoint.x, aPoint.y);
	}

	/**
	 * Answer the rho as of a polar coordinate.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#rho()
	 * @category polar coordinates
	 */
	public double rho() {
		return Math.sqrt(this.dotProduct_(this));
	}

	/**
	 * Answer the theta as of the polar coordinates.
	 * 
	 * @return double
	 * @category polar coordinates
	 */
	public double theta() {
		if (x == 0.0d) {
			if (y == 0.0d) {
				return JunAngle.Zero().rad();
			}
			if (y >= 0.0d) {
				return Math.PI * 0.5d;
			} else {
				return Math.PI * 1.5d;
			}
		}
		JunAngle theta = JunAngle.FromRad_(Math.atan(y / x));
		if (x >= 0) {
			if (y >= 0) {
				return theta.rad();
			} else {
				return theta.plus_(Math.PI * 2.0d).rad();
			}
		} else {
			return theta.plus_(Math.PI).rad();
		}
	}

	/**
	 * Print my storable string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		aWriter.write(String.valueOf(x));
		aWriter.write(" , ");
		aWriter.write(String.valueOf(y));
		aWriter.write(')');
	}

	/**
	 * Answer true if this is a kind of 2D geometry object, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is2d()
	 * @category testing
	 */
	public boolean is2d() {
		return true;
	}

	/**
	 * Answer true if this point is ZERO, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#isZero()
	 * @category testing
	 */
	public boolean isZero() {
		return x == 0.0d && y == 0.0d;
	}

	/**
	 * Answer true if this point is negative, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#negative()
	 * @category testing
	 */
	public boolean negative() {
		return this.isLessThan_(Zero());
	}

	/**
	 * Answer true if this point is positive, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#positive()
	 * @category testing
	 */
	public boolean positive() {
		return this.isGreaterThanOrEqualTo_(Zero());
	}

	/**
	 * Answer the value which side of on a plane.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @return int
	 * @category testing
	 */
	public int whichSideOf_(Jun2dLine aLine) {
		return aLine.whichSide_(this);
	}

	/**
	 * Answer the new Jun2dPoint which is rotated by aJunAngle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category transforming
	 */
	public Jun2dPoint rotatedBy_(JunAngle anAngle) {
		return this.transform_(Jun2dTransformation.Rotate_(anAngle));
	}

	/**
	 * Answer the new Jun2dPoint which is scaled by argument, aPoint.
	 * 
	 * @param aPoint double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category transforming
	 */
	public Jun2dPoint scaledBy_(double aPoint) {
		return this.transform_(Jun2dTransformation.Scale_(aPoint));
	}

	/**
	 * Answer the new Jun2dPoint which is scaled by aJunPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category transforming
	 */
	public Jun2dPoint scaledBy_(Jun2dPoint aPoint) {
		return this.transform_(Jun2dTransformation.Scale_(aPoint));
	}

	/**
	 * Apply a transformation 'aJunTransformation' to the receiver. This method
	 * is not defined in Smalltalk version.
	 * 
	 * @param aTransformation
	 *        jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category transforming
	 */
	public Jun2dPoint transform_(Jun2dTransformation aTransformation) {
		return aTransformation.applyToPoint_(this);
	}

	/**
	 * Apply a transformation to scale with the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category transforming
	 */
	public Jun2dTransformation transformationToScale() {
		return Jun2dTransformation.Scale_(this);
	}

	/**
	 * Apply a transformation to translate with the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category transforming
	 */
	public Jun2dTransformation transformationToTranslate() {
		return Jun2dTransformation.Translate_(this);
	}

	/**
	 * Answer the new Jun2dPoint which is translated by argument, aPoint.
	 * 
	 * @param aPoint double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category transforming
	 */
	public Jun2dPoint translatedBy_(double aPoint) {
		return this.transform_(Jun2dTransformation.Translate_(aPoint));
	}

	/**
	 * Answer the new Jun2dPoint which is translated by aJunPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category transforming
	 */
	public Jun2dPoint translatedBy_(Jun2dPoint aPoint) {
		return this.transform_(Jun2dTransformation.Translate_(aPoint));
	}

	/**
	 * Ciling this point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category truncation and round off
	 */
	public Jun2dPoint ceiling() {
		return new Jun2dPoint(Math.ceil(x), Math.ceil(y));
	}

	/**
	 * Answer the floor point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category truncation and round off
	 */
	public Jun2dPoint floor() {
		return new Jun2dPoint(Math.floor(x), Math.floor(y));
	}

	/**
	 * Answer the rounded Jun2dPoint.
	 * 
	 * @return Jun2dPoint
	 * @category truncation and round off
	 */
	public Jun2dPoint rounded() {
		return new Jun2dPoint((double) Math.round(x), (double) Math.round(y));
	}

	/**
	 * Answer the value that is an integer multiple of the argument, aNumber,
	 * that is nearest the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aNumber double
	 * @category truncation and round off
	 */
	public Jun2dPoint roundTo_(double aNumber) {
		return this.dividedBy_(aNumber).rounded().multipliedBy_(aNumber);
	}

	/**
	 * Answer the value that is an integer multiple of the argument, aNumber,
	 * that is nearest the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aNumber double jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category truncation and round off
	 */
	public Jun2dPoint roundTo_(Jun2dPoint aNumber) {
		return this.dividedBy_(aNumber).rounded().multipliedBy_(aNumber);
	}

	/**
	 * Answer the truncated Jun2dPoint.
	 * 
	 * @return Jun2dPoint
	 * 
	 * @category truncation and round off
	 */
	public Jun2dPoint truncated() {
		double newX;
		double newY;
		newX = (x >= 0) ? Math.ceil(x) : Math.floor(x);
		newY = (y >= 0) ? Math.ceil(y) : Math.floor(y);
		return new Jun2dPoint(newX, newY);
	}
}
