package jp.co.sra.jun.geometry.basic;

import java.io.IOException;
import java.io.Writer;

import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.curves.Jun3dPolyline;
import jp.co.sra.jun.geometry.surfaces.Jun3dTriangle;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dVertex;

/**
 * Jun3dPoint class
 * 
 *  @author    nisinaka
 *  @created   1998/09/30 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2004/10/19 (by Mitsuhiro Asada)
 *  @updated   2004/12/06 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun675 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun3dPoint.java,v 8.25 2008/02/20 06:30:55 nisinaka Exp $
 */
public class Jun3dPoint extends JunPoint {
	/** The <i>x</i> coordinate. */
	protected double x;

	/** The <i>y</i> coordinate. */
	protected double y;

	/** The <i>z</i> coordinate. */
	protected double z;

	/**
	 * Answer the new Jun3dPoint which is the same kind of point as the
	 * receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Coercing
	 */
	public static Jun3dPoint Coerce_(JunPoint aPoint) {
		double numberZ = aPoint instanceof Jun3dPoint ? ((Jun3dPoint) aPoint).z() : 0.0d;
		return new Jun3dPoint(aPoint.x(), aPoint.y(), numberZ);
	}

	/**
	 * Answer the new Jun3dPoint which is the same kind of point as the
	 * receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber double
	 * @category Coercing
	 */
	public static Jun3dPoint Coerce_(double aNumber) {
		return new Jun3dPoint(aNumber, aNumber, aNumber);
	}

	/**
	 * Answer a new Jun3dPoint with unit values.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Constants access
	 */
	public static Jun3dPoint Unity() {
		return new Jun3dPoint(1.0d, 1.0d, 1.0d);
	}

	/**
	 * Answer a new Jun3dPoint with zero values.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Constants access
	 */
	public static Jun3dPoint Zero() {
		return new Jun3dPoint(0.0d, 0.0d, 0.0d);
	}

	/**
	 * Create a new Jun3dPoint.
	 * 
	 * @param newX double
	 * @param newY double
	 * @param newZ double
	 * @category Instance creation
	 */
	public Jun3dPoint(double newX, double newY, double newZ) {
		super();
		x = newX;
		y = newY;
		z = newZ;
	}

	/**
	 * Create a new instance of Jun3dPoint and initialize it with three Numbers.
	 * 
	 * @param numberX java.lang.Number
	 * @param numberY java.lang.Number
	 * @param numberZ java.lang.Number
	 * @category Instance creation
	 */
	public Jun3dPoint(Number numberX, Number numberY, Number numberZ) {
		this(numberX.doubleValue(), numberY.doubleValue(), numberZ.doubleValue());
	}

	/**
	 * Create a new Jun3dPoint with the Jun2dPoint and z-coordinate.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param z double
	 * @category Instance creation
	 */
	public Jun3dPoint(Jun2dPoint aPoint, double z) {
		this(aPoint.x, aPoint.y, z);
	}

	/**
	 * Create a new instance of Jun3dPoint and initialize it with a Jun2dPoint
	 * and a Number.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param numberZ java.lang.Number
	 * @category Instance creation
	 */
	public Jun3dPoint(Jun2dPoint aPoint, Number numberZ) {
		this(aPoint, numberZ.doubleValue());
	}

	/**
	 * Create a new instance of Jun2dPoint and initialize it with another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public Jun3dPoint(JunPoint aPoint) {
		this(aPoint.x(), aPoint.y(), (aPoint instanceof Jun3dPoint ? ((Jun3dPoint) aPoint).z() : 0.0d));
	}

	/**
	 * Create a new instance of Jun3dPoint and initialize it with an array of double.
	 * 
	 * @param anArray double[]
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public static Jun3dPoint FromArray_(double[] anArray) {
		return new Jun3dPoint(anArray[0], anArray[1], anArray[2]);
	}

	/**
	 * Answer a new Jun3dPoint created as a copy of another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @deprecated since jun499, use constructor
	 * @category Instance creation
	 */
	public static Jun3dPoint FromPoint_(JunPoint aPoint) {
		return new Jun3dPoint(aPoint);
	}

	/**
	 * Answer my bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		return this.asBoundingBox();
	}

	/**
	 * Answer the receiver's normal unit vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint normalUnitVector() {
		return this.unitVector();
	}

	/**
	 * Answer the receiver's normal vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint normalVector() {
		return this;
	}

	/**
	 * Answer the x coordinate.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#x()
	 * @category accessing
	 */
	public double x() {
		return x;
	}

	/**
	 * Answer the y coordinate.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#y()
	 * @category accessing
	 */
	public double y() {
		return this.y;
	}

	/**
	 * Answer the z coordinate.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double z() {
		return this.z;
	}

	/**
	 * Answer the absolute Point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category arithmetic
	 */
	public Jun3dPoint abs() {
		return new Jun3dPoint(Math.abs(x), Math.abs(y), Math.abs(z));
	}

	/**
	 * Answer the result of dividing this point by delta. This method
	 * corresponds to the method '/' in Smalltalk.
	 * 
	 * @param delta double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category arithmetic
	 */
	public Jun3dPoint dividedBy_(double delta) {
		return new Jun3dPoint(x / delta, y / delta, z / delta);
	}

	/**
	 * Answer the result of dividing this point by aPoint. This method
	 * corresponds to the method '/' in Smalltalk.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category arithmetic
	 */
	public Jun3dPoint dividedBy_(Jun3dPoint aPoint) {
		return new Jun3dPoint(x / aPoint.x, y / aPoint.y, z / aPoint.z);
	}

	/**
	 * Answer the difference between this point and delta.
	 * 
	 * @param delta double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category arithmetic
	 */
	public Jun3dPoint minus_(double delta) {
		return new Jun3dPoint(x - delta, y - delta, z - delta);
	}

	/**
	 * Answer the difference between this point and aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category arithmetic
	 */
	public Jun3dPoint minus_(Jun3dPoint aPoint) {
		return new Jun3dPoint(x - aPoint.x, y - aPoint.y, z - aPoint.z);
	}

	/**
	 * Answer the result of multiplying this point by aPoint.
	 * 
	 * @param delta double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category arithmetic
	 */
	public Jun3dPoint multipliedBy_(double delta) {
		return new Jun3dPoint(x * delta, y * delta, z * delta);
	}

	/**
	 * Answer the result of multiplying this point by aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category arithmetic
	 */
	public Jun3dPoint multipliedBy_(Jun3dPoint aPoint) {
		return new Jun3dPoint(x * aPoint.x, y * aPoint.y, z * aPoint.z);
	}

	/**
	 * Answer the negation of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category arithmetic
	 */
	public Jun3dPoint negated() {
		return Jun3dPoint.Zero().minus_(this);
	}

	/**
	 * Answer the sum of this point and delta.
	 * 
	 * @param delta double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category arithmetic
	 */
	public Jun3dPoint plus_(double delta) {
		return new Jun3dPoint(x + delta, y + delta, z + delta);
	}

	/**
	 * Answer the sum of this point and aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#plus_(jp.co.sra.jun.geometry.basic.JunPoint)
	 * @category arithmetic
	 */
	public JunPoint plus_(JunPoint aPoint) {
		if (aPoint instanceof Jun3dPoint) {
			return this.plus_((Jun3dPoint) aPoint);
		}

		return this.plus_(Coerce_(aPoint));
	}

	/**
	 * Answer the sum of this point and aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category arithmetic
	 */
	public Jun3dPoint plus_(Jun3dPoint aPoint) {
		return new Jun3dPoint(x + aPoint.x, y + aPoint.y, z + aPoint.z);
	}

	/**
	 * Answer 1 divided by the receiver.
	 * Provide an error notification if the receiver is 0.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category arithmetic
	 */
	public Jun3dPoint reciprocal() {
		return Unity().dividedBy_(this);
	}

	/**
	 * Answer the number representing the ordering of the receiver in the
	 * generality hierarchy.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#generality()
	 * @category coercing
	 */
	public int generality() {
		return 220;
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || anObject.getClass() != this.getClass()) {
			return false;
		}

		Jun3dPoint aPoint = (Jun3dPoint) anObject;
		return this.isEqualNumber_to_(x, aPoint.x) && this.isEqualNumber_to_(y, aPoint.y) && this.isEqualNumber_to_(z, aPoint.z);
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || anObject.getClass() != this.getClass()) {
			return false;
		}

		Jun3dPoint aPoint = (Jun3dPoint) anObject;
		return (x == aPoint.x) && (y == aPoint.y) && (z == aPoint.z);
	}

	/**
	 * Answer true if the receiver is greater than the point.
	 * This method corresponds to the method "&gt;" in smalltalk.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category comparing
	 */
	public boolean isGreaterThan_(Jun3dPoint aPoint) {
		return (x > aPoint.x) && (y > aPoint.y) && (z > aPoint.z);
	}

	/**
	 * Answer true if the receiver is greater than or equal to the point.
	 * This method corresponds to the method "&gt;=" in smalltalk.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category comparing
	 */
	public boolean isGreaterThanOrEqualTo_(Jun3dPoint aPoint) {
		return (x >= aPoint.x) && (y >= aPoint.y) && (z >= aPoint.z);
	}

	/**
	 * Answer true if the receiver is less than aPoint. This method corresponds
	 * to the method "&lt;" in Smalltalk.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category comparing
	 */
	public boolean isLessThan_(Jun3dPoint aPoint) {
		return (x < aPoint.x) && (y < aPoint.y) && (z < aPoint.z);
	}

	/**
	 * Answer true if the receiver is less than or equal to aPoint. This method
	 * corresponds to the method "&lt;=" in Smalltalk.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category comparing
	 */
	public boolean isLessThanOrEqualTo_(Jun3dPoint aPoint) {
		return (x <= aPoint.x) && (y <= aPoint.y) && (z <= aPoint.z);
	}

	/**
	 * Answer the maximum point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category comparing
	 */
	public Jun3dPoint max_(Jun3dPoint aPoint) {
		return new Jun3dPoint(Math.max(x, aPoint.x), Math.max(this.y, aPoint.y), Math.max(this.z, aPoint.z));
	}

	/**
	 * Answer the minimum point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category comparing
	 */
	public Jun3dPoint min_(Jun3dPoint aPoint) {
		return new Jun3dPoint(Math.min(x, aPoint.x), Math.min(this.y, aPoint.y), Math.min(this.z, aPoint.z));
	}

	/**
	 * Answer the bounding box with a point as the extent.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @category compatibilities
	 */
	public Jun3dBoundingBox extent_(Jun3dPoint aPoint) {
		return Jun3dBoundingBox.Origin_extent_(this, aPoint);
	}

	/**
	 * Answer the receiver as Jun2dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#as2dPoint()
	 * @category converting
	 */
	public Jun2dPoint as2dPoint() {
		return new Jun2dPoint(this.x(), this.y());
	}

	/**
	 * Answer the receiver as Jun3dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#as3dPoint()
	 * @category converting
	 */
	public Jun3dPoint as3dPoint() {
		return this;
	}

	/**
	 * Answer the receiver as an array of double.
	 * 
	 * @return double[]
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#asArray()
	 * @category converting
	 */
	public double[] asArray() {
		return new double[] { x, y, z };
	}

	/**
	 * Convert to Jun3dBoundingBox.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @category converting
	 */
	public Jun3dBoundingBox asBoundingBox() {
		return Jun3dBoundingBox.Origin_corner_(this, this);
	}

	/**
	 * Convert to a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dVertex aVertex = new JunOpenGL3dVertex(this);
		aVertex.size_(5);
		aVertex.paint_(this.defaultColor());
		return aVertex;
	}

	/**
	 * Answer the point which difference from this point to another point. 
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category double dispatching
	 */
	public Jun3dPoint differenceFromPoint_(Jun3dPoint aPoint) {
		return new Jun3dPoint(aPoint.x - x, aPoint.y - y, aPoint.z - z);
	}

	/**
	 * Answer true if the point is equal to the receiver, otherwise false.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category double dispatching
	 */
	public boolean equalFromPoint_(Jun3dPoint aPoint) {
		Jun3dPoint p = Coerce_(aPoint);
		return p.x == x && p.y == y && p.z == z;
	}

	/**
	 * Answer the product of the Jun2dPoint and the receiver.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category double dispatching
	 */
	public Jun3dPoint productFromPoint_(Jun3dPoint aPoint) {
		Jun3dPoint p = Coerce_(aPoint);
		return new Jun3dPoint(p.x * x, p.y * y, p.z * z);
	}

	/**
	 * Answer the quo point from integer.
	 * 
	 * @param anInteger int
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category double dispatching
	 */
	public Jun3dPoint quoFromInteger_(int anInteger) {
		return new Jun3dPoint(anInteger / x, anInteger / y, anInteger / z).truncated();
	}

	/**
	 * Answer the quotient point from another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category double dispatching
	 */
	public Jun3dPoint quotientFromPoint_(Jun3dPoint aPoint) {
		Jun3dPoint p = Coerce_(aPoint);
		return new Jun3dPoint(p.x / x, p.y / y, p.z / z);
	}

	/**
	 * Answer the sum point from another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category double dispatching
	 */
	public Jun3dPoint sumFromPoint_(Jun3dPoint aPoint) {
		Jun3dPoint p = Coerce_(aPoint);
		return new Jun3dPoint(p.x + x, p.y + y, p.z + z);
	}

	/**
	 * Answer the bounding ball object with radius value.
	 * 
	 * @param radiusValue double
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBall
	 * @category functions
	 */
	public Jun3dBoundingBall ball_(double radiusValue) {
		return new Jun3dBoundingBall(this, radiusValue);
	}

	/**
	 * Answer the receiver's bisector with the another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category functions
	 */
	public JunPlane bisector_(Jun3dPoint aPoint) {
		return JunPlane.Between_and_(this, aPoint);
	}

	/**
	 * Answer the bounding box object with another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category functions
	 */
	public Jun3dBoundingBox box_(Jun3dPoint aPoint) {
		return Jun3dBoundingBox.Vertex_vertex_(this, aPoint);
	}

	/**
	 * Answer the center point between this point and another point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint center_(Jun3dPoint aPoint) {
		Jun3dPoint thePoint = Coerce_(aPoint);
		return this.plus_(thePoint.minus_(this).dividedBy_(2));
	}

	/**
	 * Answer the bounding box with a point as the corner.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dBoundingBox corner_(Jun3dPoint aPoint) {
		return Jun3dBoundingBox.Origin_corner_(this, aPoint);
	}

	/**
	 * Answer the distance from aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @category functions
	 */
	public double distance_(Jun3dPoint aPoint) {
		return this.minus_(aPoint).rho();
	}

	/**
	 * Answer the result of a dot product with aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @category functions
	 */
	public double dotProduct_(Jun3dPoint aPoint) {
		Jun3dPoint newPoint = this.multipliedBy_(aPoint);
		return newPoint.x + newPoint.y + newPoint.z;
	}

	/**
	 * Answer the line from another point to this.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category functions
	 */
	public Jun3dLine from_(JunPoint aPoint) {
		return new Jun3dLine(aPoint, this);
	}

	/**
	 * Answer the result of the inner product with aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @category functions
	 */
	public double innerProduct_(Jun3dPoint aPoint) {
		return this.dotProduct_(aPoint);
	}

	/**
	 * Answer the intersecting point with the Jun3dLine.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint intersectingPointWithLine_(Jun3dLine aLine) {
		if (aLine.containsPoint_(this)) {
			return this;
		} else {
			return null;
		}
	}

	/**
	 * Set the length from the origin.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint length_(double aNumber) {
		double length = this.length();
		if (length <= JunPoint.ACCURACY) {
			return this;
		}
		return this.multipliedBy_(aNumber / length);
	}

	/**
	 * Answer the result of a outer product with aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint outerProduct_(Jun3dPoint aPoint) {
		return this.outerProduct_(aPoint);
	}

	/**
	 * Answer the polyline with the specified two points.
	 * 
	 * @param firstPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param secondPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @category functions
	 */
	public Jun3dPolyline polyline_with_(Jun3dPoint firstPoint, Jun3dPoint secondPoint) {
		return new Jun3dPolyline(new Jun3dPoint[] { this, firstPoint, secondPoint });
	}

	/**
	 * Answer the polyline with the specified three points.
	 * 
	 * @param firstPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param secondPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param thirdPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @category functions
	 */
	public Jun3dPolyline polyline_with_with_(Jun3dPoint firstPoint, Jun3dPoint secondPoint, Jun3dPoint thirdPoint) {
		return new Jun3dPolyline(new Jun3dPoint[] { this, firstPoint, secondPoint, thirdPoint });
	}

	/**
	 * Answer the polyline with the specified four points.
	 * 
	 * @param firstPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param secondPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param thirdPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param forthPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @category functions
	 */
	public Jun3dPolyline polyline_with_with_(Jun3dPoint firstPoint, Jun3dPoint secondPoint, Jun3dPoint thirdPoint, Jun3dPoint forthPoint) {
		return new Jun3dPolyline(new Jun3dPoint[] { this, firstPoint, secondPoint, thirdPoint, forthPoint });
	}

	/**
	 * Answer the result of a product with aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint product_(Jun3dPoint aPoint) {
		return this.vectorProduct_(aPoint);
	}

	/**
	 * Answer the line from this to another point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category functions
	 */
	public Jun3dLine to_(JunPoint aPoint) {
		return new Jun3dLine(this, aPoint);
	}

	/**
	 * Answer the triangle from this to another points.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @category functions
	 */
	public Jun3dTriangle triangle_and_(JunPoint aPoint1, JunPoint aPoint2) {
		return Jun3dTriangle.On_on_on_(this, aPoint1, aPoint2);
	}

	/**
	 * Answer the unit vector derived from this point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint unitVector() {
		return this.length_(1.0d);
	}

	/**
	 * Answer the result of a vector product with aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint vectorProduct_(Jun3dPoint aPoint) {
		return new Jun3dPoint((y * aPoint.z) - (z * aPoint.y), (z * aPoint.x) - (x * aPoint.z), (x * aPoint.y) - (y * aPoint.x));
	}

	/**
	 * Computes a hash code for this object.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#hashCode()
	 * @category hash function
	 */
	public int hashCode() {
		long bitsX = Double.doubleToLongBits(x);
		int xHash = (int) (bitsX ^ (bitsX >> 32));
		long bitsY = Double.doubleToLongBits(y);
		int yHash = (int) (bitsY ^ (bitsY >> 32));
		long bitsZ = Double.doubleToLongBits(z);
		int zHash = (int) (bitsZ ^ (bitsZ >> 32));

		return (((xHash >> 2) | yHash) >> 2 | zHash);
	}

	/**
	 * Answer the phi as of the polar coordinates.
	 * 
	 * @return double
	 * @category polar coordinates
	 */
	public double phi() {
		if (x == 0.0d) {
			if (y == 0.0d) {
				return JunAngle.Zero().rad();
			}
			if (y >= 0.0d) {
				return 0.5d * Math.PI;
			} else {
				return 1.5d * Math.PI;
			}
		}
		JunAngle phi = JunAngle.FromRad_(Math.atan(y / x));
		if (x >= 0) {
			if (y >= 0) {
				return phi.rad();
			} else {
				return 2.0d * Math.PI + phi.rad();
			}
		} else {
			return 1.0d * Math.PI + phi.rad();
		}
	}

	/**
	 * Answer the rho as of a polar coordinate.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#rho()
	 * @category polar coordinates
	 */
	public double rho() {
		return Math.sqrt(this.dotProduct_(this));
	}

	/**
	 * Answer the theta as of the polar coordinates.
	 * 
	 * @return double
	 * @category polar coordinates
	 */
	public double theta() {
		double wk = Math.sqrt(x * x + y * y);
		if (z == 0.0d) {
			if (wk == 0.0d) {
				return JunAngle.Zero().rad();
			}
			return 0.5d * Math.PI;
		}
		JunAngle theta = JunAngle.FromRad_(Math.atan(wk / z));
		if (z >= 0) {
			return theta.rad();
		} else {
			return 1.0d * Math.PI + theta.rad();
		}
	}

	/**
	 * Print my storable string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		aWriter.write(String.valueOf(x));
		aWriter.write(" , ");
		aWriter.write(String.valueOf(y));
		aWriter.write(" , ");
		aWriter.write(String.valueOf(z));
		aWriter.write(')');
	}

	/**
	 * Answer true if this is a kind of 3D geometry object, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is3d()
	 * @category testing
	 */
	public boolean is3d() {
		return true;
	}

	/**
	 * Answer true if this point is ZERO, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#isZero()
	 * @category testing
	 */
	public boolean isZero() {
		return x == 0.0d && y == 0.0d && z == 0.0d;
	}

	/**
	 * Answer true if this point is negative, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#negative()
	 * @category testing
	 */
	public boolean negative() {
		return this.isLessThan_(Zero());
	}

	/**
	 * Answer true if this point is positive, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.basic.JunPoint#positive()
	 * @category testing
	 */
	public boolean positive() {
		return this.isGreaterThanOrEqualTo_(Zero());
	}

	/**
	 * Answer the new Jun3dPoint which is rotated by aJunAngle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category transforming
	 */
	public Jun3dPoint rotatedBy_(JunAngle anAngle) {
		return this.transform_(Jun3dTransformation.Rotate_(anAngle));
	}

	/**
	 * Answer the new Jun3dPoint which is scaled by argument, aPoint.
	 * 
	 * @param aPoint double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category transforming
	 */
	public Jun3dPoint scaledBy_(double aPoint) {
		return this.transform_(Jun3dTransformation.Scale_(aPoint));
	}

	/**
	 * Answer the new Jun3dPoint which is scaled by aJunPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category transforming
	 */
	public Jun3dPoint scaledBy_(Jun3dPoint aPoint) {
		return this.transform_(Jun3dTransformation.Scale_(aPoint));
	}

	/**
	 * Apply a transformation 'aJunTransformation' to the receiver. This method
	 * is not defined in Smalltalk version.
	 * 
	 * @param aTransformation
	 *        jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category transforming
	 */
	public Jun3dPoint transform_(Jun3dTransformation aTransformation) {
		return aTransformation.applyToPoint_(this);
	}

	/**
	 * Apply a transformation to scale with the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming
	 */
	public Jun3dTransformation transformationToScale() {
		return Jun3dTransformation.Scale_(this);
	}

	/**
	 * Apply a transformation to translate with the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming
	 */
	public Jun3dTransformation transformationToTranslate() {
		return Jun3dTransformation.Translate_(this);
	}

	/**
	 * Answer the new Jun3dPoint which is translated by argument, aPoint.
	 * 
	 * @param aPoint double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category transforming
	 */
	public Jun3dPoint translatedBy_(double aPoint) {
		return this.transform_(Jun3dTransformation.Translate_(aPoint));
	}

	/**
	 * Answer the new Jun3dPoint which is translated by aJunPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category transforming
	 */
	public Jun3dPoint translatedBy_(Jun3dPoint aPoint) {
		return this.transform_(Jun3dTransformation.Translate_(aPoint));
	}

	/**
	 * Ceiling this point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category truncation and round off
	 */
	public Jun3dPoint ceiling() {
		return new Jun3dPoint(Math.ceil(x), Math.ceil(y), Math.ceil(z));
	}

	/**
	 * Answer the floor point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category truncation and round off
	 */
	public Jun3dPoint floor() {
		return new Jun3dPoint(Math.floor(x), Math.floor(y), Math.floor(z));
	}

	/**
	 * Answer the rounded Jun3dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category truncation and round off
	 */
	public Jun3dPoint rounded() {
		return new Jun3dPoint((double) Math.round(x), (double) Math.round(y), (double) Math.round(z));
	}

	/**
	 * Answer the value that is an integer multiple of the argument, aNumber,
	 * that is nearest the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber double
	 * @category truncation and round off
	 */
	public Jun3dPoint roundTo_(double aNumber) {
		return this.dividedBy_(aNumber).rounded().multipliedBy_(aNumber);
	}

	/**
	 * Answer the value that is an integer multiple of the argument, aNumber,
	 * that is nearest the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber double jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category truncation and round off
	 */
	public Jun3dPoint roundTo_(Jun3dPoint aNumber) {
		return this.dividedBy_(aNumber).rounded().multipliedBy_(aNumber);
	}

	/**
	 * Answer the truncated Jun3dPoint.
	 * 
	 * @return Jun3dPoint
	 * @category truncation and round off
	 */
	public Jun3dPoint truncated() {
		double newX;
		double newY;
		double newZ;
		newX = (x >= 0) ? Math.ceil(x) : Math.floor(x);
		newY = (y >= 0) ? Math.ceil(y) : Math.floor(y);
		newZ = (z >= 0) ? Math.ceil(z) : Math.floor(z);

		return new Jun3dPoint(newX, newY, newZ);
	}
}
