package jp.co.sra.jun.geometry.boundaries;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.TreeSet;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.surfaces.Jun2dPolygon;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;

/**
 * Jun2dPolygonalBoundary class
 * 
 *  @author    nisinaka
 *  @created   1999/08/04 (by nisinaka)
 *  @updated   2006/10/10 (by nisinaka)
 *  @updated   2007/06/29 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dPolygonalBoundary.java,v 8.14 2008/02/20 06:30:55 nisinaka Exp $
 */
public class Jun2dPolygonalBoundary extends JunGeometry {

	/** The collection of polygons which is sorted with their area. */
	protected TreeSet polygons;

	/**
	 * Create a new instance of Jun2dPolygonalBoundary and initialize it.
	 *
	 * @category Instance creation
	 */
	public Jun2dPolygonalBoundary() {
		super();
	}

	/**
	 * Create a new instance of Jun2dPolygonalBoundary and initialize it.
	 *
	 * @param anArrayOfJun2dPolygon jp.co.sra.jun.geometry.surfaces.Jun2dPolygon[]
	 * @category Instance creation
	 */
	public Jun2dPolygonalBoundary(Jun2dPolygon[] anArrayOfJun2dPolygon) {
		this.setPolygons_(anArrayOfJun2dPolygon);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		polygons = new TreeSet(new Comparator() {
			public int compare(Object more, Object less) {
				return (((Jun2dPolygon) more).area() > ((Jun2dPolygon) less).area()) ? -1 : 1;
			}
		});
	}

	/**
	 * Answer my current polygons.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon[]
	 * @category accessing
	 */
	public Jun2dPolygon[] polygons() {
		return (Jun2dPolygon[]) polygons.toArray(new Jun2dPolygon[polygons.size()]);
	}

	/**
	 * Answer my area.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category accessing
	 */
	public double area() {
		double area = 0;
		Jun2dPolygon[] polygons = this.polygons();
		for (int i = 0; i < polygons.length; i++) {
			area += polygons[i].signedArea();
		}
		return area;
	}

	/**
	 * Add the polygon to the receiver.
	 * 
	 * @param aJun2dPolygon jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category adding
	 */
	public void add_(Jun2dPolygon aJun2dPolygon) {
		polygons.add(aJun2dPolygon);
	}

	/**
	 * Add the polygon as a hole to the receiver.
	 * 
	 * @param aJun2dPolygon jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category adding
	 */
	public void addHole_(Jun2dPolygon aJun2dPolygon) {
		this.add_(aJun2dPolygon.asNegativePolygon());
	}

	/**
	 * Add the polygon to the receiver.
	 * 
	 * @param aJun2dPolygon jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category adding
	 */
	public void addPolygon_(Jun2dPolygon aJun2dPolygon) {
		this.add_(aJun2dPolygon.asPositivePolygon());
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun2dPolygonalBoundary aPolygonalBoundary = (Jun2dPolygonalBoundary) anObject;
		int size = polygons.size();
		if (size != aPolygonalBoundary.polygons.size()) {
			return false;
		}

		Jun2dPolygon[] myPolygons = (Jun2dPolygon[]) polygons.toArray(new Jun2dPolygon[size]);
		Jun2dPolygon[] yourPolygons = (Jun2dPolygon[]) aPolygonalBoundary.polygons.toArray(new Jun2dPolygon[size]);
		for (int i = 0; i < size; i++) {
			if (myPolygons[i].equal_(yourPolygons[i]) == false) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun2dPolygonalBoundary aPolygonalBoundary = (Jun2dPolygonalBoundary) anObject;
		int size = polygons.size();
		if (size != aPolygonalBoundary.polygons.size()) {
			return false;
		}

		Jun2dPolygon[] myPolygons = (Jun2dPolygon[]) polygons.toArray(new Jun2dPolygon[size]);
		Jun2dPolygon[] yourPolygons = (Jun2dPolygon[]) aPolygonalBoundary.polygons.toArray(new Jun2dPolygon[size]);
		for (int i = 0; i < size; i++) {
			if (myPolygons[i].equals(yourPolygons[i]) == false) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Convert the receiver as an array of polygons.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon[]
	 * @category converting
	 */
	public Jun2dPolygon[] asArrayOfPolygons() {
		final ArrayList polygonList = new ArrayList();
		this.asArrayOfPolygonsDo_(new StBlockClosure() {
			public Object value_(Object polygon) {
				polygonList.add(polygon);
				return null;
			}
		});

		return (Jun2dPolygon[]) polygonList.toArray(new Jun2dPolygon[polygonList.size()]);
	}

	/**
	 * Convert the receiver to a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		final ArrayList objects = new ArrayList();
		this.asJunOpenGL3dObjectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				objects.add(anObject);
				return null;
			}
		});
		JunOpenGL3dObject aBody = (objects.size() == 1) ? (JunOpenGL3dObject) objects.get(0) : new JunOpenGL3dCompoundObject(objects);
		aBody.objectsDo_(new StBlockClosure() {
			public Object value_(Object object) {
				((JunOpenGL3dObject) object).paint_alpha_(Jun2dPolygonalBoundary.this.defaultColor(), Float.NaN);
				return null;
			}
		});
		return aBody;
	}

	/**
	 * Enumerate the array of polygons and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object asArrayOfPolygonsDo_(StBlockClosure aBlock) {
		ArrayList islandsCollection = new ArrayList();
		ArrayList holesCollection = new ArrayList();

		Jun2dPolygon[] thePolygons = (Jun2dPolygon[]) polygons.toArray(new Jun2dPolygon[polygons.size()]);
		for (int i = 0; i < thePolygons.length; i++) {
			Jun2dPolygon polygon = thePolygons[i];
			if (polygon.isPositive()) {
				islandsCollection.add(polygon);
			} else {
				holesCollection.add(polygon);
			}
		}

		Jun2dPolygon[] islands = (Jun2dPolygon[]) islandsCollection.toArray(new Jun2dPolygon[islandsCollection.size()]);
		Arrays.sort(islands, new Comparator() {
			public int compare(Object less, Object more) {
				return (((Jun2dPolygon) less).area() < ((Jun2dPolygon) more).area()) ? -1 : 1;
			}
		});

		Jun2dPolygon[] holes = (Jun2dPolygon[]) holesCollection.toArray(new Jun2dPolygon[holesCollection.size()]);

		HashMap holesOfIslands = new HashMap(islands.length);
		for (int i = 0; i < islands.length; i++) {
			holesOfIslands.put(islands[i], new Vector());
		}
		for (int i = 0; i < holes.length; i++) {
			Jun2dPolygon hole = holes[i];
			Jun2dPolygon theIsland = null;
			for (int j = 0; j < islands.length; j++) {
				Jun2dPolygon anotherIsland = (Jun2dPolygon) islands[j];
				if (anotherIsland.containsPoint_(hole.pointAt_(0))) {
					theIsland = anotherIsland;
					break;
				}
			}
			if (theIsland != null) {
				((Vector) holesOfIslands.get(theIsland)).addElement(hole);
			}
		}
		for (int i = 0; i < islands.length; i++) {
			Jun2dPolygon island = (Jun2dPolygon) islands[i];
			Vector holesOfTheIsland = (Vector) holesOfIslands.get(island);
			Jun2dPolygon theIsland = island;
			if (holesOfTheIsland.isEmpty() == false) {
				theIsland = this._polygonFrom_withHoles_(island, holesOfTheIsland);
			}
			Object result = aBlock.value_(theIsland);
			if (result != null) {
				return result;
			}
		}

		return null;
	}

	/**
	 * Enumerate all of the convex polygons and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object asConvexPolygonsDo_(final StBlockClosure aBlock) {
		return this.asArrayOfPolygonsDo_(new StBlockClosure() {
			public Object value_(Object p) {
				return ((Jun2dPolygon) p).asConvexPolygonsDo_(aBlock);
			}
		});
	}

	/**
	 * Enumerate all of the polygons as JunOpenGL3dObjects and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object asJunOpenGL3dObjectsDo_(final StBlockClosure aBlock) {
		return this.asConvexPolygonsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				aBlock.value_(((Jun2dPolygon) anObject).asJunOpenGL3dObject());
				return null;
			}
		});
	}

	/**
	 * Answer a copy of the receiver scaled by the specified extent.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dPolygonalBoundary
	 * @category transforming
	 */
	public Jun2dPolygonalBoundary scaledBy_(Jun2dPoint aPoint) {
		Jun2dPolygon[] myPolygons = (Jun2dPolygon[]) polygons.toArray(new Jun2dPolygon[polygons.size()]);
		Jun2dPolygon[] scaledPolygons = new Jun2dPolygon[myPolygons.length];
		if ((aPoint.x() * aPoint.y()) > 0) {
			for (int i = 0; i < myPolygons.length; i++) {
				scaledPolygons[i] = myPolygons[i].scaledBy_(aPoint);
			}
		} else {
			for (int i = 0; i < myPolygons.length; i++) {
				scaledPolygons[i] = myPolygons[i].scaledBy_(aPoint).reversed();
			}
		}
		return new Jun2dPolygonalBoundary(scaledPolygons);
	}

	/**
	 * Answer a copy of the receiver translated by the specified extent.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dPolygonalBoundary
	 * @category transforming
	 */
	public Jun2dPolygonalBoundary translatedBy_(Jun2dPoint aPoint) {
		Jun2dPolygon[] myPolygons = (Jun2dPolygon[]) polygons.toArray(new Jun2dPolygon[polygons.size()]);
		Jun2dPolygon[] scaledPolygons = new Jun2dPolygon[myPolygons.length];
		for (int i = 0; i < myPolygons.length; i++) {
			scaledPolygons[i] = myPolygons[i].translatedBy_(aPoint);
		}
		return new Jun2dPolygonalBoundary(scaledPolygons);
	}

	/**
	 * Set the polygons.
	 * 
	 * @param anArrayOfPolygon jp.co.sra.jun.geometry.surfaces.Jun2dPolygon[]
	 * @category private
	 */
	protected void setPolygons_(Jun2dPolygon[] anArrayOfPolygon) {
		for (int i = 0; i < anArrayOfPolygon.length; i++) {
			this.add_(anArrayOfPolygon[i]);
		}
	}

	/**
	 * Answer a polygon with holes.
	 * 
	 * @param aJun2dPolygon jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @param aCollectionOfJun2dPolygons java.util.Vector
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category private
	 */
	protected Jun2dPolygon _polygonFrom_withHoles_(Jun2dPolygon aJun2dPolygon, Vector aCollectionOfJun2dPolygons) {
		Jun2dPolygon[] anArrayOfJun2dPolygons = new Jun2dPolygon[aCollectionOfJun2dPolygons.size()];
		aCollectionOfJun2dPolygons.copyInto(anArrayOfJun2dPolygons);
		Vector loops = new Vector(anArrayOfJun2dPolygons.length + 1);
		loops.addElement(aJun2dPolygon.points());
		for (int i = 0; i < anArrayOfJun2dPolygons.length; i++) {
			loops.addElement(anArrayOfJun2dPolygons[i].points());
		}
		while (loops.size() > 1) {
			Jun2dPoint[] loop = (Jun2dPoint[]) loops.lastElement();
			loops.removeElementAt(loops.size() - 1);
			int maxYIndex = 0;
			Jun2dPoint maxYPoint = loop[0];
			for (int index = 1; index < loop.length; index++) {
				Jun2dPoint point = loop[index];
				if (point.y() > maxYPoint.y()) {
					maxYPoint = point;
					maxYIndex = index;
				}
			}

			Jun2dPoint minYPoint = null;
			int minYPointIndex = -1;
			int minYLoopIndex = -1;
			for (int pairLoopIndex = 0; pairLoopIndex < loops.size(); pairLoopIndex++) {
				Jun2dPoint[] pairLoop = (Jun2dPoint[]) loops.elementAt(pairLoopIndex);
				for (int i = 0; i < pairLoop.length; i++) {
					Jun2dPoint p1 = pairLoop[i];
					Jun2dPoint p2 = pairLoop[(i + 1) % pairLoop.length];
					if ((p1.x() > p2.x()) && (p1.x() > maxYPoint.x()) && (p2.x() <= maxYPoint.x())) {
						double y = (((maxYPoint.x() - p1.x()) * (p2.y() - p1.y())) / (p2.x() - p1.x())) + p1.y();
						if ((maxYPoint.y() <= y) && (minYPoint == null || y <= minYPoint.y())) {
							minYPoint = new Jun2dPoint(maxYPoint.x(), y);
							minYPointIndex = i;
							minYLoopIndex = pairLoopIndex;
						}
					}
				}
			}

			if (minYPoint == null) {
				throw SmalltalkException.Error("internal error");
			}

			Jun2dPoint[] minYLoop = (Jun2dPoint[]) loops.elementAt(minYLoopIndex);
			loops.removeElementAt(minYLoopIndex);
			Vector newLoop = new Vector();
			for (int i = 0; i <= maxYIndex; i++) {
				newLoop.addElement(loop[i]);
			}
			if (minYPoint.equals(minYLoop[(minYPointIndex + 1) % minYLoop.length]) == false) {
				newLoop.addElement(minYPoint);
			}
			for (int i = minYPointIndex + 1; i < minYLoop.length; i++) {
				newLoop.addElement(minYLoop[i]);
			}
			for (int i = 0; i <= minYPointIndex; i++) {
				newLoop.addElement(minYLoop[i]);
			}
			if (minYPoint.equals(minYLoop[minYPointIndex]) == false) {
				newLoop.addElement(minYPoint);
			}
			for (int i = maxYIndex; i < loop.length; i++) {
				newLoop.addElement(loop[i]);
			}
			Jun2dPoint[] thePoints = new Jun2dPoint[newLoop.size()];
			newLoop.copyInto(thePoints);
			if (minYLoopIndex == 0) {
				loops.insertElementAt(thePoints, 0);
			} else {
				loops.addElement(thePoints);
			}
		}

		return new Jun2dPolygon((Jun2dPoint[]) loops.firstElement());
	}

}
