package jp.co.sra.jun.geometry.solid;

import java.io.IOException;
import java.io.Writer;

import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.abstracts.JunSolid;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;

/**
 * JunSphere class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/06/29 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun610 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSphere.java,v 8.8 2008/02/20 06:30:58 nisinaka Exp $
 */
public class JunSphere extends JunSolid {
	protected double x0;
	protected double y0;
	protected double z0;
	protected double radius;

	/**
	 * Create a new instance of <code>JunSphere</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected JunSphere() {
		super();
	}

	/**
	 * Create a new instance of <code>JunSphere</code> and initialize it.
	 * 
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param radiusValue double
	 * @category Instance creation
	 */
	public JunSphere(JunPoint centerPoint, double radiusValue) {
		super();
		this.setCenter_(centerPoint);
		this.setRadius_(radiusValue);
	}

	/**
	 * Answer my area.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category accessing
	 */
	public double area() {
		return 4 * Math.PI * Math.pow(this.radius(), 2);
	}

	/**
	 * Answer the receiver's center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint center() {
		return new Jun3dPoint(this.x0(), this.y0(), this.z0());
	}

	/**
	 * Answer the receiver's radius.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double radius() {
		return radius;
	}

	/**
	 * Answer my volume.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#volume()
	 * @category accessing
	 */
	public double volume() {
		return 4 * Math.PI * Math.pow(this.radius(), 3) / 3;
	}

	/**
	 * Answer <code>true</code> if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || anObject.getClass() != this.getClass()) {
			return false;
		}

		JunSphere aSphere = (JunSphere) anObject;
		return this.isEqualNumber_to_(this.x0(), aSphere.x0()) && this.isEqualNumber_to_(this.y0(), aSphere.y0()) && this.isEqualNumber_to_(this.z0(), aSphere.z0()) && this.isEqualNumber_to_(this.r(), aSphere.r());
	}

	/**
	 * Answer <code>true</code> if the receiver is equal to the object.
	 * 
	 * @return boolean
	 * @param anObject java.lang.Object
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || anObject.getClass() != this.getClass()) {
			return false;
		}

		JunSphere aSphere = (JunSphere) anObject;
		return this.x0() == aSphere.x0() && this.z0() == aSphere.z0() && this.z0() == aSphere.z0() && this.r() == aSphere.r();
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dObject aBall = JunOpenGL3dObject.Ball_radius_center_(3, this.radius(), this.center());
		aBall.objectsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				((JunOpenGL3dObject) each).paint_(JunSphere.this.defaultColor());
				return null;
			}
		});
		return aBall;
	}

	/**
	 * Answer the receiver's r.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double r() {
		return this.radius();
	}

	/**
	 * Answer the receiver's x0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double x0() {
		return x0;
	}

	/**
	 * Answer the receiver's y0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double y0() {
		return y0;
	}

	/**
	 * Answer the receiver's z0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double z0() {
		return z0;
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		this.center().printOn_(aWriter);
		aWriter.write(" sphere: ");
		aWriter.write(Double.toString(this.radius()));
		aWriter.write(')');
	}

	/**
	 * Print my storable string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		aWriter.write(this._className().toString());
		aWriter.write(" center: ");
		this.center().storeOn_(aWriter);
		aWriter.write(" radius: ");
		aWriter.write(Double.toString(this.radius()));
		aWriter.write(')');
	}

	/**
	 * Answer <code>true</code> if the receiver is a sphere, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#isSphere()
	 * @category testing
	 */
	public boolean isSphere() {
		return true;
	}

	/**
	 * Answer the value which side of on a plane.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return int
	 * @category testing
	 */
	public int whichSideOf_(JunPlane aPlane) {
		int sign = this.center().whichSideOf_(aPlane);
		if (sign == 0) {
			return 0;
		}
		if (aPlane.distanceFromPoint_(this.center()) <= this.radius()) {
			return 0;
		}
		return sign;
	}

	/**
	 * Set the receiver's center point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category private
	 */
	protected void setCenter_(JunPoint aPoint) {
		Jun3dPoint centerPoint = Jun3dPoint.Coerce_(aPoint);
		this.setX0_(centerPoint.x());
		this.setY0_(centerPoint.y());
		this.setZ0_(centerPoint.z());
	}

	/**
	 * Set the receiver's r.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setR_(double aNumber) {
		this.setRadius_(aNumber);
	}

	/**
	 * Set the receiver's radius.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setRadius_(double aNumber) {
		radius = aNumber;
	}

	/**
	 * Set the receiver's x0.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setX0_(double aNumber) {
		x0 = aNumber;
	}

	/**
	 * Set the receiver's y0.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setY0_(double aNumber) {
		y0 = aNumber;
	}

	/**
	 * Set the receiver's z0.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setZ0_(double aNumber) {
		z0 = aNumber;
	}
}
