package jp.co.sra.jun.geometry.support;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunPointSorter class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2004/12/03 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun609 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPointSorter.java,v 8.13 2008/02/20 06:30:58 nisinaka Exp $
 */
public class JunPointSorter extends JunAbstractObject {
	protected Collection sourcePoints;
	protected Jun3dPoint eyePoint;
	protected Jun3dPoint upVector;
	protected Collection sortedPoints;

	/**
	 * Get the array of sorted points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @param pointCollection java.util.Collection
	 * @category Private
	 */
	protected static JunPoint[] _AsJunPointArray_(Collection pointCollection) {
		JunPoint[] arrayOfPoint = new JunPoint[pointCollection.size()];
		pointCollection.toArray(arrayOfPoint);
		return arrayOfPoint;
	}

	/**
	 * Create a new instance of <code>JunPointSorter</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunPointSorter() {
		super();
	}

	/**
	 * Create a new instance of <code>JunPointSorter</code> and initialize it.
	 * 
	 * @param pointCollection java.util.Collection
	 * @category Instance creation
	 */
	public JunPointSorter(Collection pointCollection) {
		this(JunPointSorter._AsJunPointArray_(pointCollection));
	}

	/**
	 * Create a new instance of <code>JunPointSorter</code> and initialize it.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @category Instance creation
	 */
	public JunPointSorter(JunPoint[] pointCollection) {
		this();
		for (int i = 0; i < pointCollection.length; i++) {
			this.add_(pointCollection[i]);
		}
	}

	/**
	 * Create a new instance of <code>JunPointSorter</code> and initialize it.
	 * 
	 * @param pointCollection java.util.Collection
	 * @param eyePoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public JunPointSorter(Collection pointCollection, JunPoint eyePoint, JunPoint upVector) {
		this(pointCollection);
		this.eyePoint_(eyePoint);
		this.upVector_(upVector);
	}

	/**
	 * Create a new instance of <code>JunPointSorter</code> and initialize it.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @param eyePoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public JunPointSorter(JunPoint[] pointCollection, JunPoint eyePoint, JunPoint upVector) {
		this(pointCollection);
		this.eyePoint_(eyePoint);
		this.upVector_(upVector);
	}

	/**
	 * Create a new instance of <code>JunPointSorter</code> and initialize it.
	 * 
	 * @param eyePoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public JunPointSorter(JunPoint eyePoint, JunPoint upVector) {
		this();
		this.eyePoint_(eyePoint);
		this.upVector_(upVector);
	}

	/**
	 * Initialize this object.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		sourcePoints = null;
		eyePoint = null;
		upVector = null;
		sortedPoints = null;
	}

	/**
	 * Get the eye point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint eyePoint() {
		if (eyePoint == null) {
			eyePoint = this.defaultEyePoint();
		}
		return eyePoint;
	}

	/**
	 * Set the eye point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category accessing
	 */
	public void eyePoint_(JunPoint aPoint) {
		eyePoint = Jun3dPoint.Coerce_(aPoint);
		sortedPoints = null;
	}

	/**
	 * Get the collection of sorted points.
	 * 
	 * @return java.util.Collection
	 * @category accessing
	 */
	public Collection sortedPoints() {
		if (sortedPoints == null) {
			sortedPoints = this.computeSortedPoints();
		}
		return sortedPoints;
	}

	/**
	 * Get the array of sorted points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @category accessing
	 */
	public JunPoint[] _sortedPoints() {
		return this._AsJunPointArray_(this.sortedPoints());
	}

	/**
	 * Get the collection of source points.
	 * 
	 * @return java.util.Collection
	 * @category accessing
	 */
	public Collection sourcePoints() {
		if (sourcePoints == null) {
			sourcePoints = new ArrayList();
		}
		return sourcePoints;
	}

	/**
	 * Get the array of source points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @category accessing
	 */
	public JunPoint[] _sourcePoints() {
		return this._AsJunPointArray_(this.sourcePoints());
	}

	/**
	 * Get the up vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint upVector() {
		if (upVector == null) {
			upVector = this.defaultUpVector();
		}
		return upVector;
	}

	/**
	 * Set the up vector.
	 * 
	 * @param point jp.co.sra.jun.geometry.basic.JunPoint
	 * @category accessing
	 */
	public void upVector_(JunPoint point) {
		upVector = Jun3dPoint.Coerce_(point);
		sortedPoints = null;
	}

	/**
	 * Answer the receiver's point size.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int size() {
		return this.sortedPoints().size();
	}

	/**
	 * Add the point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return boolean
	 * @category adding
	 */
	public boolean add_(JunPoint aPoint) {
		JunPoint[] _sourcePoints = JunPointSorter._AsJunPointArray_(this.sourcePoints());
		for (int i = 0; i < _sourcePoints.length; i++) {
			if (_sourcePoints[i].equal_(aPoint)) {
				return false;
			}
		}
		boolean result = this.sourcePoints().add(aPoint);
		sortedPoints = null;
		return result;
	}

	/**
	 * Add the point.
	 * 
	 * @param pointCollection java.util.Collection
	 * @return boolean
	 * @category adding
	 */
	public boolean addAll_(Collection pointCollection) {
		JunPoint[] _pointCollection = this._AsJunPointArray_(pointCollection);
		return this.addAll_(_pointCollection);
	}

	/**
	 * Add the point.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @return boolean
	 * @category adding
	 */
	public boolean addAll_(JunPoint[] pointCollection) {
		boolean result = false;
		for (int i = 0; i < pointCollection.length; i++) {
			result = result || this.add_(pointCollection[i]);
		}
		return result;
	}

	/**
	 * Answer the default big number.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultBigNumber() {
		return 1.0e11;
	}

	/**
	 * Answer the default eye point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category defaults
	 */
	public Jun3dPoint defaultEyePoint() {
		return new Jun3dPoint(this.defaultBigNumber(), 0, 0);
	}

	/**
	 * Answer the default eye point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category defaults
	 */
	public Jun3dPoint defaultUpVector() {
		return new Jun3dPoint(0, 1, 0);
	}

	/**
	 * Enumerate each objects and evaluate the block.
	 * 
	 * @return java.lang.Object
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating 
	 */
	public Object do_(StBlockClosure aBlock) {
		JunPoint[] points = this._sortedPoints();
		for (int i = 0; i < points.length; i++) {
			Object result = aBlock.value_(points[i]);
			if (result != null) {
				return result;
			}
		}
		return null;
	}

	/**
	 * Enumerate each objects with collections and evaluate the block.
	 * 
	 * @return java.lang.Object
	 * @param aCollection java.util.Collection
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating 
	 */
	public Object with_do_(Collection aCollection, StBlockClosure aBlock) {
		Iterator sortedPointIterator = this.sortedPoints().iterator();
		Iterator collectionIterator = aCollection.iterator();
		while (sortedPointIterator.hasNext() && collectionIterator.hasNext()) {
			Object result = aBlock.value_value_(sortedPointIterator.next(), collectionIterator.next());
			if (result != null) {
				return result;
			}
		}
		return null;
	}

	/**
	 * Compute nearest point from point at point collections.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointCollection java.util.ArrayList
	 * @param fromPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint computeNearestPoint_fromPoint_(ArrayList pointCollection, Jun3dPoint fromPoint) {
		JunPoint[] _pointCollection = this._AsJunPointArray_(pointCollection);

		ArrayList rejectedPoints = new ArrayList();
		for (int i = 0; i < _pointCollection.length; i++) {
			if (_pointCollection[i].equal_(fromPoint) == false) {
				rejectedPoints.add(_pointCollection[i]);
			}
		}
		if (rejectedPoints.isEmpty()) {
			return null;
		}
		if (rejectedPoints.size() == 1) {
			return (Jun3dPoint) rejectedPoints.get(0);
		}
		double minimumDistance = Double.NaN;
		Jun3dPoint nearestPoint = null;
		JunPoint[] _rejectedPoints = this._AsJunPointArray_(rejectedPoints);
		for (int i = 0; i < _rejectedPoints.length; i++) {
			JunPoint aPoint = _rejectedPoints[i];
			Jun3dPoint a3dPoint = aPoint.as3dPoint();
			double aDistance = a3dPoint.distance_(fromPoint);
			if (Double.isNaN(minimumDistance)) {
				minimumDistance = aDistance;
				nearestPoint = a3dPoint;
			} else {
				if (aDistance < minimumDistance) {
					minimumDistance = aDistance;
					nearestPoint = a3dPoint;
				}
			}
		}
		return nearestPoint;
	}

	/**
	 * Compute nearest point from point at point collections.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointCollection java.util.ArrayList
	 * @param fromPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param lookPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param normalVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint computeNearestPoint_fromPoint_eyePoint_upVector_(ArrayList pointCollection, Jun3dPoint fromPoint, Jun3dPoint lookPoint, Jun3dPoint normalVector) {
		if (pointCollection.isEmpty()) {
			return null;
		}
		if (pointCollection.size() == 1) {
			return (Jun3dPoint) pointCollection.get(0);
		}

		JunPoint[] _pointCollection = this._AsJunPointArray_(pointCollection);

		JunPlane horizontalPlane = fromPoint.plane_(normalVector);
		Jun3dLine aroundLine = new Jun3dLine(fromPoint, fromPoint.plus_(lookPoint.minus_(fromPoint).vectorProduct_(normalVector)));
		double angleDegrees = 0;

		ArrayList candidatePoints = null;
		while (true) {
			Jun3dTransformation aTransformation = Jun3dTransformation.Rotate_around_(JunAngle.FromDeg_(angleDegrees), aroundLine);
			JunPlane aPlane = horizontalPlane.transform_((Jun3dTransformation) aTransformation);
			candidatePoints = new ArrayList();
			for (int i = 0; i < _pointCollection.length; i++) {
				if (_pointCollection[i].whichSideOf_(aPlane) != -1) {
					candidatePoints.add(_pointCollection[i]);
				}
			}
			if (candidatePoints.isEmpty() && angleDegrees <= 180) {
				angleDegrees = angleDegrees + 10;
			} else {
				break;
			}
		}
		Jun3dPoint nearestPoint = this.computeNearestPoint_fromPoint_withLine_(candidatePoints, fromPoint, eyePoint.to_(fromPoint));
		return nearestPoint;
	}

	/**
	 * Compute nearest point from point at point collections.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointCollection java.util.ArrayList
	 * @param fromPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param previousPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint computeNearestPoint_fromPoint_previousPoint_(ArrayList pointCollection, Jun3dPoint fromPoint, Jun3dPoint previousPoint) {
		if (pointCollection.isEmpty()) {
			return null;
		}
		if (pointCollection.size() == 1) {
			return (Jun3dPoint) pointCollection.get(0);
		}
		ArrayList candidatePoints;

		try {
			Jun3dLine aLine = new Jun3dLine(fromPoint, previousPoint);
			double variableT = 0;

			while (true) {
				JunPlane aPlane = aLine.atT_(variableT).plane_(fromPoint.minus_(previousPoint));
				JunPoint[] _pointCollection = this._AsJunPointArray_(pointCollection);
				candidatePoints = new ArrayList();
				for (int i = 0; i < _pointCollection.length; i++) {
					if (_pointCollection[i].whichSideOf_(aPlane) != -1) {
						candidatePoints.add(_pointCollection[i]);
					}
				}
				if (candidatePoints.isEmpty() && variableT < 1) {
					variableT = variableT + 0.1;
				} else {
					break;
				}
			}
		} catch (Exception e) {
			candidatePoints = pointCollection;
		}

		Jun3dPoint nearestPoint = this.computeNearestPoint_fromPoint_(candidatePoints, fromPoint);
		return nearestPoint;
	}

	/**
	 * Compute nearest point from point with a line.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param pointCollection java.util.ArrayList
	 * @param fromPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category private
	 */
	protected Jun3dPoint computeNearestPoint_fromPoint_withLine_(ArrayList pointCollection, Jun3dPoint fromPoint, Jun3dLine aLine) {
		ArrayList rejectedPoints = new ArrayList();
		JunPoint[] _pointCollection = this._AsJunPointArray_(pointCollection);
		for (int i = 0; i < _pointCollection.length; i++) {
			if (_pointCollection[i].equal_(fromPoint) == false) {
				rejectedPoints.add(_pointCollection[i]);
			}
		}
		if (rejectedPoints.isEmpty()) {
			return null;
		}
		if (rejectedPoints.size() == 1) {
			return (Jun3dPoint) rejectedPoints.get(0);
		}

		double minimumDistance = Double.NaN;
		Jun3dPoint nearestPoint = null;
		JunPoint[] _rejectedPoints = this._AsJunPointArray_(rejectedPoints);
		for (int i = 0; i < _rejectedPoints.length; i++) {
			Jun3dPoint aPoint = (Jun3dPoint) _rejectedPoints[i];
			double aDistance = aLine.nearestPointFromPoint_(aPoint).distance_(fromPoint);
			if (Double.isNaN(minimumDistance)) {
				minimumDistance = aDistance;
				nearestPoint = aPoint;
			} else {
				if (aDistance < minimumDistance) {
					minimumDistance = aDistance;
					nearestPoint = aPoint;
				}
			}
		}
		return nearestPoint;
	}

	/**
	 * Compute sorted points.
	 * 
	 * @return java.util.Collection
	 * @category private
	 */
	protected Collection computeSortedPoints() {
		ArrayList pointCollection = (ArrayList) this.sourcePoints();
		if (pointCollection.isEmpty()) {
			return pointCollection;
		}
		if (pointCollection.size() == 1) {
			return pointCollection;
		}
		boolean is2d = ((JunPoint) pointCollection.get(0)).is2d();
		JunPoint[] _pointCollection = this._AsJunPointArray_(pointCollection);
		pointCollection = new ArrayList();
		for (int i = 0; i < _pointCollection.length; i++) {
			pointCollection.add(_pointCollection[i].as3dPoint());
		}

		Collection sortedCollection = this.computeSortedPoints_(pointCollection);

		JunPoint[] _sourcePoints = this._sourcePoints();
		if (is2d) {
			JunPoint[] _sortedCollection = this._AsJunPointArray_(sortedCollection);
			sortedCollection = new ArrayList();
			for (int i = 0; i < _sortedCollection.length; i++) {
				JunPoint aPoint = _sortedCollection[i];
				for (int j = 0; j < _sourcePoints.length; j++) {
					JunPoint p = _sourcePoints[j];
					if (p.x() == aPoint.x() && p.y() == aPoint.y()) {
						sortedCollection.add(p);
						break;
					}
				}
				if (sortedCollection.size() != i + 1) {
					SmalltalkException.Error("notFoundError");
				}
			}
		}
		return sortedCollection;
	}

	/**
	 * Compute sorted points with collection.
	 * 
	 * @return java.util.Collection
	 * @param pointCollection java.util.ArrayList
	 * @category private
	 */
	protected Collection computeSortedPoints_(ArrayList pointCollection) {
		if (pointCollection.isEmpty()) {
			return new ArrayList(0);
		}
		if (pointCollection.size() == 1) {
			return pointCollection;
		}

		ArrayList copiedPoints = new ArrayList(pointCollection);
		ArrayList aCollection = new ArrayList(pointCollection.size());
		Jun3dPoint firstPoint = this.computeNearestPoint_fromPoint_(copiedPoints, this.eyePoint());
		JunPoint[] _copiedPoints = this._AsJunPointArray_(copiedPoints);
		copiedPoints = new ArrayList();
		for (int i = 0; i < _copiedPoints.length; i++) {
			if (_copiedPoints[i].equal_(firstPoint) == false) {
				copiedPoints.add(_copiedPoints[i]);
			}
		}
		aCollection.add(firstPoint);
		if (copiedPoints.size() == 1) {
			aCollection.add(copiedPoints.get(copiedPoints.size() - 1));
			return aCollection;
		}

		Jun3dPoint secondPoint = this.computeNearestPoint_fromPoint_eyePoint_upVector_(copiedPoints, firstPoint, this.eyePoint(), this.upVector());
		_copiedPoints = this._AsJunPointArray_(copiedPoints);
		copiedPoints = new ArrayList();
		for (int i = 0; i < _copiedPoints.length; i++) {
			if (_copiedPoints[i].equal_(secondPoint) == false) {
				copiedPoints.add(_copiedPoints[i]);
			}
		}
		aCollection.add(secondPoint);

		Jun3dPoint previousPoint = firstPoint;
		Jun3dPoint currentEyePoint = secondPoint;

		while (copiedPoints.isEmpty() == false) {
			Jun3dPoint nearestPoint = this.computeNearestPoint_fromPoint_previousPoint_(copiedPoints, currentEyePoint, previousPoint);
			_copiedPoints = this._AsJunPointArray_(copiedPoints);
			copiedPoints = new ArrayList();
			for (int i = 0; i < _copiedPoints.length; i++) {
				if (_copiedPoints[i].equal_(nearestPoint) == false) {
					copiedPoints.add(_copiedPoints[i]);
				}
			}
			aCollection.add(nearestPoint);
			previousPoint = currentEyePoint;
			currentEyePoint = nearestPoint;
		}

		JunPlane aPlane = JunPlane.On_normalVector_(firstPoint, this.upVector());
		int firstSign = firstPoint.as3dPoint().plus_(this.upVector()).whichSideOf_(aPlane);
		int secondSign = secondPoint.as3dPoint().whichSideOf_(aPlane);
		if ((firstSign == 0 || secondSign == 0) == false) {
			if (firstSign != secondSign) {
				JunPoint[] points = this._AsJunPointArray_(aCollection);
				aCollection = new ArrayList(points.length);
				for (int i = points.length - 1; i >= 0; i--) {
					aCollection.add(points[i]);
				}
			}
		}
		return aCollection;
	}
}
