package jp.co.sra.jun.geometry.support;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.TreeMap;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.Jun3dCircle;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dVertex;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunPointsOnPlane class
 * 
 *  @author    nisinaka
 *  @created   2006/10/24 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun629 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPointsOnPlane.java,v 8.11 2008/02/20 06:30:58 nisinaka Exp $
 */
public class JunPointsOnPlane extends JunAbstractObject {

	public static final double ACCURACY = JunGeometry.ACCURACY * Math.pow(10, Math.max(0, Math.round(Math.log(1 / JunGeometry.ACCURACY) / Math.log(10) / 2)));

	protected ArrayList sourcePoints;
	protected Jun3dPoint averagePoint;
	protected Jun3dPoint positivePoint;
	protected Jun3dPoint negativePoint;
	protected Jun3dLine horizontalLine;
	protected Jun3dLine verticalLine;
	protected Jun3dLine normalLine;
	protected Jun3dPoint[] sortedPoints;

	/**
	 * Create a new instance of JunPointsOnPlane and initialize it.
	 *
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category Instance creation
	 */
	public JunPointsOnPlane(Jun3dPoint[] points) {
		super();
		this.addAll_(points);
	}

	/**
	 * Create a new instance of JunPointsOnPlane and initialize it.
	 *
	 * @param aCollection java.util.Collection
	 * @category Instance creation
	 */
	public JunPointsOnPlane(Collection aCollection) {
		super();
		this.addAll_(aCollection);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		sourcePoints = null;
		this.flushAll();
	}

	/**
	 * Answer my current number of the points.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfPoints() {
		return this._sourcePoints().size();
	}

	/**
	 * Answer my current radius value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double radiusValue() {
		return this.averagePoint().distance_(this.positivePoint());
	}

	/**
	 * Answer my current diameter value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double diameterValue() {
		return this.radiusValue() * 2;
	}

	/**
	 * Answer my current source points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing points
	 */
	public Jun3dPoint[] sourcePoints() {
		return (Jun3dPoint[]) this._sourcePoints().toArray(new Jun3dPoint[this._sourcePoints().size()]);
	}

	/**
	 * Answer the array list of the source points.
	 * 
	 * @return java.util.ArrayList
	 * @category accessing points
	 */
	protected ArrayList _sourcePoints() {
		if (sourcePoints == null) {
			sourcePoints = new ArrayList();
		}
		return sourcePoints;
	}

	/**
	 * Answer my current average point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing points
	 */
	public Jun3dPoint averagePoint() {
		if (averagePoint == null) {
			double x = 0, y = 0, z = 0;
			Jun3dPoint[] points = this.sourcePoints();
			if (points.length > 0) {
				for (int i = 0; i < points.length; i++) {
					x += points[i].x();
					y += points[i].y();
					z += points[i].z();
				}
				x /= points.length;
				y /= points.length;
				z /= points.length;
			}
			averagePoint = new Jun3dPoint(x, y, z);
		}
		return averagePoint;
	}

	/**
	 * Answer my current center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing points
	 */
	public Jun3dPoint centerPoint() {
		return this.averagePoint();
	}

	/**
	 * Answer my current positive point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing points
	 */
	public Jun3dPoint positivePoint() {
		if (positivePoint == null) {
			Jun3dPoint[] points = this.sourcePoints();
			if (points.length > 0) {
				Jun3dPoint originPoint = this.averagePoint();
				Jun3dPoint maxPoint = null;
				double maxDistance = Double.MIN_VALUE;
				for (int i = 0; i < points.length; i++) {
					double distance = originPoint.distance_(points[i]);
					if (distance >= maxDistance) {
						maxDistance = distance;
						maxPoint = points[i];
					}
				}

				if (maxPoint != null) {
					if (maxPoint.distance_(originPoint) < this.accuracy()) {
						maxPoint = originPoint.plus_(new Jun3dPoint(1, 0, 0));
					}
					positivePoint = maxPoint;
				}
			}

			if (positivePoint == null) {
				positivePoint = new Jun3dPoint(1, 0, 0);
			}
		}
		return positivePoint;
	}

	/**
	 * Answer my current negative point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing points
	 */
	public Jun3dPoint negativePoint() {
		if (negativePoint == null) {
			Jun3dPoint[] points = this.sourcePoints();
			if (points.length > 0) {
				JunPlane aPlane = this.yzPlane();
				Jun3dPoint maxPoint = null;
				double maxDistance = Double.MIN_VALUE;
				for (int i = 0; i < points.length; i++) {
					if (points[i].whichSideOf_(aPlane) < 0) {
						double distance = this.positivePoint().distance_(points[i]);
						if (distance >= maxDistance) {
							maxDistance = distance;
							maxPoint = points[i];
						}
					}
				}

				if (maxPoint != null) {
					if (maxPoint.distance_(this.positivePoint()) < this.accuracy()) {
						maxPoint = this.positivePoint().plus_(new Jun3dPoint(-1, 0, 0));
					}
					negativePoint = maxPoint;
				}
			}

			if (negativePoint == null) {
				negativePoint = new Jun3dPoint(-1, 0, 0);
			}
		}
		return negativePoint;
	}

	/**
	 * Answer my current sorted points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing points
	 */
	public Jun3dPoint[] sortedPoints() {
		if (sortedPoints == null) {
			sortedPoints = this.computeSortedPoints();
		}
		return sortedPoints;
	}

	/**
	 * Answer my current horizontal line.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category accessing lines
	 */
	public Jun3dLine horizontalLine() {
		if (horizontalLine == null) {
			horizontalLine = this.averagePoint().to_(this.positivePoint()).normalizedLine();
		}
		return horizontalLine;
	}

	/**
	 * Answer my current vertical line.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category accessing lines
	 */
	public Jun3dLine verticalLine() {
		if (verticalLine == null) {
			Jun3dPoint[] points = this.sourcePoints();
			if (points.length > 0) {
				Jun3dLine maxLine = null;
				double maxDistance = Double.MIN_VALUE;
				for (int i = 0; i < points.length; i++) {
					Jun3dPoint nearestPoint = this.horizontalLine().nearestPointFromPoint_(points[i]);
					double distance = points[i].distance_(nearestPoint);
					if (distance >= maxDistance) {
						maxDistance = distance;
						maxLine = nearestPoint.to_(points[i]);
					}
				}

				if (maxLine.from().distance_(maxLine.to()) >= this.accuracy()) {
					verticalLine = this.averagePoint().to_(this.averagePoint().plus_(maxLine.normalUnitVector()));
				}
			}
			if (verticalLine == null) {
				verticalLine = this.averagePoint().to_(this.averagePoint().plus_(new Jun3dPoint(0, 1, 0)));
			}
		}
		return verticalLine;
	}

	/**
	 * Answer my current normal line.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category accessing lines
	 */
	public Jun3dLine normalLine() {
		if (normalLine == null) {
			normalLine = this.averagePoint().to_(this.averagePoint().plus_(this.xyPlane().normalUnitVector()));
		}
		return normalLine;
	}

	/**
	 * Answer my current X coordinate line.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category accessing lines
	 */
	public Jun3dLine xLine() {
		return this.horizontalLine();
	}

	/**
	 * Answer my current Y coordinate line.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category accessing lines
	 */
	public Jun3dLine yLine() {
		return this.verticalLine();
	}

	/**
	 * Answer my current Z coordinate line.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category accessing lines
	 */
	public Jun3dLine zLine() {
		return this.normalLine();
	}

	/**
	 * Answer my current XY plane.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category accessing planes
	 */
	public JunPlane xyPlane() {
		return new JunPlane(this.averagePoint(), this.xLine().to(), this.yLine().to());
	}

	/**
	 * Answer my current YZ plane.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category accessing planes
	 */
	public JunPlane yzPlane() {
		return new JunPlane(this.averagePoint(), this.yLine().to(), this.zLine().to());
	}

	/**
	 * Answer my current ZX plane.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category accessing planes
	 */
	public JunPlane zxPlane() {
		return new JunPlane(this.averagePoint(), this.zLine().to(), this.xLine().to());
	}

	/**
	 * Answer my current horizontal vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing vectors
	 */
	public Jun3dPoint horizontalVector() {
		return this.horizontalLine().to().minus_(this.horizontalLine().from());
	}

	/**
	 * Answer my current vertical vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing vectors
	 */
	public Jun3dPoint verticalVector() {
		return this.verticalLine().to().minus_(this.verticalLine().from());
	}

	/**
	 * Answer my current normal vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing vectors
	 */
	public Jun3dPoint normalVector() {
		return this.normalLine().to().minus_(this.normalLine().from());
	}

	/**
	 * Add the point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category adding
	 */
	public void add_(Jun3dPoint aPoint) {
		for (int i = 0; i < this._sourcePoints().size(); i++) {
			Jun3dPoint p = (Jun3dPoint) this._sourcePoints().get(i);
			if (p.distance_(aPoint) < this.accuracy()) {
				return;
			}
		}

		this._sourcePoints().add(aPoint);
		this.flushAll();
	}

	/**
	 * Add all of the points.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category adding
	 */
	public void addAll_(Jun3dPoint[] points) {
		for (int i = 0; i < points.length; i++) {
			this.add_(points[i]);
		}
	}

	/**
	 * Add all points in the collection
	 * 
	 * @param aCollection java.util.Collection
	 * @category adding
	 */
	public void addAll_(Collection aCollection) {
		Object[] elements = aCollection.toArray();
		for (int i = 0; i < elements.length; i++) {
			if (elements[i] instanceof Jun3dPoint) {
				this.add_((Jun3dPoint) elements[i]);
			}
		}
	}

	/**
	 * Convert the receiver as Jun3dCircle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dCircle
	 * @category converting
	 */
	public Jun3dCircle asCircle() {
		return new Jun3dCircle(this.centerPoint(), this.radiusValue(), this.normalVector());
	}

	/**
	 * Flush all cached attributes.
	 * 
	 * @category flushing
	 */
	protected void flushAll() {
		averagePoint = null;
		positivePoint = null;
		negativePoint = null;
		horizontalLine = null;
		verticalLine = null;
		normalLine = null;
		sortedPoints = null;
	}

	/**
	 * Show the receiver with JunOpenGLDisplayModel.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category viewing
	 */
	public JunOpenGLDisplayModel show() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();

		JunOpenGL3dPolyline aPolyline = new JunOpenGL3dPolyline(this.sourcePoints());
		aPolyline.lineWidth_(1);
		aPolyline.paint_(Color.gray);
		compoundObject.add_(aPolyline);

		JunOpenGL3dVertex aVertex = new JunOpenGL3dVertex(this.averagePoint());
		aVertex.size_(5);
		aVertex.paint_(Color.gray);
		compoundObject.add_(aVertex);

		aPolyline = new JunOpenGL3dPolyline(new Jun3dPoint[] { this.positivePoint(), this.negativePoint() });
		aPolyline.lineWidth_(3);
		aPolyline.paint_(Color.cyan);
		compoundObject.add_(aPolyline);

		aPolyline = new JunOpenGL3dPolyline(new Jun3dPoint[] { this.horizontalLine().from(), this.horizontalLine().to() });
		aPolyline.paint_(Color.red);
		compoundObject.add_(aPolyline);

		aPolyline = new JunOpenGL3dPolyline(new Jun3dPoint[] { this.verticalLine().from(), this.verticalLine().to() });
		aPolyline.paint_(Color.green);
		compoundObject.add_(aPolyline);

		aPolyline = new JunOpenGL3dPolyline(new Jun3dPoint[] { this.normalLine().from(), this.normalLine().to() });
		aPolyline.paint_(Color.blue);
		compoundObject.add_(aPolyline);

		aPolyline = new JunOpenGL3dPolyline(this.sortedPoints());
		aPolyline.lineWidth_(3);
		aPolyline.paint_(Color.black);
		compoundObject.add_(aPolyline);

		Jun3dPoint[] sortedPoints = this.sortedPoints();
		for (int i = 0; i < sortedPoints.length; i++) {
			aVertex = new JunOpenGL3dVertex(sortedPoints[i]);
			aVertex.size_(5);
			aVertex.paint_(Color.getHSBColor((float) i / sortedPoints.length * 2 / 3, 1, 1));
			compoundObject.add_(aVertex);
		}

		JunOpenGL3dObject aCircle = this.asCircle().asJunOpenGL3dObjectColor_alpha_(Color.magenta, 0.15f);
		compoundObject.add_(aCircle);

		JunOpenGLDisplayModel displayModel = new JunOpenGLDisplayModel(compoundObject);
		displayModel.displayLightsAllOff();
		displayModel.defaultEyePoint_(this.zLine().atT_(100));
		displayModel.defaultSightPoint_(this.averagePoint());
		displayModel.defaultUpVector_(this.yLine().atT_(1).minus_(this.averagePoint()));
		displayModel.defaultZoomHeight_(this.diameterValue() * 1.2);
		displayModel.open();
		return displayModel;
	}

	/**
	 * Answer my accuracy.
	 * 
	 * @return double
	 * @category private
	 */
	protected double accuracy() {
		return ACCURACY;
	}

	/**
	 * Compute my current sorted points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category private
	 */
	protected Jun3dPoint[] computeSortedPoints() {
		Jun3dPoint[] points = this.sourcePoints();
		if (points.length <= 3) {
			return points;
		}

		Jun3dPoint fromPoint = this.positivePoint();
		JunPlane aPlane = new JunPlane(fromPoint, this.negativePoint(), this.normalLine().to());
		ArrayList upPoints = new ArrayList();
		ArrayList downPoints = new ArrayList();
		for (int i = 0; i < points.length; i++) {
			if (points[i].whichSideOf_(aPlane) > 0 || points[i].equal_(fromPoint)) {
				upPoints.add(points[i]);
			} else {
				downPoints.add(points[i]);
			}
		}

		upPoints = this.computeNearOrder_fromPoint_(upPoints, fromPoint);
		downPoints = this.computeFarOrder_fromPoint_(downPoints, fromPoint);

		ArrayList aList = new ArrayList();
		aList.addAll(upPoints);
		aList.addAll(downPoints);
		return (Jun3dPoint[]) aList.toArray(new Jun3dPoint[aList.size()]);
	}

	/**
	 * Compute the collection in the near order from the point.
	 * 
	 * @param aList java.util.ArrayList
	 * @param fromPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return java.util.ArrayList
	 * @category private
	 */
	protected ArrayList computeNearOrder_fromPoint_(ArrayList aList, Jun3dPoint fromPoint) {
		Jun3dLine aLine = this.positivePoint().to_(this.negativePoint());

		TreeMap aTreeMap = new TreeMap();
		for (int i = 0; i < aList.size(); i++) {
			Jun3dPoint aPoint = (Jun3dPoint) aList.get(i);
			Jun3dPoint nearestPoint = aLine.nearestPointFromPoint_(aPoint);
			double distance = fromPoint.distance_(nearestPoint);
			aTreeMap.put(new Double(distance), aPoint);
		}

		return new ArrayList(aTreeMap.values());
	}

	/**
	 * Compute the collection in the far order from the point.
	 * 
	 * @param aList java.util.ArrayList
	 * @param fromPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return java.util.ArrayList
	 * @category private
	 */
	protected ArrayList computeFarOrder_fromPoint_(ArrayList aList, Jun3dPoint fromPoint) {
		aList = this.computeNearOrder_fromPoint_(aList, fromPoint);
		Collections.reverse(aList);
		return aList;
	}

}
