package jp.co.sra.jun.geometry.surfaces;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBalls;
import jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes;
import jp.co.sra.jun.geometry.pluralities.Jun2dTriangles;
import jp.co.sra.jun.geometry.transformations.Jun2dTransformation;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;

/**
 * Jun2dTriangle class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2004/12/21 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun692 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dTriangle.java,v 8.20 2008/02/20 06:30:58 nisinaka Exp $
 */
public class Jun2dTriangle extends JunTriangle {
	protected Jun2dPoint p1;
	protected Jun2dPoint p2;
	protected Jun2dPoint p3;

	/**
	 * Create a new instance of Jun2dTriangle and initialize it.
	 * 
	 * @category Instance creation
	 */
	private Jun2dTriangle() {
		super();
	}

	/**
	 * Create a new instance of Jun2dTriangle and initialize it.
	 *
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Instance creation
	 */
	public Jun2dTriangle(Jun2dPoint aPoint1, Jun2dPoint aPoint2, Jun2dPoint aPoint3) {
		this.setP1_(aPoint1);
		this.setP2_(aPoint2);
		this.setP3_(aPoint3);
	}

	/**
	 * Create a new instance of Jun2dTriangle and initialize it.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public static Jun2dTriangle First_second_third_(JunPoint aPoint1, JunPoint aPoint2, JunPoint aPoint3) {
		return Jun2dTriangle.On_on_on_(aPoint1, aPoint2, aPoint3);
	}

	/**
	 * Create a new instance of Jun2dTriangle and initialize it.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public static Jun2dTriangle On_on_on_(JunPoint aPoint1, JunPoint aPoint2, JunPoint aPoint3) {
		Jun2dPoint thePoint1 = Jun2dPoint.Coerce_(aPoint1);
		Jun2dPoint thePoint2 = Jun2dPoint.Coerce_(aPoint2);
		Jun2dPoint thePoint3 = Jun2dPoint.Coerce_(aPoint3);
		if (thePoint1.equals(thePoint2) || thePoint1.equals(thePoint3) || thePoint2.equals(thePoint3)) {
			return null;
		}

		return new Jun2dTriangle(thePoint1, thePoint2, thePoint3);
	}

	/**
	 * Answer the receiver's bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category accessing
	 */
	public Jun2dBoundingBall boundingBall() {
		return this.asBoundingBall();
	}

	/**
	 * Answer the receiver's bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category accessing
	 */
	public Jun2dBoundingBox boundingBox() {
		return this.asBoundingBox();
	}

	/**
	 * Get the first point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint first() {
		return this.p1();
	}

	/**
	 * Get the middle point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint middle() {
		return this.p2();
	}

	/**
	 * Get the last point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint last() {
		return this.p3();
	}

	/**
	 * Get the p1 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint p1() {
		return p1;
	}

	/**
	 * Answer my p1 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunTriangle#_p1()
	 * @category accessing
	 */
	protected JunPoint _p1() {
		return this.p1();
	}

	/**
	 * Get the p2 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint p2() {
		return p2;
	}

	/**
	 * Answer my p2 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunTriangle#_p2()
	 * @category accessing
	 */
	protected JunPoint _p2() {
		return this.p2();
	}

	/**
	 * Get the p3 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint p3() {
		return p3;
	}

	/**
	 * Answer the receiver's three points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] points() {
		return new Jun2dPoint[] { this.p1(), this.p2(), this.p3() };
	}

	/**
	 * Answer my p3 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunTriangle#_p3()
	 * @category accessing
	 */
	protected JunPoint _p3() {
		return this.p3();
	}

	/**
	 * Get the second point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun2dPoint second() {
		return this.p2();
	}

	/**
	 * Get the third point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun2dPoint third() {
		return this.p3();
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun2dTriangle aTriangle = (Jun2dTriangle) anObject;
		return this.first().equal_(aTriangle.first()) && this.second().equal_(aTriangle.second()) && this.third().equal_(aTriangle.third());
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBox#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun2dTriangle aTriangle = (Jun2dTriangle) anObject;
		return this.first().equals(aTriangle.first()) && this.second().equals(aTriangle.second()) && this.third().equals(aTriangle.third());
	}

	/**
	 * Convert the receiver as an array of the <code>Jun2dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @category converting
	 */
	public Jun2dLine[] asArrayOfLines() {
		return this.asArrayOf2dLines();
	}

	/**
	 * Convert the receiver as an array of <code>Jun2dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf2dLines()
	 * @category converting
	 */
	public Jun2dLine[] asArrayOf2dLines() {
		return new Jun2dLine[] { this.p1().to_(this.p2()), this.p2().to_(this.p3()), this.p3().to_(this.p1()) };
	}

	/**
	 * Convert the receiver as an array of <code>JunPlane</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane[]
	 * @see jp.co.sra.jun.geometry.surfaces.JunTriangle#asArrayOfPlanes()
	 * @category converting
	 */
	public JunPlane[] asArrayOfPlanes() {
		return new JunPlane[] { this.asPlane() };
	}

	/**
	 * Convert to an array of point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[3]
	 * @category converting
	 */
	public Jun2dPoint[] asArrayOfPoints() {
		return new Jun2dPoint[] { this.first(), this.second(), this.third() };
	}

	/**
	 * Convert to an array of a <code>Jun3dTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category converting
	 */
	public Jun2dTriangle[] asArrayOfTriangles() {
		return new Jun2dTriangle[] { this };
	}

	/**
	 * Convert to a <code>Jun2dBoundingBall</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category converting
	 */
	public Jun2dBoundingBall asBoundingBall() {
		return new Jun2dBoundingBall(new Jun2dPoint[] { this.p1(), this.p2(), this.p3() });
	}

	/**
	 * Convert to a <code>Jun2dBoundingBalls</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBalls
	 * @category converting
	 */
	public Jun2dBoundingBalls asBoundingBalls() {
		Jun2dBoundingBalls boundingBalls = new Jun2dBoundingBalls();
		Collection aCollection = new ArrayList();
		aCollection.add(this.asBoundingBall());
		boundingBalls.boundingBalls_(aCollection);
		return boundingBalls;
	}

	/**
	 * Convert to a <code>Jun2dBoundingBox</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category converting
	 */
	public Jun2dBoundingBox asBoundingBox() {
		return new Jun2dBoundingBox(new Jun2dPoint[] { this.p1(), this.p2(), this.p3() });
	}

	/**
	 * Convert to a <code>Jun2dBoundingBoxes</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category converting
	 */
	public Jun2dBoundingBoxes asBoundingBoxes() {
		Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
		Collection aCollection = new ArrayList();
		aCollection.add(this.asBoundingBox());
		boundingBoxes.boundingBoxes_(aCollection);
		return boundingBoxes;
	}

	/**
	 * Convert to a <code>Jun2dCircle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dCircle
	 * @category converting
	 */
	public Jun2dCircle asCircle() {
		Jun2dPoint centerPoint = this.centerOfCircumcircle();
		return new Jun2dCircle(centerPoint, centerPoint.distance_(this.p1()));
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dPolygon aPolygon = new JunOpenGL3dPolygon(new Jun3dPoint[] { Jun3dPoint.Coerce_(this.p1()), Jun3dPoint.Coerce_(this.p2()), Jun3dPoint.Coerce_(this.p3()) });
		aPolygon.paint_(this.defaultColor());
		return aPolygon;
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code> with points.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObjectWithPoints()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObjectWithPoints() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		compoundObject.add_(this.asJunOpenGL3dObject());
		JunOpenGL3dObject theP1 = this.p1().asJunOpenGL3dObject();
		theP1.paint_(Color.red);
		compoundObject.add_(theP1);
		JunOpenGL3dObject theP2 = this.p2().asJunOpenGL3dObject();
		theP2.paint_(Color.green);
		compoundObject.add_(theP2);
		JunOpenGL3dObject theP3 = this.p3().asJunOpenGL3dObject();
		theP3.paint_(Color.blue);
		compoundObject.add_(theP3);
		return compoundObject;
	}

	/**
	 * Convert to a <code>JunPlane</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @see jp.co.sra.jun.geometry.surfaces.JunTriangle#asPlane()
	 * @category converting
	 */
	public JunPlane asPlane() {
		return new JunPlane(this.p1(), this.p2(), this.p3());
	}

	/**
	 * Convert to an array of point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[4]
	 * @category converting
	 */
	public Jun2dPoint[] asPointArray() {
		return new Jun2dPoint[] { this.first(), this.second(), this.third(), this.first() };
	}

	/**
	 * Convert to a <code>JunTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @category converting
	 */
	public Jun2dTriangle asTriangle() {
		return this;
	}

	/**
	 * Convert to a <code>Jun2dTriangles</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dTriangles
	 * @category converting
	 */
	public Jun2dTriangles asTriangles() {
		Collection aCollection = new ArrayList();
		aCollection.add(this);
		return new Jun2dTriangles(aCollection);
	}

	/**
	 * Answer the array of the intersecting points.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category dividing
	 */
	public Jun2dPoint[] pointsDividedBy_(Jun2dLine aLine) {
		Collection aCollection = (Collection) this.tableDividedBy_(aLine).get($("points"));
		return (Jun2dPoint[]) aCollection.toArray(new Jun2dPoint[aCollection.size()]);
	}

	/**
	 * Answer the array of the triangles in the positive side of the line.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category dividing
	 */
	public Jun2dTriangle[] positivesDividedBy_(Jun2dLine aLine) {
		Collection aCollection = (Collection) this.tableDividedBy_(aLine).get($("positives"));
		return (Jun2dTriangle[]) aCollection.toArray(new Jun2dTriangle[aCollection.size()]);
	}

	/**
	 * Answer the array of the triangles in the negative side of the line.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category dividing
	 */
	public Jun2dTriangle[] negativesDividedBy_(Jun2dLine aLine) {
		Collection aCollection = (Collection) this.tableDividedBy_(aLine).get($("negatives"));
		return (Jun2dTriangle[]) aCollection.toArray(new Jun2dTriangle[aCollection.size()]);
	}

	/**
	 * Answer the array of the triangles on the line.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category dividing
	 */
	public Jun2dTriangle[] zerosDividedBy_(Jun2dLine aLine) {
		Collection aCollection = (Collection) this.tableDividedBy_(aLine).get($("zeros"));
		return (Jun2dTriangle[]) aCollection.toArray(new Jun2dTriangle[aCollection.size()]);
	}

	/**
	 * Answer the array of the triangles on and in the positive side of the line.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category dividing
	 */
	public Jun2dTriangle[] diviedBy_(Jun2dLine aLine) {
		Map aMap = this.tableDividedBy_(aLine);
		ArrayList aList = new ArrayList();
		aList.addAll((Collection) aMap.get($("zeros")));
		aList.addAll((Collection) aMap.get($("positives")));
		return (Jun2dTriangle[]) aList.toArray(new Jun2dTriangle[aList.size()]);
	}

	/**
	 * Answer the table which contains all dividing information.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @return java.util.Map
	 * @category dividing
	 */
	public Map tableDividedBy_(Jun2dLine aLine) {
		JunPlane aPlane = new JunPlane(aLine.from().as3dPoint(), aLine.to().as3dPoint(), new Jun3dPoint(0, 0, 1));
		Map aTable = Jun3dTriangle.On_on_on_(this.p1().as3dPoint(), this.p2().as3dPoint(), this.p3.as3dPoint()).tableDividedBy_(aPlane);

		ArrayList aList = new ArrayList();
		for (Iterator i = ((List) aTable.get($("positives"))).iterator(); i.hasNext();) {
			Jun3dTriangle aTriangle = (Jun3dTriangle) i.next();
			aList.add(new Jun2dTriangle(aTriangle.p1().as2dPoint(), aTriangle.p2().as2dPoint(), aTriangle.p3().as2dPoint()));
		}
		aTable.put($("positives"), aList);

		aList = new ArrayList();
		for (Iterator i = ((List) aTable.get($("negatives"))).iterator(); i.hasNext();) {
			Jun3dTriangle aTriangle = (Jun3dTriangle) i.next();
			aList.add(new Jun2dTriangle(aTriangle.p1().as2dPoint(), aTriangle.p2().as2dPoint(), aTriangle.p3().as2dPoint()));
		}
		aTable.put($("negatives"), aList);

		aList = new ArrayList();
		for (Iterator i = ((List) aTable.get($("points"))).iterator(); i.hasNext();) {
			Jun3dPoint aPoint = (Jun3dPoint) i.next();
			aList.add(aPoint.as2dPoint());
		}
		aTable.put($("points"), aList);

		return aTable;
	}

	/**
	 * Answer the this area size
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category functions
	 */
	public double area() {
		if (this.hasArea() == false) {
			return 0.0d;
		}
		double a = this.p1().distance_(this.p2());
		double b = this.p2().distance_(this.p3());
		double c = this.p3().distance_(this.p1());
		double s = (a + b + c) / 2;
		double area = Math.sqrt(s * (s - a) * (s - b) * (s - c));
		return area;
	}

	/**
	 * Answer the receiver's area with sign.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.surfaces.JunTriangle#areaWithSign()
	 * @category functions
	 */
	public double areaWithSign() {
		double areaWithSign = 0;
		areaWithSign = areaWithSign + (this.first().x() - this.second().x()) * (this.first().y() + this.second().y());
		areaWithSign = areaWithSign + (this.second().x() - this.third().x()) * (this.second().y() + this.third().y());
		areaWithSign = areaWithSign + (this.third().x() - this.first().x()) * (this.third().y() + this.first().y());
		areaWithSign = areaWithSign / 2;
		return areaWithSign;
	}

	/**
	 * Answer the center of circumcircle of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public Jun2dPoint centerOfCircumcircle() {
		return this.circumcircle().center();
	}

	/**
	 * Answer the center of incircle of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public Jun2dPoint centerOfIncircle() {
		return this.incircle().center();
	}

	/**
	 * Answer the circumcircle of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dCircle
	 * @category functions
	 */
	public Jun2dCircle circumcircle() {
		Jun3dPoint unitVector = this.asPlane().normalUnitVector();
		Jun3dPoint middlePoint = Jun3dPoint.Coerce_(this.p1().center_(this.p2()));
		Jun3dLine firstLine = new Jun3dLine(middlePoint, middlePoint.plus_(unitVector));
		Jun3dTransformation aTransformation = Jun3dTransformation.Rotate_around_(JunAngle.FromDeg_(90), Jun3dPoint.Coerce_(this.p1()).to_(this.p2()));
		firstLine = firstLine.transform_(aTransformation);
		middlePoint = Jun3dPoint.Coerce_(this.p1().center_(this.p3()));
		Jun3dLine secondLine = new Jun3dLine(middlePoint, middlePoint.plus_(unitVector));
		aTransformation = Jun3dTransformation.Rotate_around_(JunAngle.FromDeg_(90), this.p1().to_(p3));
		secondLine = secondLine.transform_(aTransformation);
		Jun3dPoint centerPoint = firstLine.intersectingPointWithLine_(secondLine);
		double radiusValue = centerPoint.distance_(Jun3dPoint.Coerce_(this.p1()));
		return new Jun2dCircle(Jun2dPoint.Coerce_(centerPoint), radiusValue);
	}

	/**
	 * Answer the receiver's detailed triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category functions
	 */
	public Jun2dTriangle[] detailedTriangles() {
		Jun2dTriangle[] detailedTriangles = new Jun2dTriangle[4];
		Jun2dPoint[] pointArray = new Jun2dPoint[6];
		pointArray[0] = this.p1();
		pointArray[1] = this.p2();
		pointArray[2] = this.p3();
		pointArray[3] = pointArray[0].center_(pointArray[1]);
		pointArray[4] = pointArray[1].center_(pointArray[2]);
		pointArray[5] = pointArray[2].center_(pointArray[0]);
		detailedTriangles[0] = new Jun2dTriangle(pointArray[0], pointArray[3], pointArray[5]);
		detailedTriangles[1] = new Jun2dTriangle(pointArray[3], pointArray[1], pointArray[4]);
		detailedTriangles[2] = new Jun2dTriangle(pointArray[4], pointArray[2], pointArray[5]);
		detailedTriangles[3] = new Jun2dTriangle(pointArray[3], pointArray[4], pointArray[5]);
		return detailedTriangles;
	}

	/**
	 * Answer the receiver's detailed triangles with the specified level.
	 * 
	 * @param levelNumber int
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category functions
	 */
	public Jun2dTriangle[] detailedTrianglesLevel_(int levelNumber) {
		if (levelNumber <= 0) {
			return new Jun2dTriangle[] { this };
		}

		Collection aCollection = new ArrayList();
		Jun2dTriangle[] detailedTriangles = this.detailedTriangles();
		for (int i = 0; i < detailedTriangles.length; i++) {
			Jun2dTriangle[] triangles = detailedTriangles[i].detailedTrianglesLevel_(levelNumber - 1);
			for (int j = 0; j < triangles.length; j++) {
				aCollection.add(triangles[j]);
			}
		}
		return (Jun2dTriangle[]) aCollection.toArray(new Jun2dTriangle[aCollection.size()]);
	}

	/**
	 * Answer the receiver's in circle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dCircle
	 * @throws java.lang.IllegalStateException
	 * @category functions
	 */
	public Jun2dCircle incircle() {
		double t = this.peripheryOfTriangle();
		if (t < this.Accuracy()) {
			throw new IllegalStateException("unexpected triangle");
		}
		double d12 = this.p1().distance_(this.p2());
		double d23 = this.p2().distance_(this.p3());
		double d31 = this.p3().distance_(this.p1());
		double x = (d23 * this.p1().x() + d31 * this.p2().x() + d12 * this.p3().x()) / t;
		double y = (d23 * this.p1().y() + d31 * this.p2().y() + d12 * this.p3().y()) / t;
		Jun2dPoint centerPoint = new Jun2dPoint(x, y);
		double s = this.peripheryOfTriangle() / 2;
		if (s < this.Accuracy()) {
			throw new IllegalStateException("unexpected triangle");
		}
		double radiusValue = Math.sqrt((s - this.p1().distance_(this.p2())) * (s - this.p2().distance_(this.p3())) * (s - this.p3().distance_(this.p1())) / s);
		return new Jun2dCircle(centerPoint, radiusValue);
	}

	/**
	 * Answer the normal unit vector of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint normalUnitVector() {
		return this.asPlane().normalUnitVector();
	}

	/**
	 * Answer the normal vector of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint normalVector() {
		return this.asPlane().normalVector();
	}

	/**
	 * Answer the periphery of triangle.
	 * 
	 * @return double
	 * @category functions
	 */
	public double peripheryOfTriangle() {
		double perimeter = 0.0d;
		perimeter = perimeter + this.p1().distance_(this.p2());
		perimeter = perimeter + this.p2().distance_(this.p3());
		perimeter = perimeter + this.p3().distance_(this.p1());
		return perimeter;
	}

	/**
	 * Answer the three lines of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[3]
	 * @category functions
	 */
	public Jun2dLine[] threeLines() {
		Jun2dLine[] threeLines = new Jun2dLine[3];

		Jun2dLine aLine = this.p1().to_(this.p2());
		threeLines[0] = aLine;

		aLine = this.p2().to_(this.p3());
		threeLines[1] = aLine;

		aLine = this.p3().to_(this.p1());
		threeLines[2] = aLine;

		return threeLines;
	}

	/**
	 * Answer the reversed triangle of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @category functions
	 */
	public Jun2dTriangle reversed() {
		return Jun2dTriangle.On_on_on_(this.p3(), this.p2(), this.p1());
	}

	/**
	 * Answer the receiver's subdivide triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dTriangles
	 * @category subdividing
	 */
	public Jun2dTriangles subdivide() {
		return this.subdivide4();
	}

	/**
	 * Answer the receiver's subdivide triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dTriangles
	 * @category subdividing
	 */
	public Jun2dTriangles subdivide4() {
		return new Jun2dTriangles(this.detailedTriangles());
	}

	/**
	 * Answer the receiver's subdivide triangles with specified level.
	 * 
	 * @param levelNumber int
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dTriangles
	 * @category subdividing
	 */
	public Jun2dTriangles subdivide4Level_(int levelNumber) {
		return new Jun2dTriangles(this.detailedTrianglesLevel_(levelNumber));
	}

	/**
	 * Answer the receiver's subdivide triangles with specified level.
	 * 
	 * @param levelNumber int
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dTriangles
	 * @category subdividing
	 */
	public Jun2dTriangles subdivideLevel_(int levelNumber) {
		return this.subdivide4Level_(levelNumber);
	}

	/**
	 * Answer true if the receiver contains aPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(Jun2dPoint aPoint) {
		Jun2dLine[] lines = this.threeLines();
		for (int i = 0; i < lines.length; i++) {
			if (aPoint.whichSideOf_(lines[i]) < 0) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Answer true if the receiver is a 2d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is2d()
	 * @category testing
	 */
	public boolean is2d() {
		return true;
	}

	/**
	 * Answer the new Jun2dTriangle which is rotated by aJunAngle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @category transforming
	 */
	public Jun2dTriangle rotatedBy_(JunAngle anAngle) {
		return this.transform_(Jun2dTransformation.Rotate_(anAngle));
	}

	/**
	 * Answer the new Jun2dTriangle which is scaled by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @category transforming
	 */
	public Jun2dTriangle scaledBy_(double aNumber) {
		return this.transform_(Jun2dTransformation.Scale_(aNumber));
	}

	/**
	 * Answer the new Jun2dTriangle which is scaled by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @category transforming
	 */
	public Jun2dTriangle scaledBy_(Jun2dPoint aPoint) {
		return this.transform_(Jun2dTransformation.Scale_(aPoint));
	}

	/**
	 * Apply a transformation 'aJunTransformation' to the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category transforming
	 */
	public Jun2dTriangle transform_(Jun2dTransformation aTransformation) {
		return Jun2dTriangle.On_on_on_(this.p1().transform_(aTransformation), this.p2().transform_(aTransformation), this.p3().transform_(aTransformation));
	}

	/**
	 * Answer the new Jun2dTriangle which is translated by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @category transforming
	 */
	public Jun2dTriangle translatedBy_(double aNumber) {
		return this.transform_(Jun2dTransformation.Translate_(aNumber));
	}

	/**
	 * Answer the new Jun2dTriangle which is translated by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @category transforming
	 */
	public Jun2dTriangle translatedBy_(Jun2dPoint aPoint) {
		return this.transform_(Jun2dTransformation.Translate_(aPoint));
	}

	/**
	 * Set the p1 point.
	 * 
	 * @param aPpoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category private
	 */
	protected void setP1_(JunPoint aPpoint) {
		p1 = Jun2dPoint.Coerce_(aPpoint);
	}

	/**
	 * Set the p2 point.
	 * 
	 * @param aPpoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category private
	 */
	protected void setP2_(JunPoint aPpoint) {
		p2 = Jun2dPoint.Coerce_(aPpoint);
	}

	/**
	 * Set the p3 point.
	 * 
	 * @param aPpoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category private
	 */
	protected void setP3_(JunPoint aPpoint) {
		p3 = Jun2dPoint.Coerce_(aPpoint);
	}
}
