package jp.co.sra.jun.geometry.surfaces;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;

import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;

/**
 * Jun3dCircle class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2004/12/17 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun665 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun3dCircle.java,v 8.19 2008/02/20 06:30:58 nisinaka Exp $
 */
public class Jun3dCircle extends JunCircle {

	protected double x0;
	protected double y0;
	protected double z0;
	protected Jun3dPoint upVector;

	/**
	 * Create a new instance of Jun3dCircle and initialize it.
	 * 
	 * @category Instance creation
	 */
	private Jun3dCircle() {
		super();
	}

	/**
	 * Create a new instance of Jun3dCircle and initialize it.
	 * 
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param radiusValue double
	 * @category Instance creation
	 */
	public Jun3dCircle(Jun3dPoint centerPoint, double radiusValue) {
		this(centerPoint, radiusValue, new Jun3dPoint(0, 0, 1));
	}

	/**
	 * Create a new instance of Jun3dCircle and initialize it.
	 * 
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param radiusValue double
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public Jun3dCircle(Jun3dPoint centerPoint, double radiusValue, Jun3dPoint upVector) {
		super();
		this.setCenter_(centerPoint);
		this.setRadius_(radiusValue);
		this.setUpVector_(upVector);
	}

	/**
	 * Create a new instance of Jun3dCircle and initialize it.
	 * 
	 * @param radiusValue double
	 * @category Instance creation
	 */
	public Jun3dCircle(double radiusValue) {
		this(new Jun3dPoint(0, 0, 0), radiusValue, new Jun3dPoint(0, 0, 1));
	}

	/**
	 * Create a new instance of Jun3dCircle and initialize it.
	 * 
	 * @param radiusValue double
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public Jun3dCircle(double radiusValue, Jun3dPoint upVector) {
		this(new Jun3dPoint(0, 0, 0), radiusValue, upVector);
	}

	/**
	 * Answer the receiver's center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint center() {
		return new Jun3dPoint(x0, y0, z0);
	}

	/**
	 * Answer my center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.surfaces.JunCircle#_center()
	 * @category accessing
	 */
	protected JunPoint _center() {
		return this.center();
	}

	/**
	 * Answer the receiver's up vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint upVector() {
		return upVector;
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.surfaces.JunCircle#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun3dCircle aCircle = (Jun3dCircle) anObject;
		return this.isEqualNumber_to_(x0, aCircle.x0) && this.isEqualNumber_to_(y0, aCircle.y0) && this.isEqualNumber_to_(z0, aCircle.z0) && this.uv().minus_(aCircle.uv()).abs().isLessThan_(Jun3dPoint.Coerce_(this.accuracy()));
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.surfaces.JunCircle#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun3dCircle aCircle = (Jun3dCircle) anObject;
		return this.x0() == aCircle.x0() && this.y0() == aCircle.y0() && this.z0() == aCircle.z0() && this.uv().equals(aCircle.uv());
	}

	/**
	 * Convert the receiver as an array of JunPlane.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOfPlanes()
	 * @category converting
	 */
	public JunPlane[] asArrayOfPlanes() {
		return new JunPlane[] { this.asPlane() };
	}

	/**
	 * Convert to a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dObject aCircle = JunOpenGL3dObject.CircleBy_radius_upVector_(10, this.radius(), this.upVector()).translatedBy_(this.center());
		aCircle.objectsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				((JunOpenGL3dObject) each).paint_(Jun3dCircle.this.defaultColor());
				return null;
			}
		});
		return aCircle;
	}

	/**
	 * Convert to a JunPlane.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @see jp.co.sra.jun.geometry.surfaces.JunCircle#asPlane()
	 * @category converting
	 */
	public JunPlane asPlane() {
		return JunPlane.On_vertical_(this.center(), new Jun3dPoint(0, 0, 0).to_(this.upVector()));
	}

	/**
	 * Convert to a JunTriangle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @category converting
	 */
	public Jun3dTriangle asTriangle() {
		return Jun3dTriangle.On_normalVector_distance_(this.center(), this.upVector(), this.radius());
	}

	/**
	 * Answer the track points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category functions
	 */
	public Jun3dPoint[] trackPoints() {
		return this.trackPointsBy_(36);
	}

	/**
	 * Answer the receiver's track points with the specified division number.
	 * 
	 * @param divisionNumber int
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category functions
	 */
	public Jun3dPoint[] trackPointsBy_(int divisionNumber) {
		ArrayList aList = new ArrayList();
		JunPlane aPlane = this.asPlane();
		Jun3dLine axisLine = this.center().to_(this.center().plus_(this.upVector()));
		Jun3dPoint aPoint = this.center().to_(aPlane.p2()).normalized().atT_(this.radius());
		Jun3dLine standardLine = axisLine.translatedBy_(aPoint.minus_(this.center()));
		for (int i = 0; i <= divisionNumber; i++) {
			double degrees = 360.0 * i / divisionNumber;
			Jun3dTransformation aTransformation = Jun3dTransformation.Rotate_around_(JunAngle.FromDeg_(degrees), axisLine);
			Jun3dLine aLine = standardLine.transform_(aTransformation);
			aPoint = aPlane.intersectingPointWithLine_(aLine);
			aList.add(aPoint);
		}
		return (Jun3dPoint[]) aList.toArray(new Jun3dPoint[aList.size()]);
	}

	/**
	 * Answer the parameter uv.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category parameters
	 */
	public Jun3dPoint uv() {
		return this.upVector();
	}

	/**
	 * Answer the parameter x0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double x0() {
		return x0;
	}

	/**
	 * Answer the parameter y0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double y0() {
		return y0;
	}

	/**
	 * Answer the parameter z0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double z0() {
		return z0;
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.geometry.surfaces.JunCircle#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		if (this.upVector().equal_(new Jun3dPoint(0, 0, 1))) {
			super.printOn_(aWriter);
		} else {
			aWriter.write('(');
			this.center().printOn_(aWriter);
			aWriter.write(" radius: ");
			aWriter.write(String.valueOf(this.radius()));
			aWriter.write(" upVector: ");
			this.upVector().printOn_(aWriter);
			aWriter.write(')');
		}
	}

	/**
	 * Print my storable string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.geometry.surfaces.JunCircle#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		if (this.upVector().equals(new Jun3dPoint(0, 0, 1))) {
			super.storeOn_(aWriter);
		} else {
			aWriter.write('(');
			aWriter.write(this._className().toString());
			aWriter.write(" center: ");
			aWriter.write(this.center().printString());
			aWriter.write(" radius: ");
			aWriter.write(String.valueOf(this.radius()));
			aWriter.write(" upVector: ");
			aWriter.write(this.upVector().printString());
			aWriter.write(')');
		}
	}

	/**
	 * Answer the new <code>Jun3dCircle</code> which is rotated by aJunAngle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dCircle
	 * @category transforming
	 */
	public Jun3dCircle rotatedBy_(JunAngle anAngle) {
		return this.transform_(Jun3dTransformation.Rotate_(anAngle));
	}

	/**
	 * Answer the new <code>Jun3dCircle</code> which is scaled by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dCircle
	 * @category transforming
	 */
	public Jun3dCircle scaledBy_(double aNumber) {
		return this.transform_(Jun3dTransformation.Scale_(aNumber));
	}

	/**
	 * Answer the new <code>Jun3dCircle</code> which is scaled by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dCircle
	 * @category transforming
	 */
	public Jun3dCircle scaledBy_(Jun3dPoint aPoint) {
		return this.transform_(Jun3dTransformation.Scale_(aPoint));
	}

	/**
	 * Apply a transformation 'aTransformation' to the receiver.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dCircle
	 * @throws java.lang.IllegalArgumentException
	 * @category transforming
	 */
	public Jun3dCircle transform_(Jun3dTransformation aTransformation) {
		double accuracy = this.accuracy();
		Jun3dPoint[] inscribedSquare = this.trackPointsBy_(4);
		Jun3dPoint[] transformedParallelogram = new Jun3dPoint[inscribedSquare.length];
		for (int i = 0; i < inscribedSquare.length; i++) {
			transformedParallelogram[i] = inscribedSquare[i].transform_(aTransformation);
		}
		Jun3dPoint north = transformedParallelogram[0];
		Jun3dPoint east = transformedParallelogram[1];
		Jun3dPoint south = transformedParallelogram[2];
		Jun3dPoint west = transformedParallelogram[3];
		Jun3dPoint ns = south.minus_(north);
		Jun3dPoint ew = west.minus_(east);
		if (ns.length() < accuracy || ew.length() < accuracy) {
			if (ns.length() < accuracy && ew.length() < accuracy) {
				throw new IllegalArgumentException("transformed circle too small to define upVector");
			}
			throw new IllegalArgumentException("transformation on the circle does not make a circle");
		}
		boolean isRhonbus = Math.abs(ns.unitVector().dotProduct_(ew.unitVector())) < accuracy;
		boolean isSquare = isRhonbus && Math.abs(ns.length() - ew.length()) < accuracy;
		if (isSquare == false) {
			throw new IllegalArgumentException("transformation on the circle does not make a circle");
		}
		Jun3dPoint newCenter = this.center().transform_(aTransformation);
		double newRadius = (ns.length() + ew.length()) / 4;
		Jun3dPoint newUpVector = new Jun3dPoint(0, 0, 0).to_(this.upVector()).transform_(aTransformation).normalVector();
		if (newUpVector.length() < accuracy) {
			JunPlane originalPlane = new JunPlane(inscribedSquare[0], inscribedSquare[1], inscribedSquare[2]);
			if (originalPlane.normalVector().dotProduct_(this.upVector()) < 0) {
				originalPlane = originalPlane.reversed();
			}
			newUpVector = originalPlane.transform_(aTransformation).normalVector();
		}
		return new Jun3dCircle(newCenter, newRadius, newUpVector);
	}

	/**
	 * Answer the new <code>Jun3dCircle</code> which is translated by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dCircle
	 * @category transforming
	 */
	public Jun3dCircle translatedBy_(double aNumber) {
		return this.transform_(Jun3dTransformation.Translate_(aNumber));
	}

	/**
	 * Answer the new <code>Jun3dCircle</code> which is translated by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dCircle
	 * @category transforming
	 */
	public Jun3dCircle translatedBy_(Jun3dPoint aPoint) {
		return this.transform_(Jun3dTransformation.Translate_(aPoint));
	}

	/**
	 * Answer true if the receiver is a 3d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is3d()
	 * @category testing
	 */
	public boolean is3d() {
		return true;
	}

	/**
	 * Answer the value which side of on a plane.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return int
	 * @category testing
	 */
	public int whichSideOf_(JunPlane aPlane) {
		int sign = this.center().whichSideOf_(aPlane);
		if (sign == 0) {
			return 0;
		}
		if (aPlane.intersectingLineWithPlane_(this.asPlane()).distanceFromPoint_(this.center()) <= this.radius()) {
			return 0;
		}
		return sign;
	}

	/**
	 * Set the receiver's center point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected void setCenter_(Jun3dPoint aPoint) {
		Jun3dPoint centerPoint = Jun3dPoint.Coerce_(aPoint);
		this.setX0_(centerPoint.x());
		this.setY0_(centerPoint.y());
		this.setZ0_(centerPoint.z());
	}

	/**
	 * Set the receiver's up vector.
	 * 
	 * @param normalVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected void setUpVector_(Jun3dPoint normalVector) {
		upVector = Jun3dPoint.Coerce_(normalVector);
	}

	/**
	 * Set the parameter uv.
	 * 
	 * @param normalVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected void setUV_(Jun3dPoint normalVector) {
		this.setUpVector_(normalVector);
	}

	/**
	 * Set the parameter x0.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setX0_(double aNumber) {
		x0 = aNumber;
	}

	/**
	 * Set the parameter y0.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setY0_(double aNumber) {
		y0 = aNumber;
	}

	/**
	 * Set the parameter z0.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setZ0_(double aNumber) {
		z0 = aNumber;
	}
}
