package jp.co.sra.jun.goodies.bulletinboard;

import java.awt.Color;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.Vector;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.SystemResourceSupport;
import jp.co.sra.jun.system.framework.JunApplicationModel;

/**
 * JunBulletinBoard class
 * 
 *  @author    nisinaka
 *  @created   2002/11/19 (by nisinaka)
 *  @updated   2004/09/22 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun629 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunBulletinBoard.java,v 8.12 2008/02/20 06:31:11 nisinaka Exp $
 */
public class JunBulletinBoard extends JunApplicationModel {
	protected static final int DefaultBulletinHeight = 48;
	protected static final Color DefaultBackgroundColor = Color.getHSBColor(0, 0, 0.833354f);

	protected Vector bulletinElements;
	protected Color bulletinColor;
	protected Rectangle bulletinBounds;
	protected Color _backgroundColor;

	/**
	 * Create a new instance of <code>JunBulletinBoard</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunBulletinBoard() {
		this.initialize();
	}

	/**
	 * Convert the specified string to a flat string.
	 *
	 * @return java.lang.String
	 * @param aString java.lang.String
	 * @category Utilities
	 */
	public static String ConvertToFlatString_(String aString) {
		if (aString.length() == 0) {
			return aString;
		}

		if (aString.length() == 1) {
			if (Character.isWhitespace(aString.charAt(0))) {
				return String.valueOf(' ');
			} else {
				return aString;
			}
		}

		StringWriter sw = new StringWriter();
		BufferedWriter writer = null;
		BufferedReader reader = null;
		try {
			writer = new BufferedWriter(sw);
			reader = new BufferedReader(new StringReader(aString));
			String line;
			while ((line = reader.readLine()) != null) {
				if (line.length() > 0) {
					writer.write(line);
				} else {
					writer.write(' ');
				}
			}
		} catch (IOException e) {
			sw.write("!!! failed to convert to a flat string !!!");
		} finally {
			if (reader != null) {
				try {
					reader.close();
				} catch (IOException e) {
				}
			}
			if (writer != null) {
				try {
					writer.flush();
					writer.close();
				} catch (IOException e) {
				}
			}
		}

		return sw.toString();
	}

	/**
	 * Answer a default font of JunBulletinBoard.
	 * Replacement of the textStyble class method of JunBulletinBoard in Smalltalk version.
	 *
	 * @return java.awt.Font
	 * @category Utilities
	 */
	public static Font Font() {
		return FontHeight_(DefaultBulletinHeight);
	}

	/**
	 * Answer a default font with the specified height.
	 * Replacement of the textStyleHeight: class method of JunBulletinBoard in Smalltalk version.
	 *
	 * @return java.awt.Font
	 * @param height int
	 * @category Utilities
	 */
	protected static Font FontHeight_(int height) {
		Font aFont = SystemResourceSupport.getFont();
		return aFont.deriveFont(Font.PLAIN, Math.max(1, height));
	}

	/**
	 * Add the specified character as an element of the receiver.
	 *
	 * @param ch char
	 * @category adding
	 */
	public void add_(char ch) {
		this.add_(new Character(ch));
	}

	/**
	 * Add the specified object as an element of the receiver.
	 *
	 * @param anObject java.lang.Object
	 * @category adding
	 */
	public void add_(Object anObject) {
		JunBulletinElement bulletinElement = new JunBulletinElement();
		bulletinElement.entity_(anObject);
		bulletinElement.color_(this.color());
		bulletinElements.add(bulletinElement);
		this.flushBounds();
	}

	/**
	 * Answer the current background color.
	 *
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color backgroundColor() {
		return _backgroundColor;
	}

	/**
	 * Set the new background color.
	 *
	 * @param newBackgroundColor java.awt.Color
	 * @category accessing
	 */
	public void backgroundColor_(Color newBackgroundColor) {
		_backgroundColor = newBackgroundColor;
	}

	/**
	 * Ansewr the current bounds of the receiver.
	 *
	 * @return java.awt.Rectangle
	 * @category accessing
	 */
	public Rectangle bounds() {
		if (bulletinBounds == null) {
			bulletinBounds = this.preferredBounds();
		}
		return bulletinBounds;
	}

	/**
	 * Answer the current color of the receiver.
	 *
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color color() {
		return bulletinColor;
	}

	/**
	 * Set the new color of the receiver.
	 *
	 * @param newBulletinColor java.awt.Color
	 * @category accessing
	 */
	public void color_(Color newBulletinColor) {
		bulletinColor = newBulletinColor;
	}

	/**
	 * Answer the all elements as an array of JunBulletinElement.
	 *
	 * @return jp.co.sra.jun.goodies.bulletinboard.JunBulletinElement[]
	 * @category accessing
	 */
	public JunBulletinElement[] elements() {
		JunBulletinElement[] elements = new JunBulletinElement[bulletinElements.size()];
		bulletinElements.copyInto(elements);
		return elements;
	}

	/**
	 * Set the new font height.
	 * Replacement of the textStyleHeight: method of JunBulletinBoard in Smalltalk version.
	 *
	 * @param newHeight int
	 * @category accessing
	 */
	public void fontHeight_(int newHeight) {
		Font aFont = FontHeight_(newHeight);
		JunBulletinElement[] elements = this.elements();
		for (int i = 0; i < elements.length; i++) {
			elements[i].font_(aFont);
		}
		this.flushBounds();
	}

	/**
	 * Answer the current height of the receiver.
	 *
	 * @return int
	 * @category accessing
	 */
	public int height() {
		return this.bounds().height;
	}

	/**
	 * Create an image at the specified area.
	 *
	 * @return java.awt.image.BufferedImage
	 * @param location int
	 * @param width int
	 * @category accessing
	 */
	public BufferedImage imageLocation_width_(int location, int width) {
		Rectangle targetArea = new Rectangle(location, 0, width, this.height());
		BufferedImage anImage = new BufferedImage(targetArea.width, targetArea.height, BufferedImage.TYPE_INT_ARGB);
		Graphics aGraphics = anImage.getGraphics();
		try {
			aGraphics.setColor(this.backgroundColor());
			aGraphics.fillRect(0, 0, targetArea.width, targetArea.height);
			JunBulletinElement[] elements = this.elements();
			for (int i = 0; i < elements.length; i++) {
				Rectangle rectangle = elements[i].rectangle();
				if (rectangle.intersects(targetArea)) {
					int x = rectangle.x - location;
					int y = rectangle.y;
					aGraphics.setColor(elements[i].color());
					elements[i].entity().displayOn_at_(aGraphics, new Point(x, y));
				}
			}
		} finally {
			aGraphics.dispose();
		}

		return anImage;
	}

	/**
	 * Open a view according to the current view mode.
	 *
	 * @return java.awt.Frame
	 * @see jp.co.sra.smalltalk.StApplicationModel#open()
	 * @category interface opening
	 */
	public Frame open() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunBulletinBoardViewAwt.OpenOn_(this);
		} else {
			return JunBulletinBoardViewSwing.OpenOn_(this);
		}
	}

	/**
	 * Answer the current width of the receiver.
	 *
	 * @return int
	 * @category accessing
	 */
	public int width() {
		return this.bounds().width;
	}

	/**
	 * Flush the bounds information.
	 * 
	 * @category private
	 */
	protected void flushBounds() {
		bulletinBounds = null;
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		bulletinElements = new Vector();
		bulletinColor = Color.black;
		bulletinBounds = new Rectangle(0, 0, 0, 0);
		_backgroundColor = DefaultBackgroundColor;
	}

	/**
	 * Answer the preferred bounds of the receiver.
	 *
	 * @return java.awt.Rectangle
	 * @category private
	 */
	protected Rectangle preferredBounds() {
		int width = 0;
		int height = 0;
		JunBulletinElement[] elements = this.elements();
		for (int i = 0; i < elements.length; i++) {
			width += elements[i].width();
			height = Math.max(height, elements[i].height());
		}
		int x = 0;
		for (int i = 0; i < elements.length; i++) {
			int y = (height - elements[i].height()) / 2;
			Rectangle rectangle = elements[i].rectangle();
			rectangle.translate(x - rectangle.x, y - rectangle.y);
			elements[i].rectangle_(rectangle);
			x += elements[i].width();
		}
		return new Rectangle(0, 0, width, height);
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunBulletinBoardViewAwt(this);
		} else {
			return new JunBulletinBoardViewSwing(this);
		}
	}

}
