package jp.co.sra.jun.goodies.calendar;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Polygon;
import java.util.Iterator;
import java.util.Vector;

import jp.co.sra.jun.system.framework.JunAbstractViewJPanel;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StModel;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;

/**
 * JunCalendarDateViewSwing class
 * 
 *  @author    Nobuto Matsubara
 *  @created   2003/10/06 (by Nobuto Matsubara)
 *  @updated   2004/09/21 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun519 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunCalendarDateViewSwing.java,v 8.17 2008/02/20 06:31:12 nisinaka Exp $
 */
public class JunCalendarDateViewSwing extends JunAbstractViewJPanel implements JunCalendarDateView {

	protected Vector frames;

	/**
	 * Create a new instance of <code>JunCalendarDateViewSwing</code> and initialize it.
	 * 
	 * @param aCalendarDateModel
	 * @category Instance creation
	 */
	public JunCalendarDateViewSwing(JunApplicationModel aCalendarDateModel) {
		super(aCalendarDateModel);
	}

	/**
	 * Return a frames.
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector frames() {
		if (frames == null) {
			this.makeFrames();
		}
		return frames;
	}

	/**
	 * Make frames.
	 * @category private
	 */
	private void makeFrames() {
		Vector collection = new Vector();
		int width = this.getSize().width / 7;
		int height = this.getSize().height / 8;
		StRectangle button = StRectangle.Origin_extent_(new Point(0, 0), new Point(12, 16));
		int x = width * 1;
		int y = 0;
		StRectangle box = StRectangle.Origin_extent_(new Point(x, y), new Point(width * 2, height));
		collection.add(new Object[] { $("year"), box });
		button = button.align_with_(button.rightCenter(), box.leftCenter());
		collection.add(new Object[] { $("button"), $("yearDown"), button });
		button = button.align_with_(button.leftCenter(), box.rightCenter());
		collection.add(new Object[] { $("button"), $("yearUp"), button });
		x = width * 4;
		y = 0;
		box = StRectangle.Origin_extent_(new Point(x, y), new Point(width * 2, height));
		collection.add(new Object[] { $("month"), box });
		button = button.align_with_(button.rightCenter(), box.leftCenter());
		collection.add(new Object[] { $("button"), $("monthDown"), button });
		button = button.align_with_(button.leftCenter(), box.rightCenter());
		collection.add(new Object[] { $("button"), $("monthUp"), button });
		x = 0;
		y = height * 1;
		for (int index = 0; index < this.getModel().weekdayNames().length; index++) {
			box = StRectangle.Origin_extent_(new Point(x, y), new Point(width, height));
			collection.add(new Object[] { $("weekday"), new Integer(index), box });
			x = x + width;
		}
		int[][] calendarMatrix = this.getModel().calendarMatrix();
		y = height * 2;
		height = (this.getBounds().height - y) / calendarMatrix.length;
		for (int indexOfWeek = 0; indexOfWeek < calendarMatrix.length; indexOfWeek++) {
			int[] week = calendarMatrix[indexOfWeek];
			x = 0;
			for (int indexOfDay = 0; indexOfDay < calendarMatrix[indexOfWeek].length; indexOfDay++) {
				if (!(week[indexOfDay] == 0)) {
					box = StRectangle.Origin_extent_(new Point(x, y), new Point(width, height));
					collection.add(new Object[] { $("day"), new Point(indexOfDay, indexOfWeek), box });
				}
				x = x + width;
			}
			y = y + height;
		}
		frames = collection;
	}

	/**
	 * paint.
	 * @param aGraphics java.Swing.Graphics
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		this.displayBordersOn_(aGraphics);
		this.displayObjectsOn_(aGraphics);
	}

	/**
	 * Paint borders.
	 * @param aGraphics java.Swing.Graphics
	 * @category displaying
	 */
	public void displayBordersOn_(Graphics aGraphics) {
		StSymbol key;
		StRectangle box;
		StRectangle clip;
		StRectangle area;
		Iterator frames = this.frames().iterator();
		while (frames.hasNext()) {
			Object[] array = (Object[]) frames.next();
			key = (StSymbol) array[0];
			box = (StRectangle) array[array.length - 1];
			clip = new StRectangle(aGraphics.getClipBounds());
			if (clip.intersects_(box)) {
				if (key == $("year") || key == $("month") || key == $("day")) {
					aGraphics.setColor(veryLightGray);
					aGraphics.fillRect(box.x(), box.y(), box.width(), box.height());
					area = box.insetBy_(StRectangle.Origin_corner_(new Point(1, 1), new Point(2, 2)));
					aGraphics.setColor(veryDarkGray);
					aGraphics.drawRect(area.x(), area.y(), area.width(), area.height());
					if (key == $("year") || key == $("month")) {
						area = box.insetBy_(StRectangle.Origin_corner_(new Point(2, 2), new Point(3, 3)));
						aGraphics.setColor(Color.black);
						aGraphics.drawRect(area.x(), area.y(), area.width(), area.height());
					}
					if (key == $("day")) {
						Point point = (Point) array[1];
						int day = this.getModel().atPoint_(point);
						if (this.getModel().day() == day) {
							area = box.insetBy_(StRectangle.Origin_corner_(new Point(2, 2), new Point(3, 3)));
							aGraphics.setColor(Color.black);
							aGraphics.drawRect(area.x(), area.y(), area.width(), area.height());
						}
					}
				}
			}
		}
	}

	/**
	 * Paint objects.
	 * @param aGraphics java.Swing.Graphics
	 * @category displaying
	 */
	public void displayObjectsOn_(Graphics aGraphics) {
		StSymbol key;
		StRectangle box;
		Iterator frames = this.frames().iterator();
		while (frames.hasNext()) {
			Object[] array = (Object[]) frames.next();
			key = (StSymbol) array[0];
			box = (StRectangle) array[array.length - 1];
			if (key == $("year")) {
				this.displayOn_year_clip_(aGraphics, key, box);
			}
			if (key == $("month")) {
				this.displayOn_month_clip_(aGraphics, key, box);
			}
			if (key == $("weekday")) {
				this.displayOn_weekday_clip_(aGraphics, (Integer) array[1], box);
			}
			if (key == $("day")) {
				this.displayOn_day_clip_(aGraphics, (Point) array[1], box);
			}
			if (key == $("button")) {
				this.displayOn_button_clip_(aGraphics, (StSymbol) array[1], box);
			}
		}
	}

	/**
	 * Paint year.
	 * @param aGraphics Graphics
	 * @param aSymbol StSymbol
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_year_clip_(Graphics aGraphics, StSymbol aSymbol, StRectangle aRectangle) {
		int year = this.getModel().year();
		StComposedText para = new StComposedText(String.valueOf(year));
		StRectangle paraBounds = new StRectangle(para.bounds());
		StRectangle rect = paraBounds.align_with_(paraBounds.center(), aRectangle.center());
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(rect)) {
			aGraphics.setColor(Color.black);
			para.displayOn_at_(aGraphics, rect.origin());
		}
	}

	/**
	 * Paint month.
	 * @param aGraphics Graphics
	 * @param aSymbol StSymbol
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_month_clip_(Graphics aGraphics, StSymbol aSymbol, StRectangle aRectangle) {
		int month = this.getModel().month();
		StComposedText para = new StComposedText(String.valueOf(month));
		StRectangle paraBounds = new StRectangle(para.bounds());
		StRectangle rect = paraBounds.align_with_(paraBounds.center(), aRectangle.center());
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(rect)) {
			aGraphics.setColor(Color.black);
			para.displayOn_at_(aGraphics, rect.origin());
		}
	}

	/**
	 * Paint day.
	 * @param aGraphics Graphics
	 * @param aPoint Point
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_day_clip_(Graphics aGraphics, Point aPoint, StRectangle aRectangle) {
		int day = this.getModel().atPoint_(aPoint);
		StComposedText para = new StComposedText(String.valueOf(day));
		StRectangle paraBounds = new StRectangle(para.bounds());
		StRectangle rect = paraBounds.align_with_(paraBounds.center(), aRectangle.center());
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(rect)) {
			aGraphics.setColor(Color.black);
			if (aPoint.x == 0) {
				aGraphics.setColor(Color.red);
			}
			if (aPoint.x == 6) {
				aGraphics.setColor(Color.blue);
			}
			para.displayOn_at_(aGraphics, rect.origin());
		}
	}

	/**
	 * Paint weekday.
	 * @param aGraphics Graphics
	 * @param anInteger Integer
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_weekday_clip_(Graphics aGraphics, Integer anInteger, StRectangle aRectangle) {
		String weekday = this.getModel().weekdayNames()[anInteger.intValue()];
		StComposedText para = new StComposedText(weekday);
		StRectangle bounds = new StRectangle(para.bounds());
		StRectangle rect = bounds.align_with_(bounds.center(), aRectangle.center());
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(rect)) {
			aGraphics.setColor(Color.black);
			if (anInteger.intValue() == 0) {
				aGraphics.setColor(Color.red);
			}
			if (anInteger.intValue() == 6) {
				aGraphics.setColor(Color.blue);
			}
			para.displayOn_at_(aGraphics, rect.origin());
		}
	}

	/**
	 * Paint button.
	 * @param aGraphics Graphics
	 * @param aSymbol StSymbol
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_button_clip_(Graphics aGraphics, StSymbol aSymbol, StRectangle aRectangle) {
		Polygon polygon;
		StRectangle box = aRectangle;
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(box)) {
			if (aSymbol == $("yearDown") || aSymbol == $("monthDown")) {
				polygon = new Polygon();
				polygon.addPoint(box.topRight().x, box.topRight().y);
				polygon.addPoint(box.bottomRight().x, box.bottomRight().y);
				polygon.addPoint(box.leftCenter().x, box.leftCenter().y);
				polygon.addPoint(box.topRight().x, box.topRight().y);
			} else {
				polygon = new Polygon();
				polygon.addPoint(box.topLeft().x, box.topLeft().y);
				polygon.addPoint(box.bottomLeft().x, box.bottomLeft().y);
				polygon.addPoint(box.rightCenter().x, box.rightCenter().y);
				polygon.addPoint(box.topLeft().x, box.topLeft().y);
			}
			aGraphics.setColor(veryLightGray);
			aGraphics.fillPolygon(polygon);
			box = box.insetBy_(new StRectangle(1, 1, 2, 2));
			if (aSymbol == $("yearDown") || aSymbol == $("monthDown")) {
				polygon = new Polygon();
				polygon.addPoint(box.topRight().x, box.topRight().y);
				polygon.addPoint(box.bottomRight().x, box.bottomRight().y);
				polygon.addPoint(box.leftCenter().x, box.leftCenter().y);
				polygon.addPoint(box.topRight().x, box.topRight().y);
			} else {
				polygon = new Polygon();
				polygon.addPoint(box.topLeft().x, box.topLeft().y);
				polygon.addPoint(box.bottomLeft().x, box.bottomLeft().y);
				polygon.addPoint(box.rightCenter().x, box.rightCenter().y);
				polygon.addPoint(box.topLeft().x, box.topLeft().y);
			}
			aGraphics.setColor(veryDarkGray);
			aGraphics.drawPolygon(polygon);
		}
	}

	/**
	 * Update event handling.
	 * @param evt DependentEvent
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		this.makeFrames();

		if (this.isShowing() == false) {
			return;
		}

		if (evt.getAspect() == $("day")) {
			repaint();
		}
		if (evt.getAspect() == $("month")) {
			repaint();
		}
		if (evt.getAspect() == $("year")) {
			repaint();
		}
		super.update_(evt);
	}

	/**
	 * Build component.
	 * @category interface opening
	 */
	protected void buildComponent() {
		this.setPreferredSize(new Dimension(300, 200));
	}

	/**
	 * Answer my model as JunCalendarDateModel.
	 * 
	 * @return jp.co.sra.jun.goodies.calendar.JunCalendarDateModel
	 * @category model accessing
	 */
	public JunCalendarDateModel getModel() {
		return (JunCalendarDateModel) this.model();
	}

	/**
	 * Return model.
	 * @return jp.co.sra.jun.goodies.calenadr.JunCalendarDateModel
	 * @category controller accessing
	 */
	protected StController defaultController() {
		return new JunCalendarDateController();
	}

	/**
	 * Return model.
	 * @return jp.co.sra.jun.goodies.calenadr.JunCalendarDateModel
	 * @category accessing
	 */
	protected StModel defaultModel() {
		return new JunCalendarDateModel();
	}

	/**
	 * Recalculate size of frames.
	 * 
	 * @param x int
	 * @param y int
	 * @param width int
	 * @param height int
	 * @see jp.co.sra.jun.system.framework.JunAbstractViewCanvas#setBounds(int, int, int, int)
	 * @category bounds accessing
	 */
	public void setBounds(int x, int y, int width, int height) {
		super.setBounds(x, y, width, height);
		this.makeFrames();
	}

}
