package jp.co.sra.jun.goodies.drawing;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Polygon;
import java.awt.RenderingHints;
import java.awt.event.KeyEvent;
import java.awt.image.BufferedImage;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;

import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.drawing.map.JunDrawingMap;
import jp.co.sra.jun.goodies.drawing.map.JunDrawingMapController;
import jp.co.sra.jun.goodies.drawing.map.JunDrawingMapModel;
import jp.co.sra.jun.system.framework.JunApplicationModel;

/**
 * JunDrawing class
 * 
 *  @author    m-asada
 *  @created   2005/06/07 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDrawing.java,v 8.10 2008/02/20 06:31:23 nisinaka Exp $
 */
public class JunDrawing extends JunApplicationModel {
	protected JunDrawingMapModel mapModel;
	protected JunButtonModel selectButton;
	protected JunButtonModel brushButton;
	protected JunButtonModel cameraButton;
	protected JunButtonModel ellipseButton;
	protected JunButtonModel linkButton;
	protected JunButtonModel pathButton;
	protected JunButtonModel rectangleButton;
	protected JunButtonModel roundRectangleButton;
	protected JunButtonModel textButton;
	protected JunButtonModel textboxButton;

	protected StMenuBar _menuBar;

	/**
	 * Create a new instance of <code>JunDrawing</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunDrawing() {
		super();
	}

	/**
	 * Create a new instance of <code>JunDrawing</code> and initialize it.
	 * 
	 * @param aMap jp.co.sra.jun.goodies.drawing.map.JunDrawingMap
	 * @category Instance creation
	 */
	public JunDrawing(JunDrawingMap aMap) {
		super();
		this.mapObject_(aMap);
	}

	/**
	 * Initialize the JunDrawingMapModel.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		mapModel = null;
		selectButton = null;
		brushButton = null;
		cameraButton = null;
		ellipseButton = null;
		linkButton = null;
		pathButton = null;
		rectangleButton = null;
		roundRectangleButton = null;
		textButton = null;
		textboxButton = null;
		_menuBar = null;
	}

	/**
	 * Remove references to objects that may refer to the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		super.release();
		this.mapModel().release();
	}

	/**
	 * Answer the receiver's map object.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.map.JunDrawingMap
	 * @category accessing
	 */
	public JunDrawingMap mapObject() {
		return this.mapModel().mapObject();
	}

	/**
	 * Set the receiver's map object.
	 * 
	 * @param aMapObject jp.co.sra.jun.goodies.drawing.map.JunDrawingMap
	 * @category accessing
	 */
	public void mapObject_(JunDrawingMap aMapObject) {
		this.mapModel().mapObject_(aMapObject);
	}

	/**
	 * Answer the receiver's map model.
	 * 
	 * @return mapModel jp.co.sra.jun.goodies.drawing.map.JunDrawingMapModel
	 * @category aspects
	 */
	public JunDrawingMapModel mapModel() {
		if (mapModel == null) {
			mapModel = new JunDrawingMapModel(null);
		}
		return mapModel;
	}

	/**
	 * Answer a view as JunDrawingView.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.JunDrawingView
	 * @category view accessing
	 */
	public JunDrawingView getDrawingView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			Object each = dependents[i];
			if (each instanceof StView && ((StView) each).model() == this && each instanceof JunDrawingView) {
				return (JunDrawingView) each;
			}
		}

		return null;
	}

	/**
	 * Get the select button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel selectButton() {
		if (selectButton == null) {
			selectButton = new JunButtonModel(false, JunCursors.NormalCursorImage(), new StBlockClosure() {
				public Object value_(Object obj) {
					JunDrawing.this.getDrawingView().drawingMapView().editMode_(JunDrawingMapController.EDITMODE_SELECT);
					return null;
				}
			});
		}
		return selectButton;
	}

	/**
	 * Get the brush button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel brushButton() {
		if (brushButton == null) {
			brushButton = new JunButtonModel(false, JunCursors.BrushCursorImage(), new StBlockClosure() {
				public Object value_(Object obj) {
					JunDrawing.this.getDrawingView().drawingMapView().editMode_(JunDrawingMapController.EDITMODE_ADD_FREEHAND);
					return null;
				}
			});
		}
		return brushButton;
	}

	/**
	 * Get the camera button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel cameraButton() {
		if (cameraButton == null) {
			cameraButton = new JunButtonModel(false, JunCursors.CameraCursorImage(), new StBlockClosure() {
				public Object value_(Object obj) {
					JunDrawing.this.getDrawingView().drawingMapView().editMode_(JunDrawingMapController.EDITMODE_ADD_IMAGE);
					return null;
				}
			});
		}
		return cameraButton;
	}

	/**
	 * Get the oval ellipse button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel ellipseButton() {
		if (ellipseButton == null) {
			BufferedImage icon = new BufferedImage(16, 16, BufferedImage.TYPE_INT_ARGB);
			Graphics2D aGraphics = (Graphics2D) icon.getGraphics();
			try {
				aGraphics.setColor(Color.black);
				aGraphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
				aGraphics.drawOval(0, 0, 15, 15);
			} finally {
				aGraphics.dispose();
			}
			ellipseButton = new JunButtonModel(false, icon, new StBlockClosure() {
				public Object value_(Object obj) {
					JunDrawing.this.getDrawingView().drawingMapView().editMode_(JunDrawingMapController.EDITMODE_ADD_ELIPSE);
					return null;
				}
			});
		}
		return ellipseButton;
	}

	/**
	 * Get the link button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel linkButton() {
		if (linkButton == null) {
			linkButton = new JunButtonModel(false, JunCursors.ArrowCursorImage(), new StBlockClosure() {
				public Object value_(Object obj) {
					JunDrawing.this.getDrawingView().drawingMapView().editMode_(JunDrawingMapController.EDITMODE_ADD_LINK);
					return null;
				}
			});
		}
		return linkButton;
	}

	/**
	 * Get the path button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel pathButton() {
		if (pathButton == null) {
			BufferedImage icon = new BufferedImage(16, 16, BufferedImage.TYPE_INT_ARGB);
			Graphics2D aGraphics = (Graphics2D) icon.getGraphics();
			try {
				aGraphics.setColor(Color.black);
				aGraphics.drawLine(1, 8, 14, 8);
			} finally {
				aGraphics.dispose();
			}
			pathButton = new JunButtonModel(false, icon, new StBlockClosure() {
				public Object value_(Object obj) {
					JunDrawing.this.getDrawingView().drawingMapView().editMode_(JunDrawingMapController.EDITMODE_ADD_PATH);
					return null;
				}
			});
		}
		return pathButton;
	}

	/**
	 * Get the rectangle button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel rectangleButton() {
		if (rectangleButton == null) {
			BufferedImage icon = new BufferedImage(16, 16, BufferedImage.TYPE_INT_ARGB);
			Graphics2D aGraphics = (Graphics2D) icon.getGraphics();
			try {
				aGraphics.setColor(Color.black);
				aGraphics.drawRect(0, 0, 15, 15);
			} finally {
				aGraphics.dispose();
			}
			rectangleButton = new JunButtonModel(false, icon, new StBlockClosure() {
				public Object value_(Object obj) {
					JunDrawing.this.getDrawingView().drawingMapView().editMode_(JunDrawingMapController.EDITMODE_ADD_RECTANGLE);
					return null;
				}
			});
		}
		return rectangleButton;
	}

	/**
	 * Get the round rectangle button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel roundRectangleButton() {
		if (roundRectangleButton == null) {
			BufferedImage icon = new BufferedImage(16, 16, BufferedImage.TYPE_INT_ARGB);
			Graphics2D aGraphics = (Graphics2D) icon.getGraphics();
			try {
				int[] xPoints = new int[] { 3, 12, 15, 15, 12, 3, 0, 0, 3 };
				int[] yPoints = new int[] { 0, 0, 3, 12, 15, 15, 12, 3, 0 };
				aGraphics.setColor(Color.black);
				aGraphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
				aGraphics.drawPolygon(new Polygon(xPoints, yPoints, xPoints.length));
			} finally {
				aGraphics.dispose();
			}
			roundRectangleButton = new JunButtonModel(false, icon, new StBlockClosure() {
				public Object value_(Object obj) {
					JunDrawing.this.getDrawingView().drawingMapView().editMode_(JunDrawingMapController.EDITMODE_ADD_ROUND_RECTANGLE);
					return null;
				}
			});
		}
		return roundRectangleButton;
	}

	/**
	 * Answer the text button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel textButton() {
		if (textButton == null) {
			BufferedImage icon = new BufferedImage(16, 16, BufferedImage.TYPE_INT_ARGB);
			Graphics2D aGraphics = (Graphics2D) icon.getGraphics();
			try {
				aGraphics.setColor(Color.black);
				aGraphics.setStroke(new BasicStroke(2));
				aGraphics.drawLine(2, 2, 14, 2);
				aGraphics.drawLine(8, 2, 8, 14);
			} finally {
				aGraphics.dispose();
			}
			textButton = new JunButtonModel(false, icon, new StBlockClosure() {
				public Object value_(Object obj) {
					JunDrawing.this.getDrawingView().drawingMapView().editMode_(JunDrawingMapController.EDITMODE_ADD_TEXT);
					return null;
				}
			});
		}
		return textButton;
	}

	/**
	 * Get the textbox button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel textboxButton() {
		if (textboxButton == null) {
			BufferedImage icon = new BufferedImage(16, 16, BufferedImage.TYPE_INT_ARGB);
			Graphics2D aGraphics = (Graphics2D) icon.getGraphics();
			try {
				aGraphics.setColor(Color.black);
				aGraphics.drawRect(0, 0, 15, 15);
				aGraphics.setStroke(new BasicStroke(2));
				aGraphics.drawLine(4, 4, 12, 4);
				aGraphics.drawLine(8, 4, 8, 12);
			} finally {
				aGraphics.dispose();
			}
			textboxButton = new JunButtonModel(false, icon, new StBlockClosure() {
				public Object value_(Object obj) {
					JunDrawing.this.getDrawingView().drawingMapView().editMode_(JunDrawingMapController.EDITMODE_ADD_TEXTBOX);
					return null;
				}
			});
		}
		return textboxButton;
	}

	/**
	 * Answer the default view of the model.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunDrawingViewAwt(this);

		} else {
			return new JunDrawingViewSwing(this);
		}
	}

	/**
	 * Answer the receiver's window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Drawing");
	}

	/**
	 * Update the menu indication.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#updateMenuIndication()
	 * @category menu accessing
	 */
	public void updateMenuIndication() {
		this.mapModel().updateMenuIndication();
	}

	/**
	 * Create new model and open it.
	 *
	 * @return jp.co.sra.jun.goodies.drawing.JunDrawing
	 * @category menu messages
	 */
	public JunDrawing newMap() {
		JunDrawing mapModel = new JunDrawing(new JunDrawingMap());
		mapModel.open();
		return mapModel;
	}

	/**
	 * Quit this window.
	 * 
	 * @category menu messages
	 */
	public void quitDoing() {
		this.closeRequest();
	}

	/**
	 * Answer the receiver's menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createFileMenu());
			_menuBar.add((StMenu) this.mapModel()._menuBar().atNameKey_($("viewMenu")));
			_menuBar.add((StMenu) this.mapModel()._menuBar().atNameKey_($("elementMenu")));
			_menuBar.add((StMenu) this.mapModel()._menuBar().atNameKey_($("miscMenu")));
		}
		return _menuBar;
	}

	/**
	 * Create a "File" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	public StMenu _createFileMenu() {
		StMenu mapModelMenu = (StMenu) this.mapModel()._menuBar().atNameKey_($("fileMenu"));

		StMenu fileMenu = new StMenu($String("File"), $("fileMenu"));
		fileMenu.add(new StMenuItem($String("New"), $("newMap"), new MenuPerformer(this, "newMap"), KeyEvent.VK_N, KeyEvent.CTRL_MASK));
		fileMenu.add(mapModelMenu.atNameKey_($("openMap")));
		fileMenu.addSeparator();
		fileMenu.add(mapModelMenu.atNameKey_($("saveMap")));
		fileMenu.add(mapModelMenu.atNameKey_($("saveAsMap")));
		fileMenu.addSeparator();
		fileMenu.add(new StMenuItem($String("Close"), $("quitDoing"), new MenuPerformer(this, "quitDoing")));
		return fileMenu;
	}

	/**
	 * Update the receiver's tool bar with specified edit mode.
	 * 
	 * @param newEditMode int
	 * @category updating
	 */
	public void updateToolbar_(int newEditMode) {
		this.selectButton().value_(false);
		this.textButton().value_(false);
		this.textboxButton().value_(false);
		this.cameraButton().value_(false);
		this.rectangleButton().value_(false);
		this.roundRectangleButton().value_(false);
		this.ellipseButton().value_(false);
		this.pathButton().value_(false);
		this.brushButton().value_(false);
		this.linkButton().value_(false);

		switch (newEditMode) {
			case JunDrawingMapController.EDITMODE_SELECT:
				break;
			case JunDrawingMapController.EDITMODE_ADD_TEXT:
				this.textButton().value_(true);
				break;
			case JunDrawingMapController.EDITMODE_ADD_TEXTBOX:
				this.textboxButton().value_(true);
				break;
			case JunDrawingMapController.EDITMODE_ADD_IMAGE:
				this.cameraButton().value_(true);
				break;
			case JunDrawingMapController.EDITMODE_ADD_RECTANGLE:
				this.rectangleButton().value_(true);
				break;
			case JunDrawingMapController.EDITMODE_ADD_ROUND_RECTANGLE:
				this.roundRectangleButton().value_(true);
				break;
			case JunDrawingMapController.EDITMODE_ADD_ELIPSE:
				this.ellipseButton().value_(true);
				break;
			case JunDrawingMapController.EDITMODE_ADD_PATH:
				this.pathButton().value_(true);
				break;
			case JunDrawingMapController.EDITMODE_ADD_FREEHAND:
				this.brushButton().value_(true);
				break;
			case JunDrawingMapController.EDITMODE_ADD_LINK:
				this.linkButton().value_(true);
				break;
		}
	}
}
