package jp.co.sra.jun.goodies.drawing.element;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import java.io.Writer;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StModel;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.goodies.lisp.JunLispNil;

/**
 * JunDrawingVisual class
 * 
 *  @author    m-asada
 *  @created   2005/03/01 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDrawingVisual.java,v 8.11 2008/02/20 06:31:23 nisinaka Exp $
 */
public abstract class JunDrawingVisual extends StModel implements Transferable, StDisplayable {
	protected long id;

	protected static final SimpleDateFormat _UniqueNumberDateFormat = new SimpleDateFormat("yyyyMMddHHmmssSSS");
	protected static long _UniqueNumberGenerationTime = Long.MIN_VALUE;

	/**
	 * Generated the unique number and answer it.
	 * 
	 * @return long
	 * @category Utilities
	 */
	public static long GenerateUniqueNumber() {
		long currentTime;
		do {
			currentTime = System.currentTimeMillis();
		} while (_UniqueNumberGenerationTime >= currentTime);
		_UniqueNumberGenerationTime = currentTime;
		return Long.parseLong(_UniqueNumberDateFormat.format(new Date(_UniqueNumberGenerationTime)));
	}

	/**
	 * Create a new instance of JunDrawingVisual and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunDrawingVisual() {
		super();
		this.initialize();
		this.assignId();
	}

	/**
	 * Create a new instance of JunDrawingVisual and initialize it.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunDrawingVisual(JunLispList aList) {
		super();
		this.initialize();
		this.fromLispList_(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		id = Long.MIN_VALUE;
	}

	/**
	 * Assign a new id and answer it.
	 * 
	 * @return long
	 * @category initialize-release
	 */
	public long assignId() {
		id = GenerateUniqueNumber();
		return id;
	}

	/**
	 * Answer the receiver's id.
	 *
	 * @return long
	 * @category accessing
	 */
	public long id() {
		if (this.hasId() == false) {
			this.assignId();
		}
		return id;
	}

	/**
	 * Set the receiver's id.
	 *
	 * @param newId long
	 * @category accessing
	 */
	public void id_(long newId) {
		id = newId;
	}

	/**
	 * Answer the receiver's rectangle.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.smalltalk.StDisplayable#bounds()
	 * @category bounds accessing
	 */
	public Rectangle bounds() {
		return this.preferredBounds();
	}

	/**
	 * Set the receiver's rectangle.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @category bounds accessing
	 */
	public void bounds_(Rectangle aRectangle) {
		this.location_(aRectangle.getLocation());
		this.extent_(aRectangle.getSize());
	}

	/**
	 * Answer the receiver's preferred bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @category bounds accessing
	 */
	public Rectangle preferredBounds() {
		return new Rectangle(0, 0, 0, 0);
	}

	/**
	 * Answer the receiver's x value.
	 * 
	 * @return int
	 * @category bounds accessing
	 */
	public int x() {
		return this.bounds().x;
	}

	/**
	 * Answer the receiver's y value.
	 * 
	 * @return int
	 * @category bounds accessing
	 */
	public int y() {
		return this.bounds().y;
	}

	/**
	 * Answer the receiver's width size.
	 * 
	 * @return int
	 * @category bounds accessing
	 */
	public int width() {
		return this.bounds().width;
	}

	/**
	 * Answer the receiver's height size.
	 * 
	 * @return int
	 * @category bounds accessing
	 */
	public int height() {
		return this.bounds().height;
	}

	/**
	 * Answer the receiver's location point.
	 * 
	 * @return java.awt.Point
	 * @category bounds accessing
	 */
	public Point location() {
		return this.bounds().getLocation();
	}

	/**
	 * Set the receiver's location point.
	 * 
	 * @param aPoint java.awt.Point
	 * @category bounds accessing
	 */
	public abstract void location_(Point aPoint);

	/**
	 * Set the receiver's location point.
	 * 
	 * @param x int
	 * @param y int
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#location_(java.awt.Point)
	 * @category bounds accessing
	 */
	public void location_(int x, int y) {
		this.location_(new Point(x, y));
	}

	/**
	 * Answer the receiver's extent size.
	 * 
	 * @return java.awt.Dimension
	 * @category bounds accessing
	 */
	public Dimension extent() {
		return this.bounds().getSize();
	}

	/**
	 * Answer the receiver's extent point.
	 * 
	 * @return java.awt.Point
	 * @category bounds accessing
	 */
	public Point extentPoint() {
		return this.isRectangular() ? new Point(this.x() + this.width() - 1, this.y() + this.height() - 1) : new Point(this.x() + this.width(), this.y() + this.height());
	}

	/**
	 * Set the receiver's extent size.
	 * 
	 * @param aDimension java.awt.Dimension
	 * @category bounds accessing
	 */
	public abstract void extent_(Dimension aDimension);

	/**
	 * Set the receiver's extent size.
	 * 
	 * @param width int
	 * @param height int
	 * @category bounds accessing
	 */
	public void extent_(int width, int height) {
		this.extent_(new Dimension(width, height));
	}

	/**
	 * Answer the receiver's center point.
	 * 
	 * @return java.awt.Point
	 * @category bounds accessing
	 */
	public Point center() {
		return new Point(this.x() + this.width() / 2, this.y() + this.height() / 2);
	}

	/**
	 * Answer the receiver as StImage.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.smalltalk.StDisplayable#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		int width = this.width();
		int height = this.height();
		StImage anImage = new StImage(width, height);
		Graphics2D aGraphics = null;
		try {
			aGraphics = (Graphics2D) anImage.image().getGraphics();
			aGraphics.setClip(0, 0, width, height);
			aGraphics.setColor(Color.white);
			aGraphics.fillRect(0, 0, width, height);
			aGraphics.setColor(Color.black);
			this.displayOn_(aGraphics);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
		return anImage;
	}

	/**
	 * Answer a copy of receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#copy()
	 * @category copying
	 */
	public StObject copy() {
		return this.copy(new HashMap());
	}

	/**
	 * Answer a copy of receiver using the context.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @category copying
	 */
	public JunDrawingVisual copy(Map context) {
		JunDrawingVisual copy = (JunDrawingVisual) this.shallowCopy();
		context.put(this, copy);
		return copy.postCopy(context);
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#postCopy()
	 * @category copying
	 */
	public StObject postCopy() {
		return this.postCopy(new HashMap());
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		super.postCopy();
		if (this.hasId()) {
			this.assignId();
		}
		return this;
	}

	/**
	 * Get the default location.
	 *
	 * @return java.awt.Point
	 * @category defaults
	 */
	public Point defaultLocation() {
		return new Point(0, 0);
	}

	/**
	 * Get the default extent.
	 *
	 * @return java.awt.Dimension
	 * @category defaults
	 */
	public Dimension defaultExtent() {
		return new Dimension(96, 48);
	}

	/**
	 * Get the default minimum extent.
	 *
	 * @return java.awt.Dimension
	 * @category defaults
	 */
	public Dimension defaultMinimumExtent() {
		return new Dimension(20, 20);
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public abstract void displayOn_(Graphics aGraphics);

	/**
	 * Display the receiver on the graphics translated by a point.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		aGraphics.translate(aPoint.x, aPoint.y);
		try {
			this.displayOn_(aGraphics);
		} finally {
			aGraphics.translate(-aPoint.x, -aPoint.y);
		}
	}

	/**
	 * Answer the nearest point on the receiver from the point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point
	 * @category functions
	 */
	public Point nearestPointFromPoint_(Point aPoint) {
		Jun2dPoint centerPoint = new Jun2dPoint(this.center());
		Jun2dPoint anotherPoint = new Jun2dPoint(aPoint);

		double originalDistance = centerPoint.distance_(anotherPoint);
		if (originalDistance < JunGeometry.Accuracy()) {
			return centerPoint._toPoint();
		}

		Jun2dPoint[] points = new Jun2dBoundingBox(this.bounds()).intersectingPointsWithLine_(new Jun2dLine(centerPoint, anotherPoint));
		if (points != null && points.length > 0) {
			for (int i = 0; i < points.length; i++) {
				double distance = points[i].distance_(anotherPoint);
				if (distance < originalDistance) {
					return points[i]._toPoint();
				}
			}
		}
		return aPoint;
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException if failed.
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("a ");
		aWriter.write(this._className().toString());
		aWriter.write("(id=");
		aWriter.write(String.valueOf(id));
		aWriter.write(")");
	}

	/**
	 * Answer true if receiver's bounding box contains the specified a point, otherwise false.
	 * 
	 * @param aPoint java.awt.Point
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(Point aPoint) {
		return this.bounds().contains(aPoint);
	}

	/**
	 * Answer true if receiver's element intersects the specified a rectangle, otherwise false.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @return boolean
	 * @category testing
	 */
	public boolean intersects_(Rectangle aRectangle) {
		return this.bounds().intersects(aRectangle);
	}

	/**
	 * Answer true if receiver is composite element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isComposite() {
		return false;
	}

	/**
	 * Answer true if receiver is element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isElement() {
		return false;
	}

	/**
	 * Answer true if receiver is image element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isImage() {
		return false;
	}

	/**
	 * Answer true if receiver is label element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isLabel() {
		return false;
	}

	/**
	 * Answer true if receiver is link element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isLink() {
		return false;
	}

	/**
	 * Answer true if receiver is map, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isMap() {
		return false;
	}

	/**
	 * Answer true if receiver is node element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isNode() {
		return false;
	}

	/**
	 * Answer true if receiver is path element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPath() {
		return false;
	}

	/**
	 * Answer true if receiver is rectangular element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isRectangular() {
		return false;
	}

	/**
	 * Answer true if receiver is vertexes element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isVertexes() {
		return false;
	}

	/**
	 * Answer true if the receiver has an id, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasId() {
		return id != Long.MIN_VALUE;
	}

	/**
	 * Answer true if receiver has a label element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasLabel() {
		return false;
	}

	/**
	 * Returns an array of DataFlavor objects indicating the flavors the data can be provided in. The array should be
	 * ordered according to preference for providing the data (from most richly descriptive to least descriptive).
	 * 
	 * @return java.awt.datatransfer.DataFlavor[]
	 * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
	 * @category transfering
	 */
	public abstract DataFlavor[] getTransferDataFlavors();

	/**
	 * Returns whether or not the specified data flavor is supported for this object.
	 * 
	 * @param flavor java.awt.datatransfer.DataFlavor
	 * @return boolean
	 * @see java.awt.datatransfer.Transferable#isDataFlavorSupported(java.awt.datatransfer.DataFlavor)
	 * @category transfering
	 */
	public boolean isDataFlavorSupported(DataFlavor flavor) {
		DataFlavor[] flavors = this.getTransferDataFlavors();
		for (int i = 0; i < flavors.length; i++) {
			if (flavors[i].equals(flavor)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Returns an object which represents the data to be transferred. The class of the object returned is defined by the
	 * representation class of the flavor.
	 * 
	 * @param flavor java.awt.datatransfer.DataFlavor
	 * @return java.lang.Object
	 * @throws java.io.IOException
	 * @throws java.awt.datatransfer.UnsupportedFlavorException
	 * @see java.awt.datatransfer.Transferable#getTransferData(java.awt.datatransfer.DataFlavor)
	 * @category transfering
	 */
	public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException {
		if (this.isDataFlavorSupported(flavor) == false) {
			throw new UnsupportedFlavorException(flavor);
		}
		return this;
	}

	/**
	 * Open the receiver's properties
	 * 
	 * @category visual properties
	 */
	public void openProperties() {
	}

	/**
	 * Answer the kind name of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category lisp support
	 */
	protected StSymbol kindName() {
		return this._className();
	}

	/**
	 * Answer a new lisp cons cell.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons lispCons() {
		return JunLispCons.Cell();
	}

	/**
	 * Answer a new lisp nil.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispNil lispNil() {
		return JunLispNil.NullList();
	}

	/**
	 * Convert receiver to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());
		list.add_(this.idToLispList());
		return list;
	}

	/**
	 * Convert the id to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons idToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("id"));
		list.tail_(new Long(this.id()));
		return list;
	}

	/**
	 * Get the receiver from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		this.idFromLispList_(aList);
	}

	/**
	 * Get the receiver's id from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void idFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("id"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.id_(((Number) list.tail()).longValue());
	}
}
