package jp.co.sra.jun.goodies.drawing.element;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.datatransfer.DataFlavor;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.Map;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StImage;

import jp.co.sra.jun.goodies.image.support.JunImageProcessor;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunImageElement class
 * 
 *  @author    m-asada
 *  @created   2005/11/03 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunImageElement.java,v 8.11 2008/02/20 06:31:23 nisinaka Exp $
 */
public class JunImageElement extends JunRectangularElement {
	protected File file;
	protected StImage image;
	protected transient BufferedImage scaledImage;

	public static DataFlavor DataFlavor = new DataFlavor(JunImageElement.class, "JunImageElement");

	/**
	 * Create a new instance of <code>JunImageElement</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunImageElement() {
		super();
	}

	/**
	 * Create a new instance of <code>JunImageElement</code> and initialize it.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunImageElement(JunLispList aList) {
		super(aList);
	}

	/**
	 * Answer the receiver's file.
	 * 
	 * @return filename
	 * @category accessing
	 */
	public File file() {
		return file;
	}

	/**
	 * Set the receiver's file.
	 * 
	 * @param aFile java.io.File
	 * @category accessing
	 */
	public void file_(File aFile) {
		file = aFile;
		image = null;
		scaledImage = null;
	}

	/**
	 * Set the receiver's filename.
	 * 
	 * @param filename java.lang.String
	 * @category accessing
	 */
	public void filename_(String filename) {
		File aFile = new File(filename);
		if (aFile.exists() || aFile.isFile()) {
			this.file_(aFile);
		}
	}

	/**
	 * Answer the receiver's image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	public StImage image() {
		if (image == null && this.file() != null) {
			image = JunImageProcessor.ImageFromFile_(this.file());
		}
		return image;
	}

	/**
	 * Set the receiver's image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	public void image_(StImage anImage) {
		if (image != anImage) {
			image = anImage;
			file = null;
			scaledImage = null;
		}
	}

	/**
	 * Answer the receiver's scaled image.
	 * 
	 * @return java.awt.image.BufferedImage
	 * @category accessing
	 */
	protected BufferedImage scaledImage() {
		if (this.image() != null && (scaledImage == null || scaledImage.getWidth() != this.width() || scaledImage.getHeight() != this.height())) {
			BufferedImage originalImage = this.image().image();
			int originalWidth = originalImage.getWidth();
			int originalHeight = originalImage.getHeight();
			int scaledWidth = this.width();
			int scaledHeight = this.height();
			if (((float) originalWidth / originalHeight) > ((float) scaledWidth / scaledHeight)) {
				scaledHeight = Math.round((float) scaledWidth * originalHeight / originalWidth);
			} else {
				scaledWidth = Math.round((float) scaledHeight * originalWidth / originalHeight);
			}
			AffineTransformOp imageOp = new AffineTransformOp(AffineTransform.getScaleInstance(scaledWidth / (double) originalWidth, scaledHeight / (double) originalHeight), AffineTransformOp.TYPE_NEAREST_NEIGHBOR);
			scaledImage = imageOp.filter(originalImage, null);
		}
		return scaledImage;
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		super.postCopy(context);
		scaledImage = null;

		return this;
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		Rectangle clipBounds = aGraphics.getClipBounds();
		if (clipBounds != null && clipBounds.intersects(this.bounds()) == false) {
			return;
		}

		Graphics graphicsContext = aGraphics.create();
		try {
			clipBounds = (clipBounds == null) ? this.bounds() : this.bounds().intersection(clipBounds);
			graphicsContext.setClip(clipBounds);
			graphicsContext.setColor(new Color(32, 32, 32, 32));
			graphicsContext.fillRect(clipBounds.x, clipBounds.y, clipBounds.width, clipBounds.height);
			if (this.scaledImage() != null) {
				BufferedImage scaledImage = this.scaledImage();
				int deltaX = (this.width() == scaledImage.getWidth()) ? 0 : ((this.width() - scaledImage.getWidth()) / 2);
				int deltaY = (this.height() == scaledImage.getHeight()) ? 0 : ((this.height() - scaledImage.getHeight()) / 2);
				graphicsContext.drawImage(scaledImage, this.x() + deltaX, this.y() + deltaY, null);
			} else {
				StComposedText composedText = new StComposedText("no image");
				if (composedText.width() > this.width()) {
					composedText = new StComposedText("no image", this.width());
				}
				int offsetX = (composedText.width() == this.width()) ? 0 : (this.width() - composedText.width()) / 2;
				int offsetY = (this.height() - composedText.height()) / 2;
				graphicsContext.setColor(Color.black);
				composedText.displayOn_at_(graphicsContext, new Point(this.x() + offsetX, this.y() + offsetY));
			}
		} finally {
			if (graphicsContext != null) {
				graphicsContext.dispose();
			}
		}
	}

	/**
	 * Initialize the JunDrawingVisual.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		file = null;
		image = null;
		scaledImage = null;
	}

	/**
	 * Returns an array of DataFlavor objects indicating the flavors the data can be provided in. The array should be
	 * ordered according to preference for providing the data (from most richly descriptive to least descriptive).
	 * 
	 * @return java.awt.datatransfer.DataFlavor[]
	 * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#getTransferDataFlavors()
	 * @category transfering
	 */
	public DataFlavor[] getTransferDataFlavors() {
		return new DataFlavor[] { JunPathElement.DataFlavor };
	}

	/**
	 * Answer true if receiver is image element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#isImage()
	 * @category testing
	 */
	public boolean isImage() {
		return true;
	}

	/**
	 * Convert the element to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();
		if (this.file() != null) {
			list.add_(this.fileToLispList());
		} else if (this.image() != null) {
			list.add_(this.imageToLispList());
		}
		return list;
	}

	/**
	 * Convert the image file to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons fileToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("file"));
		list.tail_(this.file().getPath());
		return list;
	}

	/**
	 * Convert the image to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons imageToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("image"));
		list.tail_(this.image());
		return list;
	}

	/**
	 * Get the receiver from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#fromLispList_(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		super.fromLispList_(aList);
		this.fileFromLispList_(aList);
		this.imageFromLispList_(aList);
	}

	/**
	 * Get the image from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void fileFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("file"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.filename_((String) list.tail());
	}

	/**
	 * Get the image from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void imageFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("image"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.image_((StImage) list.tail());
	}
}
