package jp.co.sra.jun.goodies.drawing.element;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.datatransfer.DataFlavor;
import java.io.IOException;
import java.io.Writer;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.goodies.drawing.properties.JunDrawingElementPropertiesModel;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunLinkElement class
 * 
 *  @author    m-asada
 *  @created   2005/03/01 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunLinkElement.java,v 8.11 2008/02/20 06:31:23 nisinaka Exp $
 */
public class JunLinkElement extends JunDrawingElement {
	protected JunNodeElement fromElement;
	protected JunNodeElement toElement;
	protected JunPathElement pathElement;
	protected JunLabelElement labelElement;

	protected transient long fromElementId;
	protected transient long toElementId;
	protected transient long labelElementId;

	protected transient int[] nearestPointIndexesFromLabel;
	protected transient Point labelElementDeltaPoint;

	public static DataFlavor DataFlavor = new DataFlavor(JunLinkElement.class, "JunLinkElement");

	/**
	 * Create a new instance of JunLinkElement and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunLinkElement() {
		super();
	}

	/**
	 * Create a new instance of JunLinkElement and initialize it.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunLinkElement(JunLispList aList) {
		super(aList);
	}

	/**
	 * Create a new instance of JunLinkElement and initialize it.
	 * 
	 * @param newFromElement jp.co.sra.jun.goodies.drawing.element.JunNodeElement
	 * @param newToElement jp.co.sra.jun.goodies.drawing.element.JunNodeElement
	 * @throws java.lang.IllegalArgumentException
	 * @category Instance creation
	 */
	public JunLinkElement(JunNodeElement newFromElement, JunNodeElement newToElement) {
		this(newFromElement, newToElement, null);
	}

	/**
	 * Create a new instance of JunLinkElement and initialize it.
	 * 
	 * @param newFromElement jp.co.sra.jun.goodies.drawing.element.JunNodeElement
	 * @param newToElement jp.co.sra.jun.goodies.drawing.element.JunNodeElement
	 * @param newPathElement jp.co.sra.jun.goodies.drawing.element.JunPathElement
	 * @throws java.lang.IllegalArgumentException
	 * @category Instance creation
	 */
	public JunLinkElement(JunNodeElement newFromElement, JunNodeElement newToElement, JunPathElement newPathElement) {
		this(newFromElement, newToElement, newPathElement, null);
	}

	/**
	 * Create a new instance of JunLinkElement and initialize it.
	 * 
	 * @param newFromElement jp.co.sra.jun.goodies.drawing.element.JunNodeElement
	 * @param newToElement jp.co.sra.jun.goodies.drawing.element.JunNodeElement
	 * @param newPathElement jp.co.sra.jun.goodies.drawing.element.JunPathElement
	 * @param newLabelElement jp.co.sra.jun.goodies.drawing.element.JunLabelElement
	 * @throws java.lang.IllegalArgumentException
	 * @category Instance creation
	 */
	public JunLinkElement(JunNodeElement newFromElement, JunNodeElement newToElement, JunPathElement newPathElement, JunLabelElement newLabelElement) {
		this();
		if (newFromElement == null || newToElement == null) {
			throw new IllegalArgumentException("The two nodes must be exist.");
		}
		this.fromElement_(newFromElement);
		this.toElement_(newToElement);
		this.pathElement_(newPathElement);
		this.labelElement_(newLabelElement);
		this.updateFromToPoint();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		fromElement = null;
		toElement = null;
		pathElement = null;
		labelElement = null;
		fromElementId = Long.MIN_VALUE;
		toElementId = Long.MIN_VALUE;
		labelElementId = Long.MIN_VALUE;
		labelElementDeltaPoint = null;
		nearestPointIndexesFromLabel = null;
	}

	/**
	 * Answer the reciver's from element.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunNodeElement
	 * @throws java.lang.IllegalStateException
	 * @category accessing
	 */
	public JunNodeElement fromElement() {
		if (fromElement == null && this.fromElementId() > Long.MIN_VALUE) {
			JunDrawingElement anElement = this.map().findElement_(this.fromElementId());
			if (anElement != null && anElement.isNode()) {
				this.fromElement_((JunNodeElement) anElement);
			} else {
				throw new IllegalStateException("the from element id '" + this.fromElementId() + "' don't find this map object.");
			}
		}
		return fromElement;
	}

	/**
	 * Set the reciver's from element.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunNodeElement
	 * @category accessing
	 */
	public void fromElement_(JunNodeElement anElement) {
		fromElement = anElement;
		if (fromElement != null) {
			this.fromElementId_(Long.MIN_VALUE);
		}
	}

	/**
	 * Answer the reciver's to element.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunNodeElement
	 * @throws java.lang.IllegalStateException
	 * @category accessing
	 */
	public JunNodeElement toElement() {
		if (toElement == null && this.toElementId() > Long.MIN_VALUE) {
			JunDrawingElement anElement = this.map().findElement_(this.toElementId());
			if (anElement != null && anElement.isNode()) {
				this.toElement_((JunNodeElement) anElement);
			} else {
				throw new IllegalStateException("the to element id '" + this.toElementId() + "' don't find this map object.");
			}
		}
		return toElement;
	}

	/**
	 * Set the reciver's to element.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunNodeElement
	 * @category accessing
	 */
	public void toElement_(JunNodeElement anElement) {
		toElement = anElement;
		if (toElement != null) {
			this.toElementId_(Long.MIN_VALUE);
		}
	}

	/**
	 * Answer the reciver's path element.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunPathElement
	 * @category accessing
	 */
	public JunPathElement pathElement() {
		if (pathElement == null) {
			JunPathElement newPath = new JunPathElement();
			newPath.add_(this.fromElement().center());
			newPath.add_(this.toElement().center());
			newPath.beginStyle_(JunPathElement.ARROW_STYLE_SQUARE);
			newPath.endStyle_(JunPathElement.ARROW_STYLE_ARROW);
			this.pathElement_(newPath);
		}
		return pathElement;
	}

	/**
	 * Set the reciver's path element.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunPathElement
	 * @category accessing
	 */
	public void pathElement_(JunPathElement anElement) {
		if (anElement != null) {
			pathElement = anElement;
			pathElement.parent_(this);
		}
	}

	/**
	 * Answer the receiver's path points.
	 * 
	 * @return java.util.List
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#points()
	 * @category accessing
	 */
	public List points() {
		return this.pathElement().points();
	}

	/**
	 * Set the receiver's path points.
	 * 
	 * @param newPoints java.util.List
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#points_(java.util.ArrayList)
	 * @category accessing
	 */
	public void points_(List newPoints) {
		int size = this.pathElement().points().size();
		this.pathElement().points_(newPoints);
		if (newPoints.size() != size) {
			this.flushLabelElementPoints();
		}
		this.flushBounds();
	}

	/**
	 * Answer the array of receiver's path points.
	 * 
	 * @return java.awt.Point[]
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#_points()
	 * @category accessing
	 */
	public Point[] _points() {
		return this.pathElement()._points();
	}

	/**
	 * Replaces the element at the specified position in this points with the specified point.
	 * 
	 * @param anInteger int
	 * @param aPoint java.awt.Point
	 * @throws java.lang.IndexOutOfBoundsException
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#at_put_(int, java.awt.Point)
	 * @category accessing
	 */
	public void at_put_(int anInteger, Point aPoint) {
		this.pathElement().at_put_(anInteger, aPoint);
		this.flushBounds();
	}

	/**
	 * Answer the receiver's controll point with the specified position.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @return aPoint java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#controllPointAt_(jp.co.sra.smalltalk.StSymbol)
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#controllPointAt_(jp.co.sra.smalltalk.aSymbol)
	 * @category bounds accessing
	 */
	public Point controllPointAt_(StSymbol aSymbol) {
		return this.pathElement().controllPointAt_(aSymbol);
	}

	/**
	 * Replaces the receiver's controll point with the specified position.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @param aPoint java.awt.Point
	 * @throws java.lang.IllegalArgumentException
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#controllPointAt_put_(jp.co.sra.smalltalk.StSymbol, java.awt.Point)
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#controllPointAt_put_(jp.co.sra.smalltalk.aSymbol, java.awt.Point)
	 * @category bounds accessing
	 */
	public void controllPointAt_put_(StSymbol aSymbol, Point aPoint) {
		this.pathElement().controllPointAt_put_(aSymbol, aPoint);
		this.flushBounds();
	}

	/**
	 * Answer the point at the specified position in this points.
	 * 
	 * @param anIndex int
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#pointAt_(int)
	 * @category accessing
	 */
	public Point pointAt_(int anIndex) {
		return this.pathElement().pointAt_(anIndex);
	}

	/**
	 * Answer the receiver's path begin points.
	 * 
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#begin()
	 * @category accessing
	 */
	public Point begin() {
		return this.pathElement.begin();
	}

	/**
	 * Set the receiver's path begin points.
	 * 
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#begin_(java.awt.Point)
	 * @category accessing
	 */
	public void begin_(Point aPoint) {
		this.pathElement().begin_(aPoint);
		this.flushBounds();
	}

	/**
	 * Answer the receiver's path end points.
	 * 
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#end()
	 * @category accessing
	 */
	public Point end() {
		return this.pathElement().end();
	}

	/**
	 * Set the receiver's path end points.
	 * 
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#end_(java.awt.Point)
	 * @category accessing
	 */
	public void end_(Point aPoint) {
		this.pathElement().end_(aPoint);
		this.flushBounds();
	}

	/**
	 * Answer the reciver's label element.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunLabelElement
	 * @throws java.lang.IllegalStateException
	 * @category accessing
	 */
	public JunLabelElement labelElement() {
		if (labelElement == null && this.labelElementId() > Long.MIN_VALUE) {
			JunDrawingElement anElement = this.map().findElement_(this.labelElementId());
			if (anElement != null && anElement.isLabel()) {
				this.labelElement_((JunLabelElement) anElement);
			} else {
				throw new IllegalStateException("the label element id '" + this.labelElementId() + "' don't find this map object.");
			}
		}
		return labelElement;
	}

	/**
	 * Set the reciver's label element.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunLabelElementt
	 * @category accessing
	 */
	public void labelElement_(JunLabelElement anElement) {
		if (labelElement != null) {
			labelElement.baseElement_(null);
		}
		labelElement = anElement;
		this.labelElementId_(Long.MIN_VALUE);
		this.labelElementDeltaPoint_(null);
		if (labelElement != null) {
			labelElement.baseElement_(this);
		}
	}

	/**
	 * Add the new point on receiver's line segment.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#addOnLine_(java.awt.Point)
	 * @category adding
	 */
	public Point addOnLine_(Point aPoint) {
		Point point = this.pathElement().addOnLine_(aPoint);
		this.flushLabelElementPoints();
		this.updateFromToPoint();
		return point;
	}

	/**
	 * Set the receiver's location point.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#location_(java.awt.Point)
	 * @category bounds accessing
	 */
	public void location_(Point aPoint) {
		// no operation
	}

	/**
	 * Set the receiver's extent size.
	 * 
	 * @param aDimension java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#extent_(java.awt.Dimension)
	 * @category bounds accessing
	 */
	public void extent_(Dimension aDimension) {
		// no operation
	}

	/**
	 * Answer the receiver's preferred bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#preferredBounds()
	 * @category bounds accessing
	 */
	public Rectangle preferredBounds() {
		return this.pathElement().preferredBounds();
	}

	/**
	 * Answer the collection of receiver's controll point.
	 * 
	 * @return java.awt.HashMap
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#controllPoints()
	 * @category bounds accessing
	 */
	public HashMap controllPoints() {
		return this.pathElement().controllPoints();
	}

	/**
	 * Answer true if receiver's from/to/label element is exist, otherwise false.
	 * 
	 * @return boolean
	 * @category checking
	 */
	public boolean checkElement() {
		try {
			this.fromElement();
			this.toElement();
			this.labelElement();
			return true;
		} catch (IllegalStateException e) {
			return false;
		}
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#postCopy(java.util.Map)
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		JunNodeElement from = this.fromElement();
		JunNodeElement to = this.toElement();
		JunLabelElement label = this.labelElement();

		super.postCopy(context);

		if (from != null) {
			from = (JunNodeElement) (context.containsKey(from) ? context.get(from) : from.copy(context));
			fromElement = from;
			fromElementId = from.id();
		}
		if (to != null) {
			to = (JunNodeElement) (context.containsKey(to) ? context.get(to) : to.copy(context));
			toElement = to;
			toElementId = to.id();
		}
		this.pathElement_((JunPathElement) this.pathElement().copy(context));
		if (label != null) {
			label = (JunLabelElement) (context.containsKey(label) ? context.get(label) : label.copy(context));
			labelElement = label;
			labelElementId = label.id();
			label.baseElement_(this);
			this.flushLabelElementPoints();
		}
		return this;
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		this.pathElement().displayOn_(aGraphics);
	}

	/**
	 * Flush the receiver's preferred bounds.
	 * 
	 * @category flushing
	 */
	protected void flushBounds() {
		if (this.labelElement() != null) {
			int[] indexes = this.nearestPointIndexesFromLabel();
			Point targetPoint = new Jun2dLine(new Jun2dPoint(this.pointAt_(indexes[0])), new Jun2dPoint(this.pointAt_(indexes[1]))).center()._toPoint();
			Point deltaPoint = this.labelElementDeltaPoint();
			this.labelElement().moveLabel_(new Point(targetPoint.x + deltaPoint.x, targetPoint.y + deltaPoint.y));
		}
	}

	/**
	 * Flush the receiver's label element points.
	 * 
	 * @return newIndexes int[]
	 * @category flushing
	 */
	public void flushLabelElementPoints() {
		nearestPointIndexesFromLabel = null;
		labelElementDeltaPoint = null;
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException if failed.
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("a ");
		aWriter.write(this._className().toString());
		aWriter.write("(id=");
		aWriter.write(String.valueOf(id));
		aWriter.write(", (");
		this.fromElement().printOn_(aWriter);
		aWriter.write(" -> ");
		this.toElement().printOn_(aWriter);
		aWriter.write("))");
	}

	/**
	 * Remove the point in this points.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#remove(java.awt.Point)
	 * @category removing
	 */
	public Point remove_(Point aPoint) {
		Point point = this.pathElement().remove_(aPoint);
		this.flushLabelElementPoints();
		this.updateFromToPoint();
		return point;
	}

	/**
	 * Answer true if receiver is link element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#isLink()
	 * @category testing
	 */
	public boolean isLink() {
		return true;
	}

	/**
	 * Answer true if receiver has a label element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasLabel() {
		return this.labelElement() != null;
	}

	/**
	 * Answer true if receiver's bounding box contains the specified a point, otherwise false.
	 * 
	 * @param aPoint java.awt.Point
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#containsPoint_(java.awt.Point)
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#containsPoint_(java.awt.Point)
	 * @category testing
	 */
	public boolean containsPoint_(Point aPoint) {
		return this.pathElement().containsPoint_(aPoint);
	}

	/**
	 * Answer true if receiver's element intersects the specified a rectangle, otherwise false.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#intersects_(java.awt.Rectangle)
	 * @see jp.co.sra.jun.goodies.drawing.element.JunPathElement#intersects_(java.awt.Rectangle)
	 * @category testing
	 */
	public boolean intersects_(Rectangle aRectangle) {
		return this.pathElement().intersects_(aRectangle);
	}

	/**
	 * Returns an array of DataFlavor objects indicating the flavors the data can be provided in. The array should be
	 * ordered according to preference for providing the data (from most richly descriptive to least descriptive).
	 * 
	 * @return java.awt.datatransfer.DataFlavor[]
	 * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#getTransferDataFlavors()
	 * @category transfering
	 */
	public DataFlavor[] getTransferDataFlavors() {
		return new DataFlavor[] { JunLinkElement.DataFlavor };
	}

	/**
	 * Updating receiver's from-element point and to-element point.
	 * 
	 * @category updating
	 */
	public void updateFromToPoint() {
		if (this.fromElement() == null || this.toElement() == null) {
			return;
		}
		Point fromPoint, toPoint;
		if (this.pathElement().points().size() <= 2) {
			fromPoint = this.fromElement().nearestPointFromPoint_(this.toElement().center());
			toPoint = this.toElement().nearestPointFromPoint_(this.fromElement().center());
		} else {
			fromPoint = this.fromElement().nearestPointFromPoint_(this.pathElement().pointAt_(1));
			toPoint = this.toElement().nearestPointFromPoint_(this.pathElement().pointAt_(this.pathElement().points().size() - 2));
		}
		this.begin_(fromPoint);
		this.end_(toPoint);
	}

	/**
	 * Answer the receiver's properties model.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.properties.JunDrawingElementPropertiesModel
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#propertiesModel()
	 * @category visual properties
	 */
	public JunDrawingElementPropertiesModel propertiesModel() {
		return this.pathElement().propertiesModel();
	}

	/**
	 * Convert the receiver to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();
		list.add_(this.fromElementToLispList());
		list.add_(this.toElementToLispList());
		list.add_(this.pathElementToLispList());
		if (this.hasLabel()) {
			list.add_(this.labelElementToLispList());
		}
		return list;
	}

	/**
	 * Convert the from element to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons fromElementToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("from"));
		list.tail_(new Long(this.fromElement().id()));
		return list;
	}

	/**
	 * Convert the to element to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons toElementToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("to"));
		list.tail_(new Long(this.toElement().id()));
		return list;
	}

	/**
	 * Convert the path element to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons pathElementToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("path"));
		list.tail_(this.pathElement().toLispList());
		return list;
	}

	/**
	 * Convert the label element to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons labelElementToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("label"));
		list.tail_(new Long(this.labelElement().id()));
		return list;
	}

	/**
	 * Get the receiver from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#fromLispList_(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		super.fromLispList_(aList);
		this.fromElementFromLispList_(aList);
		this.toElementFromLispList_(aList);
		this.pathElementFromLispList_(aList);
		this.labelElementFromLispList_(aList);
	}

	/**
	 * Get the from element id from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void fromElementFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("from"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.fromElement_(null);
		this.fromElementId_(((Number) list.tail()).longValue());
	}

	/**
	 * Get the to element id from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void toElementFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("to"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.toElement_(null);
		this.toElementId_(((Number) list.tail()).longValue());
	}

	/**
	 * Get the path element id from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void pathElementFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("path"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.pathElement_((JunPathElement) JunDrawingElement.FromLispList_((JunLispCons) list.tail()));
	}

	/**
	 * Get the label element id from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void labelElementFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("label"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.labelElement_(null);
		this.labelElementId_(((Number) list.tail()).longValue());
	}

	/**
	 * Answer the nearest point indexes from label element.
	 * 
	 * @return int[2]
	 * @category private
	 */
	protected int[] nearestPointIndexesFromLabel() {
		if (nearestPointIndexesFromLabel == null) {
			double minDistance = Long.MAX_VALUE;

			Point[] points = this._points();
			for (int i = 1; i < points.length; i++) {
				double distance = new Jun2dLine(new Jun2dPoint(points[i - 1]), new Jun2dPoint(points[i])).center().distance_(new Jun2dPoint(this.labelElement().center()));
				if (distance < minDistance) {
					minDistance = distance;
					nearestPointIndexesFromLabel = new int[] { i - 1, i };
				}
			}
		}
		return nearestPointIndexesFromLabel;
	}

	/**
	 * Answer the delta point from label center point with receiver's center point.
	 * 
	 * @return java.awt.Point
	 * @category private
	 */
	protected Point labelElementDeltaPoint() {
		if (labelElementDeltaPoint == null && this.labelElement() != null) {
			int[] indexes = this.nearestPointIndexesFromLabel();
			Point targetPoint = new Jun2dLine(new Jun2dPoint(this.pointAt_(indexes[0])), new Jun2dPoint(this.pointAt_(indexes[1]))).center()._toPoint();
			Point labelLocation = this.labelElement().location();
			labelElementDeltaPoint = new Point(labelLocation.x - targetPoint.x, labelLocation.y - targetPoint.y);
		}
		return labelElementDeltaPoint;
	}

	/**
	 * Set the delta point from label center point with receiver's center point.
	 * 
	 * @param aPoint java.awt.Point
	 * @category private
	 */
	protected void labelElementDeltaPoint_(Point aPoint) {
		labelElementDeltaPoint = aPoint;
	}

	/**
	 * Answer the reciver's from element id.
	 * 
	 * @return long
	 * @category private
	 */
	public long fromElementId() {
		return fromElementId;
	}

	/**
	 * Set the reciver's from element id.
	 * 
	 * @param aNumber long
	 * @category private
	 */
	public void fromElementId_(long aNumber) {
		fromElementId = aNumber;
	}

	/**
	 * Answer the reciver's to element id.
	 * 
	 * @return long
	 * @category private
	 */
	public long toElementId() {
		return toElementId;
	}

	/**
	 * Set the reciver's to element id.
	 * 
	 * @param aNumber long
	 * @category private
	 */
	public void toElementId_(long aNumber) {
		toElementId = aNumber;
	}

	/**
	 * Answer the reciver's label element id.
	 * 
	 * @return long
	 * @category private
	 */
	public long labelElementId() {
		return labelElementId;
	}

	/**
	 * Set the reciver's label element id.
	 * 
	 * @param aNumber long
	 * @category private
	 */
	public void labelElementId_(long aNumber) {
		labelElementId = aNumber;
	}
}
