package jp.co.sra.jun.goodies.drawing.element;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Shape;
import java.util.Map;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.jun.goodies.drawing.properties.JunDrawingElementPropertiesModel;
import jp.co.sra.jun.goodies.drawing.properties.JunRectangularShapePropertiesModel;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunRectangularShapeElement class
 * 
 *  @author    m-asada
 *  @created   2005/03/01 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunRectangularShapeElement.java,v 8.11 2008/02/20 06:31:23 nisinaka Exp $
 */
public abstract class JunRectangularShapeElement extends JunRectangularElement {
	protected Color foregroundColor;
	protected Color backgroundColor;
	protected int lineWidth;

	protected transient Shape shape;
	protected transient BasicStroke stroke;
	protected transient JunRectangularShapePropertiesModel propertiesModel;

	public static final int MIN_LINE_WIDTH = 0;
	public static final int MAX_LINE_WIDTH = 20;

	/**
	 * Create a new instance of JunRectangularElement and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunRectangularShapeElement() {
		super();
	}

	/**
	 * Create a new instance of JunRectangularElement and initialize it.
	 *
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunRectangularShapeElement(JunLispList aList) {
		super(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		foregroundColor = Color.black;
		backgroundColor = Color.white;
		lineWidth = 1;

		shape = null;
		stroke = null;
		propertiesModel = null;
	}

	/**
	 * Release the resources of the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		super.release();

		if (propertiesModel != null) {
			propertiesModel.closeRequest();
			propertiesModel = null;
		}
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#postCopy(java.util.Map)
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		super.postCopy(context);

		this.foregroundColor_(new Color(foregroundColor.getRGB(), true));
		if (backgroundColor != null) {
			this.backgroundColor_(new Color(backgroundColor.getRGB(), true));
		}
		this.flushBounds();
		shape = null;
		stroke = null;
		propertiesModel = null;

		return this;
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		Rectangle clipBounds = aGraphics.getClipBounds();
		if (clipBounds != null && clipBounds.intersects(this.bounds()) == false) {
			return;
		}

		Graphics2D graphicsContext = (Graphics2D) aGraphics.create();
		try {
			Shape shape = this.shape();
			if (this.backgroundColor() != null) {
				graphicsContext.setColor(this.backgroundColor());
				graphicsContext.fill(shape);
			}
			if (this.foregroundColor() != null && this.lineWidth() > 0) {
				graphicsContext.setColor(this.foregroundColor());
				graphicsContext.setStroke(new BasicStroke(this.lineWidth()));
				graphicsContext.draw(shape);
			}
		} finally {
			if (graphicsContext != null) {
				graphicsContext.dispose();
			}
		}
	}

	/**
	 * Flush the receiver's preferred bounds.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunRectangularElement#flushBounds()
	 * @category flushing
	 */
	protected void flushBounds() {
		super.flushBounds();
		stroke = null;
		shape = null;
	}

	/**
	 * Answer true if receiver is shape element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isShape() {
		return true;
	}

	/**
	 * Answer true if receiver's bounding box contains the specified a point, otherwise false.
	 * 
	 * @param aPoint java.awt.Point
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#containsPoint_(java.awt.Point)
	 * @category testing
	 */
	public boolean containsPoint_(Point aPoint) {
		if (this.backgroundColor() == null) {
			return this.stroke().createStrokedShape(this.shape()).contains(aPoint);
		} else {
			return this.shape().contains(aPoint) || this.stroke().createStrokedShape(this.shape()).contains(aPoint);
		}
	}

	/**
	 * Answer true if receiver's element intersects the specified a rectangle, otherwise false.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#intersects_(java.awt.Rectangle)
	 * @category testing
	 */
	public boolean intersects_(Rectangle aRectangle) {
		if (this.backgroundColor() == null) {
			return this.stroke().createStrokedShape(this.shape()).intersects(aRectangle);
		} else {
			return this.shape().intersects(aRectangle) || this.stroke().createStrokedShape(this.shape()).intersects(aRectangle);
		}
	}

	/**
	 * Answer the receiver's foreground color.
	 * 
	 * @return java.awt.Color
	 * @category visual properties
	 */
	public Color foregroundColor() {
		return foregroundColor;
	}

	/**
	 * Set the receiver's foreground color.
	 * 
	 * @param aColor java.awt.Color
	 * @category visual properties
	 */
	public void foregroundColor_(Color aColor) {
		foregroundColor = aColor;
	}

	/**
	 * Answer the receiver's background color.
	 * 
	 * @return java.awt.Color
	 * @category visual properties
	 */
	public Color backgroundColor() {
		return backgroundColor;
	}

	/**
	 * Set the receiver's background color.
	 * 
	 * @param aColor java.awt.Color
	 * @category visual properties
	 */
	public void backgroundColor_(Color aColor) {
		backgroundColor = aColor;
	}

	/**
	 * Answer the receiver's line width.
	 * 
	 * @return int
	 * @category visual properties
	 */
	public int lineWidth() {
		return lineWidth;
	}

	/**
	 * Set the receiver's line width.
	 * 
	 * @param aNumber int
	 * @category visual properties
	 */
	public void lineWidth_(int aNumber) {
		lineWidth = Math.min(Math.max(aNumber, MIN_LINE_WIDTH), MAX_LINE_WIDTH);
		this.flushBounds();
	}

	/**
	 * Answer the receiver's properties model.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.properties.JunDrawingElementPropertiesModel
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#propertiesModel()
	 * @category visual properties
	 */
	public JunDrawingElementPropertiesModel propertiesModel() {
		if (propertiesModel == null) {
			propertiesModel = new JunRectangularShapePropertiesModel(this);
		}
		return propertiesModel;
	}

	/**
	 * Answer the receiver's shape.
	 * 
	 * @return java.awt.Shape
	 * @category private
	 */
	protected abstract Shape shape();

	/**
	 * Answer the receiver's stroke
	 * 
	 * @return java.awt.BasicStroke
	 * @category private
	 */
	protected BasicStroke stroke() {
		if (stroke == null) {
			stroke = new BasicStroke(this.lineWidth());
		}
		return stroke;
	}

	/**
	 * Convert the receiver to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();
		list.add_(this.foregroundColorToLispList());
		list.add_(this.backgroundColorToLispList());
		list.add_(this.lineWidthToLispList());
		return list;
	}

	/**
	 * Convert the foreground color to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons foregroundColorToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("foregroundColor"));
		list.tail_(this.foregroundColor());
		return list;
	}

	/**
	 * Convert the background color to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons backgroundColorToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("backgroundColor"));
		list.tail_(this.backgroundColor());
		return list;
	}

	/**
	 * Convert the line width to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons lineWidthToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("lineWidth"));
		list.tail_(new Integer(this.lineWidth()));
		return list;
	}

	/**
	 * Get the receiver from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#fromLispList_(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		super.fromLispList_(aList);
		this.foregroundColorFromLispList_(aList);
		this.backgroundColorFromLispList_(aList);
		this.lineWidthFromLispList_(aList);
	}

	/**
	 * Get the foreground color from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void foregroundColorFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("foregroundColor"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.foregroundColor_((Color) list.tail());
	}

	/**
	 * Get the background color from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void backgroundColorFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("backgroundColor"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.backgroundColor_((Color) list.tail());
	}

	/**
	 * Get the line width from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void lineWidthFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("lineWidth"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.lineWidth_(((Number) list.tail()).intValue());
	}
}
