package jp.co.sra.jun.goodies.drawing.map;

import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.drawing.element.JunDrawingElement;
import jp.co.sra.jun.goodies.drawing.element.JunLabelElement;
import jp.co.sra.jun.goodies.drawing.element.JunLinkElement;
import jp.co.sra.jun.goodies.drawing.element.JunNodeElement;
import jp.co.sra.jun.goodies.drawing.element.JunPathElement;
import jp.co.sra.jun.goodies.drawing.element.JunVertexesElement;
import jp.co.sra.jun.system.framework.JunAbstractController;
import jp.co.sra.jun.system.framework.JunDialog;

/**
 * JunDrawingMapController class
 * 
 *  @author    m-asada
 *  @created   2005/03/01 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDrawingMapController.java,v 8.12 2008/02/20 06:31:24 nisinaka Exp $
 */
public class JunDrawingMapController extends JunAbstractController {
	protected transient int dragType;
	protected transient Point startPoint;
	protected transient Point previousPoint;
	protected transient Point popupMenuPoint;

	protected transient Point offsetPoint;
	protected transient Object[][] movingElements;
	protected transient JunLinkElement[] linkedElements;
	protected transient ArrayList trackPoints;

	protected transient StSymbol controllPointSymbol;
	protected transient Jun2dPoint[][] scaleCollection;
	protected transient boolean clickElementFlag;

	protected static final int DRAG_NONE = 0;
	protected static final int DRAG_SELECT_ITEM = 1;
	protected static final int DRAG_MOVE_ITEM = 2;
	protected static final int DRAG_MOVE_POINT = 3;
	protected static final int DRAG_MOVE_LINK_POINT = 4;
	protected static final int DRAG_MOVE_LOCUS_POINT = 5;

	public static final int EDITMODE_SELECT = 0;
	public static final int EDITMODE_ADD_TEXT = 1;
	public static final int EDITMODE_ADD_TEXTBOX = 2;
	public static final int EDITMODE_ADD_IMAGE = 3;
	public static final int EDITMODE_ADD_RECTANGLE = 4;
	public static final int EDITMODE_ADD_ROUND_RECTANGLE = 5;
	public static final int EDITMODE_ADD_ELIPSE = 6;
	public static final int EDITMODE_ADD_PATH = 7;
	public static final int EDITMODE_ADD_FREEHAND = 8;
	public static final int EDITMODE_ADD_LINK = 9;

	/**
	 * Answer the receiver's start point at mouse pressed.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point startPoint() {
		return startPoint;
	}

	/**
	 * Answer the receiver's popupMenu point.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point popupMenuPoint() {
		return popupMenuPoint;
	}

	/**
	 * Set the receiver's popupMenu point.
	 * 
	 * @param aPoint java.awt.Point
	 * @category accessing
	 */
	protected void popupMenuPoint_(Point aPoint) {
		popupMenuPoint = aPoint;
	}

	/**
	 * Answer the receiver's model as JunDrawingMapModel.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.map.JunDrawingMapModel
	 * @category model accessing
	 */
	public JunDrawingMapModel getDrawingMapModel() {
		return (JunDrawingMapModel) this.model();
	}

	/**
	 * Answer the receiver's view as JunDrawingMapView
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView
	 * @category view accessing
	 */
	public JunDrawingMapView getDrawingMapView() {
		Component aComponent = this.view().toComponent();
		while (aComponent instanceof JunDrawingMapView == false) {
			aComponent = aComponent.getParent();
		}
		return (JunDrawingMapView) aComponent;
	}

	/**
	 * Add the myself as a listener of the view.
	 * 
	 * @param newView jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StController#buildListener(jp.co.sra.smalltalk.StView)
	 * @category view accessing
	 */
	protected void buildListener(StView newView) {
		Component aView = newView.toComponent();
		aView.addMouseListener(this);
		aView.addMouseMotionListener(this);
	}

	/**
	 * Initialize temporary variavles.
	 * 
	 * @category initialize-release
	 */
	protected void initializeTemporaryVariables() {
		dragType = DRAG_NONE;
		startPoint = null;
		previousPoint = null;
		offsetPoint = null;
		movingElements = null;
		linkedElements = null;
		trackPoints = new ArrayList();
		controllPointSymbol = null;
		scaleCollection = null;
		clickElementFlag = false;
	}

	/**
	 * Invoked when a mouse button has been pressed on the view.
	 * 
	 * @param event java.awt.event.MouseEvent
	 * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mousePressed(MouseEvent event) {
		Point aPoint = this.cursorPointInModel_(event.getPoint());
		JunDrawingMap aMap = this.getDrawingMapModel().mapObject();

		if (event.isPopupTrigger() || event.isMetaDown()) {
			if (startPoint != null) {
				aMap.changed_($("redisplay"));
				this.initializeTemporaryVariables();
			}
			this.popupMenuPoint_(aPoint);
			super.mousePressed(event);
			return;
		}

		if (dragType == DRAG_MOVE_LINK_POINT) {
			this.addLink_(aPoint);
			this.initializeTemporaryVariables();
			this.getDrawingMapView().editMode_(JunDrawingMapController.EDITMODE_SELECT);
			return;
		}

		switch (this.getDrawingMapView().editMode()) {
			case EDITMODE_ADD_TEXT:
				this.getDrawingMapModel().addTextElement();
				break;
			case EDITMODE_ADD_TEXTBOX:
				this.getDrawingMapModel().addTextboxElement();
				break;
			case EDITMODE_ADD_IMAGE:
				this.getDrawingMapModel().addImageElement();
				break;
			case EDITMODE_ADD_ELIPSE:
				this.getDrawingMapModel().addEllipseElement();
				break;
			case EDITMODE_ADD_RECTANGLE:
				this.getDrawingMapModel().addRectangleElement();
				break;
			case EDITMODE_ADD_ROUND_RECTANGLE:
				this.getDrawingMapModel().addRoundRectangleElement();
				break;
			case EDITMODE_ADD_PATH:
				this.getDrawingMapModel().addPathElement();
				break;
			case EDITMODE_ADD_FREEHAND:
				this.initializeTemporaryVariables();
				this.getDrawingMapView().createOffScreen();
				this.view().toComponent().setCursor(JunCursors.BrushCursor());
				this.displayLocusPoint_(aPoint);
				return;
			case EDITMODE_ADD_LINK:
				JunDrawingElement anElement = aMap.which_(aPoint);
				if (anElement == null) {
					return;
				}
				this.selectActivity_(event);
				this.getDrawingMapModel().addLinkElement();
				return;
			case EDITMODE_SELECT:
				this.initializeTemporaryVariables();

				anElement = aMap.which_(aPoint);
				if (aMap.selectedElements().contains(anElement)) {
					aMap.removeSelectedElement_(anElement);
					aMap.addSelectedElement_(anElement);
					clickElementFlag = true;
				} else {
					this.selectActivity_(event);
				}

				if (anElement != null) {
					if (anElement.containsPointInControllArea_(aPoint)) {
						StSymbol aSymbol = anElement.controllPointAreaName_(aPoint);
						if (anElement.isLink() && (aSymbol == JunVertexesElement.CONTROLL_POINT_BEGIN || aSymbol == JunVertexesElement.CONTROLL_POINT_END)) {
							return;
						} else {
							this.view().toComponent().setCursor(JunCursors.Quarters2Cursor());
							this.movePoint_(event);
							dragType = DRAG_MOVE_POINT;
						}
					} else {
						this.view().toComponent().setCursor(JunCursors.HandCursor());
						this.moveElement_with_(anElement, aPoint);
						dragType = DRAG_MOVE_ITEM;
					}
				} else {
					this.view().toComponent().setCursor(JunCursors.BullCursor());
					this.selectElements_(aPoint);
					dragType = DRAG_SELECT_ITEM;
				}

				startPoint = aPoint;
				break;
		}

		this.getDrawingMapView().editMode_(EDITMODE_SELECT);
	}

	/**
	 * Invoked when a mouse button has been released on the view.
	 * 
	 * @param event java.awt.event.MouseEvent
	 * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mouseReleased(MouseEvent event) {
		if (event.isPopupTrigger() || event.isMetaDown() || (System.getProperty("os.name").toLowerCase().matches(".*mac.*os.*x.*") && event.isControlDown())) {
			super.mouseReleased(event);
			return;
		}

		if (dragType == DRAG_MOVE_LINK_POINT) {
			return;
		}

		if (clickElementFlag == true) {
			this.selectActivity_(event);
			clickElementFlag = false;
		} else {
			if (previousPoint != null && previousPoint.equals(startPoint) == false) {
				switch (dragType) {
					case DRAG_SELECT_ITEM:
						Point aPoint = this.cursorPointInModel_(event.getPoint());
						Rectangle aBox = StRectangle.Vertex_vertex_(startPoint, aPoint).toRectangle();
						aBox.setSize(aBox.width + 1, aBox.height + 1);
						this.getDrawingMapModel().changed_with_($("redisplay"), aBox);
						break;
					case DRAG_MOVE_ITEM:
						this.getDrawingMapModel().mapObject().flushBounds();
						this.getDrawingMapModel().mapObject().changed_($("location"));
						break;
					case DRAG_MOVE_POINT:
						this.getDrawingMapModel().mapObject().flushBounds();
						this.getDrawingMapModel().mapObject().changed_($("location"));
						break;
					case DRAG_MOVE_LOCUS_POINT:
						this.getDrawingMapModel().addFreehandElement((ArrayList) trackPoints.clone());
						this.getDrawingMapView().editMode_(EDITMODE_SELECT);
						break;
					case DRAG_MOVE_LINK_POINT:
						break;
					case DRAG_NONE:
						break;
				}
			}
		}

		this.mouseMoved(event);
		this.initializeTemporaryVariables();
	}

	/**
	 * Invoked when the mouse has been clicked on the view.
	 * 
	 * @param event java.awt.event.MouseEvent
	 * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mouseClicked(MouseEvent event) {
		if (event.isPopupTrigger() || event.isMetaDown()) {
			startPoint = this.cursorPointInModel_(event.getPoint());
			super.mouseClicked(event);
			return;
		}

		if (this.getDrawingMapView().editMode() == EDITMODE_SELECT && event.getClickCount() == 2) {
			JunDrawingElement anElement = this.getDrawingMapModel().mapObject().which_(this.cursorPointInModel_(event.getPoint()));
			if (anElement == null) {
				return;
			}
			anElement.openProperties();
		}
	}

	/**
	 * Invoked when a mouse is moved on the view.
	 * 
	 * @param event java.awt.event.MouseEvent 
	 * @see java.awt.event.MouseMotionListener#mouseMoved(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseMoved(MouseEvent event) {
		Point currentPoint = this.cursorPointInModel_(event.getPoint());

		if (dragType == DRAG_MOVE_LINK_POINT) {
			this.displayLinkPointFrom_to_(startPoint, currentPoint);
			return;
		}

		JunDrawingMap aMap = this.getDrawingMapModel().mapObject();
		JunDrawingElement anElement = aMap.which_(currentPoint);
		if (anElement == null) {
			this.view().toComponent().setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		} else if (aMap.selectedElements().contains(anElement)) {
			StSymbol aSymbol = anElement.controllPointAreaName_(currentPoint);
			if (aSymbol == JunDrawingElement.CONTROLL_POINT_TOP_LEFT || aSymbol == JunDrawingElement.CONTROLL_POINT_BOTTOM_RIGHT) {
				this.view().toComponent().setCursor(JunCursors.AntiObliqueCursor());
			} else if (aSymbol == JunDrawingElement.CONTROLL_POINT_TOP_RIGHT || aSymbol == JunDrawingElement.CONTROLL_POINT_BOTTOM_LEFT) {
				this.view().toComponent().setCursor(JunCursors.NormalObliqueCursor());
			} else if (anElement.isLink() && (aSymbol == JunVertexesElement.CONTROLL_POINT_BEGIN || aSymbol == JunVertexesElement.CONTROLL_POINT_END)) {
				this.view().toComponent().setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
			} else if (aSymbol != null) {
				this.view().toComponent().setCursor(JunCursors.QuartersCursor());
			} else {
				this.view().toComponent().setCursor(JunCursors.CrossCursor());
			}
		} else {
			this.view().toComponent().setCursor(JunCursors.CrossCursor());
		}
	}

	/**
	 * Invoked when a mouse is dragged on the view.
	 *
	 * @param event java.awt.event.MouseEvent 
	 * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseDragged(MouseEvent event) {
		Point aPoint = this.cursorPointInModel_(event.getPoint());
		if (startPoint == null || aPoint.equals(previousPoint)) {
			return;
		}
		if (previousPoint == null) {
			this.getDrawingMapView().createOffScreen();
		}

		switch (dragType) {
			case DRAG_SELECT_ITEM:
				this.selectElements_(aPoint);
				break;
			case DRAG_MOVE_ITEM:
				JunDrawingElement anElement = this.getDrawingMapModel().mapObject().which_(aPoint);
				this.moveElement_with_(anElement, aPoint);
				break;
			case DRAG_MOVE_POINT:
				this.movePoint_(event);
				break;
			case DRAG_MOVE_LOCUS_POINT:
				this.displayLocusPoint_(aPoint);
			default:
				break;
		}
		clickElementFlag = false;
	}

	/**
	 * Operation of select activity.
	 * 
	 * @param event java.awt.event.MouseEvent
	 * @category manipulating
	 */
	protected void selectActivity_(final MouseEvent event) {
		JunDrawingMapModel aModel = this.getDrawingMapModel();
		JunDrawingMap aMap = aModel.mapObject();
		Point aPoint = this.cursorPointInModel_(event.getPoint());
		JunDrawingElement anElement = aMap.which_(aPoint);
		if (anElement == null) {
			Rectangle clearArea = this.getDrawingMapView().areaOfElements_(aMap._selectedElements());
			if (clearArea != null) {
				clearArea.grow(JunDrawingElement.CONTROLL_AREA_SIZE / 2, JunDrawingElement.CONTROLL_AREA_SIZE / 2);
			}
			aMap.clearSelectedElements();
			aMap.changed_with_($("selection"), clearArea);
		} else {
			Rectangle repaintArea = new Rectangle(anElement.bounds());
			repaintArea.grow(JunDrawingElement.CONTROLL_AREA_SIZE / 2, JunDrawingElement.CONTROLL_AREA_SIZE / 2);
			if (event.isShiftDown()) {
				if (aMap.selectedElements().contains(anElement)) {
					if (anElement.containsPointInControllArea_(aPoint)) {
						aMap.changed_with_($("selection"), repaintArea);
					} else {
						aMap.removeSelectedElement_(anElement);
						aMap.changed_with_($("selection"), repaintArea);
					}
				} else {
					aMap.addSelectedElement_(anElement);
					aMap.changed_with_($("selection"), repaintArea);
				}
			} else {
				Rectangle clearArea = this.getDrawingMapView().areaOfElements_(aMap._selectedElements());
				aMap.clearSelectedElements();
				aMap.addSelectedElement_(anElement);
				if (clearArea != null) {
					clearArea.grow(JunDrawingElement.CONTROLL_AREA_SIZE / 2, JunDrawingElement.CONTROLL_AREA_SIZE / 2);
					repaintArea.add(clearArea);
				}
				aMap.changed_with_($("selection"), repaintArea);
			}
		}
	}

	/**
	 * Operation of selected elements.
	 *
	 * @param currentPoint java.awt.Point
	 * @category manipulating
	 */
	protected void selectElements_(Point currentPoint) {
		if (startPoint == null) {
			return;
		}

		Rectangle oldBounds = (previousPoint != null) ? StRectangle.Vertex_vertex_(startPoint, previousPoint).toRectangle() : null;
		Rectangle newBounds = StRectangle.Vertex_vertex_(startPoint, currentPoint).toRectangle();

		if (oldBounds != null && currentPoint.equals(previousPoint) == false) {
			JunDrawingMap aMap = this.getDrawingMapModel().mapObject();
			Jun2dPoint scalePoint = this.getDrawingMapModel().scalePoint();
			boolean isZooming = this.getDrawingMapModel().isZooming();

			JunDrawingMapView mapView = this.getDrawingMapView();
			Image offScreenImage = mapView.offScreen();
			Graphics2D offScreenGraphics = mapView.graphicsWithZoomFrom_(offScreenImage);
			try {
				if (isZooming) {
					if (scalePoint.x() < 0.2 || scalePoint.y() < 0.2) {
						oldBounds.grow((int) (1 / scalePoint.x()), (int) (1 / scalePoint.y()));
					} else {
						oldBounds.grow(5, 5);
					}
				}
				offScreenGraphics.setColor(this.view().toComponent().getBackground());
				offScreenGraphics.setClip(oldBounds.x, oldBounds.y, oldBounds.width + 1, oldBounds.height + 1);
				offScreenGraphics.fillRect(oldBounds.x, oldBounds.y, oldBounds.width + 1, oldBounds.height + 1);

				mapView.displayCanvasOn_(offScreenGraphics);

				ArrayList oldSelectedElements = (ArrayList) aMap.selectedElements().clone();
				JunDrawingElement[] elements = aMap._componentElements();
				for (int i = 0; i < elements.length; i++) {
					if (elements[i].intersects_(newBounds)) {
						aMap.addSelectedElement_(elements[i]);
					} else {
						aMap.removeSelectedElement_(elements[i]);
					}
				}

				if (oldSelectedElements.equals(aMap.selectedElements()) == false) {
					aMap.changed_($("selection"));
				}

				Rectangle clipBounds = new Rectangle(newBounds);
				if (isZooming) {
					clipBounds.grow(5, 5);
				}
				offScreenGraphics.setColor(new Color(0xFF8080));
				offScreenGraphics.setClip(clipBounds.x, clipBounds.y, clipBounds.width + 1, clipBounds.height + 1);
				offScreenGraphics.drawRect(newBounds.x, newBounds.y, newBounds.width, newBounds.height);
			} finally {
				if (offScreenGraphics != null) {
					offScreenGraphics.dispose();
					offScreenGraphics = null;
				}
			}

			if (oldBounds != null) {
				newBounds.add(oldBounds);
			}

			mapView.displayCanvasOn_scaledImage_unscaledClip_(this.view().toComponent().getGraphics(), offScreenImage, newBounds);
		}
		previousPoint = currentPoint;
	}

	/**
	 * Operation of move element.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @param currentPoint java.awt.Point
	 * @category manipulating
	 */
	protected void moveElement_with_(JunDrawingElement anElement, Point currentPoint) {
		if (movingElements == null || offsetPoint == null || linkedElements == null) {
			JunDrawingMap aMap = this.getDrawingMapModel().mapObject();
			ArrayList anArray = new ArrayList(aMap.selectedElements());
			JunDrawingElement[] selectedElements = (JunDrawingElement[]) anArray.toArray(new JunDrawingElement[anArray.size()]);
			for (int i = 0; i < selectedElements.length; i++) {
				if (anElement != selectedElements[i] && selectedElements[i].isLabel() && aMap.selectedElements().contains(((JunLabelElement) selectedElements[i]).baseElement())) {
					anArray.remove(selectedElements[i]);
				}
			}
			selectedElements = (JunDrawingElement[]) anArray.toArray(new JunDrawingElement[anArray.size()]);

			movingElements = new Object[selectedElements.length][2];
			for (int i = 0; i < selectedElements.length; i++) {
				if (selectedElements[i].isLink()) {
					Point[] linkPoints = ((JunLinkElement) selectedElements[i])._points();
					Point[] deltaPoints = new Point[linkPoints.length];
					for (int j = 0; j < linkPoints.length; j++) {
						deltaPoints[j] = new Point(new Point(linkPoints[j].x - anElement.x(), linkPoints[j].y - anElement.y()));
					}
					movingElements[i] = new Object[] { selectedElements[i], deltaPoints };
				} else {
					movingElements[i] = new Object[] { selectedElements[i], new Point[] { new Point(selectedElements[i].x() - anElement.x(), selectedElements[i].y() - anElement.y()) } };
				}
			}

			offsetPoint = new Point(currentPoint.x - anElement.x(), currentPoint.y - anElement.y());

			ArrayList linkArray = new ArrayList();
			JunLinkElement[] linkElements = aMap._linkElements();
			for (int i = 0; i < linkElements.length; i++) {
				if (anArray.contains(linkElements[i]) || anArray.contains(linkElements[i].fromElement()) || anArray.contains(linkElements[i].toElement())) {
					linkArray.add(linkElements[i]);
				}
			}
			linkedElements = (JunLinkElement[]) linkArray.toArray(new JunLinkElement[linkArray.size()]);
		}

		if (previousPoint != null && currentPoint.equals(previousPoint) == false) {
			Point newLocation = new Point(currentPoint.x - offsetPoint.x, currentPoint.y - offsetPoint.y);
			Rectangle repaintArea = null;

			// clipping old linked location
			for (int i = 0; i < linkedElements.length; i++) {
				Rectangle aBox = new Rectangle(linkedElements[i].bounds());
				if (linkedElements[i].hasLabel()) {
					aBox.add(linkedElements[i].labelElement().bounds());
				}
				if (repaintArea != null) {
					repaintArea.add(aBox);
				} else {
					repaintArea = aBox;
				}
			}

			for (int i = 0; i < movingElements.length; i++) {
				JunDrawingElement arrayFirst = (JunDrawingElement) movingElements[i][0];
				Point[] arrayLast = (Point[]) movingElements[i][1];

				// clipping old location
				Rectangle aBox = new Rectangle(arrayFirst.bounds());
				if (arrayFirst.isLink() && arrayFirst.hasLabel()) {
					aBox.add(((JunLinkElement) arrayFirst).labelElement().bounds());
				}

				// move element
				if (arrayFirst.isLink()) {
					JunLinkElement linkElement = (JunLinkElement) arrayFirst;
					if (arrayLast.length > 2) {
						ArrayList newPoints = new ArrayList(arrayLast.length);
						for (int j = 0; j < arrayLast.length; j++) {
							if (j == 0 && j == arrayLast.length - 1) {
								newPoints.add(arrayLast[j]);
							} else {
								newPoints.add(new Point(Math.max(newLocation.x + arrayLast[j].x, 0), Math.max(newLocation.y + arrayLast[j].y, 0)));
							}
						}
						linkElement.points_(newPoints);
					}
				} else {
					arrayFirst.location_(new Point(newLocation.x + arrayLast[0].x, newLocation.y + arrayLast[0].y));
				}

				// clipping new location
				aBox.add(arrayFirst.bounds());
				if (arrayFirst.isLink() && ((JunLinkElement) arrayFirst).hasLabel()) {
					aBox.add(((JunLinkElement) arrayFirst).labelElement().bounds());
				}

				if (repaintArea != null) {
					repaintArea.add(aBox);
				} else {
					repaintArea = new Rectangle(aBox);
				}
			}

			// clipping new linked location
			for (int i = 0; i < linkedElements.length; i++) {
				linkedElements[i].updateFromToPoint();
				Rectangle aBox = new Rectangle(linkedElements[i].bounds());
				if (linkedElements[i].hasLabel()) {
					aBox.add(linkedElements[i].labelElement().bounds());
				}
				repaintArea.add(aBox);
			}

			// redisplay
			repaintArea.grow(JunDrawingElement.CONTROLL_AREA_SIZE / 2, JunDrawingElement.CONTROLL_AREA_SIZE / 2);
			this.getDrawingMapView()._invalidateRectangle_repairNow_(repaintArea, true);
		}
		previousPoint = currentPoint;
	}

	/**
	 * Operation of move controll point.
	 * 
	 * @param event java.awt.event.MouseEvent
	 * @category manipulating
	 */
	protected void movePoint_(MouseEvent event) {
		Point currentPoint = this.cursorPointInModel_(event.getPoint());
		JunDrawingMap aMap = this.getDrawingMapModel().mapObject();
		JunDrawingElement[] selectedElements = aMap._selectedElements();
		ArrayList selectedElementList = aMap.selectedElements();
		JunDrawingElement anElement = aMap.currentElement();

		if (scaleCollection == null) {
			scaleCollection = new Jun2dPoint[selectedElements.length][2];
			boolean isSameScale = event.isAltDown() || event.isMetaDown();
			for (int i = 0; i < selectedElements.length; i++) {
				JunDrawingElement eachElement = selectedElements[i];
				scaleCollection[i][0] = isSameScale ? new Jun2dPoint(1, 1) : new Jun2dPoint((double) eachElement.x() / anElement.x(), (double) eachElement.y() / anElement.y());
				scaleCollection[i][1] = isSameScale ? new Jun2dPoint(1, 1) : new Jun2dPoint((double) eachElement.width() / anElement.width(), (double) eachElement.height() / anElement.height());
			}
		}

		if (controllPointSymbol == null && linkedElements == null && previousPoint != null) {
			HashMap areaMap = anElement.controllPointAreas();
			for (Iterator iterator = areaMap.keySet().iterator(); iterator.hasNext();) {
				StSymbol key = (StSymbol) iterator.next();
				Rectangle value = (Rectangle) areaMap.get(key);
				if (value.contains(previousPoint)) {
					controllPointSymbol = key;
				}
			}

			ArrayList linkArray = new ArrayList();
			JunLinkElement[] linkElements = aMap._linkElements();
			for (int i = 0; i < linkElements.length; i++) {
				if (selectedElementList.contains(linkElements[i].fromElement()) || selectedElementList.contains(linkElements[i].toElement())) {
					linkArray.add(linkElements[i]);
				}
			}
			linkedElements = (JunLinkElement[]) linkArray.toArray(new JunLinkElement[linkArray.size()]);
		}

		if (previousPoint != null && currentPoint.equals(previousPoint) == false) {
			if (anElement.isLink() || anElement.isPath()) {
				// clipping old location
				Rectangle repaintArea = new Rectangle(anElement.bounds());
				if (anElement.isLink() && anElement.hasLabel()) {
					repaintArea.add(((JunLinkElement) anElement).labelElement().bounds());
				}

				// move point
				anElement.controllPointAt_put_(controllPointSymbol, currentPoint);
				if (anElement.isLink()) {
					((JunLinkElement) anElement).updateFromToPoint();
				}

				// clipping new location
				repaintArea.add(anElement.bounds());
				if (anElement.isLink() && anElement.hasLabel()) {
					repaintArea.add(((JunLinkElement) anElement).labelElement().bounds());
				}

				// redisplay
				repaintArea.grow(JunDrawingElement.CONTROLL_AREA_SIZE / 2, JunDrawingElement.CONTROLL_AREA_SIZE / 2);
				this.getDrawingMapView()._invalidateRectangle_repairNow_(repaintArea, true);
			} else if (anElement.isNode()) {
				Rectangle repaintArea = null;

				// clipping old linked location
				for (int j = 0; j < linkedElements.length; j++) {
					if (repaintArea != null) {
						repaintArea.add(linkedElements[j].bounds());
					} else {
						repaintArea = new Rectangle(linkedElements[j].bounds());
					}
					if (linkedElements[j].hasLabel()) {
						repaintArea.add(linkedElements[j].labelElement().bounds());
					}
				}

				boolean changeVertical = (controllPointSymbol == JunDrawingElement.CONTROLL_POINT_TOP_LEFT || controllPointSymbol == JunDrawingElement.CONTROLL_POINT_TOP_RIGHT);
				boolean changeHorizontal = (controllPointSymbol == JunDrawingElement.CONTROLL_POINT_TOP_LEFT || controllPointSymbol == JunDrawingElement.CONTROLL_POINT_BOTTOM_LEFT);
				int deltaWidth = Math.max(currentPoint.x, 0) - anElement.x();
				int deltaHeight = Math.max(currentPoint.y, 0) - anElement.y();

				for (int i = 0; i < selectedElements.length; i++) {
					JunDrawingElement drawElement = selectedElements[i];
					if (drawElement.isLink() == false) {
						Rectangle bounds = drawElement.bounds();
						// clipping old location
						if (repaintArea != null) {
							repaintArea.add(bounds);
						} else {
							repaintArea = new Rectangle(bounds);
						}

						// move point
						int newX = (changeHorizontal) ? (Math.max(bounds.x + (int) Math.round(deltaWidth * scaleCollection[i][1].x()), 0)) : (bounds.x);
						int newY = (changeVertical) ? (Math.max(bounds.y + (int) Math.round(deltaHeight * scaleCollection[i][1].y()), 0)) : (bounds.y);
						int newWidth = (changeHorizontal) ? (bounds.width + (bounds.x - newX)) : (Math.max((int) Math.round((currentPoint.x - anElement.x()) * scaleCollection[i][1].x()), drawElement.defaultMinimumExtent().width));
						int newHeight = (changeVertical) ? (bounds.height + (bounds.y - newY)) : (Math.max((int) Math.round((currentPoint.y - anElement.y()) * scaleCollection[i][1].y()), drawElement.defaultMinimumExtent().height));
						if (newWidth < drawElement.defaultMinimumExtent().width) {
							newX = newX + (newWidth - drawElement.defaultMinimumExtent().width);
							newWidth = drawElement.defaultMinimumExtent().width;
						}
						if (newHeight < drawElement.defaultMinimumExtent().height) {
							newY = newY + (newHeight - drawElement.defaultMinimumExtent().height);
							newHeight = drawElement.defaultMinimumExtent().height;
						}
						drawElement.bounds_(new Rectangle(newX, newY, newWidth, newHeight));

						// clipping new location
						repaintArea.add(drawElement.bounds());
					}
				}

				// clipping new linked location
				for (int i = 0; i < linkedElements.length; i++) {
					linkedElements[i].updateFromToPoint();
					repaintArea.add(linkedElements[i].bounds());
					if (linkedElements[i].hasLabel()) {
						repaintArea.add(linkedElements[i].labelElement().bounds());
					}
				}

				// redisplay
				repaintArea.grow(JunDrawingElement.CONTROLL_AREA_SIZE / 2, JunDrawingElement.CONTROLL_AREA_SIZE / 2);
				this.getDrawingMapView()._invalidateRectangle_repairNow_(repaintArea, true);
			}
		}
		previousPoint = currentPoint;
	}

	/**
	 * Add the link element.
	 * 
	 * @param aPoint java.awt.Point
	 * @category manipulating
	 */
	protected void addLink_(Point aPoint) {
		JunDrawingMapModel aMapModel = this.getDrawingMapModel();
		JunDrawingMap aMap = aMapModel.mapObject();
		JunDrawingElement fromElement = aMap.currentElement();
		JunDrawingElement toElement = aMap.which_(aPoint);
		if (fromElement == null || toElement == null) {
			aMap.changed_($("redisplay"));
			return;
		}
		if (fromElement.isNode() == false || toElement.isNode() == false) {
			JunDialog.Warn_($String("The selected element cannot create a link."));
			aMap.changed_($("redisplay"));
			return;
		}

		JunPathElement pathElement = null;
		if (fromElement == toElement) {
			Point center = fromElement.center();
			pathElement = new JunPathElement();
			pathElement.add_(center);
			pathElement.add_(new Point(center.x, fromElement.y() + fromElement.height() + 20));
			pathElement.add_(new Point(fromElement.x() + fromElement.width() + 20, fromElement.y() + fromElement.height() + 20));
			pathElement.add_(new Point(fromElement.x() + fromElement.width() + 20, center.y));
			pathElement.add_(center);
			pathElement.beginStyle_(JunPathElement.ARROW_STYLE_SQUARE);
			pathElement.beginStyle_(JunPathElement.ARROW_STYLE_ARROW);
		}
		JunLinkElement linkElement = new JunLinkElement((JunNodeElement) fromElement, (JunNodeElement) toElement, pathElement);
		aMapModel.addElement_(linkElement);

		aMap.clearSelectedElements();
		aMap.addSelectedElement_(linkElement.fromElement());
		aMap.addSelectedElement_(linkElement.toElement());
		aMap.addSelectedElement_(linkElement);
		aMap.changed_($("redisplay"));
	}

	/**
	 * Display the link on the canvas.
	 * 
	 * @param fromPoint java.awt.Point
	 * @param toPoint java.awt.Point
	 * @category displaying
	 */
	protected void displayLinkPointFrom_to_(Point fromPoint, Point toPoint) {
		Rectangle oldBounds = (previousPoint != null) ? StRectangle.Vertex_vertex_(startPoint, previousPoint).toRectangle() : null;
		JunDrawingMapView mapView = this.getDrawingMapView();
		Image offScreenImage = (previousPoint == null) ? mapView.createOffScreen() : mapView.offScreen();
		Graphics2D offScreenGraphics = mapView.graphicsWithZoomFrom_(offScreenImage);
		try {
			if (oldBounds != null) {
				if (this.getDrawingMapModel().isZooming()) {
					Jun2dPoint scalePoint = this.getDrawingMapModel().scalePoint();
					if (scalePoint.x() < 0.2 || scalePoint.y() < 0.2) {
						oldBounds.grow((int) (1 / scalePoint.x()), (int) (1 / scalePoint.y()));
					} else {
						oldBounds.grow(5, 5);
					}
				}
				offScreenGraphics.setColor(this.view().toComponent().getBackground());
				offScreenGraphics.setClip(oldBounds.x, oldBounds.y, oldBounds.width + 1, oldBounds.height + 1);
				offScreenGraphics.fillRect(oldBounds.x, oldBounds.y, oldBounds.width + 1, oldBounds.height + 1);
				mapView.displayCanvasOn_(offScreenGraphics);
			}
			Rectangle lineBounds = new Rectangle(fromPoint, new Dimension(1, 1));
			lineBounds.add(toPoint);
			offScreenGraphics.setClip(lineBounds.x, lineBounds.y, lineBounds.width, lineBounds.height);
			this.displayOn_lineSegment_(offScreenGraphics, new Point[] { fromPoint, toPoint });
		} finally {
			if (offScreenGraphics != null) {
				offScreenGraphics.dispose();
				offScreenGraphics = null;
			}
		}

		Rectangle newBounds = StRectangle.Vertex_vertex_(fromPoint, toPoint).toRectangle();
		if (oldBounds != null) {
			newBounds.add(oldBounds);
		}
		mapView.displayCanvasOn_scaledImage_unscaledClip_(this.view().toComponent().getGraphics(), offScreenImage, newBounds);

		if (previousPoint == null) {
			this.view().toComponent().setCursor(JunCursors.BullCursor());
			dragType = DRAG_MOVE_LINK_POINT;
			startPoint = fromPoint;
		}
		previousPoint = toPoint;
	}

	/**
	 * Display the locus on the canvas.
	 * 
	 * @param point java.awt.Point
	 * @category displaying
	 */
	protected void displayLocusPoint_(Point point) {
		JunDrawingMapView mapView = this.getDrawingMapView();
		Image offScreenImage = mapView.createOffScreen();
		Graphics2D offScreenGraphics = mapView.graphicsWithZoomFrom_(offScreenImage);
		if (trackPoints.size() == 0 || (trackPoints.size() > 0 && ((Point) trackPoints.get(trackPoints.size() - 1)) != point)) {
			trackPoints.add(new Point(Math.max(0, point.x), Math.max(0, point.y)));
		}
		Point[] points = (Point[]) trackPoints.toArray(new Point[trackPoints.size()]);
		try {
			this.displayOn_polyline_(offScreenGraphics, points);
		} finally {
			if (offScreenGraphics != null) {
				offScreenGraphics.dispose();
				offScreenGraphics = null;
			}
		}

		Rectangle clipBounds = null;
		for (int i = 0; i < points.length; i++) {
			if (clipBounds == null) {
				clipBounds = new Rectangle(points[i].x, points[i].y, 1, 1);
			} else {
				clipBounds.add(points[i].x, points[i].y);
			}
		}
		mapView.displayCanvasOn_scaledImage_unscaledClip_(this.view().toComponent().getGraphics(), offScreenImage, clipBounds);

		if (previousPoint == null) {
			dragType = DRAG_MOVE_LOCUS_POINT;
			startPoint = points[0];
		}
		previousPoint = points[points.length - 1];
	}

	/**
	 * Display the line segment on the specified graphics context.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param lineSegment java.awt.Point[]
	 * @category displaying
	 */
	protected void displayOn_lineSegment_(Graphics aGraphics, Point[] lineSegment) {
		if (lineSegment != null) {
			aGraphics.setColor(this.view().toComponent().getBackground());
			aGraphics.setXORMode(Color.gray);
			for (int i = 0; i < lineSegment.length - 1; i++) {
				aGraphics.drawLine(lineSegment[i].x, lineSegment[i].y, lineSegment[i + 1].x, lineSegment[i + 1].y);
			}
		}
	}

	/**
	 * Display the polyline on the specified graphics context.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param points java.awt.Point[]
	 * @category displaying
	 */
	protected void displayOn_polyline_(Graphics aGraphics, Point[] points) {
		if (points != null) {
			aGraphics.setColor(Color.blue);
			int[] xPoints = new int[points.length];
			int[] yPoints = new int[points.length];
			for (int i = 0; i < points.length; i++) {
				xPoints[i] = points[i].x;
				yPoints[i] = points[i].y;
			}
			aGraphics.drawPolyline(xPoints, yPoints, points.length);
		}
	}

	/**
	 * Answer the cursor point which transform to model.
	 * 
	 * @return java.awt.Point
	 * @category transforming
	 */
	public Point cursorPointInModel() {
		return this.cursorPointInModel_(this.cursorPoint());
	}

	/**
	 * Answer the cursor point which transform to model.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point
	 * @category transforming
	 */
	public Point cursorPointInModel_(Point aPoint) {
		return new Point((int) (aPoint.x / this.getDrawingMapModel().scalePoint().x()), (int) (aPoint.y / this.getDrawingMapModel().scalePoint().y()));
	}
}
