package jp.co.sra.jun.goodies.drawing.properties;

import java.awt.Color;
import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.jun.goodies.colors.JunColorChoiceHSB;
import jp.co.sra.jun.goodies.colors.JunColorChoiceModel;
import jp.co.sra.jun.goodies.drawing.element.JunDrawingElement;
import jp.co.sra.jun.goodies.drawing.element.JunRectangularShapeElement;
import jp.co.sra.jun.goodies.parameter.JunParameter;
import jp.co.sra.jun.goodies.parameter.JunParameterModel;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunRectangularShapePropertiesModel class
 * 
 *  @author    m-asada
 *  @created   2005/05/12 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunRectangularShapePropertiesModel.java,v 8.10 2008/02/20 06:31:24 nisinaka Exp $
 */
public class JunRectangularShapePropertiesModel extends JunDrawingElementPropertiesModel {
	protected JunColorChoiceModel foregroundColorChoiceModel;
	protected JunColorChoiceModel backgroundColorChoiceModel;
	protected JunParameterModel lineWidthParameterModel;
	protected boolean isFill;

	/**
	 * Create a new instance of JunRectangularShapePropertiesModel and initialize it.
	 * 
	 * @param rectangularShapeElement jp.co.sra.jun.goodies.drawing.element.JunRectangularShapeElement
	 * @category Instance creation
	 */
	public JunRectangularShapePropertiesModel(JunRectangularShapeElement rectangularShapeElement) {
		super(rectangularShapeElement);
	}

	/**
	 * Initialize the ApplicationModel when created.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		foregroundColorChoiceModel = null;
		backgroundColorChoiceModel = null;
		lineWidthParameterModel = null;
		isFill = false;
	}

	/**
	 * Answer the receiver's rectangular shape element.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunRectangularShapeElement
	 * @category accessing
	 */
	public JunRectangularShapeElement rectangularShapeElement() {
		return (JunRectangularShapeElement) this.drawingElement();
	}

	/**
	 * Set the receiver's rectangular shape element.
	 * 
	 * @param newDrawingElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @category accessing
	 */
	protected void drawingElement_(JunDrawingElement newDrawingElement) {
		if (newDrawingElement == null) {
			return;
		}
		super.drawingElement_(newDrawingElement);

		JunRectangularShapeElement newRectangularShapeElement = (JunRectangularShapeElement) newDrawingElement;
		this.foregroundColorChoiceModel().color_(newRectangularShapeElement.foregroundColor());
		this.lineWidthParameterModel().parameter().value_(newRectangularShapeElement.lineWidth());
		boolean isFill = (newRectangularShapeElement.backgroundColor() != null);
		if (isFill) {
			this.fill_(true);
			this.backgroundColorChoiceModel().color_(newRectangularShapeElement.backgroundColor());
		} else {
			this.fill_(false);
			this.backgroundColorChoiceModel().color_(Color.white);
		}
	}

	/**
	 * Answer the receiver's fill flag.
	 * 
	 * @return boolean
	 * @category aspects
	 */
	public boolean isFill() {
		return isFill;
	}

	/**
	 * Set the receiver's fill flag.
	 * 
	 * @param newFill boolean
	 * @category aspects
	 */
	public void fill_(boolean newFill) {
		isFill = newFill;
	}

	/**
	 * Answer the receiver's choice model of foreground color.
	 * 
	 * @return jp.co.sra.jun.goodies.colors.JunColorChoiceModel
	 * @category aspects
	 */
	public JunColorChoiceModel foregroundColorChoiceModel() {
		if (foregroundColorChoiceModel == null) {
			foregroundColorChoiceModel = new JunColorChoiceHSB();
			foregroundColorChoiceModel.compute_(new StBlockClosure() {
				public Object value() {
					JunRectangularShapePropertiesModel.this.updateForegroundColor();
					return null;
				}
			});
		}
		return foregroundColorChoiceModel;
	}

	/**
	 * Answer the receiver's choice model of background color.
	 * 
	 * @return jp.co.sra.jun.goodies.colors.JunColorChoiceModel
	 * @category aspects
	 */
	public JunColorChoiceModel backgroundColorChoiceModel() {
		if (backgroundColorChoiceModel == null) {
			backgroundColorChoiceModel = new JunColorChoiceHSB();
			backgroundColorChoiceModel.compute_(new StBlockClosure() {
				public Object value() {
					JunRectangularShapePropertiesModel.this.updateBackgroundColor();
					return null;
				}
			});
		}
		return backgroundColorChoiceModel;
	}

	/**
	 * Answer the receiver's parameter model of line width.
	 * 
	 * @return jp.co.sra.jun.goodies.parameter.JunParameterModel
	 * @category aspects
	 */
	public JunParameterModel lineWidthParameterModel() {
		if (lineWidthParameterModel == null) {
			lineWidthParameterModel = new JunParameterModel(new JunParameter(JunSystem.$String("Border width"), ((JunRectangularShapeElement) this.drawingElement()).lineWidth(), JunRectangularShapeElement.MIN_LINE_WIDTH,
					JunRectangularShapeElement.MAX_LINE_WIDTH, 1));
			lineWidthParameterModel.addDependent_(this);
		}
		return lineWidthParameterModel;
	}

	/**
	 * Answer the default view of the model.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunRectangularShapePropertiesViewAwt(this);
		} else {
			return new JunRectangularShapePropertiesViewSwing(this);
		}
	}

	/**
	 * Receive a change notice from an object of whom the receiver is a
	 * dependent.  The argument evt.getAspect() is typically a Symbol that
	 * indicates what change has occurred.
	 * 
	 * @param event jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent event) {
		if (event.getAspect() == $("value")) {
			this.updateLineWidth();
		}
	}

	/**
	 * Called when the foreground color is changed.
	 * 
	 * @category updating
	 */
	protected void updateForegroundColor() {
		if (this.foregroundColorChoiceModel().color().equals(this.rectangularShapeElement().foregroundColor()) == false) {
			this.rectangularShapeElement().foregroundColor_(this.foregroundColorChoiceModel().color());
			this.rectangularShapeElement().map().changed_with_($("redisplay"), this.rectangularShapeElement().bounds());
			this.changed_($("foregroundColor"));
		}
	}

	/**
	 * Called when the background color is changed.
	 * 
	 * @category updating
	 */
	protected void updateBackgroundColor() {
		if (this.isFill() && this.backgroundColorChoiceModel().color().equals(this.rectangularShapeElement().backgroundColor()) == false) {
			this.rectangularShapeElement().backgroundColor_(this.backgroundColorChoiceModel().color());
			this.rectangularShapeElement().map().changed_with_($("redisplay"), this.rectangularShapeElement().bounds());
			this.changed_($("backgroundColor"));
		}
	}

	/**
	 * Called when the line width is changed.
	 * 
	 * @category updating
	 */
	protected void updateLineWidth() {
		int value = (int) this.lineWidthParameterModel().parameter().value();
		if (this.rectangularShapeElement().lineWidth() != value) {
			this.rectangularShapeElement().lineWidth_(value);
			this.rectangularShapeElement().map().changed_with_($("redisplay"), this.rectangularShapeElement().bounds());
			this.changed_($("lineWidth"));
		}
	}

	/**
	 * Called when the fill is changed.
	 * 
	 * @category updating
	 */
	protected void updateFill() {
		if (this.isFill() && (this.rectangularShapeElement().backgroundColor() == null)) {
			this.rectangularShapeElement().backgroundColor_(this.backgroundColorChoiceModel().color());
			this.rectangularShapeElement().map().changed_with_($("redisplay"), this.rectangularShapeElement().bounds());
			this.changed_($("fill"));
		} else if ((this.isFill() == false) && (this.rectangularShapeElement().backgroundColor() != null)) {
			this.rectangularShapeElement().backgroundColor_(null);
			this.rectangularShapeElement().map().changed_with_($("redisplay"), this.rectangularShapeElement().bounds());
			this.changed_($("fill"));
		}
	}
}
