package jp.co.sra.jun.goodies.files;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;

import javax.swing.DefaultListModel;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.WindowSpecLayout;

import jp.co.sra.jun.system.framework.JunAbstractViewJPanel;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunFileViewSwing class
 * 
 *  @author    nisinaka
 *  @created   2003/03/07 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun433 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFileViewSwing.java,v 8.10 2008/02/20 06:31:33 nisinaka Exp $
 */
public class JunFileViewSwing extends JunAbstractViewJPanel implements JunFileView {

	/**
	 * A BoldLabel is a Label with BOLD font.
	 */
	class BoldLabel extends JLabel {
		public BoldLabel(String text) {
			super(text);
		}

		public void addNotify() {
			super.addNotify();
			this.setFont(this.getFont().deriveFont(Font.BOLD));
		}
	}

	protected JLabel fileLabel;
	protected JTextField fileName;
	protected JList fileList;
	protected JScrollPane fileListPane;
	protected JLabel fileTypeLabel;
	protected JComboBox fileType;
	protected JLabel directoryLabel;
	protected JLabel currentDirectoryLabel;
	protected JList directoryList;
	protected JScrollPane directoryListPane;
	protected JLabel driveLabel;
	protected JComboBox currentDrive;

	protected boolean doNotPropagateFileTypeChange = false;
	protected boolean doNotPropagateCurrentDriveChange = false;

	/**
	 * Create a new instance of JunFileViewSwing and initialize it.
	 *
	 * @param aFileModel jp.co.sra.jun.goodies.files.JunFileModel
	 * @category Instance creation
	 */
	public JunFileViewSwing(JunFileModel aFileModel) {
		super(aFileModel);
	}

	/**
	 * Build this component.
	 * 
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setLayout(new WindowSpecLayout());
		this.add(this.fileLabel(), WindowSpecLayout.Position(0f, 8, 0f, 4));
		this.add(this.fileName(), WindowSpecLayout.Position(0f, 8, 0f, 32, 0.5f, -4, 0f, 60));
		this.add(this.fileListPane(), WindowSpecLayout.Position(0f, 8, 0f, 68, 0.5f, -4, 1f, -72));
		this.add(this.fileTypeLabel(), WindowSpecLayout.Position(0f, 8, 1f, -62));
		this.add(this.fileType(), WindowSpecLayout.Position(0f, 8, 1f, -36, 0.5f, -4, 1f, -8));
		this.add(this.directoryLabel(), WindowSpecLayout.Position(0.5f, 4, 0f, 4));
		this.add(this.currentDirectoryLabel(), WindowSpecLayout.Position(0.5f, 4, 0f, 32, 1f, -8, 0f, 60));
		this.add(this.directoryListPane(), WindowSpecLayout.Position(0.5f, 4, 0f, 68, 1f, -8, 1f, -72));
		this.add(this.driveLabel(), WindowSpecLayout.Position(0.5f, 4, 1f, -62));
		this.add(this.currentDrive(), WindowSpecLayout.Position(0.5f, 4, 1f, -36, 1f, -8, 1f, -8));
		this.setPreferredSize(new Dimension(500, 300));

		// postBuildWith:
		this.getFileModel().currentDirectory();
		this.getFileModel().updateCurrentDirectoryLabel();
		this.getFileModel().updateDirectoryAndFileList();
		this.getFileModel().updateCurrentDrive();
	}

	/**
	 * Answer my model as JunFileModel.
	 *
	 * @return jp.co.sra.jun.goodies.files.JunFileModel
	 * @see jp.co.sra.jun.goodies.files.JunFileView#getFileModel()
	 * @category model accessing
	 */
	public JunFileModel getFileModel() {
		return (JunFileModel) this.model();
	}

	/**
	 * Answer the current directory label.
	 *
	 * @return javax.swing.JLabel
	 * @category user interface
	 */
	protected JLabel currentDirectoryLabel() {
		if (currentDirectoryLabel == null) {
			currentDirectoryLabel = new JLabel();

			this.getFileModel().currentDirectoryLabel().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					currentDirectoryLabel.setText((String) newValue);
					return null;
				}
			});
		}
		return currentDirectoryLabel;
	}

	/**
	 * Answer the current drive combo box.
	 *
	 * @return javax.swing.JComboBox
	 * @category user interface
	 */
	protected JComboBox currentDrive() {
		if (currentDrive == null) {
			currentDrive = new JComboBox();
			currentDrive.setLightWeightPopupEnabled(false);
			File[] drives = this.getFileModel().drives();
			for (int i = 0; i < drives.length; i++) {
				currentDrive.addItem(drives[i].getPath().toUpperCase());
			}
			currentDrive.addItem(this.getFileModel().DefaultDirectoryItem);
			currentDrive.setSelectedItem(((File) this.getFileModel().currentDrive().value()).getPath().toUpperCase());

			currentDrive.addItemListener(new ItemListener() {
				public void itemStateChanged(ItemEvent e) {
					if (doNotPropagateCurrentDriveChange) {
						return;
					}
					if (e.getStateChange() == ItemEvent.SELECTED) {
						getFileModel().currentDriveChanged((String) currentDrive.getSelectedItem());
					}
				}
			});

			this.getFileModel().currentDrive().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					String newItem = ((File) newValue).getPath().toUpperCase();
					if (newItem.equals(currentDrive.getSelectedItem()) == false) {
						doNotPropagateCurrentDriveChange = true;
						try {
							currentDrive.setSelectedItem(newItem);
						} finally {
							doNotPropagateCurrentDriveChange = false;
						}
					}
					return null;
				}
			});
		}
		return currentDrive;
	}

	/**
	 * Answer the directory label.
	 *
	 * @return javax.swing.JLabel
	 * @category user interface
	 */
	protected JLabel directoryLabel() {
		if (directoryLabel == null) {
			directoryLabel = new JunFileViewSwing.BoldLabel(JunSystem.$String("Directory") + ":");
		}
		return directoryLabel;
	}

	/**
	 * Answer the directory list.
	 *
	 * @return javax.swing.JList
	 * @category user interface
	 */
	protected JList directoryList() {
		if (directoryList == null) {
			directoryList = new JList(new DefaultListModel());

			directoryList.addMouseListener(new MouseAdapter() {
				public void mouseClicked(MouseEvent e) {
					if (e.getClickCount() == 2) {
						int index = directoryList.locationToIndex(e.getPoint());
						getFileModel().directoryListDoubleClicked(index, (String) directoryList.getModel().getElementAt(index));
					}
				}
			});

			this.getFileModel().directoryList().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					Object[] directories = (Object[]) newValue;
					int level = getFileModel().currentDirectoryLevel();
					String indent = "";
					DefaultListModel listModel = (DefaultListModel) directoryList.getModel();
					listModel.removeAllElements();
					for (int i = 0; i < directories.length; i++) {
						listModel.addElement(indent + directories[i]);
						if (i < level) {
							indent += "  ";
						}
					}
					return null;
				}
			});
		}
		return directoryList;
	}

	/**
	 * Answer the scroll pane fot the directory list.
	 *
	 * @return javax.swing.JScrollPane
	 * @category user interface
	 */
	protected JScrollPane directoryListPane() {
		if (directoryListPane == null) {
			directoryListPane = new JScrollPane(this.directoryList());
		}
		return directoryListPane;
	}

	/**
	 * Answer the drive label.
	 *
	 * @return javax.swing.JLabel
	 * @category user interface
	 */
	protected JLabel driveLabel() {
		if (driveLabel == null) {
			driveLabel = new JunFileViewSwing.BoldLabel(JunSystem.$String("Drive") + ":");
		}
		return driveLabel;
	}

	/**
	 * Answer the file label.
	 *
	 * @return javax.swing.JLabel
	 * @category user interface
	 */
	protected JLabel fileLabel() {
		if (fileLabel == null) {
			fileLabel = new JunFileViewSwing.BoldLabel(JunSystem.$String("File") + ":");
		}
		return fileLabel;
	}

	/**
	 * Answer the file list.
	 *
	 * @return java.awt.List
	 * @category user interface
	 */
	protected JList fileList() {
		if (fileList == null) {
			fileList = new JList(new DefaultListModel());
			fileList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

			fileList.addListSelectionListener(new ListSelectionListener() {
				public void valueChanged(ListSelectionEvent e) {
					if (e.getValueIsAdjusting()) {
						return;
					}
					if (fileList.isSelectionEmpty()) {
						return;
					}
					getFileModel().fileListChanged((String) fileList.getSelectedValue());
				}
			});

			this.getFileModel().fileList().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					Object[] files = (Object[]) newValue;
					DefaultListModel listModel = (DefaultListModel) fileList.getModel();
					listModel.clear();
					for (int i = 0; i < files.length; i++) {
						listModel.addElement(files[i]);
					}
					return null;
				}
			});

			this.getFileModel().fileName().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					fileList.setSelectedValue(newValue, true);
					return null;
				}
			});
		}
		return fileList;
	}

	/**
	 * Answer the scroll pane fot the directory list.
	 *
	 * @return javax.swing.JScrollPane
	 * @category user interface
	 */
	protected JScrollPane fileListPane() {
		if (fileListPane == null) {
			fileListPane = new JScrollPane(this.fileList());
		}
		return fileListPane;
	}

	/**
	 * Ansewr the file name text field.
	 *
	 * @return javax.swing.JTextField
	 * @category user interface
	 */
	protected JTextField fileName() {
		if (fileName == null) {
			fileName = new JTextField();
			fileName.setText((String) this.getFileModel().fileName().value());
			fileName.selectAll();

			fileName.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (fileName.getText().equals(getFileModel().fileName()) == false) {
						getFileModel().fileNameChanged(fileName.getText());
					}
				}
			});

			fileName.addFocusListener(new FocusAdapter() {
				public void focusLost(FocusEvent e) {
					if (fileName.getText().equals(getFileModel().fileName()) == false) {
						getFileModel().fileNameChanged(fileName.getText());
					}
				}
			});

			this.getFileModel().fileName().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					fileName.setText((String) newValue);
					return null;
				}
			});

		}
		return fileName;
	}

	/**
	 * Answer the file type combo box.
	 *
	 * @return javax.swing.JComboBox
	 * @category user interface
	 */
	protected JComboBox fileType() {
		if (fileType == null) {
			fileType = new JComboBox();
			fileType.setLightWeightPopupEnabled(false);
			JunFileModel.FileType[] fileTypes = this.getFileModel().fileTypes();
			for (int i = 0; i < fileTypes.length; i++) {
				fileType.addItem(fileTypes[i].name());
			}
			fileType.setSelectedItem(((JunFileModel.FileType) this.getFileModel().fileType().value()).name());

			fileType.addItemListener(new ItemListener() {
				public void itemStateChanged(ItemEvent e) {
					if (doNotPropagateFileTypeChange) {
						return;
					}
					if (e.getStateChange() == ItemEvent.SELECTED) {
						getFileModel().fileTypeChanged(fileType.getSelectedIndex());
					}
				}
			});

			this.getFileModel().fileType().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					String newItem = ((JunFileModel.FileType) newValue).name();
					if (newItem.equals(fileType.getSelectedItem()) == false) {
						doNotPropagateFileTypeChange = true;
						try {
							fileType.setSelectedItem(((JunFileModel.FileType) newValue).name());
						} finally {
							doNotPropagateFileTypeChange = false;
						}
					}
					return null;
				}
			});
		}
		return fileType;
	}

	/**
	 * Answer the file type label.
	 *
	 * @return javax.swing.JLabel
	 * @category user interface
	 */
	protected JLabel fileTypeLabel() {
		if (fileTypeLabel == null) {
			fileTypeLabel = new JunFileViewSwing.BoldLabel(JunSystem.$String("File Type") + ":");
		}
		return fileTypeLabel;
	}

}
