package jp.co.sra.jun.goodies.fourier;

/**
 * JunDiscreteFourierTransformation class
 * 
 *  @author    nisinaka
 *  @created   2007/06/21 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun664 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDiscreteFourierTransformation.java,v 8.6 2008/02/20 06:31:33 nisinaka Exp $
 */
public class JunDiscreteFourierTransformation extends JunFourierTransformation {

	protected double[] sourceData;
	protected double[] realData;
	protected double[] imaginaryData;
	protected double[] inverseData;

	/**
	 * Create a new instance of JunDiscreteFourierTransformation and initialize it.
	 *
	 * @param sourceData double[]
	 * @category Instance creation
	 */
	public JunDiscreteFourierTransformation(double[] sourceData) {
		this.sourceData_(sourceData);
	}

	/**
	 * Create a new instance of JunDiscreteFourierTransformation and initialize it.
	 *
	 * @param realData double[]
	 * @param imaginaryData double[]
	 * @category Instance creation
	 */
	public JunDiscreteFourierTransformation(double[] realData, double[] imaginaryData) {
		this.realData_(realData);
		this.imaginaryData_(imaginaryData);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.fourier.JunFourierTransformation#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		sourceData = null;
		realData = null;
		imaginaryData = null;
		inverseData = null;
	}

	/**
	 * Answer my current source data.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] sourceData() {
		return sourceData;
	}

	/**
	 * Set my new source data.
	 * 
	 * @param valueCollection double[]
	 * @category accessing
	 */
	public void sourceData_(double[] valueCollection) {
		sourceData = valueCollection;
		realData = null;
		imaginaryData = null;
		inverseData = null;
	}

	/**
	 * Answer my current real data.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] realData() {
		if (realData == null) {
			this.computeRealAndImaginaryData();
		}
		return realData;
	}

	/**
	 * Set my new real data.
	 * 
	 * @param valueCollection double[]
	 * @category accessing
	 */
	public void realData_(double[] valueCollection) {
		realData = valueCollection;
		inverseData = null;
	}

	/**
	 * Answer my current imaginary data.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] imaginaryData() {
		if (imaginaryData == null) {
			this.computeRealAndImaginaryData();
		}
		return imaginaryData;
	}

	/**
	 * Set my new imaginary data.
	 * 
	 * @param valueCollection double[]
	 * @category accessing
	 */
	public void imaginaryData_(double[] valueCollection) {
		imaginaryData = valueCollection;
		inverseData = null;
	}

	/**
	 * Answer my current inverse data.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] inverseData() {
		if (inverseData == null) {
			this.computeInverseData();
		}
		return inverseData;
	}

	/**
	 * Compute the real and imaginary data.
	 * 
	 * @category private
	 */
	protected void computeRealAndImaginaryData() {
		double[] x = this.sourceData();
		int n = x.length;
		double[] c = new double[n];
		double[] s = new double[n];
		double q = 2 * Math.PI / n;
		for (int i = 0; i < n; i++) {
			c[i] = 0;
			s[i] = 0;
			for (int j = 0; j < n; j++) {
				double xj = x[j];
				double yj = 0; // y[j];
				double qij = q * i * j;
				c[i] += xj * Math.cos(qij) - yj * Math.sin(qij);
				s[i] += xj * Math.sin(qij) + yj * Math.cos(qij);
			}
		}

		realData = c;
		imaginaryData = s;
	}

	/**
	 * Compute the inverse data.
	 * 
	 * @category private
	 */
	protected void computeInverseData() {
		int n = 200;
		double[] x = this.realData();
		double[] y = this.imaginaryData();
		double[] c = new double[n];
		double[] s = new double[n];
		double q = 2 * Math.PI / n * -1;
		for (int i = 0; i < n; i++) {
			c[i] = 0;
			s[i] = 0;
			for (int j = 0; j < n; j++) {
				double xj = x[j];
				double yj = y[j];
				double qij = q * i * j;
				c[i] += xj * Math.cos(qij) - yj * Math.sin(qij);
				s[i] += xj * Math.sin(qij) + yj * Math.cos(qij);
			}

			c[i] /= n;
			s[i] /= n;
		}

		inverseData = c;
	}

}
