package jp.co.sra.jun.goodies.gauge;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Collection;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.system.framework.JunAbstractController;
import jp.co.sra.jun.system.framework.JunAbstractViewPanel;
import jp.co.sra.jun.system.framework.JunApplicationModel;

/**
 * JunLevelGaugeViewAwt class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/04/03 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunLevelGaugeViewAwt.java,v 8.9 2008/02/20 06:31:34 nisinaka Exp $
 */
public class JunLevelGaugeViewAwt extends JunAbstractViewPanel implements JunLevelGaugeView {
	/**
	 * Create a new instance of <code>JunLevelGaugeViewAwt</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunLevelGaugeViewAwt() {
	}

	/**
	 * Create a new instance of <code>JunLevelGaugeViewAwt</code> and initialize it.
	 * 
	 * @param aLevelGauge jp.co.sra.jun.goodies.gauge.JunLevelGaugeModel
	 * @category Instance creation
	 */
	public JunLevelGaugeViewAwt(JunApplicationModel aLevelGauge) {
		super(aLevelGauge);
	}

	/**
	 * Build this component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#buildComponent()
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setLayout(null);
		if (this.verticalGauge()) {
			this.setSize(128, 256);
		} else {
			this.setSize(256, 128);
		}
	}

	/**
	 * Answer the receiver's average value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double averageValue() {
		double aValue = this.getModel().average();
		if (Double.isNaN(aValue)) {
			aValue = this.currentValue();
		}
		return aValue;
	}

	/**
	 * Answer the receiver's current value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double currentValue() {
		double aValue = this.getModel()._doubleValue();
		if (Double.isNaN(aValue)) {
			aValue = 0;
		}
		return aValue;
	}

	/**
	 * Answer the receiver's dale value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double daleValue() {
		double aValue = this.getModel().dale();
		if (Double.isNaN(aValue)) {
			aValue = this.currentValue();
		}
		return aValue;
	}

	/**
	 * Answer the receiver's hill value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double hillValue() {
		double aValue = this.getModel().hill();
		if (Double.isNaN(aValue)) {
			aValue = this.currentValue();
		}
		return aValue;
	}

	/**
	 * Answer the receiver's maximum value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double maximumValue() {
		double aValue = this.getModel().maximum();
		if (Double.isNaN(aValue)) {
			aValue = 1;
		}
		return aValue;
	}

	/**
	 * Answer the receiver's minimum value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double minimumValue() {
		double aValue = this.getModel().minimum();
		if (Double.isNaN(aValue)) {
			aValue = -1;
		}
		return aValue;
	}

	/**
	 * Answer the receiver's average box.
	 * 
	 * @return java.awt.Rectangle
	 * @category box accessing
	 */
	public Rectangle averageBox() {
		double minimumValue = this.minimumValue();
		double maximumValue = this.maximumValue();
		double aRatio = (this.averageValue() - minimumValue) / (maximumValue - minimumValue);
		return this.boxFromRatio_(aRatio);
	}

	/**
	 * Answer the receiver's dale box.
	 * 
	 * @return java.awt.Rectangle
	 * @category box accessing
	 */
	public Rectangle daleBox() {
		double minimumValue = this.minimumValue();
		double maximumValue = this.maximumValue();
		double aRatio = (this.daleValue() - minimumValue) / (maximumValue - minimumValue);
		return this.boxFromRatio_(aRatio);
	}

	/**
	 * Answer the receiver's display box.
	 * 
	 * @return java.awt.Rectangle
	 * @category box accessing
	 */
	public Rectangle displayBox() {
		Rectangle aBox;
		if (this.verticalGauge()) {
			Point aPoint = new Point((int) Math.floor(this.getWidth() / 2.0) - (int) Math.floor(this.gaugeWidth() / 2.0), Math.max(this.cellSize(), 2));
			aBox = new StRectangle(new Rectangle(0, 0, this.getWidth(), this.getHeight())).insetBy_(aPoint).toRectangle();
		} else {
			Point aPoint = new Point(Math.max(this.cellSize(), 2), (int) Math.floor(this.getHeight() / 2.0) - (int) Math.floor(this.gaugeWidth() / 2.0));
			aBox = new StRectangle(new Rectangle(0, 0, this.getWidth(), this.getHeight())).insetBy_(aPoint).toRectangle();
		}
		return aBox;
	}

	/**
	 * Answer the receiver's gauge box.
	 * 
	 * @return java.awt.Rectangle
	 * @category box accessing
	 */
	public Rectangle gaugeBox() {
		Rectangle displayBox = this.displayBox();
		Rectangle aBox;
		if (this.verticalGauge()) {
			aBox = new StRectangle(displayBox).insetBy_(new Point(0, (int) Math.floor(this.cellSize() / 2.0))).toRectangle();
		} else {
			aBox = new StRectangle(displayBox).insetBy_(new Point((int) Math.floor(this.cellSize() / 2.0), 0)).toRectangle();
		}
		return aBox;
	}

	/**
	 * Answer the receiver's hill box.
	 * 
	 * @return java.awt.Rectangle
	 * @category box accessing
	 */
	public Rectangle hillBox() {
		double minimumValue = this.minimumValue();
		double maximumValue = this.maximumValue();
		double aRatio = (this.hillValue() - minimumValue) / (maximumValue - minimumValue);
		return this.boxFromRatio_(aRatio);
	}

	/**
	 * Answer the receiver's value box.
	 * 
	 * @return java.awt.Rectangle
	 * @category box accessing
	 */
	public Rectangle valueBox() {
		double minimumValue = this.minimumValue();
		double maximumValue = this.maximumValue();
		double aRatio = (this.currentValue() - minimumValue) / (maximumValue - minimumValue);
		return this.boxFromRatio_(aRatio);
	}

	/**
	 * Answer the receiver's default controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @see jp.co.sra.smalltalk.StViewJPanel#defaultController()
	 * @category controller accessing
	 */
	public StController defaultController() {
		return new JunAbstractController();
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StViewJPanel#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics graphicsContext) {
		if (this.isShowing() == false) {
			return;
		}

		this.displayBackgroundOn_(graphicsContext);
		this.displayFrameOn_(graphicsContext);
		this.displayValueOn_(graphicsContext);
		this.displayAverageOn_(graphicsContext);
		this.displayDaleOn_(graphicsContext);
		this.displayHillOn_(graphicsContext);
	}

	/**
	 * Display the receiver's average on the specified graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	protected void displayAverageOn_(Graphics graphicsContext) {
		if (this.showAverage() == false) {
			return;
		}

		Rectangle aBox = this.averageBox();
		Color aColor = this.averageColor();
		if (this.rainbowColor()) {
			aColor = this.colorValueOfBox_withColor_(aBox, aColor);
		}
		if (this.showEdge()) {
			graphicsContext.setColor(this.edgeColor());
			graphicsContext.fillRect(aBox.x, aBox.y, aBox.width, aBox.height);
			graphicsContext.setColor(aColor);
			if (this.verticalGauge()) {
				Rectangle insetBox = new StRectangle(aBox).insetBy_(new Point(0, 1)).toRectangle();
				graphicsContext.fillRect(insetBox.x, insetBox.y, insetBox.width, insetBox.height);
			} else {
				Rectangle insetBox = new StRectangle(aBox).insetBy_(new Point(1, 0)).toRectangle();
				graphicsContext.fillRect(insetBox.x, insetBox.y, insetBox.width, insetBox.height);
			}
		} else {
			graphicsContext.setColor(aColor);
			graphicsContext.fillRect(aBox.x, aBox.y, aBox.width, aBox.height);
		}
	}

	/**
	 * Display the receiver's background on the specified graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	protected void displayBackgroundOn_(Graphics graphicsContext) {
		if (this.showBackground() == false) {
			return;
		}

		graphicsContext.setColor(this.backgroundColor());
		graphicsContext.fillRect(0, 0, this.getWidth(), this.getHeight());
	}

	/**
	 * Display the receiver's dale on the specified graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	protected void displayDaleOn_(Graphics graphicsContext) {
		if (this.showDale() == false) {
			return;
		}

		Rectangle aBox = this.daleBox();
		Color aColor = this.daleColor();
		if (this.rainbowColor()) {
			aColor = this.colorValueOfBox_withColor_(aBox, aColor);
		}
		if (this.showEdge()) {
			graphicsContext.setColor(this.edgeColor());
			graphicsContext.fillRect(aBox.x, aBox.y, aBox.width, aBox.height);
			graphicsContext.setColor(aColor);
			if (this.verticalGauge()) {
				Rectangle insetBox = new StRectangle(aBox).insetBy_(new Point(0, 1)).toRectangle();
				graphicsContext.fillRect(insetBox.x, insetBox.y, insetBox.width, insetBox.height);
			} else {
				Rectangle insetBox = new StRectangle(aBox).insetBy_(new Point(1, 0)).toRectangle();
				graphicsContext.fillRect(insetBox.x, insetBox.y, insetBox.width, insetBox.height);
			}
		} else {
			graphicsContext.setColor(aColor);
			graphicsContext.fillRect(aBox.x, aBox.y, aBox.width, aBox.height);
		}
	}

	/**
	 * Display the receiver's frame on the specified graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	protected void displayFrameOn_(Graphics graphicsContext) {
		if (this.showFrame() == false) {
			return;
		}

		graphicsContext.setColor(this.frameColor());
		Rectangle displayBox = this.displayBox();
		graphicsContext.fillRect(displayBox.x, displayBox.y, displayBox.width, displayBox.height);
	}

	/**
	 * Display the receiver's hill on the specified graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	protected void displayHillOn_(Graphics graphicsContext) {
		if (this.showHill() == false) {
			return;
		}

		Rectangle aBox = this.hillBox();
		Color aColor = this.hillColor();
		if (this.rainbowColor()) {
			aColor = this.colorValueOfBox_withColor_(aBox, aColor);
		}
		if (this.showEdge()) {
			graphicsContext.setColor(this.edgeColor());
			graphicsContext.fillRect(aBox.x, aBox.y, aBox.width, aBox.height);
			graphicsContext.setColor(aColor);
			if (this.verticalGauge()) {
				Rectangle insetBox = new StRectangle(aBox).insetBy_(new Point(0, 1)).toRectangle();
				graphicsContext.fillRect(insetBox.x, insetBox.y, insetBox.width, insetBox.height);
			} else {
				Rectangle insetBox = new StRectangle(aBox).insetBy_(new Point(1, 0)).toRectangle();
				graphicsContext.fillRect(insetBox.x, insetBox.y, insetBox.width, insetBox.height);
			}
		} else {
			graphicsContext.setColor(aColor);
			graphicsContext.fillRect(aBox.x, aBox.y, aBox.width, aBox.height);
		}
	}

	/**
	 * Display the receiver's value on the specified graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	protected void displayValueOn_(Graphics graphicsContext) {
		if (this.showValue() == false) {
			return;
		}

		StRectangle gaugeBox = new StRectangle(this.gaugeBox());
		StRectangle valueBox = new StRectangle(this.valueBox());
		Collection boxCollection = new ArrayList();
		StRectangle aBox = valueBox;
		if (this.verticalGauge()) {
			if (this.reverseDirection()) {
				aBox = aBox.align_with_(aBox.center(), gaugeBox.topCenter());
				while (aBox.cornerY() < valueBox.cornerY()) {
					boxCollection.add(aBox);
					aBox = aBox.translatedBy_(0, aBox.height() + this.cellSpace());
				}
			} else {
				aBox = aBox.align_with_(aBox.center(), gaugeBox.bottomCenter());
				while (aBox.originY() > valueBox.originY()) {
					boxCollection.add(aBox);
					aBox = aBox.translatedBy_(0, 0 - (aBox.height() + this.cellSpace()));
				}
			}
		} else {
			if (this.reverseDirection()) {
				aBox = aBox.align_with_(aBox.center(), gaugeBox.rightCenter());
				while (aBox.originX() > valueBox.originX()) {
					boxCollection.add(aBox);
					aBox = aBox.translatedBy_(0 - (aBox.width() + this.cellSpace()), 0);
				}
			} else {
				aBox = aBox.align_with_(aBox.center(), gaugeBox.leftCenter());
				while (aBox.cornerX() < valueBox.cornerX()) {
					boxCollection.add(aBox);
					aBox = aBox.translatedBy_(aBox.width() + this.cellSpace(), 0);
				}
			}
		}
		if (aBox.intersects_(valueBox)) {
			aBox = aBox.intersect_(valueBox);
			boxCollection.add(aBox);
		}
		StRectangle[] boxes = (StRectangle[]) boxCollection.toArray(new StRectangle[boxCollection.size()]);
		for (int n = 0; n < boxes.length; n++) {
			aBox = boxes[n];
			Color aColor = this.valueColor();
			if (this.rainbowColor()) {
				aColor = this.colorValueOfBox_withColor_(aBox.toRectangle(), aColor);
			}
			if (this.showEdge()) {
				graphicsContext.setColor(this.edgeColor());
				graphicsContext.fillRect(aBox.originX(), aBox.originY(), aBox.width(), aBox.height());
				graphicsContext.setColor(aColor);
				if (this.verticalGauge()) {
					aBox = aBox.insetBy_(new Point(0, 2));
					graphicsContext.fillRect(aBox.originX(), aBox.originY(), aBox.width(), aBox.height());
				} else {
					aBox = aBox.insetBy_(new Point(2, 0));
					graphicsContext.fillRect(aBox.originX(), aBox.originY(), aBox.width(), aBox.height());
				}
			} else {
				graphicsContext.setColor(aColor);
				graphicsContext.fillRect(aBox.originX(), aBox.originY(), aBox.width(), aBox.height());
			}
		}
	}

	/**
	 * Answer the receiver's model as JunLevelGaugeModel
	 * 
	 * @return jp.co.sra.jun.goodies.gauge.JunLevelGaugeModel
	 * @see jp.co.sra.jun.goodies.gauge.JunLevelGaugeView#getModel()
	 * @category model accessing
	 */
	public JunLevelGaugeModel getModel() {
		return (JunLevelGaugeModel) this.model();
	}

	/**
	 * Answer the average color.
	 * 
	 * @return java.awt.Color.
	 * @category preferences
	 */
	public Color averageColor() {
		return this.getModel().averageColor();
	}

	/**
	 * Answer the background color.
	 * 
	 * @return java.awt.Color
	 * @category preferences
	 */
	public Color backgroundColor() {
		return this.getModel().backgroundColor();
	}

	/**
	 * Answer the cell size.
	 * 
	 * @return int
	 * @category preferences
	 */
	public int cellSize() {
		return this.getModel().cellSize();
	}

	/**
	 * Answer the cell space.
	 * 
	 * @return int
	 * @category preferences
	 */
	public int cellSpace() {
		return this.getModel().cellSpace();
	}

	/**
	 * Answer the dale color.
	 * 
	 * @return java.awt.Color
	 * @category preferences
	 */
	public Color daleColor() {
		return this.getModel().daleColor();
	}

	/**
	 * Answer the edge color.
	 * 
	 * @return java.awt.Color
	 * @category preferences
	 */
	public Color edgeColor() {
		return this.getModel().edgeColor();
	}

	/**
	 * Answer the frame color.
	 * 
	 * @return java.awt.Color
	 * @category preferences
	 */
	public Color frameColor() {
		return this.getModel().frameColor();
	}

	/**
	 * Answer the gauge width.
	 * 
	 * @return int
	 * @category preferences
	 */
	public int gaugeWidth() {
		return this.getModel().gaugeWidth();
	}

	/**
	 * Answer the hill color.
	 * 
	 * @return java.awt.Color
	 * @category preferences
	 */
	public Color hillColor() {
		return this.getModel().hillColor();
	}

	/**
	 * Answer true if the receiver is horizontal gauge, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean horizontalGauge() {
		return this.getModel().horizontalGauge();
	}

	/**
	 * Answer true if the receiver is rainbow color, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean rainbowColor() {
		return this.getModel().rainbowColor();
	}

	/**
	 * Answer true if the receiver reverse direction, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean reverseDirection() {
		return this.getModel().reverseDirection();
	}

	/**
	 * Answer true if the receiver show average, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean showAverage() {
		return this.getModel().showAverage();
	}

	/**
	 * Answer true if the receiver show background, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean showBackground() {
		return this.getModel().showBackground();
	}

	/**
	 * Answer true if the receiver show dale, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean showDale() {
		return this.getModel().showDale();
	}

	/**
	 * Answer true if the receiver show edge, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean showEdge() {
		return this.getModel().showEdge();
	}

	/**
	 * Answer true if the receiver show frame, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean showFrame() {
		return this.getModel().showFrame();
	}

	/**
	 * Answer true if the receiver show hill, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean showHill() {
		return this.getModel().showHill();
	}

	/**
	 * Answer true if the receiver show value, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean showValue() {
		return this.getModel().showValue();
	}

	/**
	 * Answer the edge color.
	 * 
	 * @return java.awt.Color
	 * @category preferences
	 */
	public Color valueColor() {
		return this.getModel().valueColor();
	}

	/**
	 * Answer true if the receiver is vertical gauge, otherwise false.
	 * 
	 * @return boolean
	 * @category preferences
	 */
	public boolean verticalGauge() {
		return this.getModel().verticalGauge();
	}

	/**
	 * Receive a change notice from an object of whom the receiver is a
	 * dependent.  The argument anAspectSymbol is typically a Symbol that
	 * indicates what change has occurred.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		StSymbol aspectSymbol = evt.getAspect();
		if (aspectSymbol == $("value")) {
			Graphics aGraphics = this.getGraphics();
			try {
				this.displayOn_(aGraphics);
				// this.getModel().displayPendingInvalidation();
			} finally {
				if (aGraphics != null) {
					aGraphics.dispose();
				}
			}
			return;
		}
		super.update_(evt);
	}

	/**
	 * Answer the box from the specified ratio.
	 * 
	 * @param aRatio double
	 * @return java.awt.Rectangle
	 * @category private
	 */
	protected Rectangle boxFromRatio_(double aRatio) {
		Rectangle gaugeBox = this.gaugeBox();
		Rectangle aBox;
		if (this.verticalGauge()) {
			if (this.reverseDirection()) {
				aBox = new Rectangle(gaugeBox.x, (int) Math.round(aRatio * gaugeBox.height) + gaugeBox.y - (int) Math.floor(this.cellSize() / 2.0), gaugeBox.width, this.cellSize());
			} else {
				aBox = new Rectangle(gaugeBox.x, (int) Math.round((1 - aRatio) * gaugeBox.height) + gaugeBox.y - (int) Math.floor(this.cellSize() / 2.0), gaugeBox.width, this.cellSize());
			}
		} else {
			if (this.reverseDirection()) {
				aBox = new Rectangle((int) Math.round((1 - aRatio) * gaugeBox.width) + gaugeBox.x - (int) Math.floor(this.cellSize() / 2.0), gaugeBox.y, this.cellSize(), gaugeBox.height);
			} else {
				aBox = new Rectangle((int) Math.round(aRatio * gaugeBox.width) + gaugeBox.x - (int) Math.floor(this.cellSize() / 2.0), gaugeBox.y, this.cellSize(), gaugeBox.height);
			}
		}
		return aBox;
	}

	/**
	 * Answer the color of the specified box with a color.
	 * 
	 * @param aBox java.awt.Rectangle
	 * @param aColor java.awt.Color
	 * @return java.awt.Color
	 * @category private
	 */
	protected Color colorValueOfBox_withColor_(Rectangle aBox, Color aColor) {
		float normalizedValue = this.normalizedValueOfBox_(aBox);
		Color colorValue = Color.getHSBColor((1 - normalizedValue) * (2f / 3), 1, StColorValue._GetBrightness(aColor));
		return colorValue;
	}

	/**
	 * Answer the normalized Value of the specified box.
	 * 
	 * @param aBox java.awt.Rectangle
	 * @return float
	 * @category private
	 */
	protected float normalizedValueOfBox_(Rectangle aBox) {
		Rectangle gaugeBox = this.gaugeBox();
		float normalizedValue;
		if (this.verticalGauge()) {
			if (this.reverseDirection()) {
				normalizedValue = (aBox.y + aBox.height / 2f - gaugeBox.y) / (gaugeBox.y + gaugeBox.height - gaugeBox.y);
			} else {
				normalizedValue = (gaugeBox.y + gaugeBox.height - (aBox.y + aBox.height / 2f)) / (gaugeBox.y + gaugeBox.height - gaugeBox.y);
			}
		} else {
			if (this.reverseDirection()) {
				normalizedValue = (gaugeBox.x + gaugeBox.width - (aBox.x + aBox.width / 2f)) / (gaugeBox.x + gaugeBox.width - gaugeBox.x);
			} else {
				normalizedValue = (aBox.x + aBox.width / 2f - gaugeBox.x) / (gaugeBox.x + gaugeBox.width - gaugeBox.x);
			}
		}
		normalizedValue = Math.max(0, Math.min(normalizedValue, 1));
		return normalizedValue;
	}
}
