package jp.co.sra.jun.goodies.image.support;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;

import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StRectangle;

import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunImageBorderDecorator class
 * 
 *  @author    nisinaka
 *  @created   2006/07/14 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun576 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunImageBorderDecorator.java,v 8.12 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunImageBorderDecorator extends JunAbstractObject {

	protected StImage originalImage;
	protected Dimension imageExtent;
	protected boolean keepAspect;
	protected Color boxColor;
	protected Color borderColor;
	protected int borderWidth;

	/**
	 * Create a new instance of JunImageBorderDecorator and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunImageBorderDecorator() {
		super();
	}

	/**
	 * Create a new instance of JunImageBorderDecorator and initialize it.
	 *
	 * @param anImage java.awt.Image
	 * @category Instance creation
	 */
	public JunImageBorderDecorator(Image anImage) {
		this(new StImage(anImage));
	}

	/**
	 * Create a new instance of JunImageBorderDecorator and initialize it.
	 *
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @category Instance creation
	 */
	public JunImageBorderDecorator(StImage anImage) {
		this();
		this.originalImage_(anImage);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		originalImage = null;
		imageExtent = null;
		keepAspect = true;
		boxColor = this.defaultBoxColor();
		borderColor = this.defaultBorderColor();
		borderWidth = this.defaultBorderWidth();
	}

	/**
	 * Answer my current original image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	public StImage originalImage() {
		return originalImage;
	}

	/**
	 * Set my new original image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	public void originalImage_(StImage anImage) {
		originalImage = anImage;
	}

	/**
	 * Answer my current image extent.
	 * 
	 * @return java.awt.Dimension
	 * @category accessing
	 */
	public Dimension imageExtent() {
		if (imageExtent == null) {
			imageExtent = this.minimumImageExtent();
			if (this.originalImage() != null) {
				int width = Math.max(imageExtent.width, this.originalImage().width());
				int height = Math.max(imageExtent.height, this.originalImage().height());
				imageExtent = new Dimension(width, height);
			}
		}

		return imageExtent;
	}

	/**
	 * Set my new image extent.
	 * 
	 * @param aDimension java.awt.Dimension
	 * @category accessing
	 */
	public void imageExtent_(Dimension aDimension) {
		imageExtent = aDimension;
	}

	/**
	 * Answer true if the receiver keeps its aspect ratio when creating an image, otherwise false.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean keepAspect() {
		return keepAspect;
	}

	/**
	 * Set whether the reciever keeps its aspect ratio when creating an image or not.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void keepAspect_(boolean aBoolean) {
		keepAspect = aBoolean;
	}

	/**
	 * Answer my current box color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color boxColor() {
		return boxColor;
	}

	/**
	 * Set my new box color.
	 * 
	 * @param aColor java.awt.Color
	 * @category accessing
	 */
	public void boxColor_(Color aColor) {
		boxColor = aColor;
	}

	/**
	 * Answer my current border color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color borderColor() {
		return borderColor;
	}

	/**
	 * Set my new border color.
	 * 
	 * @param aColor java.awt.Color
	 * @category accessing
	 */
	public void borderColor_(Color aColor) {
		borderColor = aColor;
	}

	/**
	 * Answer my current border width.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int borderWidth() {
		return borderWidth;
	}

	/**
	 * Set my new border width.
	 * 
	 * @param anInteger int
	 * @category accessing
	 */
	public void borderWidth_(int anInteger) {
		borderWidth = anInteger;
	}

	/**
	 * Create a simple image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category image accessing
	 */
	public StImage simpleImage() {
		return this.decoratedImage(this.originalImage(), this.imageExtent(), Color.white, 0, Color.white);
	}

	/**
	 * Create a decorated image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category image accessing
	 */
	public StImage decoratedImage() {
		return this.decoratedImage(this.originalImage(), this.imageExtent(), this.boxColor(), this.borderWidth(), this.borderColor());
	}

	/**
	 * Create a first image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category image accessing
	 */
	public StImage firstImage() {
		return this.decoratedImage(this.originalImage(), this.imageExtent(), this.defaultFirstBoxColor(), this.borderWidth(), this.defaultFirstBorderColor());
	}

	/**
	 * Create a current image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category image accessing
	 */
	public StImage currentImage() {
		return this.decoratedImage(this.originalImage(), this.imageExtent(), this.defaultCurrentBoxColor(), this.borderWidth(), this.defaultCurrentBorderColor());
	}

	/**
	 * Create a last image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category image accessing
	 */
	public StImage lastImage() {
		return this.decoratedImage(this.originalImage(), this.imageExtent(), this.defaultLastBoxColor(), this.borderWidth(), this.defaultLastBorderColor());
	}

	/**
	 * Create a decorated image with the specified parameters.
	 * 
	 * @param baseImage jp.co.sra.smalltalk.StImage
	 * @param extent java.awt.Dimension
	 * @param backgroundColor java.awt.Color
	 * @param lineWidth int
	 * @param lineColor java.awt.Color
	 * @return jp.co.sra.smalltalk.StImage
	 * @category image accessing
	 */
	protected StImage decoratedImage(StImage baseImage, Dimension extent, Color backgroundColor, int lineWidth, Color lineColor) {
		if (baseImage == null) {
			return this.oneColorImage(backgroundColor, extent);
		}

		StRectangle aBox = new StRectangle(extent);
		Dimension adjustImageExtent = aBox.insetBy_(lineWidth).extent();
		StImage anImage = JunImageAdjuster.AdjustImage_extent_keepAspect_(baseImage, adjustImageExtent, this.keepAspect());
		StRectangle imageBounds = new StRectangle(anImage.bounds());
		StImage theImage = new StImage(extent);

		Graphics aGraphics = null;
		try {
			aGraphics = theImage.image().getGraphics();
			aGraphics.setColor(backgroundColor);
			aGraphics.fillRect(0, 0, aBox.width(), aBox.height());

			StRectangle displayRectangle = imageBounds.expandedBy_(lineWidth);
			Point displayPoint = displayRectangle.align_with_(displayRectangle.center(), aBox.center()).origin();
			// displayPoint.translate(lineWidth, lineWidth);
			aGraphics.setColor(lineColor);
			aGraphics.fillRect(displayPoint.x, displayPoint.y, displayRectangle.width(), displayRectangle.height());

			displayPoint = imageBounds.align_with_(imageBounds.center(), aBox.center()).origin();
			anImage.displayOn_at_(aGraphics, displayPoint);

		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}

		return theImage;
	}

	/**
	 * Create a one color image with the specified parameters.
	 * 
	 * @param aColor java.awt.Color
	 * @param extent java.awt.Dimension
	 * @return jp.co.sra.smalltalk.StImage
	 * @category image accessing
	 */
	protected StImage oneColorImage(Color aColor, Dimension extent) {
		StImage anImage = new StImage(extent.width, extent.height);

		Graphics aGraphics = null;
		try {
			aGraphics = anImage.image().getGraphics();
			aGraphics.setColor(aColor);
			aGraphics.fillRect(0, 0, extent.width, extent.height);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}

		return anImage;
	}

	/**
	 * Answer the default box color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultBoxColor() {
		return this.defaultBorderColor();
	}

	/**
	 * Answer the default border color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultBorderColor() {
		return Color.white;
	}

	/**
	 * Answer the default border width.
	 * 
	 * @return int
	 * @category defaults
	 */
	protected int defaultBorderWidth() {
		return 1;
	}

	/**
	 * Answer the default first box color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultFirstBoxColor() {
		return StColorValue.Blend(Color.white, StColorValue.Blend(Color.white, this.defaultFirstBorderColor()));
	}

	/**
	 * Answer the default first border color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultFirstBorderColor() {
		return Color.blue;
	}

	/**
	 * Answer the default current box color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultCurrentBoxColor() {
		return StColorValue.Blend(Color.white, StColorValue.Blend(Color.white, this.defaultCurrentBorderColor()));
	}

	/**
	 * Answer the default current border color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultCurrentBorderColor() {
		return Color.green;
	}

	/**
	 * Answer the default last box color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultLastBoxColor() {
		return StColorValue.Blend(Color.white, StColorValue.Blend(Color.white, this.defaultLastBorderColor()));
	}

	/**
	 * Answer the default last border color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultLastBorderColor() {
		return Color.red;
	}

	/**
	 * Answer the minimum image extent.
	 * 
	 * @return java.awt.Dimension
	 * @category private
	 */
	protected Dimension minimumImageExtent() {
		StRectangle aRectangle = new StRectangle(0, 0, 30, 30);
		aRectangle = aRectangle.expandedBy_(this.borderWidth());
		return aRectangle.extent();
	}

}
