package jp.co.sra.jun.goodies.itemlist;

import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.ScrollPane;
import java.awt.image.BufferedImage;
import java.util.Vector;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.StViewPanel;
import jp.co.sra.smalltalk.WindowSpecLayout;
import jp.co.sra.jun.goodies.button.JunButtonViewAwt;
import jp.co.sra.jun.goodies.cursors.JunCursors;

/**
 * JunItemListViewAwt class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2003/12/12 (by Mitsuhiro Asada)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun465 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunItemListViewAwt.java,v 8.10 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunItemListViewAwt extends JunOrderListViewAwt implements JunItemListView {
	protected Vector checkMarkRectangles;

	protected JunButtonViewAwt checkOffButtonView;
	protected JunButtonViewAwt checkOnButtonView;

	/**
	 * Create a new instance of JunItemListViewAwt.
	 * 
	 * @param newModel jp.co.sra.jun.goodies.itemlist.JunItemListModel
	 * @category Instance creation
	 */
	public JunItemListViewAwt(JunItemListModel newModel) {
		super(newModel);
	}

	/**
	 * Create a new view for the JunItemListViewAwt.
	 * The view consists of a JunItemListViewAwt on a ScrollPane.
	 * The view having up/down buttons.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @param newModel jp.co.sra.jun.goodies.itemlist.JunItemListModel
	 * @category Instance creation
	 */
	public static StView OnScrollPaneWithButtons(JunItemListModel newModel) {
		return new StViewPanel(newModel) {
			protected JunItemListViewAwt itemListView;

			protected void buildComponent() {
				itemListView = new JunItemListViewAwt((JunItemListModel) this.model());
				ScrollPane scrollPane = new ScrollPane(ScrollPane.SCROLLBARS_ALWAYS) {
					public void setBounds(int x, int y, int width, int height) {
						super.setBounds(x, y, width, height);
						itemListView.setSize(this.getViewportSize().width, this.getViewportSize().height);
					}
				};
				scrollPane.add(itemListView);

				this.setLayout(new WindowSpecLayout());
				this.add(scrollPane, WindowSpecLayout.Position(0f, 1, 0f, 1, 1f, -20, 1f, -1));
				this.add(itemListView.upButtonView(), WindowSpecLayout.Position(1f, -19, 0.3f, -19, 1f, -1, 0.3f, -1));
				this.add(itemListView.downButtonView(), WindowSpecLayout.Position(1f, -19, 0.3f, 1, 1f, -1, 0.3f, 19));
				this.add(itemListView.checkOnButtonView(), WindowSpecLayout.Position(1f, -19, 0.7f, -19, 1f, -1, 0.7f, -1));
				this.add(itemListView.checkOffButtonView(), WindowSpecLayout.Position(1f, -19, 0.7f, 1, 1f, -1, 0.7f, 19));
				this.setSize(150, 150);
			}
		};
	}

	/**
	 * Answer my check mark rectangles.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector checkMarkRectangles() {
		if (checkMarkRectangles == null) {
			checkMarkRectangles = new Vector();
		}
		return checkMarkRectangles;
	}

	/**
	 * Answer my left margin size.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int leftMargin() {
		return 23;
	}

	/**
	 * Answer my default controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @category controller accessing
	 */
	protected StController defaultController() {
		return new JunItemListController();
	}

	/**
	 * Answer my model as a JunItemListModel.
	 * 
	 * @return jp.co.sra.jun.goodies.itemlist.JunItemListModel
	 * @see jp.co.sra.jun.goodies.itemlist.JunItemListView#getItemListModel()
	 * @category model accessing
	 */
	public JunItemListModel getItemListModel() {
		return (JunItemListModel) this.model();
	}

	/**
	 * Answer my check off button view.
	 *
	 * @return jp.co.sra.jun.goodies.button.JunButtonViewAwt
	 * @category interface opening
	 */
	protected JunButtonViewAwt checkOffButtonView() {
		if (checkOffButtonView == null) {
			checkOffButtonView = new JunButtonViewAwt(this.getItemListModel().checkOffButton());
		}
		return checkOffButtonView;
	}

	/**
	 * Answer my check on button view.
	 *
	 * @return jp.co.sra.jun.goodies.button.JunButtonViewAwt
	 * @category interface opening
	 */
	protected JunButtonViewAwt checkOnButtonView() {
		if (checkOnButtonView == null) {
			checkOnButtonView = new JunButtonViewAwt(this.getItemListModel().checkOnButton());
		}
		return checkOnButtonView;
	}

	/**
	 * Paints this component.
	 * 
	 * @param aGraphicsContext java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StViewCanvas#displayOn_(Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphicsContext) {
		if (this.isShowing() == false) {
			return;
		}

		Rectangle clippingBounds = aGraphicsContext.getClipBounds();
		if (clippingBounds == null) {
			clippingBounds = this.getBounds();
		}
		Graphics graphicsContext = aGraphicsContext.create();
		try {
			graphicsContext.clearRect(clippingBounds.x, clippingBounds.y, clippingBounds.width, clippingBounds.height);
			if (this.getModel().isEmpty()) {
				super.displayOn_(graphicsContext);
				return;
			}
			for (int anIndex = 0; anIndex < this.getModel().size(); anIndex++) {
				JunItemListObject itemObject = (JunItemListObject) this.itemObjectAt_(anIndex);
				Rectangle rowBox = this.rowBoundsAt_(anIndex);
				rowBox = new Rectangle(rowBox.x + this.leftMargin(), rowBox.y, rowBox.width - this.leftMargin(), rowBox.height);
				if (this.getModel().isSelected_(anIndex)) {
					graphicsContext.setColor(this.selectionBackgroundColor());
					graphicsContext.fillRect(rowBox.x, rowBox.y, rowBox.width, rowBox.height);
				}

				Rectangle checkMarkBox = (Rectangle) this.checkMarkRectangles().get(anIndex);
				BufferedImage opaqueImage = itemObject.checkMark() ? JunCursors.CheckOnCursorImage() : JunCursors.CheckOffCursorImage();
				graphicsContext.drawImage(opaqueImage, checkMarkBox.x, checkMarkBox.y, this);

				StComposedText composedText = new StComposedText(itemObject.itemString());
				if (this.getModel().isSelected_(anIndex)) {
					graphicsContext.setColor(this.selectionForegroundColor());
				} else {
					graphicsContext.setColor(this.getForeground());
				}
				composedText.displayOn_at_(graphicsContext, new Point(rowBox.x + this.textMargin(), (rowBox.y + rowBox.height / 2) - composedText.height() / 2));
			}
		} finally {
			if (graphicsContext != null) {
				graphicsContext.dispose();
				graphicsContext = null;
			}
		}
	}

	/**
	 * Answer the index position which item object rectangle contains a point.
	 * 
	 * @return int
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.goodies.itemlist.JunItemListView#whichOfCheckMarksAt_(java.awt.Point)
	 * @category selecting
	 */
	public int whichOfCheckMarksAt_(Point aPoint) {
		Rectangle[] boxes = this._checkMarkRectangleArray();
		for (int anIndex = 0; anIndex < boxes.length; anIndex++) {
			if (boxes[anIndex].contains(aPoint)) {
				return anIndex;
			}
		}
		return -1;
	}

	/**
	 * Flush cached information.
	 * 
	 * @category private
	 */
	protected void flushCachedInformation() {
		super.flushCachedInformation();
		checkMarkRectangles = null;
	}

	/**
	 * Compute cached information.
	 * 
	 * @category private
	 */
	protected void computeCachedInformation() {
		checkMarkRectangles = new Vector();
		itemVisualRectangles = new Vector();
		itemRowRectangles = new Vector();
		boundingBox = new Rectangle(0, 0, 0, 0);
		if (this.getModel().isEmpty()) {
			return;
		}
		final int lineGrid = this.getModel().lineGrid();
		final int leftMargin = this.leftMargin();
		final Point aPoint = new Point(0, 0);
		this.getModel().do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunItemListObject itemObject = (JunItemListObject) anObject;

				Rectangle aBox = new Rectangle(aPoint.x, aPoint.y, leftMargin, lineGrid);
				BufferedImage opaqueImage = (itemObject.checkMark()) ? JunCursors.CheckOnCursorImage() : JunCursors.CheckOffCursorImage();
				aBox = new Rectangle((aBox.x + aBox.width / 2) - (opaqueImage.getWidth() / 2), (aBox.y + aBox.height / 2) - (opaqueImage.getHeight() / 2), opaqueImage.getWidth(), opaqueImage.getHeight());
				JunItemListViewAwt.this.checkMarkRectangles.add(aBox);
				JunItemListViewAwt.this.boundingBox.add(aBox);

				aBox = new Rectangle(aPoint.x + leftMargin, aPoint.y, JunItemListViewAwt.this.getBounds().width - leftMargin, lineGrid);
				StComposedText composedText = new StComposedText(itemObject.itemString());
				aBox = new Rectangle(aBox.x + JunItemListViewAwt.this.textMargin() * 2, (aBox.y + aBox.height / 2) - (composedText.height() / 2), composedText.width(), composedText.height());
				JunItemListViewAwt.this.itemVisualRectangles.add(aBox);
				JunItemListViewAwt.this.boundingBox.add(aBox);

				aBox = new Rectangle(aPoint.x, aPoint.y, JunItemListViewAwt.this.getBounds().width, lineGrid);
				JunItemListViewAwt.this.itemRowRectangles.add(aBox);
				JunItemListViewAwt.this.boundingBox.add(aBox);

				aPoint.translate(0, lineGrid);
				return null;
			}
		});

		Rectangle[] rowRects = this._itemRowRectangleArray();
		for (int i = 0; i < rowRects.length; i++) {
			rowRects[i].width = boundingBox.width - boundingBox.x;
		}
	}

}
