package jp.co.sra.jun.goodies.lisp;

import jp.co.sra.smalltalk.*;

/**
 * JunLispParser class
 * 
 *  @author    ASTI Beijing
 *  @created   1998/12/09 (by ASTI Beijing)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun394 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunLispParser.java,v 8.10 2008/02/20 06:31:49 nisinaka Exp $
 */
public class JunLispParser extends JunLispScanner {

	/** Cached values to "unNextToken()". */
	protected int prevMark;
	protected Object prevToken;
	protected StSymbol prevTokenType;
	protected StValueHolder progressValue;
	protected int readLimit;

	/**
	 * Parse the object and generate a list.
	 * 
	 * @return java.lang.Object
	 * @param anObject java.lang.Object
	 * @category Utilities
	 */
	public static Object Parse_(Object anObject) {
		return (new JunLispParser()).parse_(MakeStream_(anObject));
	}

	/**
	 * Parse the object and generate a list.
	 * 
	 * @return java.lang.Object
	 * @param anObject java.lang.Object
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category Utilities
	 */
	public static Object Parse_ifFail_(Object anObject, StBlockClosure aBlock) {
		return (new JunLispParser()).parse_ifFail_(MakeStream_(anObject), aBlock);
	}

	/**
	 * Create a read stream from the object.
	 * 
	 * @return jp.co.sra.smalltalk.StReadStream
	 * @param anObject java.lang.Object
	 * @category private
	 */
	protected static StReadStream MakeStream_(Object anObject) {
		if (anObject instanceof StReadStream) {
			return (StReadStream) anObject;
		} else if (anObject instanceof String) {
			return new StReadStream((String) anObject);
		}

		return new StReadStream(anObject.toString());
	}

	/**
	 * Get the next token from the source stream.
	 * 
	 * @return java.lang.Object
	 * @category scanning
	 */
	public Object nextToken() {
		prevMark = mark;
		prevToken = token;
		prevTokenType = tokenType;
		Object token = super.nextToken();
		this.progress_((float) source.position() / readLimit);
		return token;
	}

	/**
	 * Parse the stream and generate a lisp list.
	 * 
	 * @return java.lang.Object
	 * @param sourceStream jp.co.sra.smalltalk.StReadStream
	 * @category parsing 
	 */
	public Object parse_(StReadStream sourceStream) {
		StBlockClosure failBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				String errorMessage = (String) anObject;
				String label = errorMessage + " near " + token;
				// StString rest = (StString) source.upToEnd();
				String string = new String();
				while (!source.atEnd()) {
					string += source.next();
				}

				if (string.length() == 0) {
					string = "--> end of file";
				} else {
					string = "--> " + string.substring(0, 29) + "...";
				}

				throw SmalltalkException.Error(label + "\n" + string);
			}
		};

		return this.parse_ifFail_(sourceStream, failBlock);
	}

	/**
	 * Parse the stream and generate a lisp list.
	 * 
	 * @return java.lang.Object
	 * @param sourceStream jp.co.sra.smalltalk.StReadStream
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category parsing 
	 */
	public Object parse_ifFail_(StReadStream sourceStream, StBlockClosure aBlock) {
		this.init_ifFail_(sourceStream, aBlock);
		return this.scan();
	}

	/**
	 * Scan the source stream.
	 * 
	 * @return java.lang.Object
	 * @category scan and parse
	 */
	public Object scan() {
		if (source.atEnd()) {
			return JunLispCons.NullList();
		}
		return this.scanList();
	}

	/**
	 * Push back the token to the stream.
	 * 
	 * @category scanning
	 */
	public void unNextToken() {
		super.unNextToken();
		mark = prevMark;
		token = prevToken;
		tokenType = prevTokenType;
	}

	/**
	 * Scan the source stream.
	 * 
	 * @return java.lang.Object
	 * @category scan and parse
	 */
	protected Object scanList() {
		this.nextToken();

		if (tokenType == $("eof")) {
			return JunLispCons.NullList();
		} else if (tokenType == $("number")) {
			return token;
		} else if (tokenType == $("string")) {
			return token;
		} else if (tokenType == $("object")) {
			return token;
		} else if (tokenType == $("symbol")) {
			return token;
		} else if (tokenType == $("quote")) {
			JunLispCons expression = new JunLispCons(this.scanList(), JunLispCons.NullList());
			return new JunLispCons($("quote"), expression);
		} else if (tokenType == $("leftParenthesis")) {
			return this.scanListAux();
		}

		return failBlock.value_("Syntax error");
	}

	/**
	 * Scan the source stream for a sublist.
	 * 
	 * @return java.lang.Object
	 * @category scan and parse
	 */
	protected Object scanListAux() {
		this.nextToken();

		if (tokenType == $("eof")) {
			return JunLispCons.NullList();
		} else if (tokenType == $("rightParenthesis")) {
			return JunLispCons.NullList();
		} else if (tokenType == $("leftParenthesis")) {
			return new JunLispCons(this.scanListAux(), this.scanListAux());
		} else if (tokenType == $("number")) {
			return new JunLispCons(token, this.scanListAux());
		} else if (tokenType == $("string")) {
			return new JunLispCons(token, this.scanListAux());
		} else if (tokenType == $("object")) {
			return new JunLispCons(token, this.scanListAux());
		} else if (tokenType == $("symbol")) {
			return new JunLispCons(token, this.scanListAux());
		} else if (tokenType == $("period")) {
			Object cdr = this.scanList();
			this.nextToken();
			if (tokenType == $("rightParenthesis")) {
				return cdr;
			} else {
				return failBlock.value_("Syntax error");
			}
		} else if (tokenType == $("quote")) {
			JunLispCons cdr = new JunLispCons(this.scanList(), JunLispCons.NullList());
			cdr = new JunLispCons($("quote"), cdr);
			return new JunLispCons(cdr, this.scanListAux());
		}

		this.unNextToken();
		return failBlock.value_("Syntax error");
	}

	/**
	 * Answer the float value of the current progress value.
	 * 
	 * @return float
	 * @category progress
	 */
	public float progress() {
		return this.progressValue()._floatValue();
	}

	/**
	 * Set the float value as my new progress value.
	 * 
	 * @param normalizedNumber float
	 * @category progress
	 */
	public void progress_(float normalizedNumber) {
		if (0 <= normalizedNumber && normalizedNumber <= 1) {
			float truncatedValue = Math.round(normalizedNumber / 0.005) * 0.005f;
			if (this.progress() != truncatedValue) {
				this.progressValue().value_(truncatedValue);
			}
		}
	}

	/**
	 * Answer my progress value.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category progress
	 */
	public StValueHolder progressValue() {
		if (progressValue == null) {
			progressValue = new StValueHolder(0);
		}
		return progressValue;
	}

	/**
	 * Set my new progress value.
	 * 
	 * @param newProgressValue jp.co.sra.smalltalk.StValueHolder
	 * @category progress
	 */
	public void progressValue_(StValueHolder newProgressValue) {
		progressValue = newProgressValue;
	}

	/**
	 * Set the computational block for the progress value.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category progress
	 */
	public void compute_(StBlockClosure aBlock) {
		this.progressValue().compute_(aBlock);
	}

	/**
	 * Initialize to parse the stream.
	 * 
	 * @param sourceStream jp.co.sra.smalltalk.StReadStream
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category private
	 */
	protected void init_ifFail_(StReadStream sourceStream, StBlockClosure aBlock) {
		super.on_(sourceStream);
		failBlock = aBlock;
		int position = sourceStream.position();
		sourceStream.setToEnd();
		readLimit = sourceStream.position();
		sourceStream.position_(position);
	}

}
