package jp.co.sra.jun.goodies.multimedia;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.io.File;
import java.net.MalformedURLException;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;

import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.goodies.display.JunDisplayModel;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.icon.JunOpaqueImageIcons;
import jp.co.sra.jun.goodies.tips.JunURL;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunMultiMediaExtraModel class
 * 
 *  @author    nisinaka
 *  @created   2005/09/07 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun648 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunMultiMediaExtraModel.java,v 8.15 2008/02/20 06:31:50 nisinaka Exp $
 */
public class JunMultiMediaExtraModel extends JunDisplayModel implements JunMultiMediaViewable {

	protected File file;
	protected StPopupMenu _popupMenu;

	/**
	 * Create a new instance of JunMultiMediaExtraModel and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunMultiMediaExtraModel() {
		super();
	}

	/**
	 * Create a new instance of JunMultiMediaExtraModel and initialize it.
	 *
	 * @param aFile java.io.File
	 * @category Instance creation
	 */
	public JunMultiMediaExtraModel(File aFile) {
		this.file_(aFile);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		file = null;
	}

	/**
	 * Answer my current file.
	 * 
	 * @return java.io.File
	 * @category accessing
	 */
	public File file() {
		return file;
	}

	/**
	 * Set my new file.
	 * 
	 * @param aFile java.io.File
	 * @category accessing
	 */
	public void file_(File aFile) {
		file = aFile;

		if (file.exists()) {
			this.visual_(this.iconImage());
		}
		this.updateWindowTitle();
		this.updateMenuIndication();
	}

	/**
	 * Answer the icon image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	protected StImage iconImage() {
		StImage anImage = null;

		String aString = this.file().getName();
		int index = aString.lastIndexOf('.');
		if (index >= 0) {
			aString = aString.substring(index + 1).toLowerCase();
			if (aString.equals("pdf")) {
				anImage = PdfIcon();
			} else if (aString.equals("doc")) {
				anImage = DocIcon();
			} else if (aString.equals("xls")) {
				anImage = XlsIcon();
			} else if (aString.equals("ppt")) {
				anImage = PptIcon();
			} else {
				anImage = Icon(aString);
			}
		} else {
			anImage = OtherIcon();
		}

		aString = this.file().getName();
		aString += '\n';
		aString += FileSizeString_(this.file());
		aString += '\n';
		aString += new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").format(new Date(this.file().lastModified()));

		StComposedText aText = new StComposedText(aString);
		aText.centered();

		int width = Math.max(anImage.width(), aText.width());
		int height = anImage.height() + aText.height();
		StImage iconImage = new StImage(width, height);
		Graphics aGraphics = null;
		try {
			aGraphics = iconImage.image().getGraphics();

			StRectangle imageBounds = new StRectangle(anImage.bounds());
			anImage.displayOn_at_(aGraphics, imageBounds.align_with_(imageBounds.topCenter(), new Point(width / 2, 0)).origin());

			aGraphics.setColor(Color.black);
			StRectangle textBounds = new StRectangle(aText.bounds());
			aText.displayOn_at_(aGraphics, textBounds.align_with_(textBounds.bottomCenter(), new Point(width / 2, height)).origin());

		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}

		return iconImage;
	}

	/**
	 * Create a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunMultiMediaExtraViewAwt(this);
		} else {
			return new JunMultiMediaExtraViewSwing(this);
		}
	}

	/**
	 * Set window label.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#setWindowLabel()
	 * @category interface opening
	 */
	public void setWindowLabel() {
		// no operation
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Extra");
	}

	/**
	 * Answer my popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		if (_popupMenu == null) {
			StPopupMenu aPopupMenu = new StPopupMenu();
			aPopupMenu.add(new StMenuItem($String("Open"), new MenuPerformer(this, "openExtraFile")));
			_popupMenu = aPopupMenu;
		}
		return _popupMenu;
	}

	/**
	 * Open the extra file.
	 * 
	 * @category menu messages
	 */
	public void openExtraFile() {
		try {
			JunURL.Browse_(this.file().toURL());
		} catch (MalformedURLException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Make be invisible.
	 * 
	 * @see jp.co.sra.jun.goodies.multimedia.JunMultiMediaViewable#beInvisible()
	 * @category accessing
	 */
	public void beInvisible() {
		this.beVisible(false);
	}

	/**
	 * Make be visible.
	 * 
	 * @see jp.co.sra.jun.goodies.multimedia.JunMultiMediaViewable#beVisible()
	 * @category accessing
	 */
	public void beVisible() {
		this.beVisible(true);
	}

	/**
	 * Set the visibility.
	 * 
	 * @param beVisible boolean
	 * @category accessing
	 */
	protected void beVisible(boolean beVisible) {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			if (dependents[i] instanceof JunMultiMediaExtraView) {
				((JunMultiMediaExtraView) dependents[i]).toComponent().setVisible(beVisible);
			}
		}
	}

	/**
	 * Answer an icon which shows the specified string.
	 * 
	 * @param aString java.lang.String
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Icons
	 */
	public static StImage Icon(String aString) {
		return Icon(aString, Color.black);
	}

	/**
	 * Answer an icon which shows the specified string with the specified color.
	 * 
	 * @param aString java.lang.String
	 * @param aColor java.awt.Color
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Icons
	 */
	public static StImage Icon(String aString, Color aColor) {
		return Icon(aString, aColor, OtherIcon());
	}

	/**
	 * Answer an icon which shows the specified string with the specified color.
	 * 
	 * @param aString java.lang.String
	 * @param aColor java.awt.Color
	 * @param originalIcon jp.co.sra.smalltalk.StImage
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Icons
	 */
	public static StImage Icon(String aString, Color aColor, StImage originalIcon) {
		StImage anImage = new StImage(originalIcon.bounds().getSize());

		Graphics2D aGraphics = null;
		try {
			aGraphics = (Graphics2D) anImage.image().getGraphics();
			originalIcon.displayOn_(aGraphics);
			aGraphics.setColor(aColor);
			Font aFont = aGraphics.getFont().deriveFont(Font.BOLD);
			StComposedText aText = new StComposedText(aString, aFont);
			int x = (anImage.width() - aText.width()) / 2;
			int y = (anImage.height() - aText.height()) / 2;

			aText.displayOn_at_(aGraphics, new Point(x, y));
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}

		return anImage;
	}

	/**
	 * Answer the icon for others.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Icons
	 */
	public static StImage OtherIcon() {
		return JunOpaqueImageIcons.OtherFigure64x64();
	}

	/**
	 * Answer the icon  for DOC.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Icons
	 */
	public static StImage DocIcon() {
		return Icon("doc", StColorValue.Blend(Color.blue, Color.gray));
	}

	/**
	 * Answer the icon  for PDF.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Icons
	 */
	public static StImage PdfIcon() {
		return Icon("pdf", Color.red);
	}

	/**
	 * Answer the icon  for PPT.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Icons
	 */
	public static StImage PptIcon() {
		return Icon("ppt", StColorValue.Blend(Color.red, Color.gray));
	}

	/**
	 * Answer the icon  for XLS.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Icons
	 */
	public static StImage XlsIcon() {
		return Icon("xls", StColorValue.Blend(Color.green, Color.gray));
	}

	/**
	 * Answer the string about the file size.
	 * 
	 * @param aFile java.io.File
	 * @return java.lang.String
	 * @category Utilities
	 */
	public static String FileSizeString_(File aFile) {
		String aString = "";

		if (aFile != null && aFile.exists()) {
			long size = aFile.length();
			aString = new DecimalFormat("#,###").format(size);
			aString += " bytes ( ";
			aString += new DecimalFormat("0.00").format((double) size / (1024 * 1024));
			aString += " MB )";
		}

		return aString;
	}

	/**
	 * Ask user to choose a file and create JunMultiMediaExtraModel with it.
	 * 
	 * @return jp.co.sra.jun.goodies.multimedia.JunMultiMediaExtraModel
	 * @category Utilities
	 */
	public static JunMultiMediaExtraModel Request() {
		ArrayList aList = new ArrayList();
		aList.addAll(Arrays.asList(JunSystem.DefaultTextExtensionPatterns()));
		aList.addAll(Arrays.asList(JunSystem.DefaultImageExtensionPatterns()));
		aList.addAll(Arrays.asList(JunSystem.DefaultMovieExtensionPatterns()));
		aList.addAll(Arrays.asList(JunSystem.DefaultSoundExtensionPatterns()));
		aList.addAll(Arrays.asList(JunSystem.DefaultExtraExtensionPatterns()));
		String[] extensionPatterns = (String[]) aList.toArray(new String[aList.size()]);

		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("Multi-media files"), extensionPatterns), JunFileModel.FileType.All($String("All files")) };
		File aFile = JunFileRequesterDialog.Request($String("Select a file."), fileTypes, fileTypes[0]);
		if (aFile == null) {
			return null;
		}

		return new JunMultiMediaExtraModel(aFile);
	}

}
