package jp.co.sra.jun.goodies.spirodesign;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Collection;

import jp.co.sra.smalltalk.DependentListener;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StInputState;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.StMenu;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.image.framework.JunImageDisplayModel;
import jp.co.sra.jun.system.framework.JunApplicationModel;

/**
 * JunSpiroDesignAbstractModel class
 * 
 *  @author    m-asada
 *  @created   2006/03/31 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun676 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSpiroDesignAbstractModel.java,v 8.10 2008/02/20 06:32:03 nisinaka Exp $
 */
public abstract class JunSpiroDesignAbstractModel extends JunApplicationModel {
	/**
	 * Answer the receiver's moon circle.
	 * 
	 * @return jp.co.sra.jun.goodies.spirodesign.JunSpiroCircle
	 * @category accessing
	 */
	public abstract JunSpiroCircle moonCircle();

	/**
	 * Answer the receiver's spiro pen.
	 * 
	 * @return jp.co.sra.jun.goodies.spirodesign.JunSpiroPen
	 * @category accessing
	 */
	public abstract JunSpiroPen spiroPen();

	/**
	 * Answer the receiver's tera circle.
	 * 
	 * @return jp.co.sra.jun.goodies.spirodesign.JunSpiroCircle
	 * @category accessing
	 */
	public abstract JunSpiroCircle teraCircle();

	/**
	 * Answer the bounding box of the receiver.
	 *
	 * @return java.awt.Rectangle
	 * @category bounds accessing
	 */
	public abstract Rectangle bounds();

	/**
	 * Convert the view to an image as StImage.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		return this.asImageExtent_(new Dimension(this.bounds().getSize().width + this.defaultMarginExtent().width, this.bounds().getSize().height + this.defaultMarginExtent().height));
	}

	/**
	 * Convert the view as an image with the specified extent size.
	 * 
	 * @param extent java.awt.Dimension
	 * @return jp.co.sra.smalltalk.StImage
	 * @category converting
	 */
	public StImage asImageExtent_(Dimension extent) {
		return this.asImageExtent_withParts_(extent, false);
	}

	/**
	 * Convert the view as an image with the specified extent size and parts.
	 * 
	 * @param extent java.awt.Dimension
	 * @param aBoolean boolean
	 * @return jp.co.sra.smalltalk.StImage
	 * @category converting
	 */
	public StImage asImageExtent_withParts_(Dimension extent, boolean aBoolean) {
		StImage anImage = new StImage(extent.width, extent.height);
		Graphics aGraphics = anImage.image().getGraphics();
		try {
			aGraphics.setColor(Color.white);
			aGraphics.fillRect(0, 0, extent.width, extent.height);
			this.displayOn_at_withParts_(aGraphics, new Point((int) Math.round(anImage.width() / 2.0d - this.bounds().getCenterX()), (int) Math.round(anImage.height() / 2.0d - this.bounds().getCenterY())), aBoolean);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
		return anImage;
	}

	/**
	 * Convert the view as an image with parts.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category converting
	 */
	public StImage asImageWithParts() {
		return this.asImageExtent_withParts_(new Dimension(this.bounds().getSize().width + this.defaultMarginExtent().width, this.bounds().getSize().height + this.defaultMarginExtent().height), true);
	}

	/**
	 * Answer the default ,argin extent.
	 * 
	 * @return java.awt.Dimension
	 * @category defaults
	 */
	public Dimension defaultMarginExtent() {
		return new Dimension(16, 16);
	}

	/**
	 * Answer the receiver's default view as JunSpiroDesignView.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunSpiroDesignViewAwt(this);
		} else {
			return new JunSpiroDesignViewSwing(this);
		}
	}

	/**
	 * Display the receiver on the graphics.
	 *
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	public void displayOn_(Graphics graphicsContext) {
		StView aView = this.getView();
		if (aView == null) {
			this.displayOn_at_(graphicsContext, new Point(0, 0));
		} else {
			this.displayOn_at_(graphicsContext, ((JunSpiroDesignView) aView).scrollAmount());
		}
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 *
	 * @param graphicsContext java.awt.Graphics
	 * @param displayPoint java.awt.Point
	 * @category displaying
	 */
	public abstract void displayOn_at_(Graphics graphicsContext, Point displayPoint);

	/**
	 * Display the receiver on the graphics at the specified point.
	 *
	 * @param graphicsContext java.awt.Graphics
	 * @param displayPoint java.awt.Point
	 * @param aBoolean boolean
	 * @category displaying
	 */
	public abstract void displayOn_at_withParts_(Graphics graphicsContext, Point displayPoint, boolean aBoolean);

	/**
	 * Answer the receiver's window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Spiro Design");
	}

	/**
	 * Update the menu indication.
	 * 
	 * @param aMenu jp.co.sra.smalltalk.menu.StMenu
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category menu accessing
	 */
	public abstract StMenu updateMenuIndication_(StMenu aMenu);

	/**
	 * Spawn the receiver's image.
	 * 
	 * @category menu messages
	 */
	public void spawnAsImage() {
		StImage image = StInputState.Default().altDown() ? this.asImageWithParts() : this.asImage();
		StView view = this.getView();
		if (this.getView() == null) {
			JunImageDisplayModel.Show_(image);
		} else {
			Rectangle box = view.topComponent().getBounds();
			Rectangle area = new Rectangle(0, 0, image.width(), image.height());
			area.translate(box.x - (area.x + area.width) - 12, box.y - area.y);
			JunImageDisplayModel.Show_at_(image, area.getLocation());
		}
	}

	/**
	 * Select the spiro design and answer it.
	 * 
	 * @param spiroDesign jp.co.sra.jun.goodies.spirodesign.JunSpiroDesign
	 * @return jp.co.sra.jun.goodies.spirodesign.JunSpiroDesign
	 * @category selecting
	 */
	public JunSpiroDesign selectSpiroDesign_(JunSpiroDesign spiroDesign) {
		return null;
	}

	/**
	 * Which the spiro design with the specified point and answer it.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.goodies.spirodesign.JunSpiroDesign
	 * @category selecting
	 */
	public JunSpiroDesign whichSpiroDesign_(Point aPoint) {
		return null;
	}

	/**
	 * Set the receiver's center of moon circle with the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category setting
	 */
	public abstract void centerOfMoonCircle_(Jun2dPoint aPoint);

	/**
	 * Set the receiver's center of tera circle with the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category setting
	 */
	public abstract void centerOfTeraCircle_(Jun2dPoint aPoint);

	/**
	 * Set the receiver's point of spiro pen with the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category setting
	 */
	public abstract void pointOfSpiroPen_(Jun2dPoint aPoint);

	/**
	 * Set the receiver's radius of moon circle with the specified number.
	 * 
	 * @param aNumber double
	 * @category setting
	 */
	public abstract void radiusOfMoonCircle_(double aNumber);

	/**
	 * Set the receiver's radius of tera circle with the specified number.
	 * 
	 * @param aNumber double
	 * @category setting
	 */
	public abstract void radiusOfTeraCircle_(double aNumber);

	/**
	 * Answer true if the receiver is active spiro design process, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isActiveSpiroDesignProcess() {
		return false;
	}

	/**
	 * Answer the receiver's view as StView.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#getView()
	 * @category private
	 */
	public StView getView() {
		DependentListener[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			Object each = dependents[i];
			if (each instanceof JunSpiroDesignView && ((JunSpiroDesignView) each).model() == this) {
				return (JunSpiroDesignView) each;
			}
		}
		return null;
	}

	/**
	 * Answer the receiver's views as array of JunSpiroDesignView.
	 * 
	 * @return jp.co.sra.jun.goodies.spirodesign.JunSpiroDesignView[]
	 * @category private
	 */
	protected JunSpiroDesignView[] getSpiroDesignViews() {
		Collection viewCollection = new ArrayList();
		DependentListener[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			Object each = dependents[i];
			if (each instanceof JunSpiroDesignView && ((JunSpiroDesignView) each).model() == this) {
				viewCollection.add(each);
			}
		}
		return (JunSpiroDesignView[]) viewCollection.toArray(new JunSpiroDesignView[viewCollection.size()]);
	}
}
